import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// MCP 스키마 테스트 클라이언트
class MCPSchemaTestClient {
  constructor() {
    this.messageId = 1;
    this.serverProcess = null;
  }

  async startServer() {
    const serverPath = path.join(__dirname, '../dist/index.js');
    this.serverProcess = spawn('node', [serverPath], {
      stdio: ['pipe', 'pipe', 'pipe'],
      cwd: path.dirname(serverPath)
    });

    this.serverProcess.stderr.on('data', (data) => {
      console.error(`서버 로그: ${data}`);
    });

    // 서버 시작 대기
    await new Promise(resolve => setTimeout(resolve, 2000));
    console.log('✅ MCP 서버 시작됨');
  }

  async sendRequest(method, params = {}) {
    const request = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method,
      params
    };

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error('요청 타임아웃'));
      }, 10000);

      let buffer = '';
      const onData = (data) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                this.serverProcess.stdout.off('data', onData);
                
                if (response.error) {
                  reject(new Error(response.error.message));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류 무시
            }
          }
        }
      };

      this.serverProcess.stdout.on('data', onData);
      this.serverProcess.stdin.write(JSON.stringify(request) + '\n');
    });
  }

  async testToolSchemas() {
    console.log('\n🔍 MCP 도구 스키마 유효성 테스트');
    
    try {
      // 1. 도구 목록 가져오기
      const toolsList = await this.sendRequest('tools/list');
      console.log(`📋 총 ${toolsList.tools.length}개의 도구 발견`);
      
      // 2. 각 도구의 스키마 검증
      let validSchemas = 0;
      let invalidSchemas = 0;
      
      for (const tool of toolsList.tools) {
        console.log(`\n🔧 도구: ${tool.name}`);
        console.log(`   설명: ${tool.description}`);
        
        // 스키마 유효성 검증
        const schema = tool.inputSchema;
        if (schema && schema.type === 'object') {
          console.log(`   ✅ 스키마 유효함`);
          
          // 필수 파라미터 확인
          if (schema.required && schema.required.length > 0) {
            console.log(`   📝 필수 파라미터: ${schema.required.join(', ')}`);
          } else {
            console.log(`   📝 필수 파라미터: 없음`);
          }
          
          // 선택적 파라미터 확인
          if (schema.properties) {
            const allParams = Object.keys(schema.properties);
            const optionalParams = allParams.filter(p => !schema.required?.includes(p));
            if (optionalParams.length > 0) {
              console.log(`   📝 선택적 파라미터: ${optionalParams.join(', ')}`);
            }
          }
          
          validSchemas++;
        } else {
          console.log(`   ❌ 스키마 무효함`);
          invalidSchemas++;
        }
      }
      
      console.log(`\n📊 스키마 유효성 결과:`);
      console.log(`✅ 유효한 스키마: ${validSchemas}`);
      console.log(`❌ 무효한 스키마: ${invalidSchemas}`);
      
      return { validSchemas, invalidSchemas, totalTools: toolsList.tools.length };
      
    } catch (error) {
      console.error('스키마 테스트 실패:', error);
      return { validSchemas: 0, invalidSchemas: 0, totalTools: 0 };
    }
  }

  async testInvalidParameterHandling() {
    console.log('\n🚨 잘못된 파라미터 처리 테스트');
    
    // 필수 파라미터 누락 테스트
    const testCases = [
      {
        name: 'translate_text',
        description: '필수 파라미터 text 누락',
        args: { target_lang: 'KO' }, // text 누락
        expectError: true
      },
      {
        name: 'web_search',
        description: '필수 파라미터 query 누락',
        args: {}, // query 누락
        expectError: true
      },
      {
        name: 'file_operation',
        description: '필수 파라미터 operation, path 누락',
        args: { content: 'test' }, // operation, path 누락
        expectError: true
      }
    ];
    
    let passedTests = 0;
    let failedTests = 0;
    
    for (const testCase of testCases) {
      console.log(`\n🧪 테스트: ${testCase.name} - ${testCase.description}`);
      
             try {
         const result = await this.sendRequest('tools/call', {
           name: testCase.name,
           arguments: testCase.args
         });
         
         // isError 필드가 있으면 오류로 간주
         const isActualError = result.isError === true;
         
         if (testCase.expectError) {
           if (isActualError) {
             console.log(`   ✅ 예상대로 오류 발생: ${result.content?.[0]?.text || 'Unknown error'}`);
             passedTests++;
           } else {
             console.log(`   ❌ 예상과 다름: 오류가 발생해야 하는데 성공함`);
             failedTests++;
           }
         } else {
           if (!isActualError) {
             console.log(`   ✅ 예상대로 성공`);
             passedTests++;
           } else {
             console.log(`   ❌ 예상과 다름: 성공해야 하는데 오류 발생: ${result.content?.[0]?.text || 'Unknown error'}`);
             failedTests++;
           }
         }
        
      } catch (error) {
        if (testCase.expectError) {
          console.log(`   ✅ 예상대로 오류 발생: ${error.message}`);
          passedTests++;
        } else {
          console.log(`   ❌ 예상과 다름: 성공해야 하는데 오류 발생: ${error.message}`);
          failedTests++;
        }
      }
    }
    
    console.log(`\n📊 파라미터 유효성 검증 결과:`);
    console.log(`✅ 통과: ${passedTests}`);
    console.log(`❌ 실패: ${failedTests}`);
    
    return { passedTests, failedTests };
  }

  async cleanup() {
    if (this.serverProcess) {
      this.serverProcess.kill();
      console.log('\n🔚 MCP 서버 종료됨');
    }
  }
}

// 테스트 실행
async function runSchemaTests() {
  const client = new MCPSchemaTestClient();
  
  try {
    await client.startServer();
    
    const schemaResults = await client.testToolSchemas();
    const paramResults = await client.testInvalidParameterHandling();
    
    console.log('\n🎯 전체 테스트 결과:');
    console.log(`📋 총 도구 수: ${schemaResults.totalTools}`);
    console.log(`✅ 유효한 스키마: ${schemaResults.validSchemas}`);
    console.log(`❌ 무효한 스키마: ${schemaResults.invalidSchemas}`);
    console.log(`✅ 파라미터 검증 통과: ${paramResults.passedTests}`);
    console.log(`❌ 파라미터 검증 실패: ${paramResults.failedTests}`);
    
    const isCompliant = schemaResults.invalidSchemas === 0 && paramResults.failedTests === 0;
    console.log(`\n🏆 MCP 규칙 준수 여부: ${isCompliant ? '✅ 완벽 준수' : '❌ 개선 필요'}`);
    
  } catch (error) {
    console.error('테스트 실행 중 오류:', error);
  } finally {
    await client.cleanup();
  }
}

// 테스트 실행
runSchemaTests().catch(console.error); 