import { NextRequest, NextResponse } from 'next/server';
import * as XLSX from 'xlsx';
import { writeFile, unlink } from 'fs/promises';
import { join } from 'path';
import { tmpdir } from 'os';
import { v4 as uuidv4 } from 'uuid';

// 행 필터 연산자 설명 함수
function getOperatorDescription(operator: string): string {
  switch (operator) {
    case 'equals': return '같음';
    case 'contains': return '포함';
    case 'greaterThan': return '보다 큼';
    case 'lessThan': return '보다 작음';
    default: return operator;
  }
}

// 열 설명 생성 함수
function getColumnDescription(header: string, data: any[][], columnIndex: number): string {
  if (data.length <= 1) return '데이터 없음';
  
  // 첫 5개 데이터 샘플링해서 타입 추정
  const samples = data.slice(1, 6).map(row => row[columnIndex - 1]);
  const nonEmptySamples = samples.filter(val => val !== null && val !== undefined && val !== '');
  
  if (nonEmptySamples.length === 0) return '빈 값';
  
  // 데이터 타입 추정
  const hasNumbers = nonEmptySamples.some(val => !isNaN(Number(val)) && val !== '');
  const hasDates = nonEmptySamples.some(val => {
    const dateStr = String(val);
    return dateStr.includes('-') || dateStr.includes('/') || dateStr.includes('년') || dateStr.includes('월');
  });
  const hasEmails = nonEmptySamples.some(val => String(val).includes('@'));
  
  let description = '';
  if (hasEmails) {
    description = '이메일 주소';
  } else if (hasDates) {
    description = '날짜/시간 정보';
  } else if (hasNumbers) {
    description = '숫자 데이터';
  } else {
    description = '텍스트 데이터';
  }
  
  // 샘플 값 추가
  const sampleValue = String(nonEmptySamples[0]);
  description += ` (예: ${sampleValue.length > 20 ? sampleValue.substring(0, 20) + '...' : sampleValue})`;
  
  return description;
}

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const options = JSON.parse(formData.get('options') as string);

    if (!file) {
      return NextResponse.json(
        { success: false, error: 'No file provided' },
        { status: 400 }
      );
    }

    // 파일을 임시 디렉토리에 저장
    const buffer = Buffer.from(await file.arrayBuffer());
    const tempFilePath = join(tmpdir(), `excel-${uuidv4()}.xlsx`);
    await writeFile(tempFilePath, buffer);

    try {
      // Excel 파일 읽기
      const workbook = XLSX.read(buffer, { type: 'buffer' });
      
      // 시트 선택
      const worksheet = workbook.Sheets[options.selectedSheet];
      if (!worksheet) {
        throw new Error(`Sheet ${options.selectedSheet} not found`);
      }

      // JSON으로 변환
      let data = XLSX.utils.sheet_to_json(worksheet, { 
        header: 1,
        defval: '',
        raw: !options.preserveFormatting 
      });

      // 헤더 정규화
      if (options.headerNormalization && data.length > 0) {
        const headers = data[0] as string[];
        data[0] = headers.map(header => 
          String(header)
            .trim()
            .replace(/\s+/g, ' ')
            .replace(/[\n\r]/g, '')
            .replace(/[^\w\s가-힣]/g, '')
        );
      }

      // 열 선택
      if (options.selectedColumns && options.selectedColumns.length > 0) {
        const headers = data[0] as string[];
        const columnIndices = options.selectedColumns.map((col: string) => 
          headers.indexOf(col)
        ).filter((idx: number) => idx !== -1);

        data = data.map((row: any) => 
          columnIndices.map((idx: number) => row[idx])
        );
      }

      // 행 필터링
      if (options.rowFilter && options.rowFilter.enabled) {
        const { column, operator, value } = options.rowFilter;
        const headers = data[0] as string[];
        const columnIndex = headers.indexOf(column);

        if (columnIndex !== -1) {
          const headerRow = data[0];
          const filteredData = [headerRow];

          for (let i = 1; i < data.length; i++) {
            const cellValue = String(data[i][columnIndex]);
            let shouldInclude = false;

            switch (operator) {
              case 'equals':
                shouldInclude = cellValue === value;
                break;
              case 'contains':
                shouldInclude = cellValue.includes(value);
                break;
              case 'greaterThan':
                shouldInclude = parseFloat(cellValue) > parseFloat(value);
                break;
              case 'lessThan':
                shouldInclude = parseFloat(cellValue) < parseFloat(value);
                break;
            }

            if (shouldInclude) {
              filteredData.push(data[i]);
            }
          }

          data = filteredData;
        }
      }

      // 데이터 분할 처리
      const shouldSplit = options.dataSplitting?.enabled && options.dataSplitting.maxRowsPerFile > 0;
      const maxRowsPerFile = shouldSplit ? options.dataSplitting.maxRowsPerFile : data.length - 1;
      const preserveHeaders = shouldSplit ? options.dataSplitting.preserveHeaders : true;
      
      // 분할된 데이터 청크들 생성
      const dataChunks: any[][][] = [];
      if (shouldSplit && data.length > 1) {
        const headers = data[0];
        const dataRows = data.slice(1);
        
        // 데이터를 지정된 행 수로 분할
        for (let i = 0; i < dataRows.length; i += maxRowsPerFile) {
          const chunkRows = dataRows.slice(i, i + maxRowsPerFile);
          const chunk = preserveHeaders ? [headers, ...chunkRows] : chunkRows;
          dataChunks.push(chunk);
        }
      } else {
        // 분할하지 않는 경우 전체 데이터를 하나의 청크로
        dataChunks.push(data);
      }

      // 처리된 파일들 저장
      const processedFiles: Array<{
        name: string;
        content: string;
        type: string;
      }> = [];

      // 각 청크에 대해 Markdown 문서 생성
      for (let chunkIndex = 0; chunkIndex < dataChunks.length; chunkIndex++) {
        const chunkData = dataChunks[chunkIndex];
        
        // Markdown 문서 생성 (제목과 설명 포함)
        let markdown = '';
        if (chunkData.length > 0) {
          // 문서 메타데이터
          const originalFileName = file.name.replace(/\.[^/.]+$/, '');
          const currentDate = new Date().toLocaleDateString('ko-KR');
          const totalRows = chunkData.length - (preserveHeaders ? 1 : 0);
          const selectedColumnsCount = options.selectedColumns?.length || chunkData[0]?.length || 0;
        
        // 제목 생성 (AI 모드인 경우 AI가 생성한 제목 사용)
        let documentTitle = originalFileName;
        if (options.mode === 'ai' && options.aiRecommendations?.aiGeneratedTitle) {
          documentTitle = options.aiRecommendations.aiGeneratedTitle;
        }
        
        // 분할된 파일의 경우 파트 번호 추가
        if (shouldSplit && dataChunks.length > 1) {
          documentTitle += ` (Part ${chunkIndex + 1}/${dataChunks.length})`;
        }
        
        markdown += `# ${documentTitle}

`;
        
        // 데이터 설명 추가 (AI 자동 생성 또는 사용자 입력)
        if (options.dataDescription && options.dataDescription.trim()) {
          markdown += `## 📝 데이터 설명

`;
          markdown += `${options.dataDescription.trim()}

`;
          
          // AI 모드인 경우 추가 분석 정보 표시
          if (options.mode === 'ai' && options.aiRecommendations?.dataCategory) {
            markdown += `**데이터 분류**: ${options.aiRecommendations.dataCategory}

`;
          }
          
          // AI 핵심 인사이트 추가
          if (options.mode === 'ai' && options.aiRecommendations?.keyInsights && options.aiRecommendations.keyInsights.length > 0) {
            markdown += `**주요 인사이트**:

`;
            options.aiRecommendations.keyInsights.forEach((insight: string) => {
              markdown += `- ${insight}

`;
            });
          }
        }
        
        markdown += `## 📋 문서 정보

`;
        markdown += `- **파일명**: ${file.name}
`;
        markdown += `- **시트명**: ${options.selectedSheet}
`;
        markdown += `- **전처리 일시**: ${currentDate}
`;
        markdown += `- **총 데이터 행수**: ${totalRows}행
`;
        markdown += `- **선택된 열수**: ${selectedColumnsCount}개

`;
        
        // 전처리 옵션 정보는 RAG에 불필요하므로 제거
        
        // AI 추천 정보
        if (options.mode === 'ai' && options.aiRecommendations) {
          markdown += `## 🤖 AI 분석 결과

`;
          
          // 선택된 시트 및 이유
          if (options.aiRecommendations.selectedSheet) {
            markdown += `**선택된 시트**: ${options.aiRecommendations.selectedSheet}
`;
            markdown += `**선택 이유**: 데이터 품질과 완성도가 가장 높음 (신뢰도: ${Math.round((options.aiRecommendations.confidence || 0) * 100)}%)

`;
          }
          
          // 추천된 열 정보
          if (options.aiRecommendations.columns && options.aiRecommendations.columns.length > 0) {
            markdown += `**AI 추천 열**: ${options.aiRecommendations.columns.length}개
`;
            markdown += `- ${options.aiRecommendations.columns.join(', ')}

`;
          }
          
          // 추천 설명
          if (options.aiRecommendations.description) {
            markdown += `**분석 결과**: ${options.aiRecommendations.description}

`;
          }
          
          // 데이터 특성 분석
          if (options.aiRecommendations.processing?.suggestedFilters) {
            const filters = options.aiRecommendations.processing.suggestedFilters;
            if (filters.length > 0) {
              markdown += `**데이터 특성 분석**:
`;
              filters.forEach((filter: any) => {
                markdown += `- ${filter.description} (${filter.column})
`;
              });
              markdown += '\n';
            }
          }
        }
        
          // 분할 정보 추가 (분할된 경우에만)
          if (shouldSplit && dataChunks.length > 1) {
            markdown += `## 📄 분할 정보
            
`;
            markdown += `- **전체 파일 수**: ${dataChunks.length}개
`;
            markdown += `- **현재 파일**: Part ${chunkIndex + 1}
`;
            markdown += `- **파일당 최대 행 수**: ${maxRowsPerFile}행
`;
            markdown += `- **원본 총 행 수**: ${data.length - 1}행

`;
          }
          
          // 데이터 설명
          markdown += `## 📊 데이터 구조\n\n`;
          const headers = chunkData[0] as any[];
          markdown += `이 테이블은 **${totalRows}개의 데이터 행**과 **${headers.length}개의 열**로 구성되어 있습니다:\n\n`;
          headers.forEach((header, index) => {
            markdown += `- **${header}**: ${getColumnDescription(header, data, index + 1)}\n`;
          });
          markdown += '\n';
          
          // 테이블 시작
          markdown += `## 📈 데이터 테이블\n\n`;
          markdown += `다음은 전처리된 데이터 테이블입니다:\n\n`;
          
          // 헤더
          markdown += '| ' + headers.join(' | ') + ' |\n';
          markdown += '| ' + headers.map(() => '---').join(' | ') + ' |\n';

          // 데이터 행
          for (let i = 1; i < chunkData.length; i++) {
            const row = chunkData[i] as any[];
            const formattedRow = row.map((cell: any) => {
              if (cell === null || cell === undefined) return '';
              
              // 셀 내 개행 처리
              let cellStr = String(cell);
              if (cellStr.includes('\n')) {
                cellStr = cellStr.replace(/\n/g, '<br>');
              }
              
              // 특수 문자 이스케이프
              cellStr = cellStr
                .replace(/\|/g, '\\|')
                .replace(/\*/g, '\\*')
                .replace(/_/g, '\\_');
              
              return cellStr;
            });
            
            markdown += '| ' + formattedRow.join(' | ') + ' |\n';
          }

          // 푸터 정보
          markdown += `\n---\n`;
          markdown += `*이 문서는 Excel 파일 "${file.name}"에서 자동으로 생성되었습니다.*  \n`;
          if (shouldSplit && dataChunks.length > 1) {
            markdown += `*Part ${chunkIndex + 1}/${dataChunks.length} - 전처리 완료: ${currentDate}*\n`;
          } else {
            markdown += `*전처리 완료: ${currentDate}*\n`;
          }
        }
        
        // 처리된 파일 정보 저장
        const processedFileName = (() => {
          // 유니크한 파일명 생성
          const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace('T', '_').slice(0, 19);
          const modePrefix = options.mode === 'ai' ? 'ai' : 'manual';
          const sheetPrefix = options.selectedSheet ? `_${options.selectedSheet.replace(/[^a-zA-Z0-9가-힣]/g, '')}` : '';
          const columnsCount = options.selectedColumns?.length || 0;
          const partSuffix = shouldSplit && dataChunks.length > 1 ? `_part${chunkIndex + 1}of${dataChunks.length}` : '';
          
          return `${file.name.replace(/\.[^/.]+$/, '')}_${modePrefix}${sheetPrefix}_${columnsCount}cols${partSuffix}_${timestamp}.md`;
        })();
        
        processedFiles.push({
          name: processedFileName,
          content: Buffer.from(markdown, 'utf-8').toString('base64'),
          type: 'text/markdown'
        });
      }

      // 임시 파일 삭제
      await unlink(tempFilePath);

      // 응답 데이터 구성
      const responseData = {
        success: true,
        splitEnabled: shouldSplit,
        fileCount: processedFiles.length,
        totalRowCount: data.length - 1,
        columnCount: data[0]?.length || 0,
        processedFiles: processedFiles,
        // 하위 호환성을 위해 단일 파일 정보도 포함 (첫 번째 파일)
        ...(processedFiles.length > 0 && {
          markdown: Buffer.from(processedFiles[0].content, 'base64').toString('utf-8'),
          fileName: processedFiles[0].name,
          rowCount: shouldSplit && processedFiles.length > 1 ? maxRowsPerFile : data.length - 1,
          processedFile: processedFiles[0]
        })
      };

      return NextResponse.json({
        success: true,
        data: responseData
      });

    } finally {
      // 임시 파일이 남아있으면 삭제
      try {
        await unlink(tempFilePath);
      } catch (e) {
        // 이미 삭제되었으면 무시
      }
    }

  } catch (error) {
    console.error('Excel preprocessing error:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Processing failed' 
      },
      { status: 500 }
    );
  }
}