import { NextRequest, NextResponse } from 'next/server';
import * as fs from 'fs';
import * as path from 'path';

// 로컬 세션 디렉토리 경로
const getSessionsDir = () => {
  return path.join(process.cwd(), '.hamonize', 'sessions');
};

// 로컬 세션 파일들에서 모든 assistant 세션 로드
async function loadAllSessionsFromLocal() {
  const sessionsDir = getSessionsDir();
  
  if (!fs.existsSync(sessionsDir)) {
    return {};
  }
  
  const sessions: Record<string, any> = {};
  
  try {
    const files = fs.readdirSync(sessionsDir);
    
    for (const file of files) {
      if (file.endsWith('.json') && file.startsWith('assistant-')) {
        try {
          const sessionFile = path.join(sessionsDir, file);
          const sessionData = JSON.parse(fs.readFileSync(sessionFile, 'utf8'));
          
          // assistant- 접두어가 있는 세션만 포함
          if (sessionData.id && sessionData.id.startsWith('assistant-')) {
            sessions[sessionData.id] = sessionData;
          }
        } catch (error) {
          console.error(`세션 파일 로드 실패: ${file}`, error);
        }
      }
    }
  } catch (error) {
    console.error('세션 디렉토리 읽기 실패:', error);
  }
  
  return sessions;
}

// GET: 모든 세션 목록 조회 (로컬 처리만)
export async function GET(request: NextRequest) {
  try {
    // 로컬 세션 목록만 로드 (외부 API 서버 호출 안 함)
    const localSessions = await loadAllSessionsFromLocal();
    
    console.log(`📋 로컬에서 ${Object.keys(localSessions).length}개 세션 로드`);
    
    return NextResponse.json({
      success: true,
      data: localSessions
    });
  } catch (error) {
    console.error('세션 목록 조회 실패:', error);
    return NextResponse.json(
      { error: 'Failed to get sessions' },
      { status: 500 }
    );
  }
} 