'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { getBusinessServerUrl } from '@/config/serverConfig';
import { 
  Search,
  Filter,
  Calendar,
  DollarSign,
  MapPin,
  Tag,
  Eye,
  ArrowRight,
  RefreshCw,
  ChevronDown,
  Building2,
  Clock,
  Users,
  FileText,
  X,
  Grid3X3,
  List
} from 'lucide-react';

interface Announcement {
  id: string;
  title: string;
  description: string;
  category: string;
  source: string;
  status: string;
  application_end: string | null;
  budget: number;
  support_amount: number;
  eligibility: {
    company_size: string[];
    industry: string[];
    location: string[];
  };
  contact: {
    department: string;
    name: string;
    phone: string | null;
    email: string | null;
  };
  view_count: number;
  created_at: string;
}

interface AnnouncementsResponse {
  announcements: Announcement[];
  total: number;
  limit: number;
  offset: number;
  filters: {
    category: string | null;
    source: string | null;
    status: string;
  };
}

export default function AnnouncementsPage() {
  const router = useRouter();
  const [announcements, setAnnouncements] = useState<Announcement[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('');
  const [selectedStatus, setSelectedStatus] = useState('active');
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [total, setTotal] = useState(0);
  const [isFilterOpen, setIsFilterOpen] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [viewMode, setViewMode] = useState<'card' | 'list'>('card');

  const BIZ_API_BASE = getBusinessServerUrl();
  const ITEMS_PER_PAGE = 20;

  // 카테고리 옵션
  const categories = ['창업', '수출', '금융', '인력', '경영'];
  const statusOptions = [
    { value: 'active', label: '활성' },
    { value: 'all', label: '전체' }
  ];

  // 데이터 로드
  const loadAnnouncements = async (page: number = 1, category: string = '', status: string = 'active') => {
    try {
      setLoading(true);
      const offset = (page - 1) * ITEMS_PER_PAGE;
      
      let url = `${BIZ_API_BASE}/announcements?limit=${ITEMS_PER_PAGE}&offset=${offset}&status=${status}`;
      if (category) {
        url += `&category=${encodeURIComponent(category)}`;
      }

      const response = await fetch(url);
      if (response.ok) {
        const data: AnnouncementsResponse = await response.json();
        setAnnouncements(data.announcements || []);
        setTotal(data.total);
        setTotalPages(Math.ceil(data.total / ITEMS_PER_PAGE));
      }
    } catch (error) {
      console.error('공고 목록 로드 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  // 검색 실행
  const handleSearch = () => {
    setCurrentPage(1);
    loadAnnouncements(1, selectedCategory, selectedStatus);
  };

  // 필터 변경
  const handleFilterChange = (category: string, status: string) => {
    setSelectedCategory(category);
    setSelectedStatus(status);
    setCurrentPage(1);
    loadAnnouncements(1, category, status);
    setIsFilterOpen(false);
  };

  // 페이지 변경
  const handlePageChange = (page: number) => {
    setCurrentPage(page);
    loadAnnouncements(page, selectedCategory, selectedStatus);
  };

  // 데이터 새로고침
  const refreshData = async () => {
    setIsRefreshing(true);
    await loadAnnouncements(currentPage, selectedCategory, selectedStatus);
    setIsRefreshing(false);
  };

  // 컴포넌트 마운트시 데이터 로드
  useEffect(() => {
    loadAnnouncements();
  }, []);

  // 가격 포맷팅
  const formatPrice = (amount: number | null | undefined) => {
    if (!amount || amount === 0) {
      return '미정';
    }
    if (amount >= 100000000) {
      return `${(amount / 100000000).toFixed(1)}억원`;
    } else if (amount >= 10000) {
      return `${(amount / 10000).toFixed(0)}만원`;
    } else {
      return `${amount.toLocaleString()}원`;
    }
  };

  // 날짜 포맷팅
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('ko-KR');
  };

  // 카테고리별 색상
  const getCategoryColor = (category: string) => {
    const colors: { [key: string]: string } = {
      '창업': 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-300',
      '수출': 'bg-blue-100 text-blue-800 dark:bg-blue-900/20 dark:text-blue-300',
      '금융': 'bg-purple-100 text-purple-800 dark:bg-purple-900/20 dark:text-purple-300',
      '인력': 'bg-orange-100 text-orange-800 dark:bg-orange-900/20 dark:text-orange-300',
      '경영': 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900/20 dark:text-indigo-300',
      '기타': 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-300'
    };
    return colors[category] || colors['기타'];
  };


  // 마감일 표시 색상
  const getDeadlineColor = (dateString: string | null) => {
    if (!dateString) return 'text-gray-500';
    
    const deadline = new Date(dateString);
    const now = new Date();
    const diffDays = Math.ceil((deadline.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
    
    if (diffDays < 0) return 'text-red-600'; // 마감
    if (diffDays <= 7) return 'text-orange-600'; // 7일 이하
    if (diffDays <= 30) return 'text-yellow-600'; // 30일 이하
    return 'text-green-600'; // 30일 초과
  };

  return (
    <div className="p-6">
        {/* 상단 필터 버튼 */}
        <div className="flex justify-end mb-6">
            <button
            onClick={refreshData}
            disabled={isRefreshing}
            className="mt-4 md:mt-0 flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors"
          >
            <RefreshCw className={`w-4 h-4 ${isRefreshing ? 'animate-spin' : ''}`} />
            새로고침
          </button>
        </div>

        {/* 검색 및 필터 영역 */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6 mb-8">
          <div className="flex flex-col lg:flex-row gap-4">
            {/* 검색 입력 */}
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                <input
                  type="text"
                  placeholder="공고 제목이나 내용을 검색하세요..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                  className="w-full pl-10 pr-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                />
              </div>
            </div>

            {/* 필터 버튼 */}
            <div className="relative">
              <button
                onClick={() => setIsFilterOpen(!isFilterOpen)}
                className="flex items-center gap-2 px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                <Filter className="w-5 h-5" />
                필터
                <ChevronDown className={`w-4 h-4 transition-transform ${isFilterOpen ? 'rotate-180' : ''}`} />
              </button>

              {/* 필터 드롭다운 */}
              {isFilterOpen && (
                <div className="absolute top-full right-0 mt-2 w-80 bg-white dark:bg-gray-800 rounded-lg shadow-lg border border-gray-200 dark:border-gray-700 z-10">
                  <div className="p-4">
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="font-medium">필터 설정</h3>
                      <button
                        onClick={() => setIsFilterOpen(false)}
                        className="text-gray-400 hover:text-gray-600"
                      >
                        <X className="w-5 h-5" />
                      </button>
                    </div>

                    {/* 카테고리 필터 */}
                    <div className="mb-4">
                      <label className="block text-sm font-medium mb-2">카테고리</label>
                      <select
                        value={selectedCategory}
                        onChange={(e) => setSelectedCategory(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700"
                      >
                        <option value="">전체 카테고리</option>
                        {categories.map((category) => (
                          <option key={category} value={category}>{category}</option>
                        ))}
                      </select>
                    </div>

                    {/* 상태 필터 */}
                    <div className="mb-4">
                      <label className="block text-sm font-medium mb-2">상태</label>
                      <select
                        value={selectedStatus}
                        onChange={(e) => setSelectedStatus(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700"
                      >
                        {statusOptions.map((option) => (
                          <option key={option.value} value={option.value}>{option.label}</option>
                        ))}
                      </select>
                    </div>

                    {/* 적용 버튼 */}
                    <button
                      onClick={() => handleFilterChange(selectedCategory, selectedStatus)}
                      className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                    >
                      필터 적용
                    </button>
                  </div>
                </div>
              )}
            </div>

            {/* 검색 버튼 */}
            <button
              onClick={handleSearch}
              className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              검색
            </button>
          </div>

          {/* 활성 필터 표시 */}
          {(selectedCategory || selectedStatus !== 'active') && (
            <div className="flex items-center gap-2 mt-4 pt-4 border-t border-gray-200 dark:border-gray-700">
              <span className="text-sm text-gray-600 dark:text-gray-400">활성 필터:</span>
              {selectedCategory && (
                <span className="px-2 py-1 bg-blue-100 dark:bg-blue-900/20 text-blue-800 dark:text-blue-300 rounded-full text-xs">
                  카테고리: {selectedCategory}
                </span>
              )}
              {selectedStatus !== 'active' && (
                <span className="px-2 py-1 bg-green-100 dark:bg-green-900/20 text-green-800 dark:text-green-300 rounded-full text-xs">
                  상태: {statusOptions.find(s => s.value === selectedStatus)?.label}
                </span>
              )}
            </div>
          )}
        </div>

        {/* 결과 요약 및 뷰 모드 토글 */}
        <div className="flex items-center justify-between mb-6">
          <div className="text-sm text-gray-600 dark:text-gray-400">
            총 <span className="font-semibold text-blue-600">{total}</span>개의 공고 
            {total > 0 && (
              <span> (페이지 {currentPage} / {totalPages})</span>
            )}
          </div>
          
          {/* 뷰 모드 토글 버튼 */}
          <div className="flex items-center gap-2 bg-gray-100 dark:bg-gray-700 p-1 rounded-lg">
            <button
              onClick={() => setViewMode('list')}
              className={`flex items-center gap-1 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                viewMode === 'list'
                  ? 'bg-white dark:bg-gray-800 text-blue-600 shadow-sm'
                  : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
              }`}
            >
              <List className="w-4 h-4" />
              목록
            </button>
            <button
              onClick={() => setViewMode('card')}
              className={`flex items-center gap-1 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                viewMode === 'card'
                  ? 'bg-white dark:bg-gray-800 text-blue-600 shadow-sm'
                  : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
              }`}
            >
              <Grid3X3 className="w-4 h-4" />
              카드
            </button>
          </div>
        </div>

        {/* 공고 목록 */}
        {loading ? (
          <div className="flex items-center justify-center py-12">
            <RefreshCw className="w-8 h-8 animate-spin text-blue-600" />
          </div>
        ) : announcements.length > 0 ? (
          viewMode === 'list' ? (
            // 목록 뷰
            <div className="space-y-4 mb-8">
              {announcements.map((announcement) => (
                <div 
                  key={announcement.id}
                  className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6 hover:shadow-md transition-shadow cursor-pointer"
                  onClick={() => router.push(`/business/announcements/${announcement.id}`)}
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      {/* 카테고리 및 메타 정보 */}
                      <div className="flex items-center gap-3 mb-3">
                        <span className={`px-3 py-1 rounded-full text-sm font-medium ${getCategoryColor(announcement.category)}`}>
                          {announcement.category}
                        </span>
                        {announcement.application_end && (
                          <span className={`text-sm font-medium flex items-center gap-1 ${getDeadlineColor(announcement.application_end)}`}>
                            <Calendar className="w-4 h-4" />
                            마감: {formatDate(announcement.application_end)}
                          </span>
                        )}
                        <span className="text-sm text-gray-500 dark:text-gray-400 flex items-center gap-1">
                          <Eye className="w-4 h-4" />
                          {announcement.view_count}회
                        </span>
                      </div>

                      {/* 제목 */}
                      <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100 mb-2 line-clamp-1">
                        {announcement.title}
                      </h3>

                      {/* 설명 */}
                      <p className="text-gray-600 dark:text-gray-400 mb-4 line-clamp-2">
                        {announcement.description}
                      </p>

                      {/* 상세 정보 */}
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                        <div className="flex items-center gap-2">
                          <DollarSign className="w-4 h-4 text-green-600" />
                          <span className="text-gray-600 dark:text-gray-400">지원금:</span>
                          <span className="font-medium text-green-600">{formatPrice(announcement.support_amount)}</span>
                        </div>

                        <div className="flex items-center gap-2">
                          <Users className="w-4 h-4 text-blue-600" />
                          <span className="text-gray-600 dark:text-gray-400">대상:</span>
                          <span className="font-medium">{announcement.eligibility.company_size.join(', ')}</span>
                        </div>

                        <div className="flex items-center gap-2">
                          <MapPin className="w-4 h-4 text-purple-600" />
                          <span className="text-gray-600 dark:text-gray-400">지역:</span>
                          <span className="font-medium">{announcement.eligibility.location.join(', ')}</span>
                        </div>
                      </div>

                      {/* 담당기관 */}
                      <div className="mt-3 pt-3 border-t border-gray-200 dark:border-gray-700 flex items-center gap-2">
                        <Building2 className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-600 dark:text-gray-400">
                          {announcement.contact.department} • {announcement.contact.name}
                        </span>
                      </div>
                    </div>

                    <ArrowRight className="w-6 h-6 text-gray-400 ml-6" />
                  </div>
                </div>
              ))}
            </div>
          ) : (
            // 카드 뷰 (4x1 배치)
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mb-8">
              {announcements.map((announcement) => (
                <div 
                  key={announcement.id}
                  className="bg-gray-50 dark:bg-gray-900 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow cursor-pointer overflow-hidden"
                  onClick={() => router.push(`/business/announcements/${announcement.id}`)}
                >
                  {/* 카드 헤더 */}
                  <div className="p-4 border-b border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-2">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${getCategoryColor(announcement.category)}`}>
                        {announcement.category}
                      </span>
                      <span className="text-xs text-gray-500 dark:text-gray-400 flex items-center gap-1">
                        <Eye className="w-3 h-3" />
                        {announcement.view_count}
                      </span>
                    </div>
                    <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100 line-clamp-2 min-h-[3rem]">
                      {announcement.title}
                    </h3>
                  </div>

                  {/* 카드 본문 */}
                  <div className="p-4">
                    <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-3 mb-4 min-h-[4.5rem]">
                      {announcement.description}
                    </p>

                    {/* 지원금 */}
                    <div className="flex items-center gap-1 mb-3">
                      <DollarSign className="w-4 h-4 text-green-600" />
                      <span className="text-sm font-medium text-green-600">{formatPrice(announcement.support_amount)}</span>
                    </div>

                    {/* 마감일 */}
                    {announcement.application_end && (
                      <div className="flex items-center gap-1 mb-3">
                        <Calendar className="w-4 h-4 text-gray-500" />
                        <span className={`text-sm ${getDeadlineColor(announcement.application_end)}`}>
                          {formatDate(announcement.application_end)}
                        </span>
                      </div>
                    )}

                    {/* 담당기관 */}
                    <div className="flex items-center gap-1 text-xs text-gray-500 dark:text-gray-400">
                      <Building2 className="w-3 h-3" />
                      <span className="truncate">{announcement.contact.department}</span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )
        ) : (
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-12 text-center">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">검색 결과가 없습니다</h3>
            <p className="text-gray-600 dark:text-gray-400">다른 검색어나 필터를 시도해보세요.</p>
          </div>
        )}

        {/* 페이지네이션 */}
        {totalPages > 1 && (
          <div className="flex justify-center items-center gap-2">
            <button
              onClick={() => handlePageChange(Math.max(1, currentPage - 1))}
              disabled={currentPage === 1}
              className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg disabled:opacity-50 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
            >
              이전
            </button>

            {[...Array(Math.min(5, totalPages))].map((_, i) => {
              const page = Math.max(1, Math.min(totalPages - 4, currentPage - 2)) + i;
              return (
                <button
                  key={page}
                  onClick={() => handlePageChange(page)}
                  className={`px-3 py-2 border rounded-lg transition-colors ${
                    currentPage === page
                      ? 'bg-blue-600 text-white border-blue-600'
                      : 'border-gray-300 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700'
                  }`}
                >
                  {page}
                </button>
              );
            })}

            <button
              onClick={() => handlePageChange(Math.min(totalPages, currentPage + 1))}
              disabled={currentPage === totalPages}
              className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg disabled:opacity-50 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
            >
              다음
            </button>
          </div>
        )}
    </div>
  );
}