'use client';

import { useState, useEffect, useRef, useCallback } from 'react';
import { useRouter } from 'next/navigation';
import { useAuth } from '@/hooks/useAuth';
import { useLanguage } from '@/contexts/LanguageContext';
import type { 
  NodeData, 
  Connection, 
  WorkflowData, 
  NodeType, 
  ProcessType,
  ExecutionResult 
} from '@/types/flowai';
import FlowCanvas from '@/components/flowai/FlowCanvas';
import Header from '@/components/flowai/Header';
import Sidebar from '@/components/flowai/Sidebar';
import SlidingPanel from '@/components/flowai/SlidingPanel';


export default function FlowAIPage() {
  const { user, isAuthenticated, isInitialized } = useAuth();
  const { t } = useLanguage();
  const router = useRouter();

  // 상태 관리
  const [workflow, setWorkflow] = useState<WorkflowData | null>(null);
  const [nodes, setNodes] = useState<NodeData[]>([]);
  const [connections, setConnections] = useState<Connection[]>([]);
  const [selectedNode, setSelectedNode] = useState<NodeData | null>(null);
  const [isExecuting, setIsExecuting] = useState(false);
  const [executionResults, setExecutionResults] = useState<ExecutionResult[]>([]);

  const [error, setError] = useState<string | null>(null);
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const [showSaveModal, setShowSaveModal] = useState(false);
  const [saveModalName, setSaveModalName] = useState('');
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);
  
  // 새로운 UI 상태
  const [isSidebarCollapsed, setIsSidebarCollapsed] = useState(false);
  const [slidingPanelType, setSlidingPanelType] = useState<'config' | 'execution' | null>(null);
  const [workflows, setWorkflows] = useState<WorkflowData[]>([]);
  const [recentWorkflows, setRecentWorkflows] = useState<WorkflowData[]>([]);

  const canvasRef = useRef<HTMLDivElement>(null);

  // 인증 확인 및 초기 데이터 로드
  useEffect(() => {
    if (isInitialized && !isAuthenticated) {
      router.push('/login');
    } else if (isInitialized && isAuthenticated) {
      loadWorkflows();
    }
  }, [isInitialized, isAuthenticated, router]);

  // 워크플로우 목록 로드
  const loadWorkflows = async () => {
    try {
      const response = await fetch('/api/flowai/workflows', {
        method: 'GET',
        credentials: 'include',
      });

      if (response.ok) {
        const result = await response.json();
        setWorkflows(result.data || []);
        
        // 최근 워크플로우 설정 (업데이트 시간 기준 상위 5개)
        const recent = (result.data || [])
          .sort((a: WorkflowData, b: WorkflowData) => 
            new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime()
          )
          .slice(0, 5);
        setRecentWorkflows(recent);
      }
    } catch (error) {
      console.error('워크플로우 목록 로드 오류:', error);
    }
  };

  // 노드 추가 (중앙에 추가)
  const addNode = useCallback((type: NodeType, position?: { x: number; y: number }) => {
    const defaultPosition = position || { 
      x: 400 + Math.random() * 200, 
      y: 200 + Math.random() * 100 
    };
    
    const newNode: NodeData = {
      id: `node-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      type,
      position: defaultPosition,
      data: {
        label: getNodeLabel(type),
        description: getNodeDescription(type),
        config: getDefaultConfig(type),
      },
      inputs: type !== 'input' ? [] : undefined,
      outputs: type !== 'output' ? [] : undefined,
    };

    setNodes(prev => [...prev, newNode]);
    setSelectedNode(newNode);
    setSlidingPanelType('config');
  }, []);

  // 노드 라벨 가져오기
  const getNodeLabel = (type: NodeType): string => {
    switch (type) {
      case 'input':
        return '입력';
      case 'process':
        return '생성';
      case 'output':
        return '출력';
      case 'condition':
        return '분기';
      case 'loop':
        return '반복';
      default:
        return '노드';
    }
  };

  // 노드 설명 가져오기
  const getNodeDescription = (type: NodeType): string => {
    switch (type) {
      case 'input':
        return '사용자 입력 또는 데이터 소스';
      case 'process':
        return 'AI 모델 생성 및 콘텐츠 제작';
      case 'output':
        return '결과 표시 또는 데이터 내보내기';
      case 'condition':
        return '조건부 논리 및 분기 처리';
      case 'loop':
        return '반복 및 재처리';
      default:
        return '';
    }
  };

  // 기본 설정 가져오기
  const getDefaultConfig = (type: NodeType): any => {
    switch (type) {
      case 'input':
        return {
          inputType: 'text',
          placeholder: '텍스트를 입력하세요',
        };
      case 'process':
        return {
          temperature: 0.7,
          maxTokens: 2000,
        };
      case 'output':
        return {
          outputFormat: 'text',
          filename: 'output',
        };
      case 'condition':
        return {
          condition: '',
          operator: 'equals',
        };
      case 'loop':
        return {
          loopCount: 3,
          loopVariable: 'item',
        };
      default:
        return {};
    }
  };

  // 노드 업데이트
  const updateNode = useCallback((nodeId: string, updates: Partial<NodeData>) => {
    setNodes(prev => prev.map(node => 
      node.id === nodeId 
        ? { ...node, ...updates, data: { ...node.data, ...updates.data } }
        : node
    ));

    if (selectedNode?.id === nodeId) {
      setSelectedNode(prev => prev ? { ...prev, ...updates, data: { ...prev.data, ...updates.data } } : null);
    }
  }, [selectedNode]);

  // 노드 삭제
  const deleteNode = useCallback((nodeId: string) => {
    setNodes(prev => prev.filter(node => node.id !== nodeId));
    setConnections(prev => prev.filter(conn => 
      conn.source !== nodeId && conn.target !== nodeId
    ));
    if (selectedNode?.id === nodeId) {
      setSelectedNode(null);
      setSlidingPanelType(null);
    }
  }, [selectedNode]);

  // 개별 노드 실행
  const executeNode = useCallback(async (nodeId: string) => {
    const node = nodes.find(n => n.id === nodeId);
    if (!node) return;

    // 노드 실행 상태 업데이트
    setNodes(prev => prev.map(n => 
      n.id === nodeId 
        ? { ...n, data: { ...n.data, isRunning: true, hasError: false } }
        : n
    ));

    try {
      // 노드 테스트 API 호출
      const response = await fetch('/api/flowai/test-node', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({
          nodeId: node.id,
          type: node.type,
          data: node.data
        })
      });

      if (!response.ok) {
        throw new Error('노드 실행 실패');
      }

      const result = await response.json();
      
      // 실행 결과 업데이트
      setExecutionResults(prev => [
        ...prev.filter(r => r.nodeId !== nodeId),
        {
          nodeId: nodeId,
          status: 'success',
          startTime: new Date().toISOString(),
          endTime: new Date().toISOString(),
          output: result.data,
          error: undefined
        }
      ]);

      // 성공 상태 업데이트
      setNodes(prev => prev.map(n => 
        n.id === nodeId 
          ? { ...n, data: { ...n.data, isRunning: false, hasError: false } }
          : n
      ));

      setSuccessMessage('노드가 성공적으로 실행되었습니다');
      setSlidingPanelType('execution');
    } catch (error) {
      // 에러 상태 업데이트
      setNodes(prev => prev.map(n => 
        n.id === nodeId 
          ? { ...n, data: { ...n.data, isRunning: false, hasError: true } }
          : n
      ));

      setError(error instanceof Error ? error.message : '노드 실행 중 오류 발생');
    }
  }, [nodes]);

  // 노드 실행 중지
  const stopNode = useCallback((nodeId: string) => {
    setNodes(prev => prev.map(n => 
      n.id === nodeId 
        ? { ...n, data: { ...n.data, isRunning: false } }
        : n
    ));
  }, []);

  // 노드 복제
  const duplicateNode = useCallback((nodeId: string) => {
    const nodeToDuplicate = nodes.find(n => n.id === nodeId);
    if (!nodeToDuplicate) return;

    const newNode: NodeData = {
      ...nodeToDuplicate,
      id: `node-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      position: {
        x: nodeToDuplicate.position.x + 50,
        y: nodeToDuplicate.position.y + 50
      },
      data: {
        ...nodeToDuplicate.data,
        label: `${nodeToDuplicate.data.label} (복사본)`
      }
    };

    setNodes(prev => [...prev, newNode]);
    setSelectedNode(newNode);
    setSlidingPanelType('config');
    setSuccessMessage('노드가 복제되었습니다');
  }, [nodes]);

  // 연결 추가
  const addConnection = useCallback((source: string, target: string, sourceHandle?: string, targetHandle?: string) => {
    try {
      console.log('🔗 addConnection called:', { source, target, currentConnections: connections });
      
      // 유효성 검사
      if (!source || !target) {
        console.error('🔗 Invalid source or target:', { source, target });
        return;
      }
      
      if (source === target) {
        console.error('🔗 Cannot connect node to itself');
        return;
      }
      
      // 중복 연결 확인 (같은 핸들 간의 중복 연결 방지)
      const exists = connections.some(conn => 
        conn.source === source && 
        conn.target === target &&
        conn.sourceHandle === sourceHandle &&
        conn.targetHandle === targetHandle
      );

      if (!exists) {
        const newConnection: Connection = {
          id: `conn-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
          source,
          target,
          sourceHandle,
          targetHandle,
        };
        console.log('🔗 Adding new connection:', newConnection);
        setConnections(prev => {
          const updated = [...prev, newConnection];
          console.log('🔗 Updated connections:', updated);
          return updated;
        });
      } else {
        console.log('🔗 Connection already exists');
      }
    } catch (error) {
      console.error('🔗 Error in addConnection:', error);
    }
  }, [connections]);

  // 연결 삭제
  const deleteConnection = useCallback((connectionId: string) => {
    setConnections(prev => prev.filter(conn => conn.id !== connectionId));
  }, []);

  // 워크플로우 실행
  const executeWorkflow = async () => {
    if (nodes.length === 0) {
      setError('실행할 노드가 없습니다');
      return;
    }

    // 워크플로우가 저장되지 않은 경우 먼저 저장
    if (!workflow?.id) {
      setError('먼저 워크플로우를 저장해주세요');
      return;
    }

    setIsExecuting(true);
    setExecutionResults([]);
    setSlidingPanelType('execution');
    setError(null);

    // 실행 과정 시뮬레이션을 위해 초기 상태 설정
    const initialResults: ExecutionResult[] = nodes.map(node => ({
      nodeId: node.id,
      status: 'pending',
      startTime: undefined,
      endTime: undefined,
      output: undefined,
      error: undefined,
    }));
    
    setExecutionResults(initialResults);

    try {
      // 실행 과정 시뮬레이션
      for (let i = 0; i < nodes.length; i++) {
        const node = nodes[i];
        
        // 노드 실행 시작
        setExecutionResults(prev => prev.map(result => 
          result.nodeId === node.id 
            ? { ...result, status: 'running', startTime: new Date().toISOString() }
            : result
        ));
        
        // 실행 시간 시뮬레이션 (0.5-2초)
        await new Promise(resolve => setTimeout(resolve, 500 + Math.random() * 1500));
      }

      // 실제 워크플로우 실행 API 호출
      const response = await fetch(`/api/flowai/workflows/${workflow.id}/execute`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({
          variables: workflow.variables || {},
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error?.message || '워크플로우 실행 실패');
      }

      const result = await response.json();
      
      // 실제 실행 결과로 업데이트
      setExecutionResults(result.data.results || []);
      setSuccessMessage(`워크플로우가 성공적으로 실행되었습니다 (${result.data.summary.totalNodes}개 노드 완료)`);
      
    } catch (error) {
      console.error('워크플로우 실행 오류:', error);
      setError(error instanceof Error ? error.message : '워크플로우 실행 중 오류가 발생했습니다');
      
      // 에러 발생 시 실행 중이던 노드들을 에러 상태로 설정
      setExecutionResults(prev => prev.map(result => 
        result.status === 'running' || result.status === 'pending'
          ? { ...result, status: 'error', error: '실행 중단됨', endTime: new Date().toISOString() }
          : result
      ));
    } finally {
      setIsExecuting(false);
    }
  };

  // 이전 함수들은 새로운 워크플로우 실행 API로 대체됨

  // 워크플로우 저장
  const saveWorkflow = async (newName?: string, saveAsNew?: boolean) => {
    // 저장 방식이 지정되지 않았고 변경사항이 있으면 모달 표시
    if (!newName && !saveAsNew && hasUnsavedChanges) {
      setShowSaveModal(true);
      setSaveModalName(workflow?.name || '');
      return;
    }
    
    // 새 워크플로우이거나 "Untitled"인 경우 모달 표시
    if (!newName && (!workflow?.id || workflow?.name === 'Untitled')) {
      setShowSaveModal(true);
      setSaveModalName('');
      return;
    }

    const workflowName = newName || workflow?.name;
    if (!workflowName) {
      setError('워크플로우 이름을 입력해주세요');
      return;
    }

    try {
      // Clean the data to remove any React refs or circular references
      const cleanNodes = nodes.map(node => ({
        id: node.id,
        type: node.type,
        position: { x: node.position.x, y: node.position.y },
        data: {
          label: node.data.label,
          description: node.data.description,
          config: node.data.config,
          processType: node.data.processType,
          ragEnabled: node.data.ragEnabled,
          webSearchEnabled: node.data.webSearchEnabled,
          apiEndpoint: node.data.apiEndpoint,
          headers: node.data.headers,
          body: node.data.body,
          condition: node.data.condition,
          loopCount: node.data.loopCount,
          loopVariable: node.data.loopVariable,
        },
        inputs: node.inputs,
        outputs: node.outputs,
      }));

      const cleanConnections = connections.map(conn => ({
        id: conn.id,
        source: conn.source,
        target: conn.target,
        sourceHandle: conn.sourceHandle,
        targetHandle: conn.targetHandle,
      }));

      let workflowData: WorkflowData;
      let isNewWorkflow = false;
      
      if (newName || saveAsNew) {
        // 다른 이름으로 저장 - 새 워크플로우 생성
        isNewWorkflow = true;
        workflowData = {
          id: `workflow-${Date.now()}`,
          name: newName || workflowName,
          description: workflow?.description || '',
          nodes: cleanNodes,
          connections: cleanConnections,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
          userId: user?.id,
        };
      } else if (workflow?.id) {
        // 기존 워크플로우 업데이트
        isNewWorkflow = false;
        workflowData = {
          id: workflow.id,
          name: workflow.name,
          description: workflow.description || '',
          nodes: cleanNodes,
          connections: cleanConnections,
          createdAt: workflow.createdAt,
          updatedAt: new Date().toISOString(),
          userId: workflow.userId,
        };
      } else {
        // 새 워크플로우 생성 (첫 저장)
        isNewWorkflow = true;
        workflowData = {
          id: `workflow-${Date.now()}`,
          name: workflowName,
          description: workflow?.description || '',
          nodes: cleanNodes,
          connections: cleanConnections,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
          userId: user?.id,
        };
      }

      const url = isNewWorkflow ? '/api/flowai/workflows' : `/api/flowai/workflows/${workflowData.id}`;
      const response = await fetch(url, {
        method: isNewWorkflow ? 'POST' : 'PUT',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify(workflowData),
      });

      if (!response.ok) {
        throw new Error('워크플로우 저장 실패');
      }

      const result = await response.json();
      setWorkflow(result.data);
      resetUnsavedChanges(); // 변경 상태 초기화
      
      if (isNewWorkflow) {
        setSuccessMessage(newName ? `"${newName}" 이름으로 새로 저장되었습니다` : '워크플로우가 저장되었습니다');
      } else {
        setSuccessMessage('워크플로우가 업데이트되었습니다');
      }
      
      // 워크플로우 목록 새로고침
      loadWorkflows();
    } catch (error) {
      console.error('워크플로우 저장 오류:', error);
      setError('워크플로우 저장 중 오류가 발생했습니다');
    }
  };

  // 워크플로우 불러오기
  const loadWorkflow = async (workflowId: string) => {
    try {
      const response = await fetch(`/api/flowai/workflows/${workflowId}`, {
        method: 'GET',
        credentials: 'include',
      });

      if (!response.ok) {
        throw new Error('워크플로우 불러오기 실패');
      }

      const result = await response.json();
      const workflowData = result.data as WorkflowData;
      
      setWorkflow(workflowData);
      setNodes(workflowData.nodes);
      setConnections(workflowData.connections);
      resetUnsavedChanges(); // 워크플로우 로드 후 변경 상태 초기화

      setSelectedNode(null);
      setSlidingPanelType(null);
      
      // 최근 워크플로우 목록 업데이트
      setRecentWorkflows(prev => {
        const filtered = prev.filter(w => w.id !== workflowId);
        return [workflowData, ...filtered].slice(0, 5);
      });
    } catch (error) {
      console.error('워크플로우 불러오기 오류:', error);
      setError('워크플로우 불러오기 중 오류가 발생했습니다');
    }
  };

  // 새 워크플로우 생성
  const createNewWorkflow = () => {
    setWorkflow({
      id: '',
      name: 'Untitled',
      description: '',
      nodes: [],
      connections: [],
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
    });
    setNodes([]);
    setConnections([]);
    setSelectedNode(null);
    setExecutionResults([]);
    setSlidingPanelType(null);
    resetUnsavedChanges(); // 새 워크플로우 시작 시 변경 상태 초기화
  };

  // 워크플로우 템플릿 적용 이벤트 리스너
  useEffect(() => {
    const handleApplyTemplate = (e: any) => {
      const template = e.detail;
      
      // 새 워크플로우 생성
      const newWorkflow: WorkflowData = {
        id: `workflow-${Date.now()}`,
        name: template.name,
        description: template.description,
        nodes: [],
        connections: [],
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
      };
      
      setWorkflow(newWorkflow);
      
      // 노드들 생성
      const newNodes: NodeData[] = template.nodes.map((nodeConfig: any, index: number) => ({
        id: `node-${Date.now()}-${index}`,
        type: nodeConfig.type,
        position: nodeConfig.position,
        data: {
          label: getNodeLabel(nodeConfig.type),
          description: getNodeDescription(nodeConfig.type),
          config: { ...getDefaultConfig(nodeConfig.type), ...nodeConfig.config },
        },
        inputs: nodeConfig.type !== 'input' ? [] : undefined,
        outputs: nodeConfig.type !== 'output' ? [] : undefined,
      }));
      
      setNodes(newNodes);
      setConnections([]);
      setSelectedNode(null);
      setSuccessMessage(`${template.name} 템플릿이 적용되었습니다`);
    };
    
    window.addEventListener('applyWorkflowTemplate', handleApplyTemplate);
    return () => window.removeEventListener('applyWorkflowTemplate', handleApplyTemplate);
  }, []);

  // 워크플로우 변경 감지
  useEffect(() => {
    // 워크플로우가 로드되었을 때는 변경으로 간주하지 않음
    if (!workflow) return;

    // 새 워크플로우이거나 노드/연결이 변경되었을 때 변경 상태로 설정
    setHasUnsavedChanges(true);
  }, [nodes, connections]);

  // 워크플로우 저장 후 변경 상태 초기화
  const resetUnsavedChanges = () => {
    setHasUnsavedChanges(false);
  };

  // 에러 및 성공 메시지 자동 숨김
  useEffect(() => {
    if (error) {
      const timer = setTimeout(() => setError(null), 5000);
      return () => clearTimeout(timer);
    }
  }, [error]);

  useEffect(() => {
    if (successMessage) {
      const timer = setTimeout(() => setSuccessMessage(null), 3000);
      return () => clearTimeout(timer);
    }
  }, [successMessage]);



  if (!isInitialized) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-lg">로딩 중...</div>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-screen" style={{ backgroundColor: 'var(--body-bg)' }}>
      {/* 상단 헤더 */}
      <Header
        onNodeAdd={(type) => addNode(type)}
        onToggleSidebar={() => setIsSidebarCollapsed(!isSidebarCollapsed)}
        isExecuting={isExecuting}
        isSidebarCollapsed={isSidebarCollapsed}
        nodesCount={nodes.length}
        workflowName={workflow?.name}
        onWorkflowNameUpdate={async (newName) => {
          if (workflow) {
            // 먼저 로컬 상태 업데이트
            setWorkflow(prev => prev ? { ...prev, name: newName } : null);
            
            // 기존 워크플로우 업데이트 (새 워크플로우 생성하지 않음)
            if (workflow.id) {
              try {
                const response = await fetch(`/api/flowai/workflows/${workflow.id}`, {
                  method: 'PUT',
                  headers: { 'Content-Type': 'application/json' },
                  credentials: 'include',
                  body: JSON.stringify({
                    ...workflow,
                    name: newName,
                    updatedAt: new Date().toISOString(),
                  }),
                });

                if (response.ok) {
                  setSuccessMessage('워크플로우 이름이 변경되었습니다');
                  loadWorkflows(); // 워크플로우 목록 새로고침
                } else {
                  throw new Error('워크플로우 이름 변경 실패');
                }
              } catch (error) {
                console.error('워크플로우 이름 변경 오류:', error);
                setError('워크플로우 이름 변경 중 오류가 발생했습니다');
                // 오류 시 이름 되돌리기
                setWorkflow(prev => prev ? { ...prev, name: workflow.name } : null);
              }
            }
          }
        }}
      />

      {/* 메인 컨텐츠 영역 */}
      <div className="flex flex-1 overflow-hidden" style={{ height: 'calc(100vh - 6rem)' }}>
        {/* 왼쪽 사이드바 */}
        <Sidebar
          isCollapsed={isSidebarCollapsed}
          workflows={workflows}
          recentWorkflows={recentWorkflows}
          executionHistory={executionResults}
          onWorkflowSelect={loadWorkflow}
          onNewWorkflow={createNewWorkflow}
          onShowTemplates={() => {}}
          onShowWorkflowList={() => {}}
        />

        {/* 메인 캔버스 영역 */}
        <div className="flex-1 relative">
          <FlowCanvas
            nodes={nodes}
            connections={connections}
            selectedNode={selectedNode}
            onNodeSelect={(node) => {
              setSelectedNode(node);
              if (node) {
                setSlidingPanelType('config');
              } else {
                setSlidingPanelType(null);
              }
            }}
            onNodeUpdate={updateNode}
            onNodeDelete={deleteNode}
            onNodeExecute={executeNode}
            onNodeStop={stopNode}
            onNodeDuplicate={duplicateNode}
            onConnectionAdd={addConnection}
            onConnectionDelete={deleteConnection}
            isDragging={false}
            draggedNodeType={null}
            onDrop={(type, position) => {
              if (type) {
                addNode(type, position);
              }
            }}
          />

          {/* 플로팅 액션 버튼들 - 왼쪽 상단 */}
          <div className="absolute top-8 left-4 flex space-x-2 z-10">
            <button
              onClick={executeWorkflow}
              disabled={isExecuting || nodes.length === 0}
              className={`flex items-center space-x-2 px-4 py-2.5 text-sm font-medium rounded-xl border transition-all duration-150 ${
                isExecuting || nodes.length === 0
                  ? 'bg-transparent text-gray-400 border-gray-200 dark:border-gray-700 cursor-not-allowed'
                  : 'bg-transparent text-emerald-600 dark:text-emerald-400 border-emerald-200 dark:border-emerald-800 hover:bg-emerald-50 dark:hover:bg-emerald-900/20 hover:scale-[1.02] active:scale-[0.98]'
              }`}
              title="워크플로우 실행"
            >
              {isExecuting ? (
                <>
                  <div className="w-4 h-4 border-2 border-emerald-600 dark:border-white/70 border-t-transparent rounded-full animate-spin"></div>
                  <span className="font-semibold">실행 중</span>
                </>
              ) : (
                <>
                  <span className="text-base group-hover:animate-pulse">▶️</span>
                  <span className="font-semibold">실행</span>
                </>
              )}
            </button>

            <button
              onClick={() => {
                // 변경사항이 있으면 항상 모달 표시
                saveWorkflow();
              }}
              disabled={nodes.length === 0}
              className={`flex items-center space-x-2 px-4 py-2.5 text-sm font-medium rounded-xl border transition-all duration-150 ${
                nodes.length === 0
                  ? 'bg-transparent text-gray-400 border-gray-200 dark:border-gray-700 cursor-not-allowed'
                  : 'bg-transparent text-blue-600 dark:text-blue-400 border-blue-200 dark:border-blue-800 hover:bg-blue-50 dark:hover:bg-blue-900/20 hover:scale-[1.02] active:scale-[0.98]'
              }`}
              title={workflow?.id && workflow?.name !== 'Untitled' ? "워크플로우 저장 (우클릭: 다른 이름으로 저장)" : "워크플로우 저장"}
            >
              <span className="text-base group-hover:animate-bounce">💾</span>
              <span className="font-semibold">저장</span>
            </button>
          </div>

          {/* 에러 메시지 */}
          {error && (
            <div className="absolute top-4 left-1/2 transform -translate-x-1/2 bg-red-500 text-white px-4 py-2 rounded-xl shadow-lg z-30">
              {error}
            </div>
          )}

          {/* 성공 메시지 */}
          {successMessage && (
            <div className="absolute top-4 left-1/2 transform -translate-x-1/2 bg-green-500 text-white px-4 py-2 rounded-xl shadow-lg z-30">
              {successMessage}
            </div>
          )}
        </div>
      </div>

      {/* 슬라이딩 패널 */}
      <SlidingPanel
        isOpen={slidingPanelType !== null}
        type={slidingPanelType}
        selectedNode={selectedNode}
        executionResults={executionResults}
        isExecuting={isExecuting}
        onNodeUpdate={updateNode}
        onNodeDelete={deleteNode}
        onClose={() => {
          setSlidingPanelType(null);
          setSelectedNode(null);
        }}
      />



      {/* 저장 모달 */}
      {showSaveModal && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-2xl p-6 w-full max-w-md mx-4">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                워크플로우 저장
              </h3>
              <button
                onClick={() => setShowSaveModal(false)}
                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            <div className="mb-6">
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                워크플로우 이름
              </label>
              <input
                type="text"
                value={saveModalName}
                onChange={(e) => setSaveModalName(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                placeholder="워크플로우 이름을 입력하세요"
                autoFocus
              />
            </div>

            <div className="flex space-x-3">
              {workflow?.id && workflow?.name !== 'Untitled' ? (
                <>
                  <button
                    onClick={() => {
                      // 기존 워크플로우 업데이트 (현재 이름 유지)
                      saveWorkflow(undefined, false);
                      setShowSaveModal(false);
                    }}
                    className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors"
                  >
                    저장
                  </button>
                  <button
                    onClick={() => {
                      if (saveModalName.trim()) {
                        // 새 워크플로우로 저장
                        saveWorkflow(saveModalName.trim(), true);
                        setShowSaveModal(false);
                      }
                    }}
                    className="flex-1 bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-lg font-medium transition-colors"
                    disabled={!saveModalName.trim()}
                  >
                    다른 이름으로 저장
                  </button>
                </>
              ) : (
                <button
                  onClick={() => {
                    if (saveModalName.trim()) {
                      // 새 워크플로우 저장
                      saveWorkflow(saveModalName.trim());
                      setShowSaveModal(false);
                    }
                  }}
                  className="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors"
                  disabled={!saveModalName.trim()}
                >
                  저장
                </button>
              )}
            </div>

            <div className="mt-3">
              <button
                onClick={() => setShowSaveModal(false)}
                className="w-full text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 py-2 text-sm"
              >
                취소
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}