import React, { useState, useEffect } from 'react';
import Prism from 'prismjs';
import 'prismjs/components/prism-python';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-tsx';
import 'prismjs/components/prism-typescript';
import 'prismjs/components/prism-json';
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-javascript';
import { useTheme } from 'next-themes';

interface CodeViewerProps {
  code: string;
  language?: string;
  title?: string;
  className?: string;
}

export const CodeViewer: React.FC<CodeViewerProps> = ({ 
  code, 
  language = 'python', 
  title,
  className = ''
}) => {
  const { resolvedTheme } = useTheme();
  const [mounted, setMounted] = useState(false);
  const [isDark, setIsDark] = useState(false);
  
  // 클라이언트 사이드에서만 테마 확인
  useEffect(() => {
    setMounted(true);
    
    // 다크모드 감지 - 여러 방법으로 확인
    const checkDarkMode = () => {
      // 1. next-themes의 resolvedTheme 확인
      if (resolvedTheme === 'dark') return true;
      
      // 2. HTML 요소의 dark 클래스 확인
      if (document.documentElement.classList.contains('dark')) return true;
      
      // 3. data-theme 속성 확인
      if (document.documentElement.getAttribute('data-theme') === 'dark') return true;
      
      // 4. CSS media query 확인
      if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
        // 시스템 설정이 다크모드이고 테마가 설정되지 않은 경우
        return !resolvedTheme || resolvedTheme === 'system';
      }
      
      return false;
    };
    
    setIsDark(checkDarkMode());
  }, [resolvedTheme]);
  
  // 테마 변경 감지
  useEffect(() => {
    if (!mounted) return;
    
    const handleThemeChange = () => {
      setIsDark(
        resolvedTheme === 'dark' || 
        document.documentElement.classList.contains('dark') ||
        document.documentElement.getAttribute('data-theme') === 'dark'
      );
    };
    
    // MutationObserver로 DOM 변경 감지
    const observer = new MutationObserver(handleThemeChange);
    observer.observe(document.documentElement, {
      attributes: true,
      attributeFilter: ['class', 'data-theme']
    });
    
    return () => observer.disconnect();
  }, [mounted, resolvedTheme]);

  // Prism 하이라이트
  let highlighted = Prism.highlight(code, Prism.languages[language] || Prism.languages.plaintext, language);
  
  // 다크모드일 때 토큰 색상 변경
  if (isDark) {
    highlighted = highlighted
      .replace(/class="token comment"/g, 'class="token comment" style="color: #6b7280"')
      .replace(/class="token punctuation"/g, 'class="token punctuation" style="color: #9ca3af"')
      .replace(/class="token (property|tag|boolean|number|constant|symbol|deleted)"/g, 'class="token $1" style="color: #f87171"')
      .replace(/class="token (selector|attr-name|string|char|builtin|inserted)"/g, 'class="token $1" style="color: #34d399"')
      .replace(/class="token (operator|entity|url)"/g, 'class="token $1" style="color: #60a5fa"')
      .replace(/class="token (atrule|attr-value|keyword)"/g, 'class="token $1" style="color: #a78bfa"')
      .replace(/class="token (function|class-name)"/g, 'class="token $1" style="color: #fbbf24"')
      .replace(/class="token (regex|important|variable)"/g, 'class="token $1" style="color: #f59e0b"');
  }
  
  const highlightedLines = highlighted.split('\n');
  const codeLines = code.split('\n');

  const handleCopy = () => {
    navigator.clipboard.writeText(code);
    // 복사 성공 피드백 (선택사항)
    const button = document.activeElement as HTMLButtonElement;
    if (button) {
      const originalText = button.textContent;
      button.textContent = '복사됨!';
      setTimeout(() => {
        button.textContent = originalText;
      }, 1000);
    }
  };

  // 서버 사이드 렌더링 시에는 스켈레톤 표시
  if (!mounted) {
    return (
      <div className={`relative group rounded-lg border overflow-hidden ${className} border-gray-200 bg-gray-50`}>
        <div className="px-3 py-2 text-xs font-medium border-b bg-gray-100 text-gray-600 border-gray-200">
          <div className="flex items-center space-x-2">
            {title && <span className="font-semibold">{title}</span>}
            {language && language !== 'plaintext' && (
              <span className="px-2 py-1 rounded text-xs bg-gray-200 text-gray-600">
                {language}
              </span>
            )}
          </div>
        </div>
        <div className="bg-gray-50 p-4">
          <div className="animate-pulse">
            <div className="h-4 bg-gray-200 rounded mb-2"></div>
            <div className="h-4 bg-gray-200 rounded mb-2 w-3/4"></div>
            <div className="h-4 bg-gray-200 rounded mb-2 w-1/2"></div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className={`relative group rounded-lg border overflow-hidden ${className} ${
      isDark ? 'border-gray-700 bg-gray-900' : 'border-gray-200 bg-gray-50'
    }`}>
      {/* 헤더 - 제목과 언어 표시 */}
      <div className={`px-3 py-2 text-xs font-medium border-b flex items-center justify-between ${
        isDark 
          ? 'bg-gray-800 text-gray-300 border-gray-700' 
          : 'bg-gray-100 text-gray-600 border-gray-200'
      }`}>
        <div className="flex items-center space-x-2">
          {title && (
            <span className="font-semibold">{title}</span>
          )}
          {language && language !== 'plaintext' && (
            <span className={`px-2 py-1 rounded text-xs ${
              isDark 
                ? 'bg-gray-700 text-gray-300' 
                : 'bg-gray-200 text-gray-600'
            }`}>
              {language}
            </span>
          )}
        </div>
        
        {/* 복사 버튼 */}
        <button
          className={`px-2 py-1 text-xs rounded transition-colors ${
            isDark
              ? 'bg-gray-700 text-gray-300 hover:bg-blue-600 hover:text-white'
              : 'bg-gray-200 text-gray-600 hover:bg-blue-600 hover:text-white'
          }`}
          onClick={handleCopy}
          title="코드 복사"
        >
          복사
        </button>
      </div>
      
      {/* 코드 영역 */}
      <div className={`${isDark ? 'bg-gray-900' : 'bg-gray-50'}`}>
        <pre className="m-0 p-0 font-mono overflow-x-auto" style={{ fontSize: '13px', lineHeight: '1.6' }}>
          {codeLines.map((line, i) => (
            <div 
              key={i} 
              className={`flex transition-colors ${
                isDark 
                  ? 'bg-gray-900 hover:bg-gray-800' 
                  : 'bg-gray-50 hover:bg-gray-100'
              }`}
            >
              <span
                className={`select-none text-right pr-3 pl-3 ${
                  isDark 
                    ? 'text-gray-500 bg-gray-800' 
                    : 'text-gray-400 bg-gray-100'
                }`}
                style={{ 
                  minWidth: '3em', 
                  fontSize: '13px', 
                  lineHeight: '1.6',
                  borderRight: isDark ? '1px solid #374151' : '1px solid #e5e7eb'
                }}
              >
                {i + 1}
              </span>
              <code
                className={`pl-3 pr-4 flex-1 block ${
                  isDark ? 'text-gray-100' : 'text-gray-900'
                }`}
                style={{ 
                  fontSize: '13px', 
                  lineHeight: '1.6',
                  minHeight: '1.6em',
                  fontFamily: 'inherit',
                  backgroundColor: isDark ? '#111827' : '#f9fafb' // 명시적 배경색 설정
                }}
                dangerouslySetInnerHTML={{ 
                  __html: highlightedLines[i] || '<span>&nbsp;</span>'
                }}
              />
            </div>
          ))}
        </pre>
      </div>
    </div>
  );
}; 