import { useState } from 'react';
import type { NodeConfigProps } from './types';

export default function LoopNodeConfig({ node, onNodeUpdate }: NodeConfigProps) {
  const [localConfig, setLocalConfig] = useState(node.data.config || {});
  const [localData, setLocalData] = useState(node.data);

  const updateConfig = (key: string, value: any) => {
    const newConfig = { ...localConfig, [key]: value };
    setLocalConfig(newConfig);
    onNodeUpdate(node.id, {
      data: {
        ...localData,
        config: newConfig,
      },
    });
  };

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    onNodeUpdate(node.id, {
      data: newData,
    });
  };

  return (
    <div className="space-y-4">
      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          반복 타입
        </label>
        <select
          value={localConfig.loopType || 'count'}
          onChange={(e) => updateConfig('loopType', e.target.value)}
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        >
          <option value="count">횟수 기반</option>
          <option value="condition">조건 기반</option>
          <option value="array">배열 순회</option>
        </select>
      </div>

      {localConfig.loopType === 'count' && (
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            반복 횟수
          </label>
          <input
            type="number"
            value={localData.loopCount || 3}
            onChange={(e) => updateData('loopCount', parseInt(e.target.value))}
            min="1"
            max="100"
            className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
          />
        </div>
      )}

      {localConfig.loopType === 'condition' && (
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            반복 조건
          </label>
          <textarea
            value={localConfig.loopCondition || ''}
            onChange={(e) => updateConfig('loopCondition', e.target.value)}
            placeholder="반복을 계속할 조건을 작성하세요 (예: result.length < 10)"
            className="w-full px-3 py-2 bg-transparent border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 h-20 resize-none"
          />
        </div>
      )}

      {localConfig.loopType === 'array' && (
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            배열 변수명
          </label>
          <input
            type="text"
            value={localConfig.arrayVariable || 'items'}
            onChange={(e) => updateConfig('arrayVariable', e.target.value)}
            placeholder="순회할 배열의 변수명"
            className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
          />
        </div>
      )}

      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          반복 변수명
        </label>
        <input
          type="text"
          value={localData.loopVariable || 'item'}
          onChange={(e) => updateData('loopVariable', e.target.value)}
          placeholder="반복 시 사용할 변수명"
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        />
      </div>

      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          최대 반복 횟수 (안전장치)
        </label>
        <input
          type="number"
          value={localConfig.maxIterations || 100}
          onChange={(e) => updateConfig('maxIterations', parseInt(e.target.value))}
          min="1"
          max="1000"
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        />
      </div>

      <div className="flex items-center">
        <input
          type="checkbox"
          id="breakOnError"
          checked={localConfig.breakOnError || true}
          onChange={(e) => updateConfig('breakOnError', e.target.checked)}
          className="w-4 h-4 text-blue-600 bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 focus:ring-2 mr-3"
        />
        <label htmlFor="breakOnError" className="text-sm font-medium text-gray-700 dark:text-gray-300">
          오류 시 반복 중단
        </label>
      </div>
    </div>
  );
}