// 서버 설정 중앙 관리
export interface ServerConfig {
  // Next.js 웹 서버
  nextjs: {
    port: number;
    url: string;
  };
  
  // AIRUN API 서버
  api: {
    port: number;
    url: string;
    wsUrl: string;
  };
  
  // RAG 서버
  rag: {
    port: number;
    url: string;
  };
  
  // 웹검색 서버
  websearch: {
    port: number;
    url: string;
  };
  
  // 사업분석 서버
  business: {
    port: number;
    url: string;
  };
  
  // Ollama 서버
  ollama: {
    port: number;
    url: string;
    proxyUrl?: string;
    model: string;
  };
  
  // 데이터베이스
  database: {
    host: string;
    port: number;
    user: string;
    password: string;
    name: string;
  };
  
  // Redis
  redis: {
    port: number;
    url: string;
  };
  
  // 환경 정보
  environment: {
    isDevelopment: boolean;
    isProduction: boolean;
  };
}

// 환경변수에서 서버 설정 읽기
export function getServerConfig(): ServerConfig {
  const isDevelopment = process.env.NODE_ENV === 'development';
  const isProduction = process.env.NODE_ENV === 'production';
  
  // 통일된 NEXT_PUBLIC_ 환경변수 사용
  const apiServerUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
  
  // Ollama 설정
  const ollamaUrl = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || 
                   process.env.OLLAMA_URL || 
                   'http://localhost:11434';
  
  const ollamaModel = process.env.OLLAMA_MODEL || 'hamonize:latest';
  
  return {
    nextjs: {
      port: parseInt(process.env.PORT || '5700'),
      url: process.env.NEXT_PUBLIC_SITE_URL || `http://localhost:${process.env.PORT || '5700'}`
    },
    
    api: {
      port: 5500,
      url: apiServerUrl,
      wsUrl: process.env.NEXT_PUBLIC_WS_SERVER_URL || apiServerUrl.replace('http', 'ws')
    },
    
    rag: {
      port: 5600,
      url: process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600'
    },

    websearch: {
      port: 5610,
      url: process.env.NEXT_PUBLIC_WEB_API_SERVER || 'http://localhost:5610'
    },

    business: {
      port: 5640,
      url: process.env.NEXT_PUBLIC_BUSINESS_API_SERVER || 'http://localhost:5640'
    },
    
    ollama: {
      port: 11434,
      url: ollamaUrl,
      proxyUrl: process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER,
      model: ollamaModel
    },
    
    database: {
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5433'),
      user: process.env.DB_USER || 'ivs',
      password: process.env.DB_PASSWORD || '1234',
      name: process.env.DB_NAME || 'airun'
    },
    
    redis: {
      port: parseInt(process.env.REDIS_PORT || '6379'),
      url: process.env.REDIS_URL || 'redis://localhost:6379'
    },
    
    environment: {
      isDevelopment,
      isProduction
    }
  };
}

// 전역 설정 인스턴스
export const serverConfig = getServerConfig();

// 통일된 환경변수 사용으로 클라이언트/서버 구분 불필요

// RAG 서버 URL 결정
function getRagServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';
}

// 웹검색 서버 URL 결정
function getWebsearchServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_WEB_API_SERVER || 'http://localhost:5610';
}

// API 서버 URL 결정
function getApiServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
}

// 리포트 서버 URL 결정
function getReportServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_REPORT_API_SERVER || 'http://localhost:5620';
}

// LLM 파인튜닝 서버 URL 결정
function getLLMFinetuneServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_LLM_FINETUNE_API_SERVER || 'http://localhost:5630';
}

// 사업분석 서버 URL 결정
function getBusinessServerUrlDynamic(): string {
  return process.env.NEXT_PUBLIC_BUSINESS_API_SERVER || 'http://localhost:5640';
}

// 개별 설정 접근 헬퍼 함수들
export const getApiServerUrl = () => getApiServerUrlDynamic();
export const getRagServerUrl = () => getRagServerUrlDynamic();
export const getWebsearchServerUrl = () => getWebsearchServerUrlDynamic();
export const getReportServerUrl = () => getReportServerUrlDynamic();
export const getLLMFinetuneServerUrl = () => getLLMFinetuneServerUrlDynamic();
export const getBusinessServerUrl = () => getBusinessServerUrlDynamic();
export const getNextjsServerUrl = () => serverConfig.nextjs.url;
export const getOllamaServerUrl = () => serverConfig.ollama.url;
export const getWebSocketUrl = () => serverConfig.api.wsUrl;
export const getDatabaseConfig = () => serverConfig.database;
export const getRedisUrl = () => serverConfig.redis.url;

// 환경별 설정 확인
export const isDevelopment = () => serverConfig.environment.isDevelopment;
export const isProduction = () => serverConfig.environment.isProduction;

// 디버그용 설정 출력
export const logServerConfig = () => {
  if (isDevelopment()) {
    console.log('🔧 Server Configuration:');
    console.log('  - Next.js:', serverConfig.nextjs.url);
    console.log('  - API Server:', serverConfig.api.url);
    console.log('  - RAG Server:', serverConfig.rag.url);
    console.log('  - Websearch Server:', serverConfig.websearch.url);
    console.log('  - Business Server:', serverConfig.business.url);
    console.log('  - Ollama:', serverConfig.ollama.url);
    console.log('  - Database:', `${serverConfig.database.host}:${serverConfig.database.port}`);
    console.log('  - Redis:', serverConfig.redis.url);
    if (serverConfig.ollama.proxyUrl) {
      console.log('  - Ollama Proxy:', serverConfig.ollama.proxyUrl);
    }
  }
}; 