import { requireAuth } from './auth.js';

// apiMonitorPage 함수를 export
export async function apiMonitorPage(chatMessages, translations) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // API 서버 상태 확인
    try {
        const taskId = getUnique();
        await new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                reject(new Error('Request timeout'));
            }, 5000);

            window.electronAPI.receive('response', (response) => {
                if (response.taskId === taskId) {
                    clearTimeout(timeout);
                    if (response.error) {
                        reject(new Error(response.error));
                    } else {
                        resolve(response.arg);
                    }
                }
            });
            
            window.electronAPI.send('request', {
                mode: 'apiServerHealth',
                taskId: taskId,
                arg: {}
            });
        });
    } catch (error) {
        // API 서버가 실행되지 않은 경우 안내 메시지 표시
        chatMessages.innerHTML = `
            <div class="page-container api-monitor-page">
                <div class="page-header">
                    <div class="header-content">
                        <h2 class="page-title">${translations?.apiMonitor?.title || 'API Server Monitor'}</h2>
                    </div>
                </div>
                <div class="server-error-container" style="text-align: center; padding: 40px; margin: 20px; background-color: var(--bg-secondary); border-radius: 8px;">
                    <div style="font-size: 48px; margin-bottom: 20px;">🔌</div>
                    <h3 style="color: var(--error-color); margin-bottom: 20px;">${translations?.apiMonitor?.serverOffline || 'API 서버가 실행되지 않았습니다'}</h3>
                    <p style="color: var(--text-secondary);">${translations?.apiMonitor?.serverOfflineDesc || 'API 서버를 시작한 후 다시 프로그램을 재시작 하세요.'}</p>
                </div>
            </div>
        `;
        return;
    }

    // Chart.js와 플러그인 로드
    if (!window.Chart) {
        await new Promise((resolve, reject) => {
            // Chart.js 로드
            const chartScript = document.createElement('script');
            chartScript.src = './static/chart.umd.min.js';
            chartScript.onload = () => {
                // date-fns 로드
                const dateFnsScript = document.createElement('script');
                dateFnsScript.src = './static/index.min.js';
                dateFnsScript.onload = () => {
                    // date-fns 어댑터 로드
                    const adapterScript = document.createElement('script');
                    adapterScript.src = './static/chartjs-adapter-date-fns.bundle.min.js';
                    adapterScript.onload = () => {
                        // chartjs-chart-matrix 플러그인 로드
                        const matrixScript = document.createElement('script');
                        matrixScript.src = './static/chartjs-chart-matrix.min.js';
                        matrixScript.onload = () => {
                            // Matrix 컨트롤러 등록
                            if (window.Chart && window.Chart.register && window.Chart.controllers) {
                                const MatrixController = window.Chart.controllers.matrix || window.Chart.Matrix;
                                if (MatrixController) {
                                    window.Chart.register(MatrixController);
                                }
                            }
                            resolve();
                        };
                        matrixScript.onerror = reject;
                        document.head.appendChild(matrixScript);
                    };
                    adapterScript.onerror = reject;
                    document.head.appendChild(adapterScript);
                };
                dateFnsScript.onerror = reject;
                document.head.appendChild(dateFnsScript);
            };
            chartScript.onerror = reject;
            document.head.appendChild(chartScript);
        });
    } else if (!Chart.controllers.matrix) {
        // Chart.js는 이미 로드되어 있지만 Matrix 플러그인이 없는 경우
        await new Promise((resolve, reject) => {
            const matrixScript = document.createElement('script');
            matrixScript.src = './static/chartjs-chart-matrix.min.js';
            matrixScript.onload = () => {
                // Matrix 컨트롤러 등록
                if (window.Chart && window.Chart.register && window.Chart.controllers) {
                    const MatrixController = window.Chart.controllers.matrix || window.Chart.Matrix;
                    if (MatrixController) {
                        window.Chart.register(MatrixController);
                    }
                }
                resolve();
            };
            matrixScript.onerror = reject;
            document.head.appendChild(matrixScript);
        });
    }

    // html2pdf.js 로드
    if (!window.html2pdf) {
        await new Promise((resolve, reject) => {
            const html2pdfScript = document.createElement('script');
            html2pdfScript.src = './static/html2pdf.bundle.min.js';
            html2pdfScript.onload = resolve;
            html2pdfScript.onerror = reject;
            document.head.appendChild(html2pdfScript);
        });
    }

    // CSS 파일 로드
    if (!document.querySelector('link[href="./pages/apiMonitor.css"]')) {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = './pages/apiMonitor.css';
        document.head.appendChild(link);
    }

    // 페이지 컨테이너 생성
    chatMessages.innerHTML = `
        <div class="page-container api-monitor-page">
            <div class="page-header">
                <div class="header-content">
                    <h2 class="page-title">${translations?.apiMonitor?.title || 'API Server Monitor'}</h2>
                    <p class="page-description">${translations?.apiMonitor?.description || 'Monitor API server status and usage'}</p>
                </div>
            </div>
            
            <!-- API 요약 섹션 -->
            <div class="api-summary-section">
                <div class="summary-cards">
                    <div class="summary-card total-calls">
                        <h3>${translations?.apiMonitor?.totalCalls || 'Total API Calls'}</h3>
                        <div class="amount" id="totalCalls">0</div>
                        <div class="trend" id="callsTrend"></div>
                    </div>
                    <div class="summary-card limited-calls">
                        <h3>${translations?.apiMonitor?.limitedCalls || 'API Plan Calls'}</h3>
                        <div class="amount" id="limitedCalls">0</div>
                        <div class="trend" id="limitedCallsTrend"></div>
                    </div>
                    <div class="summary-card avg-response">
                        <h3>${translations?.apiMonitor?.avgResponse || 'Avg Response Time'}</h3>
                        <div class="amount" id="avgResponse">0ms</div>
                        <div class="trend" id="responseTrend"></div>
                    </div>
                    <div class="summary-card success-rate">
                        <h3>${translations?.apiMonitor?.successRate || 'Success Rate'}</h3>
                        <div class="amount" id="successRate">0%</div>
                        <div class="trend" id="successTrend"></div>
                    </div>
                </div>
            </div>

            <!-- API 분석 섹션 -->
            <div class="api-analysis-section">
                <h3>${translations?.apiMonitor?.analysis || 'API Analysis'}</h3>
                <div class="chart-grid">
                    <!-- 엔드포인트별 사용량 차트 -->
                    <div class="chart-container">
                        <h4>${translations?.apiMonitor?.endpointUsage || 'Endpoint Usage'}</h4>
                        <canvas id="endpointUsageChart"></canvas>
                    </div>
                    
                    <!-- 시간대별 API 호출 히트맵 -->
                    <div class="chart-container">
                        <h4>${translations?.apiMonitor?.callsHeatmap || 'API Calls Heatmap'}</h4>
                        <canvas id="apiCallsHeatmap"></canvas>
                    </div>
                    
                    <!-- 응답 시간 추이 차트 -->
                    <div class="chart-container">
                        <h4>${translations?.apiMonitor?.responseTime || 'Response Time Trend'}</h4>
                        <canvas id="responseTimeChart"></canvas>
                    </div>
                    
                    <!-- 오류율 추이 차트 -->
                    <div class="chart-container">
                        <h4>${translations?.apiMonitor?.errorRate || 'Error Rate Trend'}</h4>
                        <canvas id="errorRateChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- API 로그 테이블 -->
            <div class="api-logs-section">
                <div class="section-header">
                    <h3>${translations?.apiMonitor?.logs || 'API Logs'}</h3>
                    <button class="download-btn" onclick="downloadApiLogs()">
                        ${translations?.apiMonitor?.downloadLogs || 'Download CSV'}
                    </button>
                </div>
                <div class="table-container">
                    <table id="apiLogsTable">
                        <thead>
                            <tr>
                                <th>${translations?.apiMonitor?.timestamp || 'Timestamp'}</th>
                                <th>${translations?.apiMonitor?.endpoint || 'Endpoint'}</th>
                                <th>${translations?.apiMonitor?.method || 'Method'}</th>
                                <th>${translations?.apiMonitor?.status || 'Status'}</th>
                                <th>${translations?.apiMonitor?.responseTime || 'Response Time'}</th>
                                <th>${translations?.apiMonitor?.apiType || 'API 유형'}</th>
                            </tr>
                        </thead>
                        <tbody id="apiLogsTableBody">
                        </tbody>
                    </table>
                    <div class="pagination" id="apiLogsTablePagination">
                        <!-- 페이지네이션 컨트롤이 여기에 추가됨 -->
                    </div>
                </div>
            </div>
        </div>
    `;

    // 데이터 로드 및 차트 초기화
    try {
        const apiData = await getApiMonitorData();
        updateDashboard(apiData, translations);
    } catch (error) {
        console.error('Error loading API monitor data:', error);
        // 데이터 로드 실패 시 오류 메시지 표시
        const errorContainer = document.createElement('div');
        errorContainer.className = 'error-container';
        errorContainer.style.textAlign = 'center';
        errorContainer.style.padding = '20px';
        errorContainer.style.color = 'var(--error-color)';
        errorContainer.innerHTML = `
            <div style="font-size: 24px; margin-bottom: 10px;">⚠️</div>
            <p>${translations?.apiMonitor?.loadError || 'API 모니터링 데이터를 불러오는데 실패했습니다'}</p>
            <button onclick="location.reload()" 
                    style="margin-top: 15px; padding: 8px 15px; background-color: var(--accent-color); color: var(--bg-primary); border: none; border-radius: 4px; cursor: pointer;">
                ${translations?.common?.retry || '다시 시도'}
            </button>
        `;
        chatMessages.querySelector('.api-monitor-page').appendChild(errorContainer);
    }
}

// 데이터 로드 함수
async function getApiMonitorData() {
    const taskId = getUnique();
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
            reject(new Error('Request timeout'));
        }, 10000);

        window.electronAPI.receive('response', (response) => {
            if (response.taskId === taskId) {
                clearTimeout(timeout);
                if (response.error) {
                    reject(new Error(response.error));
                } else {
                    resolve(response.arg);
                }
            }
        });
        
        window.electronAPI.send('request', {
            mode: 'getApiMonitorData',
            taskId: taskId,
            arg: {}
        });
    });
}

// 대시보드 업데이트 함수
function updateDashboard(apiData, translations) {
    updateSummaryCards(apiData);
    updateEndpointUsageChart(apiData);
    updateApiCallsHeatmap(apiData);
    updateResponseTimeChart(apiData);
    updateErrorRateChart(apiData);
    updateApiLogsTable(apiData);
}

// 요약 카드 업데이트
function updateSummaryCards(apiData) {
    // console.log('[Debug] Updating summary cards with data:', apiData);
    
    // 총 호출 수는 endpointStats의 모든 호출 수의 합
    const totalCalls = Object.entries(apiData.endpointStats)
        .reduce((sum, [endpoint, stats]) => sum + stats.calls, 0);
    document.getElementById('totalCalls').textContent = totalCalls.toLocaleString();
    
    // 제한 대상 API 호출 수 계산 (제외 대상이 아닌 모든 호출)
    const limitedCalls = Object.entries(apiData.endpointStats)
        .filter(([endpoint]) => {
            const isExcluded = isApiExcluded(endpoint);
            // console.log(`[Debug] Endpoint ${endpoint}: excluded=${isExcluded}`);
            return !isExcluded;
        })
        .reduce((sum, [_, stats]) => sum + stats.calls, 0);
    
    // console.log(`[Debug] Total calls: ${totalCalls}, Limited calls: ${limitedCalls}`);
    document.getElementById('limitedCalls').textContent = limitedCalls.toLocaleString();
    
    document.getElementById('avgResponse').textContent = `${apiData.avgResponseTime}ms`;
    document.getElementById('successRate').textContent = `${(apiData.successRate * 100).toFixed(1)}%`;
}

// 엔드포인트별 사용량 차트 업데이트
function updateEndpointUsageChart(apiData) {
    const canvas = document.getElementById('endpointUsageChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const existingChart = Chart.getChart(canvas);
    if (existingChart) {
        existingChart.destroy();
    }

    new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(apiData.endpointStats),
            datasets: [{
                label: 'API Calls',
                data: Object.values(apiData.endpointStats).map(stat => stat.calls),
                backgroundColor: 'rgba(75, 192, 192, 0.8)'
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true
                }
            }
        }
    });
}

// 히트맵 차트 업데이트 함수도 수정
function updateApiCallsHeatmap(apiData) {
    const ctx = document.getElementById('apiCallsHeatmap').getContext('2d');
    const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    const hours = Array.from({ length: 24 }, (_, i) => `${i}:00`);
    
    // 데이터 포맷 변환
    const data = [];
    apiData.callsHeatmap.forEach((dayData, dayIndex) => {
        dayData.forEach((value, hourIndex) => {
            data.push({
                x: hourIndex,
                y: dayIndex,
                v: value
            });
        });
    });

    // 기존 차트 제거
    const existingChart = Chart.getChart(ctx.canvas);
    if (existingChart) {
        existingChart.destroy();
    }

    // 새 차트 생성
    return new Chart(ctx, {
        type: 'matrix',
        data: {
            datasets: [{
                label: 'API Calls',
                data: data,
                backgroundColor(context) {
                    const value = context.dataset.data[context.dataIndex].v;
                    const alpha = value === 0 ? 0.1 : Math.min(0.2 + (value / 10), 1);
                    return `rgba(54, 162, 235, ${alpha})`;
                },
                borderColor: 'rgba(54, 162, 235, 0.2)',
                borderWidth: 1,
                width: ({ chart }) => (chart.chartArea || {}).width / 24 - 1,
                height: ({ chart }) => (chart.chartArea || {}).height / 7 - 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                tooltip: {
                    callbacks: {
                        title() {
                            return '';
                        },
                        label(context) {
                            const v = context.dataset.data[context.dataIndex];
                            return [
                                `${days[v.y]}, ${hours[v.x]}`,
                                `Calls: ${v.v}`
                            ];
                        }
                    }
                },
                legend: {
                    display: false
                }
            },
            scales: {
                x: {
                    type: 'linear',
                    offset: true,
                    min: -0.5,
                    max: 23.5,
                    ticks: {
                        stepSize: 4,
                        callback: (value) => hours[value] || ''
                    },
                    grid: {
                        display: false
                    }
                },
                y: {
                    type: 'linear',
                    offset: true,
                    min: -0.5,
                    max: 6.5,
                    ticks: {
                        stepSize: 1,
                        callback: (value) => days[value] || ''
                    },
                    grid: {
                        display: false
                    }
                }
            }
        }
    });
}

// 응답 시간 추이 차트 업데이트
function updateResponseTimeChart(apiData) {
    const canvas = document.getElementById('responseTimeChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const existingChart = Chart.getChart(canvas);
    if (existingChart) {
        existingChart.destroy();
    }

    new Chart(ctx, {
        type: 'line',
        data: {
            labels: apiData.timeLabels,
            datasets: [{
                label: 'Response Time (ms)',
                data: apiData.responseTimes,
                borderColor: 'rgb(75, 192, 192)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true
                }
            }
        }
    });
}

// 오류율 추이 차트 업데이트
function updateErrorRateChart(apiData) {
    const canvas = document.getElementById('errorRateChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const existingChart = Chart.getChart(canvas);
    if (existingChart) {
        existingChart.destroy();
    }

    new Chart(ctx, {
        type: 'line',
        data: {
            labels: apiData.timeLabels,
            datasets: [{
                label: 'Error Rate (%)',
                data: apiData.errorRates,
                borderColor: 'rgb(255, 99, 132)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    max: 100
                }
            }
        }
    });
}

// 페이지네이션 설정
const PAGE_SIZE = 10;
let currentPage = 1;
let totalPages = 1;
let currentLogs = [];

// API 로그 테이블 업데이트
function updateApiLogsTable(apiData) {
    const tbody = document.getElementById('apiLogsTableBody');
    const paginationDiv = document.getElementById('apiLogsTablePagination');
    if (!tbody || !paginationDiv) return;

    // 전체 로그 데이터 저장
    currentLogs = apiData.logs || [];
    totalPages = Math.ceil(currentLogs.length / PAGE_SIZE);
    
    // 현재 페이지가 총 페이지 수를 초과하지 않도록
    currentPage = Math.min(currentPage, totalPages);
    
    // 현재 페이지의 로그만 표시
    const startIndex = (currentPage - 1) * PAGE_SIZE;
    const endIndex = Math.min(startIndex + PAGE_SIZE, currentLogs.length);
    const pageData = currentLogs.slice(startIndex, endIndex);

    // 테이블 내용 업데이트
    tbody.innerHTML = pageData.map(log => {
        const isExcluded = isApiExcluded(log.endpoint);
        return `
            <tr>
                <td>${new Date(log.timestamp).toLocaleString()}</td>
                <td>${log.endpoint}</td>
                <td>${log.method}</td>
                <td class="${log.status < 400 ? 'success' : 'error'}">${log.status}</td>
                <td>${log.responseTime}ms</td>
                <td class="api-type ${!isExcluded ? 'limited' : 'unlimited'}">${!isExcluded ? 'API Plan' : '제외 대상'}</td>
            </tr>
        `;
    }).join('');

    // 페이지네이션 UI 업데이트
    updatePagination(paginationDiv, totalPages, currentPage, 'handlePageChange');
}

// 페이지 변경 함수
window.handlePageChange = function(newPage) {
    if (newPage >= 1 && newPage <= totalPages && newPage !== currentPage) {
        currentPage = newPage;
        // 현재 저장된 로그 데이터로 테이블 업데이트
        updateApiLogsTable({ logs: currentLogs });
    }
};

// 페이지네이션 UI 업데이트
function updatePagination(container, totalPages, currentPage, handlerName) {
    const maxVisiblePages = 5;
    let pagesHtml = '';
    
    // 이전 페이지 버튼
    pagesHtml += `
        <button 
            ${currentPage === 1 ? 'disabled' : ''} 
            onclick="window.${handlerName}(${currentPage - 1})">
            ${translations?.apiMonitor?.pagination?.previous || '이전'}
        </button>
    `;
    
    // 페이지 번호 버튼
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage + 1 < maxVisiblePages) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    for (let i = startPage; i <= endPage; i++) {
        pagesHtml += `
            <button 
                class="${i === currentPage ? 'active' : ''}"
                onclick="window.${handlerName}(${i})">
                ${i}
            </button>
        `;
    }
    
    // 다음 페이지 버튼
    pagesHtml += `
        <button 
            ${currentPage === totalPages ? 'disabled' : ''} 
            onclick="window.${handlerName}(${currentPage + 1})">
            ${translations?.apiMonitor?.pagination?.next || '다음'}
        </button>
    `;
    
    // 페이지 정보 표시
    pagesHtml += `
        <span class="page-info">
            ${currentPage} / ${totalPages} ${translations?.apiMonitor?.pagination?.page || '페이지'}
        </span>
    `;

    container.innerHTML = pagesHtml;
}

// CSV 다운로드 함수
window.downloadApiLogs = async function() {
    if (!currentLogs) return;

    const notification = createProgressNotification(translations?.apiMonitor?.generatingCSV || 'CSV 파일 생성 중...');
    
    try {
        const headers = [
            translations?.apiMonitor?.csv?.timestamp || 'Timestamp',
            translations?.apiMonitor?.csv?.endpoint || 'Endpoint',
            translations?.apiMonitor?.csv?.method || 'Method',
            translations?.apiMonitor?.csv?.status || 'Status',
            translations?.apiMonitor?.csv?.responseTime || 'Response Time',
            translations?.apiMonitor?.csv?.apiType || 'API Type'
        ];

        const csvRows = [headers.join(',')];
        const batchSize = 1000;
        const totalRecords = currentLogs.length;
        
        for (let i = 0; i < totalRecords; i += batchSize) {
            const batch = currentLogs.slice(i, i + batchSize);
            const batchRows = batch.map(log => {
                const isExcluded = isApiExcluded(log.endpoint);
                return [
                    new Date(log.timestamp).toLocaleString(),
                    log.endpoint,
                    log.method,
                    log.status,
                    `${log.responseTime}ms`,
                    isExcluded ? '제외 대상' : 'API Plan'
                ].map(value => `"${value}"`).join(',');
            });

            csvRows.push(...batchRows);
            await new Promise(resolve => setTimeout(resolve, 0));
        }

        const csvContent = '\ufeff' + csvRows.join('\n');
        downloadCSVFile(csvContent, 'api_monitor_logs.csv');
        showSuccessNotification(notification);
    } catch (error) {
        console.error('Error generating CSV:', error);
        showErrorNotification(notification);
    }
};

// 알림 관련 유틸리티 함수들
function createProgressNotification(message) {
    const notification = document.createElement('div');
    notification.className = 'pdf-notification';
    notification.textContent = message;
    document.body.appendChild(notification);
    return notification;
}

function showSuccessNotification(notification) {
    notification.textContent = translations?.apiMonitor?.csvDownloadComplete || 'CSV 파일이 다운로드되었습니다';
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => notification.remove(), 300);
    }, 2000);
}

function showErrorNotification(notification) {
    notification.style.backgroundColor = 'var(--error-color)';
    notification.textContent = translations?.apiMonitor?.csvGenerationFailed || 'CSV 파일 생성에 실패했습니다';
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

function downloadCSVFile(content, filename) {
    const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

// API 제외 여부 확인 함수 (api_server.js와 동일한 로직)
function isApiExcluded(path) {
    // 모든 경로를 /api/v1/ 형식으로 변환
    const fullPath = path.startsWith('/api/v1/') ? path : 
                    path.startsWith('/') ? `/api/v1${path}` : 
                    `/api/v1/${path}`;
    // console.log(`[Debug] Checking path: ${path} -> ${fullPath}`);

    // 문서화 관련 패턴
    const docPatterns = [
        /^\/api\/v1\/docs/,  // 모든 docs 관련 요청 제외
        /^\/api\/v1\/docs\.json$/  // OpenAPI 스펙
    ];

    // 시스템 관련 패턴
    const systemPatterns = [
        /^\/api\/v1\/health$/,
        /^\/api\/v1\/license\/validate$/,
        /^\/api\/v1\/config(\/.*)?$/,
        /^\/api\/v1\/usage$/,
        /^\/api\/v1\/sessions$/,
        /^\/api\/v1\/cache\/clear$/,
        /^\/api\/v1\/monitor\/data$/
    ];

    // RAG 관련 패턴
    const ragPatterns = [
        /^\/api\/v1\/rag\/(search|add|path|list|init)$/
    ];

    // 웹 검색 관련 패턴
    const webPatterns = [
        /^\/api\/v1\/web\/search$/
    ];

    // 모든 패턴을 하나의 배열로 통합
    const allPatterns = [
        ...docPatterns,
        ...systemPatterns,
        ...ragPatterns,
        ...webPatterns
    ];

    // 패턴 매칭 확인
    const isExcluded = allPatterns.some(pattern => {
        const matches = pattern.test(fullPath);
        // if (matches) {
        //     console.log(`[Debug] Matched pattern: ${pattern}`);
        // }
        return matches;
    });

    // console.log(`[Debug] Is excluded: ${isExcluded}`);
    return isExcluded;
}

// 유니크 ID 생성 함수
function getUnique() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

// PDF 다운로드 함수
window.downloadPDF = async function() {
    const container = document.querySelector('.page-container');
    if (!container) return;

    const notification = createProgressNotification(translations?.apiMonitor?.generatingPDF || 'Generating PDF...');
    
    try {
        const element = container.cloneNode(true);
        element.querySelector('.download-pdf-btn')?.remove();
        
        const pdf = await html2pdf().from(element).save(`api_monitor_${new Date().toISOString().split('T')[0]}.pdf`);
        
        notification.textContent = translations?.apiMonitor?.pdfSuccess || 'PDF has been downloaded';
        notification.classList.add('success');
        
        setTimeout(() => {
            notification.remove();
        }, 3000);
    } catch (error) {
        console.error('Error generating PDF:', error);
        notification.textContent = translations?.apiMonitor?.pdfError || 'Failed to generate PDF';
        notification.classList.add('error');
        
        setTimeout(() => {
            notification.remove();
        }, 3000);
    }
} 