// 로그인 함수
export async function login(reqAPI, username, password) {
    console.log('[DEBUG] 로그인 시도:', { username });
    
    try {
        // 요청 데이터를 직접 전달
        const response = await reqAPI('auth/login', {
            username: username,
            password: password
        });

        console.log('[DEBUG] 로그인 응답:', response);

        if (!response || !response.success) {
            const errorMessage = response?.error?.message || '로그인 실패';
            console.error('[DEBUG] 로그인 실패:', errorMessage);
            throw new Error(errorMessage);
        }

        // 응답 데이터 구조 확인
        const userData = response.data;
        if (!userData || !userData.token || !userData.user) {
            console.error('[DEBUG] 잘못된 응답 데이터 구조:', response);
            throw new Error('서버 응답에 필요한 데이터가 없습니다.');
        }

        // 토큰과 사용자 정보 저장
        localStorage.setItem('userToken', userData.token);
        localStorage.setItem('userRole', userData.user.role);
        localStorage.setItem('userName', userData.user.name || userData.user.username);
        localStorage.setItem('username', userData.user.username);
        localStorage.setItem('userId', userData.user.id.toString());

        // Main 프로세스에 현재 사용자 ID 설정
        try {
            await reqAPI('setCurrentUser', { userId: userData.user.id });
            console.log('[DEBUG] Main 프로세스에 사용자 ID 설정 완료:', userData.user.id);
        } catch (error) {
            console.warn('[DEBUG] Main 프로세스 사용자 ID 설정 실패:', error);
        }

        console.log('[DEBUG] 사용자 정보 저장 완료:', {
            username: userData.user.username,
            role: userData.user.role,
            token: userData.token.substring(0, 10) + '...'
        });
        
        return {
            success: true,
            user: userData.user
        };
    } catch (error) {
        console.error('[DEBUG] 로그인 실패:', error);
        return {
            success: false,
            error: error.message || '로그인 처리 중 오류가 발생했습니다.'
        };
    }
}

// 로그인 상태 체크
export async function checkAuthStatus(reqAPI) {
    const token = localStorage.getItem('userToken');
    
    console.log('[DEBUG] 저장된 토큰:', token);
    
    if (!token) {
        console.log('[DEBUG] 토큰 없음');
        return {
            isAuthenticated: false,
            isAdmin: false
        };
    }

    try {
        console.log('[DEBUG] 토큰 검증 요청');
        // 토큰 유효성 검증
        const response = await reqAPI('auth/me', {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });

        console.log('[DEBUG] 토큰 검증 응답:', response);

        if (!response || !response.success) {
            throw new Error('인증 실패');
        }

        const userData = response.data;
        if (!userData) {
            console.error('[DEBUG] 사용자 데이터 없음:', response);
            throw new Error('사용자 정보를 찾을 수 없습니다.');
        }

        const userRole = userData.role;
        
        console.log('[DEBUG] 검증된 사용자 정보:', userData);
        
        return {
            isAuthenticated: true,
            isAdmin: userRole === 'admin',
            user: userData
        };
    } catch (error) {
        console.error('[DEBUG] 토큰 검증 실패:', error);
        // 토큰이 유효하지 않은 경우 로컬 스토리지 클리어
        localStorage.removeItem('userToken');
        localStorage.removeItem('userRole');
        localStorage.removeItem('userName');
        localStorage.removeItem('username');
        
        return {
            isAuthenticated: false,
            isAdmin: false,
            error: error.message
        };
    }
}

// 일반 사용자 인증 체크 함수
export async function requireAuth(reqAPI) {
    try {
        // 토큰이 저장될 때까지 잠시 대기
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // 토큰 확인
        const token = localStorage.getItem('userToken');
        console.log('[DEBUG] requireAuth - 토큰 확인:', token ? '토큰 있음' : '토큰 없음');
        
        if (!token) {
            console.log('[DEBUG] requireAuth - 토큰 없음, 로그인 페이지로 이동');
            window.navigateToPage('login');
            return false;
        }

        // 토큰 유효성 검증
        const response = await reqAPI('auth/me', {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });

        if (!response || !response.success) {
            console.log('[DEBUG] requireAuth - 토큰 검증 실패');
            // 토큰이 유효하지 않으면 로그인 페이지로 이동
            localStorage.removeItem('userToken');
            localStorage.removeItem('userRole');
            localStorage.removeItem('userName');
            localStorage.removeItem('username');
            localStorage.removeItem('userId');
            window.navigateToPage('login');
            return false;
        }

        console.log('[DEBUG] requireAuth - 인증 성공');
        return true;
    } catch (error) {
        console.error('Auth check error:', error);
        window.navigateToPage('login');
        return false;
    }
}

// 관리자 권한 체크
export async function requireAdmin(reqAPI, redirectToLogin = true) {
    console.log('[DEBUG] 관리자 권한 체크 시작');
    const authStatus = await checkAuthStatus(reqAPI);
    
    console.log('[DEBUG] 관리자 권한 상태:', authStatus);
    
    if (!authStatus.isAuthenticated) {
        console.log('[DEBUG] 인증되지 않음, 리다이렉트:', redirectToLogin);
        if (redirectToLogin) {
            window.navigateToPage('login');
        }
        return false;
    }

    if (!authStatus.isAdmin) {
        console.log('[DEBUG] 관리자 권한 없음');
        await reqAPI('errnotify', '관리자 권한이 필요합니다.');
        if (redirectToLogin) {
            window.navigateToPage('login');
        }
        return false;
    }
    
    return true;
}

// 로그아웃
export function logout() {
    console.log('[DEBUG] 로그아웃 실행');
    localStorage.removeItem('userToken');
    localStorage.removeItem('userRole');
    localStorage.removeItem('userName');
    localStorage.removeItem('username');
    window.navigateToPage('login');
}

// 페이지 접근 권한 정의
export const PAGE_ACCESS = {
    PUBLIC: 'public',      // 로그인 불필요
    PROTECTED: 'protected', // 로그인 필요
    ADMIN: 'admin'         // 관리자 권한 필요
};

// 페이지별 필요 권한 정의
export const PAGE_PERMISSIONS = {
    'login': PAGE_ACCESS.PUBLIC,
    'help': PAGE_ACCESS.PUBLIC,
    'chat': PAGE_ACCESS.PROTECTED,
    'config': PAGE_ACCESS.PROTECTED,
    'task': PAGE_ACCESS.PROTECTED,
    'logs': PAGE_ACCESS.PROTECTED,
    'apiMonitor': PAGE_ACCESS.PROTECTED,
    'report': PAGE_ACCESS.PROTECTED,
    'prompt': PAGE_ACCESS.PROTECTED,
    'user': PAGE_ACCESS.ADMIN
};

// 페이지 접근 권한 체크 함수
export async function checkPageAccess(reqAPI, pageName) {
    const permission = PAGE_PERMISSIONS[pageName];
    
    if (permission === PAGE_ACCESS.PUBLIC) {
        return true;
    }
    
    const authStatus = await checkAuthStatus(reqAPI);
    
    if (!authStatus.isAuthenticated) {
        if (pageName !== 'login') {
            window.navigateToPage('login');
        }
        return false;
    }
    
    if (permission === PAGE_ACCESS.ADMIN && !authStatus.isAdmin) {
        await reqAPI('errnotify', '관리자 권한이 필요합니다.');
        return false;
    }
    
    return true;
} 