import { AI_PROVIDERS } from '../providers.js';
import { requireAuth } from './auth.js';

export async function configPage(chatMessages, translations, reqAPI, getText, showCustomDialog, getModelInformation, aiIndicator, abortAllTask, refreshList, startNewTalk, updateUILanguage) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // 기존 내용 초기화
    chatMessages.innerHTML = '';

    const configContainer = document.createElement('div');
    configContainer.id = 'configuration';

    // 페이지 제목 추가
    const pageTitle = document.createElement('div');
    pageTitle.innerText = getText('settings', 'title');
    pageTitle.className = 'configuration_title';
    pageTitle.style.marginTop = '10px';
    configContainer.appendChild(pageTitle);

    const description = document.createElement('p');
    description.className = 'prompt-description';
    description.innerText = getText('settings', 'description');
    description.style.marginBottom = '20px';
    description.style.paddingLeft = '20px';
    configContainer.appendChild(description);

    async function makeConfigLine(container, { title, description, type, list, keyname, placeholder, vendorKey, handleRadioChange, handleButtonClick, defaultValue, inputType }) {
        const lineContainer = document.createElement('div');
        lineContainer.className = 'configuration_config-line';
        if (vendorKey) {
            lineContainer.classList.add('configuration_hidden');
        }
        container.appendChild(lineContainer);

        const configKeyTitle = document.createElement('div');
        configKeyTitle.innerText = title;
        configKeyTitle.className = 'configuration_config-key';
        lineContainer.appendChild(configKeyTitle);

        const configValueUI = document.createElement('div');
        configValueUI.className = 'configuration_config-value';
        lineContainer.appendChild(configValueUI);

        if (type === 'input') {
            const inputContainer = document.createElement('div');
            inputContainer.style.display = 'flex';
            inputContainer.style.gap = '10px';
            inputContainer.style.width = '100%';
            configValueUI.appendChild(inputContainer);

            const inputElement = document.createElement('input');
            inputElement.classList.add('configuration_input');
            inputElement.type = inputType || 'text';
            inputElement.___name = vendorKey;
            inputElement.placeholder = placeholder;
            inputElement.value = defaultValue || await reqAPI('getconfig', { key: keyname });
            inputElement.style.flex = '1';
            inputElement.style.minWidth = '400px';
            inputContainer.appendChild(inputElement);

            // API 키 입력 필드인 경우 토글 버튼과 Save 버튼 추가
            if (inputType === 'password') {
                // 비밀번호 표시/숨김 토글 버튼
                const toggleButton = document.createElement('button');
                toggleButton.innerHTML = '👁️';
                toggleButton.className = 'configuration_button';
                toggleButton.style.minWidth = '40px';
                toggleButton.style.background = 'transparent';
                toggleButton.style.border = 'none';
                toggleButton.style.cursor = 'pointer';
                toggleButton.style.padding = '5px';
                toggleButton.title = getText('settings', 'togglePassword');
                toggleButton.addEventListener('click', () => {
                    if (inputElement.type === 'password') {
                        inputElement.type = 'text';
                        toggleButton.style.opacity = '1';
                    } else {
                        inputElement.type = 'password';
                        toggleButton.style.opacity = '0.7';
                    }
                });
                inputContainer.appendChild(toggleButton);

                // Save 버튼
                const saveButton = document.createElement('button');
                saveButton.innerText = getText('common', 'save');
                saveButton.className = 'configuration_button';
                saveButton.style.minWidth = '60px';
                saveButton.addEventListener('click', async () => {
                    try {
                        const value = inputElement.value;
                        console.log('Saving config:', { keyname, value }); // 디버깅용
                        
                        // 설정 저장
                        await reqAPI('setconfig', { key: keyname, value: value });
                        
                        // 성공 다이얼로그 표시
                        await showCustomDialog({
                            title: getText('dialog', 'successTitle'),
                            message: getText('settings', 'settingsSaved'),
                            type: 'alert',
                            okText: getText('common', 'ok')
                        });
                    } catch (error) {
                        console.error('Save error:', error); // 디버깅용
                        
                        // 에러 다이얼로그 표시
                        await showCustomDialog({
                            title: getText('dialog', 'error'),
                            message: getText('settings', 'settingsSaveError'),
                            type: 'alert',
                            okText: getText('common', 'ok')
                        });
                    }
                });
                inputContainer.appendChild(saveButton);
            }
        } else if (type === 'radio' && list) {
            const radioContainer = document.createElement('div');
            radioContainer.className = 'configuration_radio-container';
            configValueUI.appendChild(radioContainer);

            if (description) {
                const descel = document.createElement('div');
                descel.innerText = description;
                descel.style.padding = '5px';
                descel.style.display = 'block';
                radioContainer.appendChild(descel);
            }

            const defaultValue = (await reqAPI('getconfig', { key: keyname }))?.toLowerCase();
            
            list.forEach((item, index) => {
                const radioInput = document.createElement('input');
                radioInput.type = 'radio';
                radioInput.name = keyname;
                radioInput.value = typeof item === 'string' ? item : item.keyAsValue;
                radioInput.id = `radio-${keyname}-${typeof item === 'string' ? item : item.keyAsValue}`;
                
                if (defaultValue ? 
                    (typeof item === 'string' ? item.toLowerCase() : item.keyAsValue.toLowerCase()) === defaultValue 
                    : index === 0) {
                    radioInput.checked = true;
                }

                radioInput.addEventListener('change', async () => {
                    const value = typeof item === 'string' ? item : item.keyAsValue;
                    await reqAPI('setconfig', { key: keyname, value: value });
                    if (handleRadioChange) await handleRadioChange(value, keyname);
                });

                radioContainer.appendChild(radioInput);

                const radioLabel = document.createElement('label');
                radioLabel.htmlFor = `radio-${keyname}-${typeof item === 'string' ? item : item.keyAsValue}`;
                radioLabel.innerText = typeof item === 'string' ? item : item.displayName;
                radioContainer.appendChild(radioLabel);
            });
        } else if (type === 'button' && list) {
            const buttonContainer = document.createElement('div');
            buttonContainer.className = 'configuration_button-container';
            configValueUI.appendChild(buttonContainer);

            if (description) {
                const descel = document.createElement('div');
                descel.innerText = description;
                descel.style.padding = '5px';
                descel.style.display = 'block';
                buttonContainer.appendChild(descel);
            }

            list.forEach((item) => {
                const button = document.createElement('button');
                button.className = 'configuration_button';
                button.setAttribute('data-action', item.keyAsValue);
                button.innerText = item.displayName;
                
                button.addEventListener('click', async () => {
                    if (handleButtonClick) await handleButtonClick(item.keyAsValue);
                });
                
                buttonContainer.appendChild(button);
            });
        }

        return lineContainer;
    }

    function createSettingsGroup(title, description = '') {
        const group = document.createElement('div');
        group.className = 'settings-group';
        
        const header = document.createElement('div');
        header.className = 'settings-group-header';
        header.innerHTML = `
            <div>
                <div>${title}</div>
                ${description ? `<div class="settings-group-description">${description}</div>` : ''}
            </div>
            <span class="settings-group-toggle">▼</span>
        `;

        const content = document.createElement('div');
        content.className = 'settings-group-content';

        header.addEventListener('click', () => {
            group.classList.toggle('collapsed');
            content.classList.toggle('collapsed');
        });

        group.appendChild(header);
        group.appendChild(content);
        return { group, content };
    }

    // AI 모델 설정 그룹
    const aiModelGroup = createSettingsGroup(
        getText('settings', 'aiModelSettings'),
        getText('settings', 'aiModelSettingsDesc')
    );
    
    // 벤더 목록을 AI_PROVIDERS에서 가져오도록 수정
    const vendorList = Object.entries(AI_PROVIDERS).map(([key, provider]) => ({
        displayName: provider.displayName,
        keyAsValue: key
    }));

    // 벤더 라디오 버튼 생성
    const vendorRadio = await makeConfigLine(aiModelGroup.content, {
        title: getText('settings', 'aiVendor'),
        type: 'radio',
        list: vendorList,
        keyname: 'USE_LLM'
    });

    // 모델 선택 컨테이너 생성
    const modelContainer = document.createElement('div');
    modelContainer.className = 'model-container';
    aiModelGroup.content.appendChild(modelContainer);

    // 벤더 선택 이벤트 핸들러 설정
    const radioInputs = vendorRadio.querySelectorAll('input[type="radio"]');
    radioInputs.forEach(input => {
        input.addEventListener('change', async function() {
            if (this.checked) {
                const selectedVendor = this.value;
                
                // API 키 입력 필드 표시/숨김 처리
                apiKeyLines.forEach(line => {
                    const input = line.querySelector('.configuration_input');
                    if (input) {
                        if (input.___name === selectedVendor) {
                            line.classList.remove('configuration_hidden');
                        } else {
                            line.classList.add('configuration_hidden');
                        }
                    }
                });

                // 벤더 설정 저장
                await reqAPI('setconfig', { key: 'USE_LLM', value: selectedVendor });
                // console.log('Config - Vendor selected:', selectedVendor);

                // 모델 목록 업데이트
                const modelInfo = await getModelInformation(selectedVendor, reqAPI);
                if (modelInfo) {
                    modelContainer.innerHTML = ''; // 기존 모델 선택 UI 제거
                    
                    // Ollama의 경우 현재 선택된 모델 확인
                    let currentModel;
                    if (selectedVendor === 'ollama' && modelInfo.modelList.length > 0) {
                        currentModel = await reqAPI('getconfig', { key: 'OLLAMA_MODEL' });
                        // 선택된 모델이 없는 경우에만 첫 번째 모델을 자동 선택
                        if (!currentModel) {
                            currentModel = modelInfo.modelList[0];
                            await reqAPI('setconfig', { key: 'OLLAMA_MODEL', value: currentModel });
                            // console.log('Config - Auto-selected first Ollama model:', currentModel);
                        }
                    }
                    
                    // 인디케이터 즉시 업데이트
                    const poweredbyElement = document.querySelector('#poweredby');
                    await aiIndicator(true, reqAPI, poweredbyElement);
                    
                    await makeConfigLine(modelContainer, {
                        title: getText('settings', 'model'),
                        type: 'radio',
                        keyname: modelInfo.keyname,
                        list: modelInfo.modelList.map(modelName => ({
                            displayName: modelName,
                            keyAsValue: modelName
                        })),
                        handleRadioChange: async function(model_name, config_key_model) {
                            await reqAPI('setconfig', { key: config_key_model, value: model_name });
                            // console.log('Config - Model selected:', model_name);
                            const poweredbyElement = document.querySelector('#poweredby');
                            await aiIndicator(true, reqAPI, poweredbyElement);  // 모델 선택 시 인디케이터 업데이트
                        }
                    });
                }
            }
        });
    });

    // API 키 입력 필드들을 AI_PROVIDERS에서 가져오도록 수정
    const apiKeyFields = Object.entries(AI_PROVIDERS)
        .filter(([_, provider]) => provider.keyName) // keyName이 있는 프로바이더만 필터링
        .map(([key, provider]) => ({
            title: getText('settings', `${key}ApiKey`),
            keyname: provider.keyName,
            vendorKey: key,
            type: 'password'
        }));

    // API 키 입력 필드 생성
    const apiKeyLines = [];
    for (const field of apiKeyFields) {
        const line = await makeConfigLine(aiModelGroup.content, {
            title: field.title,
            type: 'input',
            keyname: field.keyname,
            placeholder: 'Please input API Key',
            vendorKey: field.vendorKey,
            inputType: field.type
        });
        apiKeyLines.push(line);
    }

    // 현재 선택된 벤더의 API KEY 입력폼만 표시
    const currentVendor = await reqAPI('getconfig', { key: 'USE_LLM' }) || 'openai';
    apiKeyLines.forEach(line => {
        const input = line.querySelector('.configuration_input');
        if (input) {
            if (input.___name === currentVendor) {
                line.classList.remove('configuration_hidden');
            } else {
                line.classList.add('configuration_hidden');
            }
        }
    });

    // 현재 선택된 벤더의 초기 모델 목록 표시를 위한 이벤트 발생
    if (currentVendor) {
        const radioInput = vendorRadio.querySelector(`input[value="${currentVendor}"]`);
        if (radioInput) {
            radioInput.checked = true;
            radioInput.dispatchEvent(new Event('change'));
        }
    }

    // 대화 설정 그룹
    const chatGroup = createSettingsGroup(
        getText('settings', 'conversationSettings'),
        getText('settings', 'conversationSettingsDesc')
    );

    // AI 모드 설정 추가
    await makeConfigLine(chatGroup.content, {
        title: getText('settings', 'aiMode'),
        description: getText('settings', 'aiModeDesc'),
        type: 'radio',
        list: [
            { displayName: getText('settings', 'executionMode'), keyAsValue: 'code' },
            { displayName: getText('settings', 'chatMode'), keyAsValue: 'chat' },
            // { displayName: getText('settings', 'agentMode'), keyAsValue: 'agent' },
            // { displayName: getText('settings', 'reportMode'), keyAsValue: 'report' }
        ],
        keyname: 'AIMODE',
        handleRadioChange: async function(mode) {
            try {
                if (mode === 'chat') {
                    await reqAPI('resetRagPrompt');
                } else {
                    await reqAPI('resetPrompt');
                }
                await showCustomDialog({
                    title: getText('dialog', 'successTitle'),
                    message: getText('settings', 'modeChangeSuccess'),
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            } catch (error) {
                await showCustomDialog({
                    title: getText('dialog', 'errorTitle'),
                    message: getText('settings', 'modeChangeError') + ': ' + error.message,
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            }
        }
    });

    await makeConfigLine(chatGroup.content, {
        title: getText('settings', 'review'),
        description: getText('settings', 'reviewDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'USE_REVIEW'
    });

    await makeConfigLine(chatGroup.content, {
        title: getText('settings', 'autoExecute'),
        description: getText('settings', 'autoExecuteDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'AUTO_EXECUTE'
    });

    await makeConfigLine(chatGroup.content, {
        title: getText('settings', 'theme'),
        type: 'radio',
        list: [
            { displayName: getText('settings', 'darkTheme'), keyAsValue: 'dark' },
            { displayName: getText('settings', 'lightTheme'), keyAsValue: 'light' }
        ],
        keyname: 'THEME',
        handleRadioChange: async function(theme) {
            document.documentElement.setAttribute('data-theme', theme);
        }
    });

    await makeConfigLine(chatGroup.content, {
        title: getText('settings', 'language'),
        type: 'radio',
        list: [
            { displayName: '한국어', keyAsValue: 'ko' },
            { displayName: 'English', keyAsValue: 'en' },
            { displayName: '日本語', keyAsValue: 'ja' },
            { displayName: '中文', keyAsValue: 'zh' },
            { displayName: 'Español', keyAsValue: 'es' },
            { displayName: 'Français', keyAsValue: 'fr' }
        ],
        keyname: 'LANGUAGE',
        handleRadioChange: async function(lang) {
            try {
                await updateUILanguage(lang);
            } catch (error) {
                console.error('Error updating UI language:', error);
                await reqAPI('errnotify', getText('settings', 'languageUpdateError'));
            }
        }
    });

    // RAG 설정 그룹
    const ragGroup = createSettingsGroup(
        getText('settings', 'ragSettings'),
        getText('settings', 'ragSettingsDesc')
    );
    await makeConfigLine(ragGroup.content, {
        title: getText('settings', 'useRAG'),
        description: getText('settings', 'useRAGDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'USE_RAG'
    });

    // RAG 문서 경로 설정
    const currentRagDocsPath = await reqAPI('getconfig', { key: 'RAG_DOCS_PATH' }) || '';
    await makeConfigLine(ragGroup.content, {
        title: getText('settings', 'ragDocsPath'),
        type: 'input',
        keyname: 'RAG_DOCS_PATH',
        placeholder: getText('settings', 'enterDirectoryPath'),
        defaultValue: currentRagDocsPath,
        inputType: 'text'
    });

    // 입력 필드에 자동 저장 이벤트 리스너 추가
    const ragPathInput = ragGroup.content.querySelector('input[type="text"]');
    if (ragPathInput) {
        let saveTimeout;
        
        const savePath = async () => {
            try {
                const inputPath = ragPathInput.value.trim();
                if (inputPath === currentRagDocsPath) return;
                
                // 경로 유효성 검사
                if (!inputPath) {
                    throw new Error('경로를 입력해주세요.');
                }
                
                // 설정 저장
                await reqAPI('setconfig', { 
                    key: 'RAG_DOCS_PATH', 
                    value: inputPath,
                    force: true
                });
            } catch (error) {
                console.error('Path save error:', error);
                await showCustomDialog({
                    title: getText('dialog', 'error'),
                    message: error.message || '경로 설정에 실패했습니다.',
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
                // 이전 값으로 복원
                ragPathInput.value = currentRagDocsPath;
            }
        };

        // 입력 중일 때는 타이머를 재설정
        ragPathInput.addEventListener('input', () => {
            clearTimeout(saveTimeout);
        });

        // 입력이 멈춘 후 1초 뒤에 저장
        ragPathInput.addEventListener('change', () => {
            clearTimeout(saveTimeout);
            saveTimeout = setTimeout(savePath, 1000);
        });

        // 포커스를 잃을 때도 저장
        ragPathInput.addEventListener('blur', () => {
            clearTimeout(saveTimeout);
            savePath();
        });
    }

    // RAG DB 초기화 버튼
    await makeConfigLine(ragGroup.content, {
        title: getText('settings', 'ragDatabaseInit'),
        description: getText('settings', 'ragDatabaseInitDesc'),
        type: 'button',
        list: [
            { displayName: getText('settings', 'processDocuments'), keyAsValue: 'process' }
        ],
        async handleButtonClick(action) {
            try {
                if (action === 'process') {
                    const confirmed = await showCustomDialog({
                        title: getText('dialog', 'processRagDocumentsTitle'),
                        message: getText('dialog', 'processRagDocumentsMessage'),
                        okText: getText('dialog', 'processRagDocumentsOk'),
                        cancelText: getText('dialog', 'processRagDocumentsCancel')
                    });
                
                    if (confirmed) {
                        // 터미널 창 생성
                        const terminalWindow = document.createElement('div');
                        terminalWindow.className = 'terminal-window';
                        terminalWindow.innerHTML = `
                            <div class="terminal-header">
                                <span>${getText('dialog', 'processingDocuments')}</span>
                                <button class="close-button">×</button>
                            </div>
                            <div class="terminal-content" id="ragTerminal"></div>
                        `;
                        document.body.appendChild(terminalWindow);
                
                        // 닫기 버튼 이벤트
                        const closeButton = terminalWindow.querySelector('.close-button');
                        closeButton.addEventListener('click', () => {
                            terminalWindow.remove();
                        });
                
                        // 진행 상황을 수신할 이벤트 리스너 설정
                        const progressHandler = (...args) => {
                            const data = args[0];  // 첫 번째 인자를 데이터로 사용
                            console.log('Progress event received:', data); // 디버깅용 로그
                            const terminal = document.getElementById('ragTerminal');
                            console.log('Terminal element:', terminal);
                            
                            if (terminal && data && data.message) {
                                let lineClass = 'terminal-line';
                                let prefix = '';
                                
                                // 메시지 타입에 따른 스타일 적용
                                switch (data.type) {
                                    case 'error':
                                        lineClass += ' error';
                                        prefix = '❌ ';
                                        break;
                                    case 'start':
                                        lineClass += ' info';
                                        prefix = '🚀 ';
                                        break;
                                    case 'complete':
                                        lineClass += ' success';
                                        prefix = '✅ ';
                                        break;
                                    case 'file':
                                        lineClass += ' file';
                                        prefix = '📄 ';
                                        break;
                                    case 'extract':
                                        lineClass += ' extract';
                                        prefix = '  ↳ 📑 ';
                                        break;
                                    case 'chunk':
                                        lineClass += ' chunk';
                                        prefix = '  ↳ ✂️ ';
                                        break;
                                    case 'vectorize':
                                        lineClass += ' vectorize';
                                        prefix = '  ↳ 🔄 ';
                                        break;
                                    default:
                                        prefix = '  ';
                                }
                                
                                const line = `<div class="${lineClass}">${prefix}${data.message}</div>`;
                                console.log('Adding line to terminal:', line);
                                terminal.innerHTML += line;
                                terminal.scrollTop = terminal.scrollHeight;
                            } else {
                                console.log('Terminal element not found or invalid data:', { terminal, data });
                            }
                        };

                        // 이벤트 리스너 등록 전 로그
                        console.log('Registering rag-progress event listener');
                        window.electronAPI.receive('rag-progress', progressHandler);
                        console.log('Event listener registered');
                        
                        // 초기 메시지 표시
                        const terminal = document.getElementById('ragTerminal');
                        terminal.innerHTML = `<div class="terminal-line">🚀 ${getText('dialog', 'processingDocuments')}...</div>`;

                        try {
                            // 문서 처리 요청
                            const result = await reqAPI('processRagDocuments');
                            
                            // 성공 메시지 표시
                            if (result.success) {
                                terminal.innerHTML += `<div class="terminal-line success">✅ ${getText('dialog', 'documentsProcessedSuccess').replace('{count}', result.processedFiles)}</div>`;
                            }
                        } catch (error) {
                            console.error('Error during RAG processing:', error);
                            // error.message가 객체인 경우를 처리
                            let errorMessage = error?.message;
                            if (typeof errorMessage === 'object') {
                                errorMessage = JSON.stringify(errorMessage);
                            }
                            terminal.innerHTML += `<div class="terminal-line error">❌ ${errorMessage || 'Unknown error'}</div>`;
                            throw error;
                        } finally {
                            // 이벤트 리스너 제거
                            window.electronAPI.send('remove-rag-progress-listener');
                        }
                    }
                }
            } catch (error) {
                await showCustomDialog({
                    title: getText('dialog', 'errorTitle'),
                    message: `${getText('settings', 'ragOperationError')}: ${error.message}`,
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            }
        }
    });

    // SMTP 설정 그룹 추가
    const smtpGroup = createSettingsGroup(
        getText('settings', 'smtpSettings'),
        getText('settings', 'smtpSettingsDesc')
    );

    // SMTP 설정값 초기화
    const smtpSettings = {
        USE_SMTP: await reqAPI('getconfig', { key: 'USE_SMTP' }),
        SMTP_HOST: await reqAPI('getconfig', { key: 'SMTP_HOST' }),
        SMTP_PORT: await reqAPI('getconfig', { key: 'SMTP_PORT' }),
        SMTP_USERNAME: await reqAPI('getconfig', { key: 'SMTP_USERNAME' }),
        SMTP_PASSWORD: await reqAPI('getconfig', { key: 'SMTP_PASSWORD' }),
        SMTP_SECURE: await reqAPI('getconfig', { key: 'SMTP_SECURE' })
    };

    // SMTP 사용 여부 설정
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'useSmtp'),
        description: getText('settings', 'useSmtpDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'USE_SMTP',
        defaultValue: smtpSettings.USE_SMTP || 'NO'
    });

    // SMTP 서버 설정
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'smtpHost'),
        description: getText('settings', 'smtpHostDesc'),
        type: 'input',
        keyname: 'SMTP_HOST',
        placeholder: 'smtp.gmail.com',
        defaultValue: smtpSettings.SMTP_HOST,
        inputType: 'text'
    });

    // SMTP 포트 설정
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'smtpPort'),
        description: getText('settings', 'smtpPortDesc'),
        type: 'input',
        keyname: 'SMTP_PORT',
        placeholder: '587',
        defaultValue: smtpSettings.SMTP_PORT,
        inputType: 'text'
    });

    // SMTP 사용자 이름
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'smtpUsername'),
        description: getText('settings', 'smtpUsernameDesc'),
        type: 'input',
        keyname: 'SMTP_USERNAME',
        placeholder: 'your-email@gmail.com',
        defaultValue: smtpSettings.SMTP_USERNAME,
        inputType: 'text'
    });

    // SMTP 비밀번호
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'smtpPassword'),
        description: getText('settings', 'smtpPasswordDesc'),
        type: 'input',
        keyname: 'SMTP_PASSWORD',
        placeholder: 'your-password-or-app-password',
        defaultValue: smtpSettings.SMTP_PASSWORD,
        inputType: 'password'
    });

    // SMTP 설정 필드들에 대한 이벤트 리스너 추가
    const smtpInputs = smtpGroup.content.querySelectorAll('input[type="text"], input[type="password"]');
    smtpInputs.forEach(input => {
        const configLine = input.closest('.configuration_config-line');
        if (!configLine) return;

        // 입력 필드에 설정 키 데이터 속성 추가
        const configKey = configLine.querySelector('.configuration_config-key').textContent;
        if (configKey === getText('settings', 'smtpUsername')) {
            input.dataset.configKey = 'SMTP_USERNAME';
        } else if (configKey === getText('settings', 'smtpPassword')) {
            input.dataset.configKey = 'SMTP_PASSWORD';
        } else if (configKey === getText('settings', 'smtpHost')) {
            input.dataset.configKey = 'SMTP_HOST';
        } else if (configKey === getText('settings', 'smtpPort')) {
            input.dataset.configKey = 'SMTP_PORT';
        }

        input.addEventListener('change', async () => {
            try {
                const settingKey = input.dataset.configKey;
                if (settingKey) {
                    // SMTP 설정이 변경될 때 USE_SMTP를 YES로 설정
                    await reqAPI('setconfig', { key: 'USE_SMTP', value: 'YES' });
                    
                    // 설정 값 저장
                    const result = await reqAPI('setconfig', { key: settingKey, value: input.value });
                    
                    if (!result || result.error) {
                        throw new Error(result?.error || '설정 저장에 실패했습니다.');
                    }

                    // 콘솔에 로그 출력 (디버깅용)
                    console.log(`SMTP 설정 저장 완료: ${settingKey} = ${input.value}`);
                }
            } catch (error) {
                console.error('Save error:', error);
                await showCustomDialog({
                    title: getText('dialog', 'error'),
                    message: getText('settings', 'settingsSaveError'),
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            }
        });
    });

    // SMTP 보안 연결 설정
    await makeConfigLine(smtpGroup.content, {
        title: getText('settings', 'smtpSecure'),
        description: getText('settings', 'smtpSecureDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'SMTP_SECURE',
        defaultValue: smtpSettings.SMTP_SECURE || 'NO'
    });

    // 추가 기능 그룹
    const additionalGroup = createSettingsGroup(
        getText('settings', 'additionalSettings'),
        getText('settings', 'additionalSettingsDesc')
    );
    
    // 웹 검색 설정
    await makeConfigLine(additionalGroup.content, {
        title: getText('settings', 'useWebSearch'),
        description: getText('settings', 'useWebSearchDesc'),
        type: 'radio',
        list: ['YES', 'NO'],
        keyname: 'USE_WEB_SEARCH'
    });

    // 세션 초기화 버튼
    await makeConfigLine(additionalGroup.content, {
        title: getText('settings', 'clearSession'),
        description: getText('settings', 'clearSessionDesc'),
        type: 'button',
        list: [
            { displayName: getText('settings', 'clearAllSessions'), keyAsValue: 'clear' }
        ],
        handleButtonClick: async function(action) {
            if (action === 'clear') {
                await clearSessions();
            }
        }
    });

    // 설정 초기화 버튼
    await makeConfigLine(additionalGroup.content, {
        title: getText('settings', 'resetSettings'),
        description: getText('settings', 'resetSettingsDesc'),
        type: 'button',
        list: [
            { displayName: getText('settings', 'resetAllSettings'), keyAsValue: 'reset' }
        ],
        handleButtonClick: async function(action) {
            if (action === 'reset') {
                await resetConfiguration();
            }
        }
    });

    // 모든 설정 그룹을 configContainer에 추가
    configContainer.appendChild(aiModelGroup.group);
    configContainer.appendChild(chatGroup.group);
    configContainer.appendChild(ragGroup.group);
    configContainer.appendChild(smtpGroup.group);
    configContainer.appendChild(additionalGroup.group);

    // configContainer를 chatMessages에 추가
    chatMessages.appendChild(configContainer);

    // 유틸리티 함수들
    async function resetConfiguration() {
        const result = await showCustomDialog({
            title: getText('dialog', 'resetSettingsTitle'),
            message: getText('dialog', 'resetSettingsMessage'),
            okText: getText('dialog', 'resetSettingsOk'),
            cancelText: getText('dialog', 'resetSettingsCancel')
        });
    
        if (result) {
            // 로딩 스피너 표시
            const { loadingContainer } = initPage();
            
            try {
                await abortAllTask();
                await reqAPI('resetconfig');
                await prepareVENV();
                
                // 로딩 스피너 제거
                document.body.removeChild(loadingContainer);
                
                // configPage 다시 로드
                chatMessages.innerHTML = '';
                await configPage(
                    chatMessages,
                    translations,
                    reqAPI,
                    getText,
                    showCustomDialog,
                    getModelInformation,
                    aiIndicator,
                    abortAllTask,
                    refreshList,
                    startNewTalk,
                    updateUILanguage
                );
                
                let els = await refreshList();
                if (els[0]) els[0]?.click();
            } catch (error) {
                // 에러 발생 시 로딩 스피너 제거
                if (document.body.contains(loadingContainer)) {
                    document.body.removeChild(loadingContainer);
                }
                console.error('Reset configuration failed:', error);
            }
        }
    }
    
    async function clearSessions() {
        const result = await showCustomDialog({
            title: getText('dialog', 'clearSessionsTitle'),
            message: getText('dialog', 'clearSessionsMessage'),
            okText: getText('dialog', 'clearSessionsOk'),
            cancelText: getText('dialog', 'clearSessionsCancel')
        });
    
        if (result) {
            try {
                await abortAllTask();
                await reqAPI('clearSessionLogs');
                await startNewTalk();
                await refreshList();
            } catch (error) {
                await reqAPI('errnotify', getText('settings', 'clearSessionError'));
            }
        }
    }

    return configContainer;
}

// 모델 정보 관련 함수 수정
export async function getModelInformation(vendorKey, reqAPI) {
    const provider = AI_PROVIDERS[vendorKey];
    if (!provider) {
        throw new Error(`Unknown vendor: ${vendorKey}`);
    }

    if (vendorKey === 'ollama') {
        try {
            const modelList = await reqAPI('ollamamodellist');
            return {
                keyname: provider.modelVar,
                modelList: modelList || []
            };
        } catch (error) {
            throw new Error('Unable to connect to Ollama service. Please make sure the service is running and try again.');
        }
    }

    return {
        keyname: provider.modelVar,
        modelList: provider.models
    };
}

// AI 모델 표시 관련 함수
export async function aiIndicator(load = false, reqAPI, poweredby) {
    poweredby.innerHTML = [
        `<div style="text-align:center;padding:5px;">`,
        `<span style="opacity:0.5;font-size:0.8em;">Model: </span>`,
        `<span style="font-size:0.8em;opacity:0;">checking</span>`,
        `</div>`,
    ].join('');
    
    try {
        const vendorName = await reqAPI('getconfig', { key: 'USE_LLM' }) || 'openai';  // 기본값으로 openai 설정
        if (vendorName && !load) {
            // 벤더가 선택되었지만 load가 false인 경우에도 모델 정보를 표시
            let modelName = await getModelNamed(reqAPI, vendorName);
            if (!modelName) {
                // 모델이 선택되지 않은 경우 기본 모델 설정
                const modelInfo = await getModelInformation(vendorName, reqAPI);
                if (modelInfo && modelInfo.modelList.length > 0) {
                    modelName = modelInfo.modelList[0];
                    await reqAPI('setconfig', { key: modelInfo.keyname, value: modelName });
                }
            }
            
            if (modelName) {
                updateModelDisplay(poweredby, modelName);
                return;
            }
        }
        
        if (!load) return;

        let modelName = await getModelNamed(reqAPI, vendorName);
        if (!modelName) {
            // 모델이 선택되지 않은 경우 기본 모델 설정
            const modelInfo = await getModelInformation(vendorName, reqAPI);
            if (modelInfo && modelInfo.modelList.length > 0) {
                modelName = modelInfo.modelList[0];
                await reqAPI('setconfig', { key: modelInfo.keyname, value: modelName });
            }
        }
        
        if (!modelName) {
            console.log('aiIndicator - No model configured');
            return;
        }
        
        updateModelDisplay(poweredby, modelName);
    } catch (error) {
        console.error('Error in aiIndicator:', error);
        poweredby.innerHTML = [
            `<div style="text-align:center;padding:5px;">`,
            `<span style="opacity:0.5;font-size:0.8em;">Model: </span>`,
            `<span style="font-size:0.8em;color:var(--error-color);">Error</span>`,
            `</div>`,
        ].join('');
    }
}

// 모델 표시 업데이트 함수
function updateModelDisplay(poweredby, modelName) {
    poweredby.style.cursor = 'pointer';
    poweredby.innerHTML = [
        `<div style="text-align:center;padding:5px;" class="model-indicator">`,
        `<span style="opacity:0.5;font-size:0.8em;">Model: </span>`,
        `<span style="font-size:0.8em;">${modelName}</span>`,
        `</div>`,
    ].join('');

    // 모델 표시 클릭 이벤트 리스너 추가
    const modelIndicator = poweredby.querySelector('.model-indicator');
    if (modelIndicator) {
        modelIndicator.addEventListener('click', () => {
            const settingsMenu = document.querySelector('#majore li[data-menu="settings"]');
            if (settingsMenu) {
                settingsMenu.click();
            }
        });
    }
}

// Ollama 모델 캐시
let ollamaModelCache = null;
let ollamaModelCacheTime = 0;
const CACHE_DURATION = 30000; // 30초 캐시

// 모델 이름 가져오기
async function getModelNamed(reqAPI, vendorName) {
    let modelName = '';
    
    // Ollama의 경우 모델 리스트에서 선택된 모델 확인
    if (vendorName === 'ollama') {
        try {
            // 캐시된 모델 리스트 사용 또는 새로 가져오기
            let ollamaModels;
            const now = Date.now();
            if (ollamaModelCache && (now - ollamaModelCacheTime) < CACHE_DURATION) {
                ollamaModels = ollamaModelCache;
            } else {
                ollamaModels = await reqAPI('ollamamodellist');
                ollamaModelCache = ollamaModels;
                ollamaModelCacheTime = now;
            }
            
            modelName = await reqAPI('getconfig', { key: 'OLLAMA_MODEL' });
            
            // 모델 이름에서 태그 제거
            const cleanModelName = modelName?.split(':')[0];
            const cleanOllamaModels = ollamaModels.map(m => m.split(':')[0]);
            
            // 선택된 모델이 실제 존재하는지 확인
            if (cleanModelName && cleanOllamaModels.includes(cleanModelName)) {
                return `${cleanModelName}:latest`;
            }
            
            // 모델이 없는 경우 첫 번째 사용 가능한 모델을 자동 선택
            if (ollamaModels.length > 0) {
                const firstModel = ollamaModels[0];
                await reqAPI('setconfig', { key: 'OLLAMA_MODEL', value: firstModel });
                return firstModel;
            }
            
            return null;
        } catch (error) {
            console.error('Error getting Ollama model:', error);
            return null;
        }
    }
    
    // 다른 벤더들의 경우 기본 모델 설정
    const modelConfig = {
        'openai': { key: 'OPENAI_MODEL', default: 'gpt-4o-mini' },
        'groq': { key: 'GROQ_MODEL', default: 'meta-llama/llama-4-maverick-17b-128e-instruct' },
        'gemini': { key: 'GEMINI_MODEL', default: 'gemini-2.5-flash' },
        'anthropic': { key: 'ANTHROPIC_MODEL', default: 'claude-3-5-sonnet-20241022' }
    };

    const config = modelConfig[vendorName];
    if (config) {
        modelName = await reqAPI('getconfig', { key: config.key });
        if (!modelName) {
            // 모델이 설정되지 않은 경우 기본값 설정
            modelName = config.default;
            await reqAPI('setconfig', { key: config.key, value: modelName });
        }
    }
    
    return modelName;
} 