// Using PostgreSQL with pgvector for vector database operations
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import { getVarVal } from '../../configuration.js';
import os from 'os';
import { spawn } from 'child_process';
import { pipeline } from '@xenova/transformers';
import chalk from 'chalk';
import axios from 'axios';
import { debugLog, logger } from '../../utils/logger.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// 프로젝트 루트 디렉토리 찾기 (plugins/rag 상위 디렉토리의 상위 디렉토리)
const projectRoot = path.resolve(__dirname, '../..');

// 캐시 디렉토리 설정
const CACHE_DIR = path.join(os.homedir(), '.airun', 'cache');

// Electron 환경 체크 함수
function isElectron() {
    return typeof process !== 'undefined' && 
           typeof process.versions !== 'undefined' && 
           typeof process.versions.electron !== 'undefined';
}

class RAGCache {
    constructor() {
        this.cachePath = path.join(os.homedir(), '.airun', 'rag_cache.json');
        this.maxSize = 1000;
        this.ttl = 72 * 60 * 60 * 1000; // 3일
        this.cache = new Map();
        this.collectionInfoCache = null;
        
        // NO_CACHE 옵션이 활성화되지 않은 경우에만 캐시 로드
        if (!global.NO_CACHE) {
            this.loadCache();
        }
    }

    loadCache() {
        try {
            if (fs.existsSync(this.cachePath)) {
                const data = JSON.parse(fs.readFileSync(this.cachePath, 'utf8'));
                const cache = new Map();
                for (const [key, item] of Object.entries(data)) {
                    cache.set(key, {
                        data: item.data,
                        timestamp: item.timestamp,
                        collectionInfo: item.collectionInfo
                    });
                }
                // console.log(`RAG 캐시 로드 완료: ${cache.size}개 항목`);
                this.cache = cache;
            }
        } catch (error) {
            console.error('RAG cache load failed:', error);
        }
    }

    saveCache() {
        try {
            const cacheDir = path.dirname(this.cachePath);
            if (!fs.existsSync(cacheDir)) {
                fs.mkdirSync(cacheDir, { recursive: true });
            }
            const data = Object.fromEntries(this.cache);
            fs.writeFileSync(this.cachePath, JSON.stringify(data), 'utf8');
            // console.log(`[DEBUG] RAG cache saved to: ${this.cachePath}`);
        } catch (error) {
            console.error('✗ RAG cache save failed:', error);
        }
    }

    generateKey(query, userId = null) {
        // 쿼리 정규화: 소문자 변환, 공백 정리, 특수문자 제거
        const normalizedQuery = query.toLowerCase()
            .trim()
            .replace(/\s+/g, ' ')
            .replace(/[?!.,]/g, '')
            .replace(/[은는이가을를의]/g, '');
        
        // 사용자 ID가 있는 경우 캐시 키에 추가
        if (userId) {
            return `${normalizedQuery}_${userId}`;
        }
        
        return normalizedQuery;
    }

    get(query, currentCollectionInfo, userId = null) {
        const key = this.generateKey(query, userId);
        const item = this.cache.get(key);
        
        if (!item) return null;
        
        // TTL 체크
        if (Date.now() - item.timestamp > this.ttl) {
            this.cache.delete(key);
            this.saveCache();
            return null;
        }

        // 컬렉션 정보 캐시 사용
        if (this.collectionInfoCache && 
            JSON.stringify(this.collectionInfoCache) === JSON.stringify(currentCollectionInfo)) {
            return item.data;
        }

        // 문서 컬렉션이 변경되었는지 확인
        if (JSON.stringify(item.collectionInfo) !== JSON.stringify(currentCollectionInfo)) {
            this.cache.delete(key);
            this.saveCache();
            return null;
        }
        
        // 컬렉션 정보 캐시 업데이트
        this.collectionInfoCache = currentCollectionInfo;
        
        return item.data;
    }

    set(query, results, collectionInfo, userId = null) {
        const key = this.generateKey(query, userId);
        
        // 캐시 크기 제한 관리
        if (this.cache.size >= this.maxSize) {
            const oldestKey = Array.from(this.cache.entries())
                .sort(([,a], [,b]) => a.timestamp - b.timestamp)[0][0];
            this.cache.delete(oldestKey);
        }
        
        this.cache.set(key, {
            data: results,
            timestamp: Date.now(),
            collectionInfo: collectionInfo
        });
        
        // 컬렉션 정보 캐시 업데이트
        this.collectionInfoCache = collectionInfo;
        
        this.saveCache();
    }

    clear() {
        this.cache.clear();
        this.saveCache();
    }
}

class EmbeddingClient {
    constructor() {
        this.baseUrl = 'http://localhost:5600';
        this.isAvailable = false;
        this.isChecked = false;
        // 생성자에서 checkAvailability 호출하지 않음
    }

    async checkAvailability() {
        // try {
        //     if (typeof logger !== 'undefined') {
        //         logger.debug('🔍 임베딩 서비스 가용성 확인 시작');
        //     }
        // } catch (error) {
        //     // logger 사용 불가 시 console.log 사용
        //     console.log('🔍 임베딩 서비스 가용성 확인 시작');
        // }
        
        try {
            // /health 엔드포인트가 빈 응답을 반환하므로 /embed_search 엔드포인트를 테스트
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 5000); // 5초 타임아웃

            const response = await fetch(`${this.baseUrl}/embed_search`, {
                method: 'POST',
                signal: controller.signal,
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({ text: 'test' })
            }).finally(() => clearTimeout(timeoutId));

            if (!response.ok) {
                try {
                    if (typeof logger !== 'undefined') {
                        logger.warn(`임베딩 서비스가 비정상 상태로 응답했습니다. (상태 코드: ${response.status})`);
                    }
                } catch (error) {
                    console.warn(`임베딩 서비스가 비정상 상태로 응답했습니다. (상태 코드: ${response.status})`);
                }
                this.isAvailable = false;
                this.isChecked = true;
                return false;
            }

            // 응답을 JSON으로 파싱
            const result = await response.json();
            
            // 성공적인 임베딩 응답인지 확인
            if (result.success && result.embedding && Array.isArray(result.embedding)) {
                try {
                    if (typeof logger !== 'undefined') {
                        logger.debug('임베딩 서비스 사용 가능 확인됨');
                    }
                } catch (error) {
                    console.log('임베딩 서비스 사용 가능 확인됨');
                }
                this.isAvailable = true;
                this.isChecked = true;
                return true;
            } else {
                try {
                    if (typeof logger !== 'undefined') {
                        logger.warn('임베딩 서비스 응답 형식이 올바르지 않습니다:', result);
                    }
                } catch (error) {
                    console.warn('임베딩 서비스 응답 형식이 올바르지 않습니다:', result);
                }
                this.isAvailable = false;
                this.isChecked = true;
                return false;
            }

        } catch (error) {
            try {
                if (typeof logger !== 'undefined') {
                    logger.error('임베딩 서비스 확인 중 오류 발생:', error.message);
                }
            } catch (loggerError) {
                console.error('임베딩 서비스 확인 중 오류 발생:', error.message);
            }
            this.isAvailable = false;
            this.isChecked = true;
            return false;  // 에러를 던지지 않고 false 반환
        }
    }

    async getEmbedding(text) {
        try {
            if (!this.isAvailable) {
                throw new Error('Embedding service not available');
            }

            const response = await fetch(`${this.baseUrl}/embed`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ text })
            });

            if (!response.ok) {
                throw new Error('Failed to get embedding');
            }

            const result = await response.json();
            if (!result.success) {
                throw new Error(result.error || 'Unknown error');
            }

            return result.embedding;
        } catch (error) {
            console.log('임베딩 생성 중 오류:', error.message);
            throw error;  // 상위에서 처리하도록 에러를 전파
        }
    }

    async getSearchEmbedding(text) {
        try {
            // 검색 전용 엔드포인트 사용 (블로킹 방지)
            const response = await fetch(`${this.baseUrl}/embed_search`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ text })
            });

            if (!response.ok) {
                // 검색 전용 엔드포인트가 실패하면 기본 엔드포인트로 폴백
                console.log('검색 전용 임베딩 엔드포인트 실패, 기본 엔드포인트 사용');
                return await this.getEmbedding(text);
            }

            const result = await response.json();
            if (!result.success) {
                // 검색 전용 엔드포인트가 실패하면 기본 엔드포인트로 폴백
                console.log('검색 전용 임베딩 결과 실패, 기본 엔드포인트 사용');
                return await this.getEmbedding(text);
            }

            return result.embedding;
        } catch (error) {
            console.log('검색 전용 임베딩 생성 중 오류, 기본 엔드포인트 사용:', error.message);
            // 에러 발생 시 기본 엔드포인트로 폴백
            return await this.getEmbedding(text);
        }
    }
}

export class RAGService {
    static instance = null;
    static electronInstance = null;
    static initialized = false;
    static electronInitialized = false;
    static cache = new RAGCache();
    static collectionInfo = null;
    static isEnabled = false;  // RAG 활성화 상태 추적
    static embeddingCache = null;  // 임베딩 모델 캐시
    static embeddingClient = new EmbeddingClient();
    static embeddingServiceUrl = 'http://localhost:5600';  // 임베딩 서비스 URL 추가

    constructor() {
        this.embeddingServiceUrl = RAGService.embeddingServiceUrl;
    }

    // RAG 활성화 상태 확인
    static async checkIfEnabled(forceEnabled) {
        // forceEnabled가 명시적으로 전달된 경우 해당 값을 사용
        if (forceEnabled !== undefined) {
            RAGService.isEnabled = forceEnabled;
            return forceEnabled;
        }

        try {
            const useRAG = await getVarVal('USE_RAG');
            if (!useRAG) return false;
            
            const value = typeof useRAG === 'string' ? useRAG.toLowerCase() : useRAG;
            RAGService.isEnabled = value === 'true' || value === true || value === 'yes';
            return RAGService.isEnabled;
        } catch (error) {
            console.error('Error checking RAG status:', error);
            return false;
        }
    }

    static async getInstance() {
        if (!RAGService.instance) {
            RAGService.instance = new RAGService();
        }
        return RAGService.instance;
    }

    static async getElectronInstance() {
        if (!RAGService.electronInstance) {
            RAGService.electronInstance = new RAGService();
        }
        return RAGService.electronInstance;
    }

    // 로그 출력 함수 추가
    static log(message) {
        console.log(`[JS-RAG] ${message}`);
    }

    static error(message, error = null) {
        console.error(`[JS-RAG] ${message}`, error || '');
    }

    // PostgreSQL 기반 임베딩 서비스 초기화 메소드
    static async initializeEmbeddingOnly() {
        const isElectronEnv = isElectron();
        
        // 이미 초기화된 경우 임베딩 서비스 연결만 확인
        if ((isElectronEnv && RAGService.electronInitialized) || 
            (!isElectronEnv && RAGService.initialized)) {
            try {
                await RAGService.embeddingClient.checkAvailability();
                return;
            } catch (error) {
                if (isElectronEnv) {
                    RAGService.electronInitialized = false;
                } else {
                    RAGService.initialized = false;
                }
            }
        }
        
        try {
            // 임베딩 서비스 연결 확인
            const isAvailable = await RAGService.embeddingClient.checkAvailability();
            if (!isAvailable) {
                throw new Error('임베딩 서비스에 연결할 수 없습니다.');
            }
            
            if (isElectronEnv) {
                RAGService.electronInitialized = true;
            } else {
                RAGService.initialized = true;
            }
            
        } catch (error) {
            RAGService.error('Error in RAG search initialization:', error);
            throw error;
        }
    }

    // 서버 시작 시 호출되는 초기화 메서드
    static async initializeOnStartup() {
        try {
            if (!RAGService.instance) {
                RAGService.instance = new RAGService();
            }
            
            // 이미 초기화되었는지 확인
            if (RAGService.initialized) {
                try {
                    await RAGService.embeddingClient.checkAvailability();
                    RAGService.log('RAG Service already initialized');
                    return true;
                } catch (error) {
                    RAGService.initialized = false;
                }
            }
            
            // PostgreSQL 기반 임베딩 서비스 초기화
            await RAGService.initializeEmbeddingOnly();
            
            RAGService.log('RAG Service initialized successfully on startup');
            return true;
        } catch (error) {
            RAGService.error('Failed to initialize RAG Service on startup:', error);
            console.error('Connection check failed:', error);
            return false;
        }
    }

    async ensureInitialized() {
        try {
            // 이미 초기화된 경우 임베딩 서비스 연결만 확인
            if (RAGService.initialized) {
                try {
                    await RAGService.embeddingClient.checkAvailability();
                    return;
                } catch (error) {
                    RAGService.initialized = false;
                }
            }
            
            // 임베딩 서비스 연결 확인
            const isAvailable = await RAGService.embeddingClient.checkAvailability();
            if (!isAvailable) {
                throw new Error('임베딩 서비스에 연결할 수 없습니다.');
            }
            
            RAGService.initialized = true;
            
        } catch (error) {
            console.error('Error in RAG initialization:', error);
            throw error;
        }
    }

    async query(text, userId) {
        await this.ensureInitialized();

        try {
            // 임베딩 서비스 사용 가능 여부 확인
            const useEmbeddingService = await RAGService.embeddingClient.checkAvailability();
            if (!useEmbeddingService) {
                throw new Error('임베딩 서비스에 연결할 수 없습니다.');
            }

            // 임베딩 생성
            const embedding = await RAGService.embeddingClient.getSearchEmbedding(text);
            
            const searchPayload = {
                embedding: embedding,
                query: text,
                user_id: String(userId || 'all')
            };

            // PostgreSQL 기반 검색 실행
            const searchResponse = await axios.post(`${RAGService.embeddingServiceUrl}/search`, searchPayload);
            
            if (!searchResponse.data || !searchResponse.data.results) {
                return [];
            }
            
            return searchResponse.data.results;

        } catch (error) {
            console.error('PostgreSQL 검색 오류:', error);
            throw error;
        }
    }

    static async searchDocuments(query, userId, tempSettings = null, ragSearchScope = 'personal') {
        const startTime = Date.now();  // 시작 시간 기록
        try {
            // query 파라미터 검증
            if (!query || typeof query !== 'string') {
                throw new Error('검색어가 유효하지 않습니다.');
            }

            // RAGService 인스턴스 먼저 가져오기
            const ragService = await RAGService.getInstance();
            if (!ragService) return null;

            // userId 안전 처리 - 간단하게 처리
            let effectiveUserId = userId;
            
            // null, undefined, 빈 문자열 처리
            if (effectiveUserId === null || effectiveUserId === undefined || effectiveUserId === '') {
                effectiveUserId = 'all';
            } else {
                // 문자열로 변환 (정수형이 들어올 경우 대비)
                effectiveUserId = String(effectiveUserId);
                
                // 특수문자 정리 (안전한 filename/identifier로 변환)
                effectiveUserId = effectiveUserId.trim().replace(/[^\w\-_]/g, '_');
            }
            
            // try {
            //     logger.info(`최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // } catch (error) {
            //     console.log(`최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // }
            
            // 프롬프트 전처리
            const cleanQuery = query.replace(/^airun\s*/, '').trim();
            const actualQuery = cleanQuery.match(/"([^"]+)"/) ?
                cleanQuery.match(/"([^"]+)"/)[1] : 
                cleanQuery;

            // 캐시 키 생성
            const cacheKey = `search:${actualQuery}:${effectiveUserId}`;
            
            // 충분히 빠른 속도이기 때문에 캐시를 사용하지 않음 - 2025-06-22 Kevin Kim
            // 캐시 확인
            // const cachedResult = RAGService.cache.get(cacheKey);
            // if (cachedResult) {
            //     console.log('캐시된 검색 결과를 사용합니다.');
            //     return cachedResult;
            // }

            // 임베딩 서비스 사용 가능 여부 확인 생략 (RAG 서비스가 내부적으로 임베딩 처리)
            const useEmbeddingService = true;
            // try {
            //     if (typeof logger !== 'undefined') {
            //         logger.debug('🔍 임베딩 서비스 사용 가능:', useEmbeddingService);
            //     }
            // } catch (error) {
            //     console.log('🔍 임베딩 서비스 사용 가능:', useEmbeddingService);
            // }
            
            let searchResults;
            
                            if (useEmbeddingService) {
                try {
                    try {
                        if (typeof logger !== 'undefined') {
                            logger.debug('🔍 지식 검색 시작...');
                            logger.debug('- Search method: embedding_service');
                            logger.debug('- User ID:', effectiveUserId);
                        }
                    } catch (error) {
                        console.log('🔍 지식 검색 시작...');
                        console.log('- Search method: embedding_service');
                        console.log('- User ID:', effectiveUserId);
                    }
                    // 임베딩을 제거하고 쿼리만 사용하여 RAG 서비스가 내부적으로 임베딩을 생성하도록 함
                    const searchPayload = {
                        query: actualQuery,
                        user_id: String(effectiveUserId),
                        tempSettings: tempSettings,
                        ragSearchScope: ragSearchScope
                    };

                    // 2. 임베딩을 사용하여 문서 검색
                    const searchStartTime = Date.now();
                    const searchResponse = await axios.post(`${RAGService.embeddingServiceUrl}/search`, searchPayload);
                    const searchEndTime = Date.now();
                    
                    const endTime = Date.now();
                    try {
                        if (typeof logger !== 'undefined') {
                            logger.debug('- Search time:', `${((searchEndTime - searchStartTime) / 1000).toFixed(3)}s`);
                            logger.debug('- Total execution time:', `${((endTime - startTime) / 1000).toFixed(3)}s`);
                        }
                    } catch (error) {
                        console.log('- Search time:', `${((searchEndTime - searchStartTime) / 1000).toFixed(3)}s`);
                        console.log('- Total execution time:', `${((endTime - startTime) / 1000).toFixed(3)}s`);
                    }

                    if (!searchResponse.data || !searchResponse.data.results) {
                        try {
                            if (typeof logger !== 'undefined') {
                                logger.debug('검색 결과가 없습니다.');
                            }
                        } catch (error) {
                            console.log('검색 결과가 없습니다.');
                        }
                        searchResults = {
                            documents: [[]],
                            metadatas: [[]],
                            distances: [[]]
                        };
                    } else {
                        const results = searchResponse.data.results;
                        try {
                            if (typeof logger !== 'undefined') {
                                logger.debug('Search results:', results.length);
                            }
                        } catch (error) {
                            console.log('Search results:', results.length);
                        }

                        // 검색 결과 포맷팅
                        searchResults = {
                            documents: [results.map(r => r.content)],
                            metadatas: [results.map(r => ({
                                source: r.source || 'Unknown',
                                page_number: r.page_number,
                                chunk_index: r.chunk_index,
                                total_chunks: r.total_chunks,
                                timestamp: r.timestamp,
                                modification_date: r.modification_date,
                                user_id: r.user_id || 'all',
                                final_score: r.final_score || 0,
                                // 새로운 4점수 시스템 (RAG 서비스에서 반환되는 경로로 접근)
                                core_scores: {
                                    keyword_precision: r.core_scores?.keyword_precision || 0,
                                    semantic_relevance: r.core_scores?.semantic_relevance || 0,
                                    content_quality: r.core_scores?.content_quality || 0,
                                    recency: r.core_scores?.recency || 0
                                },
                                // 기존 호환성을 위해 유지
                                main_scores: {
                                    similarity_score: r.main_scores?.similarity_score || 0,
                                    exact_match_score: r.main_scores?.exact_match_score || 0,
                                    relevance_score: r.main_scores?.relevance_score || 0,
                                    date_score: r.main_scores?.date_score || 0
                                },
                                sub_scores: {
                                    structure_score: r.sub_scores?.structure_score || 0,
                                    content_score: r.sub_scores?.content_score || 0,
                                    length_score: r.sub_scores?.length_score || 0,
                                    keyword_quality: r.sub_scores?.keyword_quality || 0,
                                    partial_match_score: r.sub_scores?.partial_match_score || 0
                                },
                                matches: r.matches || {
                                    exact_matches: [],
                                    partial_matches: []
                                },
                                thresholds: r.thresholds || {},
                                main_weights: r.main_weights || {},
                                sub_weights: r.sub_weights || {}
                            }))],
                            distances: [results.map(r => 1 - (r.main_scores?.similarity_score || 0))]
                        };
                    }

                } catch (error) {
                    try {
                        if (typeof logger !== 'undefined') {
                            logger.error('임베딩 서비스 검색 중 오류:', error);
                            logger.debug('기본 검색으로 전환합니다...');
                        }
                    } catch (loggerError) {
                        console.error('임베딩 서비스 검색 중 오류:', error);
                        console.log('기본 검색으로 전환합니다...');
                    }
                }
            }
            
            if (!searchResults) {
            throw new Error('검색 서비스에 연결할 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
        }

        return searchResults;
            
        } catch (error) {
            const endTime = Date.now();
            const duration = (endTime - startTime) / 1000;
            try {
                if (typeof logger !== 'undefined') {
                    logger.debug(`검색 소요 시간: ${duration.toFixed(3)}초 (오류 발생)`);
                    logger.error('Search error:', error);
                }
            } catch (loggerError) {
                console.log(`검색 소요 시간: ${duration.toFixed(3)}초 (오류 발생)`);
                console.error('Search error:', error);
            }
            return {
                documents: [[]],
                metadatas: [[]],
                distances: [[]]
            };
        }
    }

    // 컬렉션 정보 캐시 초기화 메서드 추가
    static clearCollectionInfoCache() {
        RAGService.collectionInfo = null;
        console.log('Collection info cache cleared.');
    }

    // 기존의 clearCache 메서드 수정
    static clearCache() {
        console.log('RAG search cache cleared...');
        this.cache.clear();
        this.clearCollectionInfoCache();
        // PostgreSQL 기반 연결로 초기화 완료
        console.log('Cache cleared.');
    }

    // PostgreSQL 기반 연결 상태 확인 메서드
    static async checkConnection() {
        try {
            return await RAGService.embeddingClient.checkAvailability();
        } catch (error) {
            console.error('PostgreSQL 연결 확인 오류:', error);
            return false;
        }
    }

    static async preCacheRAG() {
        try {
            console.log('Starting RAG cache initialization...');
            
            // Python 스크립트 경로 설정
            const ragPluginPath = path.join(projectRoot, 'plugins', 'rag');
            const scriptPath = path.join(ragPluginPath, 'rag_cache_initializer.py');
            
            // 가상환경 경로 가져오기
            const venvPath = path.join(os.homedir(), '.airun_venv');
            const pythonPath = process.platform === 'win32'
                ? path.join(venvPath, 'Scripts', 'python.exe')
                : path.join(venvPath, 'bin', 'python');

            // 스크립트 실행
            console.log('Executing RAG cache initializer script...');
            
            return new Promise((resolve, reject) => {
                const pythonProcess = spawn(pythonPath, [
                    '-W', 'ignore',  // 모든 경고 무시
                    '-u',  // 버퍼링 비활성화
                    scriptPath
                ], {
                    env: { 
                        ...process.env, 
                        PYTHONUNBUFFERED: '1',
                        PYTHONWARNINGS: 'ignore'  // Python 경고 무시
                    },
                    stdio: ['ignore', 'pipe', 'pipe']
                });

                let errorOutput = '';

                pythonProcess.stdout.on('data', (data) => {
                    process.stdout.write(data);
                });

                pythonProcess.stderr.on('data', (data) => {
                    errorOutput += data;
                    process.stderr.write(data);
                });

                pythonProcess.on('close', (code) => {
                    if (code === 0) {
                        resolve({
                            status: 'success',
                            message: 'RAG cache generation completed successfully'
                        });
                    } else {
                        resolve({
                            status: 'error',
                            message: errorOutput || `Process exited with code ${code}`
                        });
                    }
                });

                pythonProcess.on('error', (error) => {
                    reject(error);
                });
            });
            
        } catch (error) {
            console.error('Error in RAG cache generation:', error);
            return {
                status: 'error',
                message: error.message
            };
        }
    }

    static async listDocuments(userId = null) {
        try {
            // userId 안전 처리 - 정수형 ID가 들어와도 안전하게 처리
            let effectiveUserId = userId;
            
            if (effectiveUserId !== null && effectiveUserId !== undefined && effectiveUserId !== '') {
                // 문자열로 변환 (정수형이 들어올 경우 대비)
                effectiveUserId = String(effectiveUserId);
                
                // 특수문자 정리 (안전한 filename/identifier로 변환)
                effectiveUserId = effectiveUserId.trim().replace(/[^\w\-_]/g, '_');
            }
            
            // try {
            //     logger.info(`[listDocuments] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // } catch (error) {
            //     console.log(`[listDocuments] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // }
            
            // 먼저 임베딩 서비스의 /list 엔드포인트 시도
            try {
                const url = new URL('http://localhost:5600/list');
                if (effectiveUserId) {
                    url.searchParams.append('user_id', effectiveUserId);
                }
                const response = await fetch(url);
                if (response.ok) {
                    const result = await response.json();
                    if (result.status === 'success') {
                        return result.collection;
                    }
                }
            } catch (error) {
                console.log('임베딩 서비스의 문서 목록 조회 실패, 기존 방식으로 전환:', error.message);
            }

            // 임베딩 서비스 사용 실패시 에러 발생
            throw new Error('문서 목록을 가져올 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
            
            if (result.status === 'error') {
                throw new Error(result.message);
            }

            return {
                name: result.collection.name,
                total_chunks: result.collection.total_chunks,
                unique_files: result.collection.unique_files,
                metadata: result.collection.metadata,
                documents: result.collection.documents,
                user_id: result.collection.user_id
            };
        } catch (error) {
            console.error('RAG 문서 목록 조회 실패:', error);
            throw error;
        }
    }

    static async addDocuments(path) {
        try {
            // RAG 서비스 인스턴스 가져오기
            const ragService = await RAGService.getInstance();
            if (!ragService) {
                throw new Error('Failed to initialize RAG service');
            }

            // RAG 문서 경로 가져오기
            const { getVarVal } = await import('../../configuration.js');
            const base_path = await getVarVal('RAG_DOCS_PATH') || path.join(os.homedir(), '.airun', 'rag');

            // 서비스가 실행 중인지 확인
            try {
                const response = await fetch('http://localhost:5600/health', {
                    timeout: 1000  // 1초 타임아웃
                });
                
                if (response.ok) {
                    // 서비스를 통해 문서 추가
                    console.log('Using RAG service for document processing...');
                    const addResponse = await fetch('http://localhost:5600/add', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            path: path,
                            base_path: base_path
                        })
                    });

                    if (!addResponse.ok) {
                        throw new Error(`Service request failed: ${addResponse.statusText}`);
                    }

                    const result = await addResponse.json();
                    return {
                        success: true,
                        data: result
                    };
                }
            } catch (error) {
                throw new Error('RAG 서비스에 연결할 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
            }

            // 문서가 추가된 후 캐시 초기화
            if (result.success) {
                RAGService.clearCollectionInfoCache();
            }

            return {
                success: true,
                data: {
                    status: "success",
                    message: `문서 처리가 완료되었습니다. (처리된 파일: ${result.files_count || 0}개)`,
                    files_count: result.files_count,
                    original_path: result.original_path,
                    rag_docs_path: base_path,
                    summary: result.summary
                }
            };
        } catch (error) {
            console.error('RAG 문서 추가 오류:', error);
            return {
                success: false,
                error: {
                    message: error.message,
                    details: error.stack
                }
            };
        }
    }

    static async addDocumentsWithUserId(path, userId) {
        try {
            // userId 안전 처리 - 간단하게 처리
            let effectiveUserId = userId;
            
            if (effectiveUserId !== null && effectiveUserId !== undefined && effectiveUserId !== '') {
                // 문자열로 변환 (정수형이 들어올 경우 대비)
                effectiveUserId = String(effectiveUserId);
                
                // 특수문자 정리 (안전한 filename/identifier로 변환)
                effectiveUserId = effectiveUserId.trim().replace(/[^\w\-_]/g, '_');
            }
            
            // try {
            //     logger.info(`[addDocumentsWithUserId] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // } catch (error) {
            //     console.log(`[addDocumentsWithUserId] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // }

            // RAG 서비스 인스턴스 가져오기
            const ragService = await RAGService.getInstance();
            if (!ragService) {
                throw new Error('Failed to initialize RAG service');
            }

            // RAG 문서 경로 가져오기
            const { getVarVal } = await import('../../configuration.js');
            const base_path = await getVarVal('RAG_DOCS_PATH') || path.join(os.homedir(), '.airun', 'rag');

            // 서비스가 실행 중인지 확인
            try {
                const response = await fetch('http://localhost:5600/health', {
                    timeout: 1000  // 1초 타임아웃
                });
                
                if (response.ok) {
                    // 서비스를 통해 문서 추가
                    console.log('Using RAG service for document processing...');
                    const addResponse = await fetch('http://localhost:5600/add', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            path: path,
                            user_id: effectiveUserId || '',
                            base_path: base_path
                        })
                    });

                    if (!addResponse.ok) {
                        throw new Error(`Service request failed: ${addResponse.statusText}`);
                    }

                    const result = await addResponse.json();
                    return {
                        success: true,
                        data: result
                    };
                }
            } catch (error) {
                throw new Error('RAG 서비스에 연결할 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
            }

            // 문서가 추가된 후 캐시 초기화
            if (result.success) {
                RAGService.clearCollectionInfoCache();
            }

            return {
                success: true,
                data: {
                    status: "success",
                    message: `문서 처리가 완료되었습니다. (처리된 파일: ${result.files_count || 0}개)`,
                    files_count: result.files_count,
                    original_path: result.original_path,
                    rag_docs_path: base_path,
                    user_id: effectiveUserId || '',  // undefined인 경우 빈 문자열로 변환
                    summary: result.summary
                }
            };
        } catch (error) {
            console.error('RAG 문서 추가 오류:', error);
            return {
                success: false,
                error: {
                    message: error.message,
                    details: error.stack
                }
            };
        }
    }

    static async setDocumentsPath(path) {
        try {
            // 경로 유효성 검사
            if (!path || typeof path !== 'string') {
                throw new Error('유효하지 않은 문서 경로입니다.');
            }

            // 경로가 존재하는지 확인
            if (!fs.existsSync(path)) {
                throw new Error('지정된 경로가 존재하지 않습니다.');
            }

            // 경로가 디렉토리인지 확인
            const stats = fs.statSync(path);
            if (!stats.isDirectory()) {
                throw new Error('지정된 경로는 디렉토리가 아닙니다.');
            }

            // 설정에 경로 저장
            const { setVarVal } = await import('../../configuration.js');
            await setVarVal('RAG_DOCS_PATH', path, true);

            return {
                success: true,
                data: {
                    path,
                    message: 'RAG 문서 경로가 성공적으로 설정되었습니다.'
                }
            };
        } catch (error) {
            console.error('RAG 문서 경로 설정 오류:', error);
            return {
                success: false,
                error: {
                    message: error.message
                }
            };
        }
    }

    static async removeDocument(filePath, userId = null) {
        try {
            // userId 안전 처리 - 간단하게 처리
            let effectiveUserId = userId;
            
            if (effectiveUserId !== null && effectiveUserId !== undefined && effectiveUserId !== '') {
                // 문자열로 변환 (정수형이 들어올 경우 대비)
                effectiveUserId = String(effectiveUserId);
                
                // 특수문자 정리 (안전한 filename/identifier로 변환)
                effectiveUserId = effectiveUserId.trim().replace(/[^\w\-_]/g, '_');
            }
            
            // try {
            //     logger.info(`[removeDocument] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // } catch (error) {
            //     console.log(`[removeDocument] 최종 사용자 ID 설정: ${userId} -> ${effectiveUserId}`);
            // }

            // RAG 서비스 인스턴스 가져오기
            const ragService = await RAGService.getInstance();
            if (!ragService) {
                throw new Error('Failed to initialize RAG service');
            }

            // 서비스가 실행 중인지 확인
            try {
                const response = await fetch('http://localhost:5600/health', {
                    timeout: 1000  // 1초 타임아웃
                });
                
                if (!response.ok) {
                    throw new Error('RAG service health check failed');
                }

                // 서비스를 통해 문서 삭제
                console.log('Using RAG service for document deletion...');
                const deleteResponse = await fetch('http://localhost:5600/delete', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        file_paths: Array.isArray(filePath) ? filePath : [filePath],
                        user_id: effectiveUserId
                    })
                });

                if (!deleteResponse.ok) {
                    throw new Error(`RAG service delete request failed: ${deleteResponse.statusText}`);
                }

                const result = await deleteResponse.json();
                return result;  // 서버 응답을 그대로 반환

            } catch (error) {
                console.error('RAG service delete error:', error);
                // 네트워크 오류인지 응답 오류인지 구분
                if (error.message.includes('Failed to fetch') || error.message.includes('ECONNREFUSED')) {
                    throw new Error('RAG 서비스에 연결할 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
                } else {
                    // 다른 오류는 그대로 전달
                    throw error;
                }
            }
        } catch (error) {
            console.error('문서 삭제 중 오류:', error);
            throw error;
        }
    }
    
    // 문서별 처리 상태 반환
    static async getProcessingStatus() {
        try {
            const response = await fetch('http://localhost:5600/status');
            if (response.ok) {
                const result = await response.json();
                return result;
            }
            throw new Error('Processing status unavailable');
        } catch (error) {
            console.error('처리 상태 조회 오류:', error);
            throw new Error('RAG 서비스에 연결할 수 없습니다. PostgreSQL 기반 임베딩 서비스를 확인하세요.');
        }
    }
}

// 지원되는 확장자 목록을 가져오는 함수 추가
export async function getSupportedExtensions() {
    try {
        // rag_process.py에서 정의된 SUPPORTED_EXTENSIONS 사용
        return [
            // 텍스트 기반 파일
            '.txt', '.md', '.text', '.markdown', '.mdx',
            '.rst', '.adoc', '.org',
            
            // 문서 파일
            '.pdf',
            '.doc', '.docx',
            '.hwp', '.hwpx',
            '.ppt', '.pptx',
            '.xls', '.xlsx',
            
            // 프로그래밍 언어 파일
            '.py',                     // Python
            '.js', '.jsx',            // JavaScript
            '.ts', '.tsx',            // TypeScript
            '.java',                  // Java
            '.cpp', '.c', '.h',       // C/C++
            '.cs',                    // C#
            '.php',                   // PHP
            '.rb',                    // Ruby
            '.swift',                 // Swift
            '.go',                    // Go
            '.rs',                    // Rust
            '.scala',                 // Scala
            '.kt',                    // Kotlin
            '.m',                     // Objective-C
            
            // 새로 추가된 파일 유형
            '.proto',                 // Protocol Buffers
            '.tex',                   // LaTeX
            '.html', '.htm',          // HTML
            '.sol',                   // Solidity
            '.cob', '.cbl',           // COBOL
            '.lua',                   // Lua
            '.pl', '.pm',             // Perl
            '.hs',                    // Haskell
            '.ex', '.exs',            // Elixir
            '.ps1', '.psm1',          // PowerShell
            '.sql'                    // SQL
        ];
    } catch (error) {
        console.error('Error getting supported extensions:', error);
        return [];
    }
}