# AI.RUN Client Setup Script for Windows
# PowerShell 스크립트 실행 정책 설정이 필요할 수 있습니다:
# Set-ExecutionPolicy RemoteSigned -Scope CurrentUser

# 기본 설정
$API_SERVER = if ($env:API_SERVER) { $env:API_SERVER } else { "http://localhost:5500" }
$AIRUN_HOME = "$env:USERPROFILE\.airun"
$VENV_PATH = "$env:USERPROFILE\.airun_venv"
$CONFIG_FILE = "$AIRUN_HOME\airun.conf.bat"

Write-Host "AI.RUN Client Environment Setup" -ForegroundColor Green

# API 서버 연결 확인 및 설정 가져오기
function Check-ApiServer {
    Write-Host "API 서버 연결을 확인합니다..." -ForegroundColor Green
    
    try {
        $configResponse = Invoke-RestMethod -Uri "$API_SERVER/api/v1/client/config" -Method Get
        $script:SMTP_HOST = $configResponse.data.smtp.host
        $script:SMTP_PORT = $configResponse.data.smtp.port
        $script:UTILS_VERSION = $configResponse.data.utils_version
        Write-Host "API 서버 연결 성공" -ForegroundColor Green
    }
    catch {
        Write-Host "API 서버에 연결할 수 없습니다." -ForegroundColor Red
        Write-Host "API_SERVER 환경변수를 설정하여 서버 주소를 지정할 수 있습니다:" -ForegroundColor Yellow
        Write-Host '$env:API_SERVER = "http://your-server:port"'
        exit 1
    }
}

# Python 버전 체크
function Check-Python {
    Write-Host "Python 환경을 확인합니다..." -ForegroundColor Green
    
    try {
        $pythonVersion = python -c "import sys; print('.'.join(map(str, sys.version_info[:3])))"
        
        # API 서버에서 받은 Python 버전 정보 사용
        $configResponse = Invoke-RestMethod -Uri "$API_SERVER/api/v1/client/config" -Method Get
        $minVersion = $configResponse.data.python.min_version
        $recommendedVersion = $configResponse.data.python.recommended_version
        
        if ([version]$pythonVersion -lt [version]$minVersion) {
            Write-Host "Python 버전이 너무 낮습니다." -ForegroundColor Red
            Write-Host "최소 버전: $minVersion" -ForegroundColor Red
            Write-Host "현재 버전: $pythonVersion" -ForegroundColor Red
            Write-Host "권장 버전: $recommendedVersion" -ForegroundColor Yellow
            exit 1
        }
        
        if ([version]$pythonVersion -lt [version]$recommendedVersion) {
            Write-Host "경고: 현재 Python 버전($pythonVersion)이 권장 버전($recommendedVersion)보다 낮습니다." -ForegroundColor Yellow
        }
        
        Write-Host "Python 버전 확인 완료: $pythonVersion" -ForegroundColor Green
    }
    catch {
        Write-Host "Python이 설치되어 있지 않습니다. Python을 먼저 설치해 주세요." -ForegroundColor Red
        Write-Host "https://www.python.org/downloads/" -ForegroundColor Yellow
        exit 1
    }
}

# 디렉토리 구조 생성
function Create-Directories {
    Write-Host "디렉토리 구조를 생성합니다..." -ForegroundColor Green
    
    New-Item -ItemType Directory -Force -Path $AIRUN_HOME | Out-Null
    New-Item -ItemType Directory -Force -Path "$AIRUN_HOME\cache" | Out-Null
    New-Item -ItemType Directory -Force -Path "$AIRUN_HOME\logs" | Out-Null
    
    Write-Host "디렉토리 생성 완료" -ForegroundColor Green
}

# 가상환경 설정
function Setup-Venv {
    Write-Host "Python 가상환경을 설정합니다..." -ForegroundColor Green
    
    if (Test-Path $VENV_PATH) {
        Write-Host "기존 가상환경을 제거하고 새로 생성합니다..." -ForegroundColor Yellow
        Remove-Item -Recurse -Force $VENV_PATH
    }
    
    python -m venv $VENV_PATH
    & "$VENV_PATH\Scripts\Activate.ps1"
    python -m pip install --upgrade pip
    Write-Host "가상환경 설정 완료" -ForegroundColor Green
}

# 필수 패키지 설치
function Install-Packages {
    Write-Host "필수 패키지를 설치합니다..." -ForegroundColor Green
    
    $packages = @(
        "numpy==1.24.4",
        "pandas",
        "matplotlib",
        "requests",
        "pillow",
        "reportlab",
        "aiohttp",
        "sqlalchemy",
        "fpdf",
        "cairosvg"
    )
    
    foreach ($package in $packages) {
        pip install $package
        if ($LASTEXITCODE -ne 0) {
            Write-Host "패키지 설치에 실패했습니다: $package" -ForegroundColor Red
            exit 1
        }
    }
    
    Write-Host "패키지 설치 완료" -ForegroundColor Green
}

# utils 패키지 설치
function Install-Utils {
    Write-Host "utils 패키지를 설치합니다..." -ForegroundColor Green
    
    # Python 버전 정보 가져오기
    $pythonVersion = python -c "import sys; print(f'{sys.version_info.major}{sys.version_info.minor}')"
    
    # Windows용 파일명
    $utilsFilename = "utils.cp${pythonVersion}-win_amd64.pyd"
    
    # site-packages 경로 확인
    $sitePackages = python -c "import site; print(site.getsitepackages()[0])"
    
    # utils 패키지 디렉토리 생성
    $utilsDir = "$sitePackages\utils"
    New-Item -ItemType Directory -Force -Path $utilsDir | Out-Null
    
    # 컴파일된 utils 파일 다운로드
    Write-Host "컴파일된 utils 파일을 다운로드합니다... ($utilsFilename)" -ForegroundColor Green
    
    $utilsUrl = "$API_SERVER/api/v1/client/utils/windows/$pythonVersion/$utilsFilename"
    Write-Host "다운로드 URL: $utilsUrl" -ForegroundColor Yellow
    
    try {
        Invoke-WebRequest -Uri $utilsUrl -OutFile "$utilsDir\$utilsFilename"
    }
    catch {
        Write-Host "utils 파일 다운로드에 실패했습니다." -ForegroundColor Red
        exit 1
    }
    
    # __init__.py 생성
    @"
import os
import importlib.util

# 현재 디렉토리의 .pyd 파일 찾기
module_dir = os.path.dirname(__file__)
for file in os.listdir(module_dir):
    if file.startswith('utils.') and file.endswith('.pyd'):
        # 동적으로 모듈 로드
        spec = importlib.util.spec_from_file_location("utils", os.path.join(module_dir, file))
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        # 모든 속성을 현재 네임스페이스로 가져오기
        for attr in dir(module):
            if not attr.startswith('_'):
                globals()[attr] = getattr(module, attr)
        break

__version__ = "$UTILS_VERSION"
"@ | Set-Content "$utilsDir\__init__.py"
    
    # user_tools.py 설치
    Write-Host "user_tools.py를 설치합니다..." -ForegroundColor Green
    try {
        Invoke-WebRequest -Uri "$API_SERVER/api/v1/client/user-tools" -OutFile "$sitePackages\user_tools.py"
    }
    catch {
        Write-Host "user_tools.py 다운로드에 실패했습니다." -ForegroundColor Red
        exit 1
    }
}

# 실행 스크립트 생성
function Create-Runner {
    Write-Host "실행 스크립트를 생성합니다..." -ForegroundColor Green
    
    # run.bat 생성
    @"
@echo off
set AIRUN_HOME=%USERPROFILE%\.airun
set VENV_PATH=%USERPROFILE%\.airun_venv

:: 가상 환경 활성화
call %VENV_PATH%\Scripts\activate.bat

:: 환경 변수 로드
if exist "%AIRUN_HOME%\airun.conf.bat" (
    call %AIRUN_HOME%\airun.conf.bat
)

:: Python 스크립트 실행
python %*

:: 가상 환경 비활성화
call %VENV_PATH%\Scripts\deactivate.bat
"@ | Set-Content "$AIRUN_HOME\run.bat"
    
    Write-Host "실행 스크립트 생성 완료" -ForegroundColor Green
}

# 설정 파일 생성
function Create-Config {
    Write-Host "설정 파일을 생성합니다..." -ForegroundColor Green
    
    @"
@echo off
set SMTP_SERVER=$SMTP_HOST
set SMTP_PORT=$SMTP_PORT
set SMTP_USERNAME=
set SMTP_PASSWORD=
set API_SERVER=$API_SERVER
set PYTHON_VENV_PATH=$VENV_PATH
set AIRUN_HOME=$AIRUN_HOME
set AIRUN_CACHE_DIR=%AIRUN_HOME%\cache
set AIRUN_LOGS_DIR=%AIRUN_HOME%\logs
set USE_REVIEW=NO
set AUTO_EXECUTE=NO
set HIDE_CODE=NO
"@ | Set-Content $CONFIG_FILE
}

# 메인 설치 프로세스
function Main {
    Write-Host "AI.RUN 클라이언트 환경 설정을 시작합니다..." -ForegroundColor Green
    
    Check-ApiServer
    Check-Python
    Create-Directories
    Setup-Venv
    Install-Packages
    Install-Utils
    Create-Config
    Create-Runner
    
    Write-Host "`n설치가 완료되었습니다!" -ForegroundColor Green
    Write-Host "`n설정 방법:" -ForegroundColor Yellow
    Write-Host "1. SMTP 설정: $CONFIG_FILE 파일을 편집하여 이메일 설정을 구성하세요."
    Write-Host "2. 실행 방법: $AIRUN_HOME\run.bat <python_file>"
    
    # 설치 결과 확인
    Write-Host "`n설치 확인:" -ForegroundColor Green
    
    $checkMark = [char]0x221A
    $crossMark = [char]0x2718
    
    Write-Host "1. 가상환경: $(if (Test-Path $VENV_PATH) { $checkMark } else { $crossMark })"
    Write-Host "2. 설정 파일: $(if (Test-Path $CONFIG_FILE) { $checkMark } else { $crossMark })"
    Write-Host "3. 실행 스크립트: $(if (Test-Path "$AIRUN_HOME\run.bat") { $checkMark } else { $crossMark })"
    
    # utils 패키지 설치 확인
    $utilsCheck = python -c "import utils" 2>$null
    Write-Host "4. utils 패키지: $(if ($?) { $checkMark } else { $crossMark })"
    
    # user_tools.py 설치 확인
    $userToolsCheck = python -c "import user_tools" 2>$null
    Write-Host "5. user_tools: $(if ($?) { $checkMark } else { $crossMark })"
}

# 스크립트 실행
Main 