#!/bin/bash

# Color definitions
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# 스크립트 디렉토리 확인
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"

# 프로젝트 루트로 이동
cd "$PROJECT_ROOT"

echo -e "${GREEN}AI.RUN bash completion 설치를 시작합니다...${NC}"

# OS 확인
if [[ "$(uname)" != "Linux" ]] && [[ "$(uname)" != "Darwin" ]]; then
    echo -e "${RED}이 스크립트는 Linux 또는 macOS에서만 실행할 수 있습니다.${NC}"
    exit 1
fi

# completion 스크립트 존재 확인
BASH_COMPLETION_SCRIPT="scripts/airun-completion.bash"
ZSH_COMPLETION_SCRIPT="scripts/airun-completion.zsh"

if [ ! -f "$BASH_COMPLETION_SCRIPT" ]; then
    echo -e "${RED}Bash completion 스크립트를 찾을 수 없습니다: $BASH_COMPLETION_SCRIPT${NC}"
    exit 1
fi

if [ ! -f "$ZSH_COMPLETION_SCRIPT" ]; then
    echo -e "${RED}Zsh completion 스크립트를 찾을 수 없습니다: $ZSH_COMPLETION_SCRIPT${NC}"
    exit 1
fi

# Bash completion 설치
install_bash_completion() {
    echo -e "${GREEN}Bash completion 설치 중...${NC}"
    
    # 사용자별 completion 디렉토리 설정
    COMPLETION_DIR="$HOME/.local/share/bash-completion/completions"
    mkdir -p "$COMPLETION_DIR"
    
    # completion 스크립트 복사
    cp "$BASH_COMPLETION_SCRIPT" "$COMPLETION_DIR/airun"
    chmod 644 "$COMPLETION_DIR/airun"
    
    echo -e "${GREEN}✓ Bash completion 파일 설치: $COMPLETION_DIR/airun${NC}"
    
    # .bashrc에 completion 로딩 코드 추가 (중복 방지)
    BASHRC_FILE="$HOME/.bashrc"
    
    # 기존 설정 확인
    if ! grep -q "AI.RUN bash completion" "$BASHRC_FILE" 2>/dev/null; then
        echo "" >> "$BASHRC_FILE"
        echo "# AI.RUN bash completion" >> "$BASHRC_FILE"
        echo "if [ -f $COMPLETION_DIR/airun ]; then" >> "$BASHRC_FILE"
        echo "    source $COMPLETION_DIR/airun" >> "$BASHRC_FILE"
        echo "fi" >> "$BASHRC_FILE"
        echo -e "${GREEN}✓ .bashrc에 completion 설정 추가${NC}"
    else
        echo -e "${YELLOW}이미 .bashrc에 completion 설정이 존재합니다${NC}"
    fi
}

# Zsh completion 설치
install_zsh_completion() {
    echo -e "${GREEN}Zsh completion 설치 중...${NC}"
    
    # 사용자별 zsh completion 디렉토리 설정
    ZSH_COMPLETION_DIR="$HOME/.zsh/completion"
    mkdir -p "$ZSH_COMPLETION_DIR"
    
    # completion 스크립트 복사
    cp "$ZSH_COMPLETION_SCRIPT" "$ZSH_COMPLETION_DIR/_airun"
    chmod 644 "$ZSH_COMPLETION_DIR/_airun"
    
    echo -e "${GREEN}✓ Zsh completion 파일 설치: $ZSH_COMPLETION_DIR/_airun${NC}"
    
    # .zshrc에 completion 설정 추가 (중복 방지)
    ZSHRC_FILE="$HOME/.zshrc"
    
    if ! grep -q "AI.RUN zsh completion" "$ZSHRC_FILE" 2>/dev/null; then
        echo "" >> "$ZSHRC_FILE"
        echo "# AI.RUN zsh completion" >> "$ZSHRC_FILE"
        echo "# completion 시스템 초기화" >> "$ZSHRC_FILE"
        echo "autoload -Uz compinit" >> "$ZSHRC_FILE"
        echo "compinit" >> "$ZSHRC_FILE"
        echo "" >> "$ZSHRC_FILE"
        echo "# completion 스타일 설정" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*' menu select" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*' matcher-list 'm:{a-zA-Z}={A-Za-z}'" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*' list-colors \${(s.:.)LS_COLORS}" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*' verbose yes" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*:descriptions' format '%F{green}-- %d --%f'" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*:messages' format '%F{yellow}-- %d --%f'" >> "$ZSHRC_FILE"
        echo "zstyle ':completion:*:warnings' format '%F{red}-- no matches found --%f'" >> "$ZSHRC_FILE"
        echo "" >> "$ZSHRC_FILE"
        echo "# AI.RUN completion 경로 설정" >> "$ZSHRC_FILE"
        echo "fpath=($ZSH_COMPLETION_DIR \$fpath)" >> "$ZSHRC_FILE"
        echo -e "${GREEN}✓ .zshrc에 completion 설정 추가${NC}"
    else
        echo -e "${YELLOW}이미 .zshrc에 completion 설정이 존재합니다${NC}"
    fi
}

# 설치 결과 검증
verify_installation() {
    echo -e "\n${GREEN}설치 결과 확인:${NC}"
    
    # Bash completion 파일 확인
    if [ -f "$HOME/.local/share/bash-completion/completions/airun" ]; then
        echo -e "${GREEN}✓ Bash completion 파일 설치됨${NC}"
    else
        echo -e "${RED}✗ Bash completion 파일 설치 실패${NC}"
    fi
    
    # Zsh completion 파일 확인
    if [ -f "$HOME/.zsh/completion/_airun" ]; then
        echo -e "${GREEN}✓ Zsh completion 파일 설치됨${NC}"
    else
        echo -e "${RED}✗ Zsh completion 파일 설치 실패${NC}"
    fi
    
    # .bashrc 설정 확인
    if grep -q "AI.RUN bash completion" "$HOME/.bashrc" 2>/dev/null; then
        echo -e "${GREEN}✓ .bashrc에 completion 설정 확인됨${NC}"
    else
        echo -e "${RED}✗ .bashrc에 completion 설정 없음${NC}"
    fi
    
    # .zshrc 설정 확인  
    if grep -q "AI.RUN zsh completion" "$HOME/.zshrc" 2>/dev/null; then
        echo -e "${GREEN}✓ .zshrc에 completion 설정 확인됨${NC}"
    else
        echo -e "${RED}✗ .zshrc에 completion 설정 없음${NC}"
    fi
}

# 메인 설치 프로세스
main() {
    # 현재 쉘 확인
    CURRENT_SHELL=$(basename "$SHELL")
    
    echo -e "${YELLOW}현재 쉘: $CURRENT_SHELL${NC}"
    
    # 쉘별 설치
    case "$CURRENT_SHELL" in
        bash)
            install_bash_completion
            ;;
        zsh)
            install_zsh_completion
            # zsh 환경에서도 bash completion 함께 설치 (호환성을 위해)
            install_bash_completion
            ;;
        *)
            echo -e "${YELLOW}지원되는 모든 쉘에 대해 completion을 설치합니다...${NC}"
            install_bash_completion
            install_zsh_completion
            ;;
    esac
    
    # 설치 결과 확인
    verify_installation
    
    # 사용자 안내
    echo -e "\n${YELLOW}설치 완료!${NC}"
    echo -e "${YELLOW}자동완성을 사용하려면 다음 중 하나를 실행하세요:${NC}"
    echo -e "  • 새로운 터미널 세션 시작"
    echo -e "  • Bash 사용자: ${GREEN}source ~/.bashrc${NC}"
    echo -e "  • Zsh 사용자: ${GREEN}source ~/.zshrc${NC}"
    echo -e "\n${YELLOW}확인 방법:${NC}"
    echo -e "  터미널에서 ${GREEN}airun <TAB><TAB>${NC}을 입력하여 자동완성이 작동하는지 확인하세요."
}

# 스크립트 실행
main 