import AuthService from './AuthService.js';
import PermissionService from './PermissionService.js';
import dbManager from '../database/index.js';

let authService;
let permissionService;

export const initializeAuth = () => {
    authService = new AuthService();
    permissionService = new PermissionService();
};

export const authMiddleware = async (req, res, next) => {
    if (!authService) {
        return res.status(500).json({
            success: false,
            error: {
                message: '인증 서비스가 초기화되지 않았습니다.'
            }
        });
    }

    try {
        const authHeader = req.headers.authorization;
        if (!authHeader || !authHeader.startsWith('Bearer ')) {
            return res.status(401).json({
                success: false,
                error: {
                    message: '인증 토큰이 필요합니다.'
                }
            });
        }

        const token = authHeader.split(' ')[1];
        const decoded = await authService.verifyToken(token);
        
        // username이 없는 경우 사용자 정보 다시 조회
        if (!decoded.username) {
            const user = await authService.getUser(decoded.id);
            decoded.username = user.username;
        }
        
        // IP 주소 로그 기록 (비동기로 처리하여 인증 흐름 차단 방지)
        const ipAddress = req.ip || req.connection.remoteAddress;
        setImmediate(async () => {
            try {
                await dbManager.query(
                    'INSERT INTO user_ip_logs (user_id, ip_address) VALUES ($1, $2)',
                    [decoded.id, ipAddress]
                );
            } catch (logError) {
                console.warn('IP 로그 기록 실패:', logError.message);
                // IP 로그 실패는 인증에 영향을 주지 않음
            }
        });
        
        req.user = decoded;
        next();
    } catch (error) {
        res.status(401).json({
            success: false,
            error: {
                message: '유효하지 않은 토큰입니다.'
            }
        });
    }
};

export const checkPermission = (requiredPermission) => {
    return async (req, res, next) => {
        if (!permissionService) {
            return res.status(500).json({
                success: false,
                error: {
                    message: '권한 서비스가 초기화되지 않았습니다.'
                }
            });
        }

        try {
            const hasPermission = await permissionService.checkPermission(
                req.user.id, 
                requiredPermission
            );
            
            if (!hasPermission) {
                return res.status(403).json({
                    success: false,
                    error: {
                        message: '권한이 없습니다.'
                    }
                });
            }
            next();
        } catch (error) {
            res.status(500).json({
                success: false,
                error: {
                    message: '권한 확인 중 오류가 발생했습니다.'
                }
            });
        }
    };
};

