import fs from 'fs';
import path from 'path';
import os from 'os';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);
/**
 * 앱의 설정 디렉토리 경로를 가져옵니다.
 */
export async function getAppConfigPath() {
    const homedir = os.homedir();
    const configDir = path.join(homedir, '.airun');
    const tasksPath = path.join(configDir, 'tasks.json');
    const fontPath = path.join(configDir, 'Pretendard-Regular.ttf');
    
    // 디렉토리가 없으면 생성
    if (!fs.existsSync(configDir)) {
        await fs.promises.mkdir(configDir, { recursive: true });
    }
    
    // 폰트 파일이 없으면 assets에서 복사
    if (!fs.existsSync(fontPath)) {
        const templateFontPath = path.join(__dirname, '../../assets', 'Pretendard-Regular.ttf');
        try {
            await fs.promises.copyFile(templateFontPath, fontPath);
        } catch (error) {
            console.error('Error copying font file:', error);
        }
    }
        
    // tasks.json 파일이 없으면 템플릿 파일 복사
    if (!fs.existsSync(tasksPath)) {
        const templatePath = path.join(__dirname, '../../task_template.json');
        try {
            const templateContent = await fs.promises.readFile(templatePath, 'utf8');
            await fs.promises.writeFile(tasksPath, templateContent, 'utf8');
        } catch (error) {
            console.error('Error copying template file:', error);
            // 기본 빈 배열로 초기화
            await fs.promises.writeFile(tasksPath, '[]', 'utf8');
        }
    }

    return configDir;
}

/**
 * 실행 로그 파일의 경로를 가져옵니다.
 */
export async function getExecutionLogsPath() {
    const configDir = await getAppConfigPath();
    const logsPath = path.join(configDir, 'execution_logs.json');
    
    // configDir이 존재하지 않으면 생성
    if (!fs.existsSync(configDir)) {
        await fs.promises.mkdir(configDir, { recursive: true });
    }
    
    // logsPath가 디렉토리인 경우 삭제
    try {
        const stats = fs.existsSync(logsPath) ? await fs.promises.stat(logsPath) : null;
        if (stats && stats.isDirectory()) {
            await fs.promises.rm(logsPath, { recursive: true, force: true });
        }
    } catch (error) {
        console.error('Error checking/removing logs path:', error);
    }
    
    // 파일이 존재하지 않으면 빈 배열로 초기화
    if (!fs.existsSync(logsPath)) {
        try {
            await fs.promises.writeFile(logsPath, '[]', 'utf8');
        } catch (error) {
            console.error('Error initializing logs file:', error);
        }
    }
    
    return logsPath;
}

/**
 * 세션 디렉토리 경로를 가져옵니다.
 */
export async function getSessionPath() {
    const configDir = await getAppConfigPath();
    const sessionDir = path.join(configDir, 'sessions');
    
    if (!fs.existsSync(sessionDir)) {
        await fs.promises.mkdir(sessionDir, { recursive: true });
    }
    
    return sessionDir;
}

/**
 * 작업 목록을 가져옵니다.
 */
export async function getTasks() {
    try {
        const configDir = await getAppConfigPath();
        const tasksPath = path.join(configDir, 'tasks.json');
        
        if (!fs.existsSync(tasksPath)) {
            await fs.promises.writeFile(tasksPath, '[]', 'utf8');
            return [];
        }
        
        const tasks = JSON.parse(await fs.promises.readFile(tasksPath, 'utf8'));
        return tasks;
    } catch (error) {
        console.error('Failed to get tasks:', error);
        return [];
    }
} 