import sys
import os
import json
import asyncio
import traceback
import importlib.metadata
import signal
from utils import debug_print

# 현재 스크립트의 디렉토리를 Python 경로에 추가
current_dir = os.path.dirname(os.path.abspath(__file__))
if current_dir not in sys.path:
    sys.path.insert(0, current_dir)

# 프로젝트 루트 디렉토리 찾기 (services/report 상위 디렉토리)
project_root = os.path.dirname(os.path.dirname(current_dir))
if project_root not in sys.path:
    sys.path.insert(0, project_root)

# 디버그 정보 출력
# debug_print(f"현재 디렉토리: {os.getcwd()}")
# debug_print(f"스크립트 디렉토리: {current_dir}")
# debug_print(f"프로젝트 루트: {project_root}")
# debug_print(f"Python 경로: {sys.executable}")
# debug_print(f"Python 버전: {sys.version}")
# debug_print(f"sys.path: {sys.path}")

# 개발/배포 환경에 따른 모듈 로딩 전략
def load_report_generator():
    """
    개발 환경에서는 Python 소스 파일을 우선 사용하고,
    배포 환경에서는 컴파일된 파일을 사용합니다.
    """
    # 1. 개발 환경: Python 소스 파일 확인
    report_generator_py = os.path.join(current_dir, 'report_generator.py')
    
    if os.path.exists(report_generator_py):
        # debug_print(f"✓ 개발 모드: Python 소스 파일 사용 - {report_generator_py}")
        try:
            # Python 소스 파일을 직접 로드
            import importlib.util
            spec = importlib.util.spec_from_file_location("report_generator", report_generator_py)
            report_generator_module = importlib.util.module_from_spec(spec)
            spec.loader.exec_module(report_generator_module)
            
            # 모듈이 성공적으로 로드되었는지 확인
            BizPlanContentGenerator = getattr(report_generator_module, 'BizPlanContentGenerator', None)
            _setup_logger = getattr(report_generator_module, '_setup_logger', None)
            
            if BizPlanContentGenerator is None or _setup_logger is None:
                raise ImportError("필수 클래스/함수를 찾을 수 없습니다")
                
            # debug_print("✓ Python 소스 파일에서 모듈 로드 성공")
            return report_generator_module, BizPlanContentGenerator, _setup_logger
            
        except Exception as e:
            debug_print(f"⚠ Python 소스 파일 로드 실패: {str(e)}")
            debug_print("컴파일된 모듈 사용으로 전환합니다...")
    else:
        debug_print(f"✓ 배포 모드: Python 소스 파일이 없습니다 - {report_generator_py}")
    
    # 2. 배포 환경: 컴파일된 모듈 사용
    try:
        # debug_print("컴파일된 report_generator 모듈 로드 시도...")
        from report_generator import BizPlanContentGenerator, _setup_logger
        import report_generator as report_generator_module
        
        # debug_print(f"✓ 컴파일된 모듈 사용 - {report_generator_module.__file__}")
        return report_generator_module, BizPlanContentGenerator, _setup_logger
        
    except ImportError as e:
        debug_print(f"✗ 컴파일된 모듈 로드도 실패: {str(e)}")
        raise ImportError(f"report_generator 모듈을 로드할 수 없습니다: {str(e)}")

try:
    # 모듈 로딩 전략 실행
    report_generator, BizPlanContentGenerator, _setup_logger = load_report_generator()
    # debug_print(f"최종 사용 모듈: {report_generator.__file__ if hasattr(report_generator, '__file__') else 'unknown'}")
    
    # 개발/배포 모드 표시
    if os.path.exists(os.path.join(current_dir, 'report_generator.py')):
        print("🔧 개발 모드: Python 소스 파일을 사용합니다.")
    else:
        print("📦 배포 모드: 컴파일된 모듈을 사용합니다.")
        
except Exception as e:
    print(f"[ERROR] 모듈 가져오기 실패: {str(e)}")
    print(f"[ERROR] 상세 오류: {traceback.format_exc()}")
    sys.exit(1)

from datetime import datetime

def signal_handler(signum, frame):
    """시그널 처리 함수"""
    if signum == signal.SIGINT:
        print("\n\n⚠️ Ctrl+C가 감지되었습니다. 작업을 안전하게 중단합니다...")
        print("[DEBUG] 사용자가 Ctrl+C로 작업 중단을 요청했습니다")
        # 프로그램 종료
        sys.exit(1)

# 시그널 핸들러 등록 (메인 스레드에서만)
try:
    # 메인 스레드 체크
    import threading
    if threading.current_thread() is threading.main_thread():
        signal.signal(signal.SIGINT, signal_handler)
    else:
        # 메인 스레드가 아닌 경우 signal 핸들러 등록하지 않음
        pass
except (ValueError, AttributeError):
    # 메인 스레드가 아니거나 signal 설정 불가능한 경우 무시
    pass

def check_environment():
    """필요한 패키지 설치 여부 확인"""
    try:
        # 필요한 패키지 목록
        required_packages = {
            'pyhwp': 'not installed',
            'numpy': 'not installed',
            'pandas': 'not installed'
        }

        # 설치된 패키지 버전 확인
        for package_name in required_packages.keys():
            try:
                version = importlib.metadata.version(package_name)
                required_packages[package_name] = version
            except importlib.metadata.PackageNotFoundError:
                pass

        # 환경 정보 구성
        environment_info = {
            'success': True,
            'python_version': sys.version,
            'python_path': sys.executable,
            'packages': required_packages
        }

        # 필수 패키지 설치 여부 확인
        missing_packages = [pkg for pkg, ver in required_packages.items() 
                          if ver == 'not installed']
        
        if missing_packages:
            environment_info['success'] = False
            environment_info['error'] = f"Missing required packages: {', '.join(missing_packages)}"

        return environment_info

    except Exception as e:
        return {
            'success': False,
            'error': f"Environment check failed: {str(e)}"
        }

async def generate_business_plan(data, progress_callback=None):
    """사업계획서 최종 생성"""
    try:
        debug_print("===== 사업계획서 생성 시작 =====")
        
        # 진행률 콜백 호출
        if progress_callback:
            await progress_callback("사업계획서 생성 환경 검사 중...", 5)
        
        # 환경 검사 수행
        env_check = check_environment()
        if not env_check['success']:
            debug_print(f"환경 검사 실패: {env_check['error']}")
            return {'success': False, 'error': env_check['error']}
        debug_print("환경 검사 완료")
        
        # output 디렉토리 생성 (프로젝트 루트 기준)
        output_base_dir = os.path.join(project_root, "output")
        debug_print(f"출력 디렉토리 경로: {output_base_dir}")
        
        try:
            os.makedirs(output_base_dir, exist_ok=True)
            debug_print(f"출력 디렉토리 생성 성공: {output_base_dir}")
        except Exception as e:
            debug_print(f"출력 디렉토리 생성 실패: {str(e)}")
            raise
        
        # 실행 디렉토리 생성 (사용자별 구분)
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        
        # 사용자 ID가 있는 경우 사용자별 디렉토리 생성
        user_id = None
        try:
            input_data = json.loads(data) if isinstance(data, str) else data
            user_id = input_data.get('user_id')
        except:
            pass
        
        if user_id:
            # 사용자별 출력 디렉토리: output/{user_id}/
            user_output_dir = os.path.join(output_base_dir, user_id)
            os.makedirs(user_output_dir, exist_ok=True)
            run_dir = os.path.join(user_output_dir, f"bizplan_{timestamp}")
            debug_print(f"사용자별 출력 디렉토리 생성: {run_dir}")
        else:
            # 기존 방식 (호환성 유지)
            run_dir = os.path.join(output_base_dir, f"bizplan_{timestamp}")
            debug_print(f"기본 출력 디렉토리 생성: {run_dir}")
        
        os.makedirs(run_dir, exist_ok=True)

        # 입력 데이터 처리
        try:
            input_data = json.loads(data) if isinstance(data, str) else data
        except Exception as e:
            debug_print(f"입력 데이터 파싱 실패: {str(e)}")
            raise ValueError("입력 데이터 형식이 올바르지 않습니다")

        # 입력 데이터에서 format, project_hash, job_id, log_file, user_id 가져오기
        output_format = input_data.get('format', 'pdf')
        project_hash = input_data.get('project_hash')
        job_id = input_data.get('job_id')
        log_file = input_data.get('log_file')
        user_id = input_data.get('user_id')  # 사용자 ID 추가

        # 사용자별 로그 파일 경로 설정
        if user_id and project_hash:
            # 사용자별 캐시 경로에 로그 저장: ~/.airun/cache/report/{username}/{project_hash}/logs/
            user_cache_dir = os.path.join(os.path.expanduser('~/.airun/cache/report'), user_id)
            log_dir = os.path.join(user_cache_dir, project_hash, 'logs')
            os.makedirs(log_dir, exist_ok=True)
            log_filename = f"report_generator_{project_hash}.log"
            log_file_path = os.path.join(log_dir, log_filename)
            debug_print(f"사용자별 로그 파일 경로: {log_file_path}")
        else:
            log_file_path = log_file
        
        # AI 모델 설정 추출
        provider = input_data.get('provider')
        model = input_data.get('model')
        
        # BizPlanContentGenerator를 사용하여 사업계획서 생성
        logger = _setup_logger(job_id=job_id, project_hash=project_hash, log_file=log_file_path)
        content_generator = BizPlanContentGenerator(output_format=output_format, logger=logger, provider=provider, model=model)
        
        # 사용자 ID가 있는 경우 사용자별 디렉토리 구조 적용
        if user_id:
            content_generator.set_user_id(user_id)
            debug_print(f"사용자별 디렉토리 적용: {user_id}")
        
        # template 변수 정의
        template = input_data.get('template')
        
        # 진행률 콜백 호출
        if progress_callback:
            await progress_callback("BizPlan 생성기 초기화 완료", 10)
        
        # 문서 생성 (output_path 대신 output_dir만 전달, 실제 파일명은 create_business_plan에서 결정)
        input_data['output_dir'] = run_dir
        input_data['timestamp'] = timestamp
        result = await content_generator.create_business_plan(
            executive_summary=input_data.get('executive_summary', ''),
            output_dir=run_dir,
            output_format=output_format,
            timestamp=timestamp,
            use_cache=True,
            project_hash=project_hash,
            template=template,
            progress_callback=progress_callback  # progress_callback 전달
        )
        
        debug_print(f"create_business_plan 결과: {result}")
        
        # 파일 경로 확인 (결과에서 file_path 가져오기)
        if result.get('success') and result.get('file_path'):
            final_output_path = result.get('file_path')
            
            # 파일 존재 여부 확인
            if os.path.exists(final_output_path):
                file_size = os.path.getsize(final_output_path)
                debug_print(f"생성된 파일 확인: {final_output_path} (크기: {file_size:,} bytes)")
                
                if file_size > 0:
                    debug_print("✔ 요청한 문서가 생성되었습니다.")
                    debug_print(f"📁 최종 결과 파일: {final_output_path}")
                    
                    # result에서 필요한 모든 정보를 포함
                    return {
                        'success': True,
                        'error': None,
                        'project_hash': result.get('project_hash'),
                        'result': {
                            'file_path': final_output_path,
                            'format': output_format,
                            'status': 'completed',
                            'file_size': file_size,
                            'sections': result.get('sections', {}),
                            'cache_dir': result.get('cache_dir'),
                            'statistics': result.get('statistics', {})
                        }
                    }
        
        # 파일이 없거나 크기가 0이면 실패로 처리
        error_msg = result.get('error', '알 수 없는 오류')
        debug_print(f"문서 생성 실패: {error_msg}")
        return {
            'success': False,
            'error': error_msg,
            'result': None
        }
            
    except Exception as e:
        debug_print(f"사업계획서 생성 실패: {str(e)}")
        debug_print(f"상세 오류: {traceback.format_exc()}")
        return {
            'success': False,
            'error': str(e),
            'result': None
        }

if __name__ == '__main__':
    if len(sys.argv) > 1 and sys.argv[1] == '--check-env':
        # 환경 검사만 수행
        result = check_environment()
        print(json.dumps(result))
        sys.exit(0 if result['success'] else 1)
        
    if len(sys.argv) < 2:
        print(json.dumps({
            'success': False,
            'error': 'No input data provided'
        }))
        sys.exit(1)
    
    try:
        debug_print("스크립트 시작...")
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        result = loop.run_until_complete(generate_business_plan(sys.argv[1]))
        print(json.dumps(result))
        loop.close()
    except Exception as e:
        error_traceback = traceback.format_exc()
        debug_print(f"치명적 오류: {str(e)}\n{error_traceback}")
        print(json.dumps({
            'success': False,
            'error': f"Fatal Error: {str(e)}\nTraceback:\n{error_traceback}"
        }))
        sys.exit(1) 