import TaskStep from '../models/TaskStep.js';

class DependencyAnalyzer {
    parseTaskBreakdown(input) {
        // 이미 구조화된 데이터가 전달된 경우
        if (Array.isArray(input) && input.every(step => typeof step === 'object')) {
            const steps = input.map((step, index) => {
                const taskStep = new TaskStep({
                    id: step.id || String(index),
                    number: step.number || (index + 1),  // 원래 번호 유지, 없을 경우에만 생성
                    description: step.description,
                    dependencies: step.dependencies || [],
                    request: { description: step.description }
                });
                
                // 순차적 의존성 추가
                if (index > 0) {
                    taskStep.dependencies = [(index - 1).toString()];
                }
                
                return taskStep;
            });
            
            return steps;
        }

        // 텍스트 형식의 데이터가 전달된 경우
        const steps = [];
        
        // 텍스트를 줄 단위로 분석
        const lines = typeof input === 'string' ? input.split('\n') : [];
        
        for (const line of lines) {
            const trimmedLine = line.trim();
            
            // 숫자로 시작하는 단계 매칭 (1., 2., 등)
            const match = trimmedLine.match(/^(\d+)\.\s*(.+)$/);
            if (match) {
                const stepNumber = parseInt(match[1]);  // 원래 번호 사용
                const description = match[2];
                
                const step = new TaskStep({
                    id: String(stepNumber - 1),  // ID는 0-based로 유지
                    number: stepNumber,  // 원래 번호 유지
                    description,
                    dependencies: stepNumber > 1 ? [(stepNumber - 2).toString()] : [],  // 의존성도 조정
                    request: { description }
                });
                
                steps[stepNumber - 1] = step;  // 인덱스도 0-based로 조정
            }
        }
        
        // 빈 슬롯 제거
        const finalSteps = steps.filter(step => step !== undefined);
        
        // 순차적 의존성만 확인하고 번호는 유지
        for (let i = 0; i < finalSteps.length; i++) {
            const step = finalSteps[i];
            
            // 첫 번째 단계는 의존성이 없음
            if (i === 0) {
                step.dependencies = [];
            } else {
                // 나머지 단계들은 이전 단계에 의존
                step.dependencies = [(i - 1).toString()];
            }
        }
        
        return finalSteps;
    }

    shouldCombineSteps(lastDescription, currentDescription) {
        return false;
    }
}

export default DependencyAnalyzer; 