class TaskContext {
    constructor(originalRequest = '', paths = { inputs: [], output: '' }) {
        this.originalRequest = originalRequest;
        this.paths = paths;
        this.taskSteps = [];
        this.variables = new Map();
        this.executionHistory = [];
        this.currentStepId = null;
    }

    setVariable(key, value) {
        this.variables.set(key, value);
    }

    getVariable(key) {
        return this.variables.get(key);
    }

    addExecutionHistory(step) {
        this.executionHistory.push({
            timestamp: new Date(),
            stepId: step.id,
            status: step.status,
            result: step.result,
            error: step.error
        });
    }

    setTaskSteps(steps) {
        this.taskSteps = steps.map(step => ({
            id: step.id,
            number: step.number,
            description: step.description,
            status: 'pending',
            result: null,
            startTime: null,
            endTime: null,
            error: null
        }));
    }

    updateStepStatus(stepId, status, result = null, error = null) {
        const step = this.taskSteps.find(s => s.id === stepId);
        if (step) {
            step.status = status;
            if (status === 'running') {
                step.startTime = new Date().toISOString();
            } else if (status === 'completed' || status === 'failed') {
                step.endTime = new Date().toISOString();
                if (status === 'completed') {
                    step.result = result;
                }
                if (status === 'failed') {
                    step.error = error;
                }
            }
        }
    }

    getPreviousStepResult() {
        if (!this.currentStepId) return null;
        const currentIndex = this.taskSteps.findIndex(s => s.id === this.currentStepId);
        if (currentIndex <= 0) return null;
        return this.taskSteps[currentIndex - 1]?.result || null;
    }

    toJSON() {
        return {
            originalRequest: this.originalRequest,
            paths: this.paths,
            taskSteps: this.taskSteps,
            currentStepId: this.currentStepId,
            executionHistory: this.executionHistory
        };
    }
}

export default TaskContext; 