#!/usr/bin/env python3
"""
AIRUN 웹검색 서비스 - 진짜 최적화 버전 (Monkey Patch)
- 포트: 5610
- 기능: 구글, 네이버, 다음 통합 웹검색
- 특징: utils.py 함수들을 monkey patch로 최적화
"""

import asyncio
import json
import logging
import os
import sys
import time
import traceback
import warnings
import threading
import signal
from contextlib import asynccontextmanager
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any
from concurrent.futures import ThreadPoolExecutor
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By

# 외부 라이브러리 경고 억제
import warnings
warnings.filterwarnings("ignore", category=UserWarning)

# trafilatura 경고 억제 (구글 검색 시 발생)
import logging as trafilatura_logging
trafilatura_logging.getLogger('trafilatura.core').setLevel(logging.ERROR)

# urllib3 연결 경고 억제 (Selenium WebDriver 연결 시 발생)
trafilatura_logging.getLogger('urllib3.connectionpool').setLevel(logging.ERROR)

# Selenium 관련 경고 억제
trafilatura_logging.getLogger('selenium').setLevel(logging.ERROR)

# 프로젝트 루트 경로 설정
project_root = Path(__file__).parent.parent.parent
sys.path.insert(0, str(project_root))

# utils.py를 직접 임포트
import utils

# 필요한 패키지들 임포트
try:
    import uvicorn
    from fastapi import FastAPI, HTTPException, Query
    from fastapi.middleware.cors import CORSMiddleware
    from pydantic import BaseModel, Field
    import requests
    import urllib.parse
    import re
    from bs4 import BeautifulSoup
except ImportError as e:
    print(f"필요한 패키지가 설치되지 않았습니다: {e}")
    sys.exit(1)

# 로깅 설정 - 개별 로그 파일에 기록
log_dir = Path.home() / ".airun" / "logs"
log_dir.mkdir(parents=True, exist_ok=True)
log_file = log_dir / "airun-websearch.log"

# 로거 설정
logger = logging.getLogger("airun-websearch-server")
logger.setLevel(logging.INFO)

# 파일 핸들러 생성
file_handler = logging.FileHandler(log_file, encoding='utf-8')
file_handler.setLevel(logging.INFO)

# 콘솔 핸들러 생성 (선택적)
console_handler = logging.StreamHandler()
console_handler.setLevel(logging.WARNING)  # 콘솔에는 WARNING 이상만 출력

# 포맷터 설정
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
file_handler.setFormatter(formatter)
console_handler.setFormatter(formatter)

# 핸들러 추가
logger.addHandler(file_handler)
logger.addHandler(console_handler)

# 다른 로거들의 중복 출력 방지
logger.propagate = False

# 루트 로거의 핸들러들 제거 (중복 방지)
root_logger = logging.getLogger()
for handler in root_logger.handlers[:]:
    root_logger.removeHandler(handler)

# 모든 경고 메시지 무시
warnings.filterwarnings('ignore')

# 전역 변수들
driver_pool: Optional['UltraFastDriverPool'] = None
websearch_cache: Optional['WebSearchCache'] = None
original_get_selenium_driver = None

# 서버 통계
server_stats = {
    "start_time": None,
    "total_requests": 0,
    "successful_searches": 0,
    "failed_searches": 0,
    "cache_hits": 0,
    "cache_misses": 0,
    "avg_response_time": 0.0,
    "driver_reuse_count": 0
}

# =============================================================================
# 초고속 드라이버 풀 (Monkey Patch 방식)
# =============================================================================

class UltraFastDriverPool:
    """초고속 드라이버 풀 (utils.py monkey patch)"""
    
    def __init__(self, pool_size: int = 3):
        self.pool_size = pool_size
        self.drivers = []
        self.available_drivers = []
        self.lock = threading.Lock()
        self.initialized = False
        
    def initialize(self) -> bool:
        """드라이버 풀 초기화 및 monkey patch 적용"""
        try:
            logger.info(f"🚀 초고속 드라이버 풀 초기화 시작 (크기: {self.pool_size})")
            
            # 원본 함수 백업
            global original_get_selenium_driver
            original_get_selenium_driver = utils.get_selenium_driver
            
            # 미리 여러 개의 드라이버 생성
            for i in range(self.pool_size):
                try:
                    logger.info(f"🔧 드라이버 {i+1}/{self.pool_size} 생성 중...")
                    driver = original_get_selenium_driver()
                    
                    if driver:
                        self.drivers.append(driver)
                        self.available_drivers.append(driver)
                        logger.info(f"✅ 드라이버 {i+1} 생성 완료")
                    else:
                        logger.warning(f"⚠️ 드라이버 {i+1} 생성 실패")
                        
                except Exception as e:
                    logger.error(f"❌ 드라이버 {i+1} 생성 중 오류: {e}")
            
            self.initialized = len(self.drivers) > 0
            
            if self.initialized:
                # Monkey patch 적용!
                utils.get_selenium_driver = self._get_driver_from_pool
                logger.info(f"✅ Monkey Patch 적용 완료: utils.get_selenium_driver -> driver_pool")
                logger.info(f"✅ 초고속 드라이버 풀 초기화 완료: {len(self.drivers)}개 드라이버 준비")
            else:
                logger.error("❌ 드라이버 풀 초기화 실패")
            
            return self.initialized
            
        except Exception as e:
            logger.error(f"❌ 드라이버 풀 초기화 실패: {e}")
            logger.error(f"상세 오류:\n{traceback.format_exc()}")
            return False
    
    def _get_driver_from_pool(self):
        """개선된 Monkey patch 함수: 풀에서 드라이버 가져오기"""
        # 즉시 사용 가능한 드라이버 확인
        with self.lock:
            if self.available_drivers:
                driver = self.available_drivers.pop(0)
                server_stats["driver_reuse_count"] += 1
                logger.debug(f"🚗 드라이버 풀에서 할당 (재사용 #{server_stats['driver_reuse_count']}, 남은: {len(self.available_drivers)})")
                return driver
        
        # 풀이 비어있으면 대기 없이 바로 새 드라이버 생성
        logger.info("⚡ 드라이버 풀이 비어있음 - 새 드라이버 즉시 생성")
        try:
            new_driver = original_get_selenium_driver()
            server_stats["driver_create_count"] = server_stats.get("driver_create_count", 0) + 1
            return new_driver
        except Exception as e:
            logger.error(f"새 드라이버 생성 실패: {e}")
            return None
    
    def return_driver(self, driver):
        """개선된 드라이버 풀 반환"""
        if not driver:
            return
            
        try:
            # 풀에서 생성된 드라이버인지 확인
            if driver in self.drivers:
                with self.lock:
                    if driver not in self.available_drivers:
                        # 드라이버 상태 확인 후 반환
                        try:
                            driver.current_url  # 드라이버가 살아있는지 확인
                            self.available_drivers.append(driver)
                            logger.debug(f"🔄 드라이버 풀에 반환 (사용 가능: {len(self.available_drivers)})")
                        except:
                            # 드라이버가 죽어있으면 풀에서 제거
                            if driver in self.drivers:
                                self.drivers.remove(driver)
                            logger.warning("💀 죽은 드라이버 발견 - 풀에서 제거")
            else:
                # 풀에서 생성되지 않은 드라이버는 바로 종료
                try:
                    driver.quit()
                    logger.debug("🗑️ 임시 드라이버 종료")
                except:
                    pass
        except Exception as e:
            logger.error(f"드라이버 반환 중 오류: {e}")
            # 오류 발생 시 드라이버 강제 종료
            try:
                driver.quit()
            except:
                pass
    
    def cleanup(self):
        """드라이버 풀 정리 및 monkey patch 복원"""
        logger.info("🧹 드라이버 풀 정리 시작")
        
        # Monkey patch 복원
        if original_get_selenium_driver:
            utils.get_selenium_driver = original_get_selenium_driver
            logger.info("✅ Monkey Patch 복원 완료")
        
        # 드라이버들 정리
        with self.lock:
            for driver in self.drivers:
                try:
                    if hasattr(driver, 'quit'):
                        driver.quit()
                except Exception as e:
                    logger.error(f"드라이버 정리 중 오류: {e}")
            
            self.drivers.clear()
            self.available_drivers.clear()
            self.initialized = False
        logger.info("✅ 드라이버 풀 정리 완료")

# =============================================================================
# 초고속 웹검색 엔진 (utils.py 직접 사용)
# =============================================================================

class WebSearch:
    """utils.py 함수들을 직접 사용하는 웹검색 엔진"""
    
    def __init__(self, driver_pool: UltraFastDriverPool):
        self.driver_pool = driver_pool
    
    def search_duckduckgo(self, query: str, max_results: int = None) -> List[Dict[str, Any]]:
        """DuckDuckGo 검색을 수행하고 결과를 반환합니다."""
        try:
            results = []
            max_results = max_results or 10
            
            # DuckDuckGo 검색 URL
            headers = {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            }
            
            # DuckDuckGo HTML 검색
            search_url = f"https://html.duckduckgo.com/html/?q={urllib.parse.quote_plus(query)}"
            logger.debug(f"DuckDuckGo 검색 URL: {search_url}")
            
            response = requests.get(search_url, headers=headers, timeout=10)
            response.raise_for_status()
            
            soup = BeautifulSoup(response.text, 'html.parser')
            
            # DuckDuckGo HTML 결과 파싱
            search_results = soup.find_all('div', class_='result')
            logger.debug(f"DuckDuckGo 결과 블록 수: {len(search_results)}")
            
            for result in search_results[:max_results]:
                try:
                    # 제목과 URL 추출
                    link_elem = result.find('a', class_='result__a')
                    if not link_elem:
                        continue
                        
                    title = self._clean_text(link_elem.get_text().strip())
                    url = link_elem.get('href', '')
                    
                    # URL 정리
                    if url.startswith('/l/?uddg='):
                        # DuckDuckGo redirect URL 디코딩
                        url = urllib.parse.unquote(url.split('uddg=')[1])
                    
                    # 설명 추출
                    desc_elem = result.find('a', class_='result__snippet')
                    description = ""
                    if desc_elem:
                        description = self._clean_text(desc_elem.get_text().strip())
                    
                    # URL 검증
                    if not self._is_valid_url(url):
                        continue
                    
                    # 제목이나 URL이 비어있는 경우 스킵
                    if not title or not url:
                        continue
                    
                    # 중복 제거
                    if not any(r["url"] == url for r in results):
                        results.append({
                            'title': title,
                            'url': url,
                            'description': description or title,
                            'type': 'web',
                            'engine': 'duckduckgo'
                        })
                        
                except Exception as e:
                    logger.debug(f"DuckDuckGo 결과 처리 중 오류: {str(e)}")
                    continue
            
            logger.info(f"✅ DuckDuckGo 검색 완료: {len(results)}개 결과")
            return results
            
        except Exception as e:
            logger.error(f"❌ DuckDuckGo 검색 중 오류: {str(e)}")
            return []
    
    def _clean_text(self, text):
        """검색용 텍스트 정리"""
        if not text:
            return ""
        # HTML 태그 제거
        text = re.sub(r'<[^>]+>', '', text)
        # 특수문자 및 연속 공백 정리  
        text = re.sub(r'\s+', ' ', text)
        return text.strip()
    
    def _is_valid_url(self, url):
        """URL 유효성 검사"""
        return url and url.startswith(('http://', 'https://'))
        
    async def search(self, query: str, max_results: int = 10, engine: str = "auto") -> List[Dict[str, Any]]:
        """스마트 웹검색 - 성능 최적화된 검색 전략"""
        start_time = time.time()
        
        try:
            logger.info(f"⚡ 웹검색 시작: '{query}' (엔진: {engine}, 최대 {max_results}개)")
            
            if not self.driver_pool.initialized:
                logger.error("❌ 드라이버 풀이 초기화되지 않음")
                return []
            
            if engine == "google":
                # 구글만 사용
                return await self._search_single_engine("google", query, max_results)
                    
            elif engine == "naver":
                # 네이버만 사용
                return await self._search_single_engine("naver", query, max_results)
                    
            elif engine == "daum":
                # 다음만 사용
                return await self._search_single_engine("daum", query, max_results)
                    
            elif engine == "youtube":
                # 유튜브만 사용
                return await self._search_single_engine("youtube", query, max_results)
                    
            elif engine == "parallel":
                # 명시적 병렬 검색 요청 - 모든 엔진 병렬 실행
                logger.info("🔄 명시적 병렬 검색 요청 - 모든 엔진 동시 실행")
                return await self._search_parallel(query, max_results, start_time)
                    
            elif engine == "auto":
                # 🎯 AI 최적화 검색: 구글 3개 + 네이버 3개 (고품질 상세설명 보장)
                logger.info("🎯 AI 최적화 검색 전략: 구글 3개 + 네이버 3개 병렬 실행")
                
                # 병렬 실행을 위한 태스크 생성
                tasks = []
                
                # 구글 검색 태스크 (3개 - 글로벌 고품질 정보)
                google_task = asyncio.create_task(
                    self._search_single_engine_silent("google", query, 3)
                )
                tasks.append(("google", google_task))
                
                # 네이버 검색 태스크 (3개 - 한국 컨텐츠 + 상세설명 개선)
                naver_task = asyncio.create_task(
                    self._search_single_engine_silent("naver", query, 3)
                )
                tasks.append(("naver", naver_task))
                
                daum_task = asyncio.create_task(
                    self._search_single_engine_silent("daum", query, 3)
                )
                tasks.append(("daum", daum_task))
                
                youtube_task = asyncio.create_task(
                    self._search_single_engine_silent("youtube", query, 3)
                )
                tasks.append(("youtube", youtube_task))                                
                
                logger.info("🚀 구글 + 네이버 AI 최적화 병렬 검색 시작")
                
                # 병렬 실행 및 결과 수집
                all_results = []
                google_count = 0
                naver_count = 0
                daum_count = 0
                youtube_count = 0

                for engine_name, task in tasks:
                    try:
                        engine_results = await task
                        if engine_results:
                            all_results.extend(engine_results)
                            if engine_name == "google":
                                google_count = len(engine_results)
                            elif engine_name == "naver":
                                naver_count = len(engine_results)
                            elif engine_name == "daum":
                                daum_count = len(engine_results)
                            elif engine_name == "youtube":
                                youtube_count = len(engine_results)
                    except Exception as e:
                        logger.error(f"❌ {engine_name} 검색 실패: {e}")
                
                # 각 엔진별 결과 로그
                logger.info(f"✅ Google: {google_count}개 결과")
                logger.info(f"✅ Naver: {naver_count}개 결과")
                
                # auto 모드에서는 6개까지 허용 (구글 3개 + 네이버 3개)
                results = all_results[:10]
                
                elapsed = time.time() - start_time
                total_count = len(results)
                logger.info(f"🎯 AI 최적화 검색 완료: {total_count}개 결과 ({elapsed:.3f}초)")
                
                return results
                
            else:
                # 알 수 없는 엔진인 경우 auto 모드로 처리
                logger.warning(f"⚠️ 알 수 없는 엔진 '{engine}' - auto 모드로 처리")
                return await self.search(query, max_results, "auto")
                
        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"❌ 웹검색 실패: {str(e)} ({execution_time:.2f}초)")
            logger.error(f"상세 오류:\n{traceback.format_exc()}")
            return []
    
    async def _search_single_engine(self, engine_name: str, query: str, max_results: int) -> List[Dict[str, Any]]:
        """단일 엔진 검색"""
        try:
            if engine_name == "google":
                # Google 검색 시도, 실패시 DuckDuckGo로 fallback
                results = utils.search_google(query, max_results)
                if not results or len(results) == 0:
                    logger.info(f"🔄 Google 검색 실패, DuckDuckGo로 fallback...")
                    results = self.search_duckduckgo(query, max_results)
            elif engine_name == "naver":
                # 최적화된 네이버 검색 사용
                results = await self._search_naver_optimized(query, max_results, silent=False)
            elif engine_name == "daum":
                results = utils.search_daum(query, max_results)
            elif engine_name == "youtube":
                results = utils.search_youtube(query, max_results)
            else:
                return []
                
            if results and isinstance(results, list):
                for result in results:
                    result['engine'] = engine_name
                logger.info(f"✅ {engine_name.capitalize()}: {len(results)}개 결과")
                return results
            else:
                logger.warning(f"⚠️ {engine_name.capitalize()}: 결과 없음")
                return []
        except Exception as e:
            logger.error(f"❌ {engine_name.capitalize()} 검색 실패: {str(e)}")
            return []
    
    async def _search_naver_optimized(self, query: str, max_results: int, silent: bool = False) -> List[Dict[str, Any]]:
        """최적화된 네이버 검색 (실제 동작하는 버전)"""
        if not silent:
            logger.info(f"🚀 최적화된 네이버 검색 시작: '{query}'")
        
        driver = None
        results = []
        
        try:
            # utils.py와 동일한 방식으로 드라이버 가져오기
            driver = utils.get_selenium_driver()
            if not driver:
                if not silent:
                    logger.error("❌ 드라이버를 가져올 수 없음")
                return []
            
            # 네이버 검색 URL 생성
            search_url = f"https://search.naver.com/search.naver?where=nexearch&query={query}"
            
            # 페이지 로드
            driver.get(search_url)
            
            # 페이지 로딩 대기 (비동기 방식으로 변경)
            await asyncio.sleep(2)  # 3초에서 2초로 단축하고 비동기로 변경
            
            # 검색 결과 찾기 - 여러 선택자 시도
            elements = []
            
            # 1. 일반 웹 검색 결과
            web_results = driver.find_elements(By.CSS_SELECTOR, 'div.sc_new')
            elements.extend(web_results)
            
            # 2. API 기반 검색 결과  
            api_results = driver.find_elements(By.CSS_SELECTOR, 'div.api_subject_bx')
            elements.extend(api_results)
            
            # 3. 통합 검색 결과
            total_results = driver.find_elements(By.CSS_SELECTOR, 'div.total_wrap')
            elements.extend(total_results)
            
            if not elements:
                if not silent:
                    logger.warning("⚠️ 네이버 검색 결과 요소를 찾을 수 없음")
                return []
            
            # 결과 처리
            processed_urls = set()
            
            for element in elements:
                try:
                    # 광고 및 필터링 스킵
                    element_text = element.text
                    if ('파워링크' in element_text or '광고' in element_text or 
                        '관련도순' in element_text or '최신순' in element_text):
                        continue
                    
                    # 더 정확한 선택자로 제목과 링크 찾기
                    title_links = []
                    
                    # 다양한 제목 선택자 시도
                    selectors = [
                        'dt a',  # 일반 검색 결과
                        'a.link_tit',  # 블로그/카페 결과
                        'a.name',  # 뉴스 결과
                        'a.sub_txt',  # 기타 결과
                        'h3 a',  # 제목 링크
                        'h4 a',  # 서브 제목 링크
                        'a[href*="http"]'  # 모든 외부 링크
                    ]
                    
                    for selector in selectors:
                        found_links = element.find_elements(By.CSS_SELECTOR, selector)
                        title_links.extend(found_links)
                    
                    # 일반 링크도 찾기
                    all_links = element.find_elements(By.TAG_NAME, 'a')
                    title_links.extend(all_links)
                    
                    for link in title_links:
                        try:
                            href = link.get_attribute('href')
                            text = link.text.strip()
                            
                            # 유효한 외부 링크인지 확인
                            if (href and text and 
                                not href.startswith('javascript') and 
                                not href.startswith('#') and
                                'naver.com' not in href and
                                'search.naver.com' not in href and
                                len(text) > 3 and
                                href not in processed_urls):
                                
                                processed_urls.add(href)
                                
                                # 개선된 설명 추출
                                description = ""
                                try:
                                    # 1. 부모 요소의 전체 텍스트에서 설명 추출
                                    parent_element = link.find_element(By.XPATH, '..')
                                    parent_text = parent_element.text.strip()
                                    
                                    if parent_text and parent_text != text:
                                        # 제목 다음에 오는 텍스트를 설명으로 사용
                                        if text in parent_text:
                                            desc_start = parent_text.find(text) + len(text)
                                            remaining_text = parent_text[desc_start:].strip()
                                            if remaining_text:
                                                description = remaining_text[:300]  # 최대 300자
                                        
                                    # 2. 설명이 없으면 더 상위 요소에서 찾기
                                    if not description:
                                        try:
                                            grandparent = parent_element.find_element(By.XPATH, '..')
                                            grandparent_text = grandparent.text.strip()
                                            if grandparent_text and grandparent_text != parent_text:
                                                # 제목과 URL을 제외한 나머지 텍스트 추출
                                                lines = grandparent_text.split('\n')
                                                desc_lines = []
                                                for line in lines:
                                                    line = line.strip()
                                                    if (line and 
                                                        line != text and 
                                                        not line.startswith('http') and
                                                        len(line) > 10 and
                                                        '광고' not in line and
                                                        '파워링크' not in line):
                                                        desc_lines.append(line)
                                                
                                                if desc_lines:
                                                    description = ' '.join(desc_lines)[:300]
                                        except:
                                            pass
                                    
                                    # 3. 특정 클래스에서 설명 찾기
                                    if not description:
                                        try:
                                            desc_selectors = [
                                                '.txt_inline',
                                                '.dsc_txt',
                                                '.desc',
                                                '.description',
                                                '.txt',
                                                '.content'
                                            ]
                                            
                                            for desc_selector in desc_selectors:
                                                desc_elements = element.find_elements(By.CSS_SELECTOR, desc_selector)
                                                for desc_elem in desc_elements:
                                                    desc_text = desc_elem.text.strip()
                                                    if desc_text and len(desc_text) > 10:
                                                        description = desc_text[:300]
                                                        break
                                                if description:
                                                    break
                                        except:
                                            pass
                                    
                                except Exception as e:
                                    if not silent:
                                        logger.debug(f"설명 추출 실패: {e}")
                                
                                # 설명 정리
                                if description:
                                    # 불필요한 문자 제거
                                    description = description.replace('\n', ' ').replace('\r', ' ')
                                    description = ' '.join(description.split())  # 중복 공백 제거
                                    
                                    # 제목이 설명에 포함되어 있으면 제거
                                    if text in description:
                                        description = description.replace(text, '').strip()
                                    
                                    # 날짜 패턴 제거 (예: 2024.01.01, 2024-01-01)
                                    import re
                                    description = re.sub(r'\d{4}[-./]\d{1,2}[-./]\d{1,2}', '', description).strip()
                                    
                                    # 빈 설명이 되지 않도록 최소 길이 확인
                                    if len(description) < 10:
                                        description = ""
                                
                                # 제목 정리 (개행문자 및 불필요한 공백 제거)
                                clean_title = text.replace('\n', ' ').replace('\r', ' ')
                                clean_title = ' '.join(clean_title.split())  # 중복 공백 제거
                                
                                results.append({
                                    'title': clean_title,
                                    'url': href,
                                    'description': description
                                })
                                
                                if len(results) >= max_results:
                                    break
                        except Exception as e:
                            continue
                    
                    if len(results) >= max_results:
                        break
                        
                except Exception as e:
                    continue
            
            # 결과 후처리: 중복 제거 및 품질 향상
            unique_results = []
            seen_titles = set()
            
            for result in results:
                title_lower = result['title'].lower()
                if title_lower not in seen_titles:
                    seen_titles.add(title_lower)
                    unique_results.append(result)
            
            results = unique_results[:max_results]
            
            if not silent:
                logger.info(f"🚀 최적화된 네이버 검색 완료: {len(results)}개 결과")
                if results:
                    desc_count = len([r for r in results if r.get('description')])
                    logger.info(f"📄 설명이 있는 결과: {desc_count}개/{len(results)}개")
            
        except Exception as e:
            if not silent:
                logger.error(f"❌ 네이버 검색 중 오류: {e}")
        finally:
            # 드라이버 반환 (monkey patch 사용)
            if driver and hasattr(self, 'driver_pool'):
                try:
                    self.driver_pool.return_driver(driver)
                except:
                    # 반환 실패 시 종료
                    try:
                        driver.quit()
                    except:
                        pass
        
        return results
    
    def _clean_text_for_search(self, text: str) -> str:
        """검색용 텍스트 정리"""
        if not text:
            return ""
        return text.strip().replace('\n', ' ').replace('\r', ' ')
    
    def _is_valid_url(self, url: str) -> bool:
        """URL 유효성 검사"""
        if not url:
            return False
        return url.startswith(('http://', 'https://')) and len(url) > 10
    
    async def _search_smart(self, query: str, max_results: int, start_time: float) -> List[Dict[str, Any]]:
        """스마트 검색 전략: 구글 우선 → 부족하면 다른 엔진으로 보완"""
        # 1단계: 구글로 먼저 검색 (가장 빠름)
        logger.info("🎯 1단계: 구글 검색 (최고 성능)")
        google_results = await self._search_single_engine("google", query, max_results)
        
        if len(google_results) >= max_results:
            # 구글 결과만으로 충분
            execution_time = time.time() - start_time
            logger.info(f"✅ 구글만으로 충분: {len(google_results)}개 결과 ({execution_time:.2f}초)")
            return google_results[:max_results]
        
        # 2단계: 구글 결과가 부족하면 다음으로 보완 (두 번째로 빠름)
        needed = max_results - len(google_results)
        logger.info(f"🔄 2단계: 다음 검색으로 보완 (부족: {needed}개)")
        
        daum_results = await self._search_single_engine("daum", query, needed)
        all_results = google_results + daum_results
        
        if len(all_results) >= max_results:
            # 구글 + 다음으로 충분
            execution_time = time.time() - start_time
            logger.info(f"✅ 구글+다음으로 충분: {len(all_results)}개 결과 ({execution_time:.2f}초)")
            return all_results[:max_results]
        
        # 3단계: 여전히 부족하면 네이버로 최종 보완
        still_needed = max_results - len(all_results)
        logger.info(f"🔄 3단계: 네이버 검색으로 최종 보완 (부족: {still_needed}개)")
        
        naver_results = await self._search_single_engine("naver", query, still_needed)
        final_results = all_results + naver_results
        
        execution_time = time.time() - start_time
        logger.info(f"✅ 스마트 검색 완료: {len(final_results)}개 결과 ({execution_time:.2f}초)")
        return final_results[:max_results]
    
    async def _search_parallel(self, query: str, max_results: int, start_time: float) -> List[Dict[str, Any]]:
        """병렬 검색 (모든 엔진 동시 실행)"""
        with ThreadPoolExecutor(max_workers=4) as executor:
            # 각 검색 엔진별 태스크 생성 (유튜브 추가)
            search_tasks = {
                executor.submit(utils.search_google, query, max_results): "google",
                executor.submit(utils.search_naver, query, max_results): "naver", 
                executor.submit(utils.search_daum, query, max_results): "daum",
                executor.submit(utils.search_youtube, query, max_results): "youtube"
            }
            
            # 결과 수집
            all_results = []
            seen_urls = set()
            
            for future in search_tasks:
                try:
                    engine_name = search_tasks[future]
                    results = future.result(timeout=8)  # 8초 타임아웃
                    
                    if results and isinstance(results, list):
                        logger.info(f"✅ {engine_name.capitalize()}: {len(results)}개 결과")
                        
                        for result in results:
                            url = result.get('url', '')
                            if url and url not in seen_urls:
                                seen_urls.add(url)
                                # 엔진 정보 추가
                                result['engine'] = engine_name
                                all_results.append(result)
                    else:
                        logger.warning(f"⚠️ {engine_name.capitalize()}: 결과 없음")
                        
                except Exception as e:
                    engine_name = search_tasks[future]
                    logger.warning(f"❌ {engine_name.capitalize()} 검색 실패: {str(e)}")
            
            # 결과 정렬 및 제한
            final_results = all_results[:max_results]
            execution_time = time.time() - start_time
            logger.info(f"⚡ 병렬 검색 완료: {len(final_results)}개 결과 ({execution_time:.2f}초)")
            return final_results

    async def _search_single_engine_silent(self, engine_name: str, query: str, max_results: int) -> List[Dict[str, Any]]:
        """단일 엔진 검색 (로그 없음 - auto 모드용)"""
        try:
            if engine_name == "google":
                # Google 검색 시도, 실패시 DuckDuckGo로 fallback
                results = utils.search_google(query, max_results)
                if not results or len(results) == 0:
                    logger.info(f"🔄 Google 검색 실패, DuckDuckGo로 fallback...")
                    results = self.search_duckduckgo(query, max_results)
            elif engine_name == "naver":
                # 최적화된 네이버 검색 사용
                results = await self._search_naver_optimized(query, max_results, silent=True)
            elif engine_name == "daum":
                results = utils.search_daum(query, max_results)
            elif engine_name == "youtube":
                results = utils.search_youtube(query, max_results)
            else:
                return []
                
            if results and isinstance(results, list):
                for result in results:
                    result['engine'] = engine_name
                return results
            else:
                return []
                
        except Exception as e:
            # 에러 로그는 출력 (중요한 정보)
            logger.error(f"❌ {engine_name} 검색 실패: {e}")
            return []

# =============================================================================
# API 모델들
# =============================================================================

class WebSearchRequest(BaseModel):
    """웹검색 요청"""
    query: str = Field(..., description="검색 쿼리")
    max_results: int = Field(default=5, description="최대 결과 수", ge=1, le=20)
    engine: str = Field(default="auto", description="검색 엔진 (auto: AI 최적화 - 구글3+네이버3, google, naver, daum, youtube, parallel: 명시적 병렬)")
    use_cache: bool = Field(default=True, description="캐시 사용 여부")

class WebSearchResponse(BaseModel):
    """웹검색 응답"""
    success: bool
    results: List[Dict[str, Any]]
    query: str
    total_results: int
    execution_time: float
    cached: bool = False
    engine_stats: Optional[Dict[str, int]] = None
    error: Optional[str] = None
    optimization_info: Optional[Dict[str, Any]] = None

class HealthResponse(BaseModel):
    """상태 확인 응답"""
    status: str
    service: str
    version: str
    timestamp: str
    uptime: str
    stats: Dict[str, Any]

# =============================================================================
# 캐시 시스템
# =============================================================================

class WebSearchCache:
    """초고속 메모리 캐시"""
    
    def __init__(self):
        self.cache = {}
        self.max_size = 2000
        self.ttl = 1800  # 30분
        
    def _generate_key(self, query: str, max_results: int, engine: str) -> str:
        """캐시 키 생성"""
        return f"{query.strip().lower()}:{max_results}:{engine}"
    
    def get(self, query: str, max_results: int, engine: str) -> Optional[List[Dict]]:
        """캐시에서 결과 조회"""
        key = self._generate_key(query, max_results, engine)
        item = self.cache.get(key)
        
        if not item:
            return None
            
        # TTL 체크
        if time.time() - item['timestamp'] > self.ttl:
            del self.cache[key]
            return None
            
        server_stats["cache_hits"] += 1
        return item['data']
    
    def set(self, query: str, max_results: int, engine: str, results: List[Dict]):
        """캐시에 결과 저장"""
        if len(self.cache) >= self.max_size:
            # 가장 오래된 항목 제거
            oldest_key = min(self.cache.keys(), key=lambda k: self.cache[k]['timestamp'])
            del self.cache[oldest_key]
            
        key = self._generate_key(query, max_results, engine)
        self.cache[key] = {
            'data': results,
            'timestamp': time.time()
        }
        
        server_stats["cache_misses"] += 1
    
    def clear(self):
        """캐시 전체 삭제"""
        self.cache.clear()

# =============================================================================
# 초고속 검색 실행 함수
# =============================================================================

async def execute_ultra_fast_search(query: str, max_results: int = 5, engine: str = "auto", use_cache: bool = True) -> Dict[str, Any]:
    """초고속 웹검색 실행 (utils.py + monkey patch)"""
    start_time = time.time()
    
    try:
        # 캐시 확인
        cached_results = None
        if use_cache and websearch_cache:
            cached_results = websearch_cache.get(query, max_results, engine)
            if cached_results:
                execution_time = time.time() - start_time
                logger.info(f"캐시 히트: {len(cached_results)}개 결과 ({execution_time*1000:.1f}ms)")
                
                return {
                    "success": True,
                    "results": cached_results,
                    "query": query,
                    "total_results": len(cached_results),
                    "execution_time": execution_time,
                    "cached": True,
                    "engine_stats": None,
                    "error": None,
                    "optimization_info": {
                        "cache_hit": True,
                        "monkey_patch_used": False,
                        "performance": "⚡ ULTRA_FAST"
                    }
                }
        
        # 드라이버 풀이 초기화되지 않은 경우
        if not driver_pool or not driver_pool.initialized:
            raise RuntimeError("드라이버 풀이 초기화되지 않았습니다")
        
        # 웹검색 엔진으로 검색 수행
        web_search_engine = WebSearch(driver_pool)
        results = await web_search_engine.search(query, max_results, engine)
        
        # 엔진별 결과 통계
        if engine == "google":
            engine_stats = {"google": len(results), "naver": 0, "daum": 0, "youtube": 0}
        elif engine == "naver":
            engine_stats = {"google": 0, "naver": len(results), "daum": 0, "youtube": 0}
        elif engine == "daum":
            engine_stats = {"google": 0, "naver": 0, "daum": len(results), "youtube": 0}
        elif engine == "youtube":
            engine_stats = {"google": 0, "naver": 0, "daum": 0, "youtube": len(results)}
        elif engine == "auto":
            # auto 모드: 구글 + 네이버만 (AI 최적화)
            engine_stats = {
                "google": len([r for r in results if r.get('engine') == 'google']),
                "naver": len([r for r in results if r.get('engine') == 'naver']),
                "daum": 0,  # auto 모드에서는 제외
                "youtube": 0  # auto 모드에서는 제외
            }
        else:  # parallel
            # parallel 모드: 모든 엔진 사용
            engine_stats = {
                "google": len([r for r in results if r.get('engine') == 'google']),
                "naver": len([r for r in results if r.get('engine') == 'naver']),
                "daum": len([r for r in results if r.get('engine') == 'daum']),
                "youtube": len([r for r in results if r.get('engine') == 'youtube'])
            }
        
        # 캐시에 저장
        if use_cache and websearch_cache and results:
            websearch_cache.set(query, max_results, engine, results)
        
        execution_time = time.time() - start_time
        
        # 평균 응답 시간 업데이트
        if server_stats["total_requests"] > 0:
            server_stats["avg_response_time"] = (
                (server_stats["avg_response_time"] * (server_stats["total_requests"] - 1) + execution_time) /
                server_stats["total_requests"]
            )
        
        return {
            "success": True,
            "results": results,
            "query": query,
            "total_results": len(results),
            "execution_time": execution_time,
            "cached": False,
            "engine_stats": engine_stats,
            "error": None,
            "optimization_info": {
                "cache_hit": False,
                "monkey_patch_used": True,
                "driver_reuse_count": server_stats["driver_reuse_count"],
                "available_drivers": len(driver_pool.available_drivers),
                "performance": "⚡ ULTRA_FAST" if execution_time < 3 else ("🚀 FAST" if execution_time < 6 else "🐢 SLOW")
            }
        }
        
    except Exception as e:
        execution_time = time.time() - start_time
        error_msg = f"초고속 검색 실행 오류: {str(e)}"
        logger.error(error_msg)
        logger.error(f"상세 오류:\n{traceback.format_exc()}")
        
        return {
            "success": False,
            "results": [],
            "query": query,
            "total_results": 0,
            "execution_time": execution_time,
            "cached": False,
            "engine_stats": None,
            "error": error_msg,
            "optimization_info": {
                "cache_hit": False,
                "monkey_patch_used": False,
                "performance": "❌ ERROR"
            }
        }

# =============================================================================
# FastAPI 앱 설정
# =============================================================================

@asynccontextmanager
async def lifespan(app: FastAPI):
    """앱 라이프사이클 관리"""
    global websearch_cache, driver_pool
    
    # 시작
    logger.info("🚀 AIRUN 초고속 웹검색 서비스 시작 (Monkey Patch 최적화)")
    server_stats["start_time"] = datetime.now()
    
    # 캐시 초기화
    websearch_cache = WebSearchCache()
    
    # 초고속 드라이버 풀 초기화 (Monkey Patch 적용!)
    driver_pool = UltraFastDriverPool(pool_size=5)  # 3에서 5로 증가하여 병렬 처리 개선
    if not driver_pool.initialize():
        logger.error("❌ 초고속 드라이버 풀 초기화 실패 - 서비스를 종료합니다")
        sys.exit(1)
    
    logger.info("⚡ AIRUN 초고속 웹검색 서비스 준비 완료 (Monkey Patch 최적화)")
    
    yield
    
    # 종료
    logger.info("🛑 AIRUN 초고속 웹검색 서비스 종료")
    if driver_pool:
        driver_pool.cleanup()

app = FastAPI(
    title="AIRUN 초고속 웹검색 서비스 (Monkey Patch)",
    description="utils.py Monkey Patch로 초고속 웹검색 API",
    version="5.0.0",
    lifespan=lifespan
)

# CORS 설정
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# =============================================================================
# API 엔드포인트
# =============================================================================

@app.get("/health", response_model=HealthResponse)
async def health_check():
    """서비스 상태 확인"""
    uptime = "0초"
    if server_stats["start_time"]:
        uptime_delta = datetime.now() - server_stats["start_time"]
        uptime = str(uptime_delta).split('.')[0]
    
    return HealthResponse(
        status="healthy",
        service="airun-websearch-server-monkeypatch",
        version="5.0.0",
        timestamp=datetime.now().isoformat(),
        uptime=uptime,
        stats={
            "total_requests": server_stats["total_requests"],
            "successful_searches": server_stats["successful_searches"],
            "failed_searches": server_stats["failed_searches"],
            "cache_hits": server_stats["cache_hits"],
            "cache_misses": server_stats["cache_misses"],
            "cache_hit_rate": (
                server_stats["cache_hits"] / max(1, server_stats["cache_hits"] + server_stats["cache_misses"])
            ) * 100,
            "avg_response_time": server_stats["avg_response_time"],
            "driver_reuse_count": server_stats["driver_reuse_count"],
            "driver_pool_initialized": driver_pool.initialized if driver_pool else False,
            "available_drivers": len(driver_pool.available_drivers) if driver_pool else 0,
            "total_drivers": len(driver_pool.drivers) if driver_pool else 0,
            "cache_size": len(websearch_cache.cache) if websearch_cache else 0
        }
    )

@app.post("/search", response_model=WebSearchResponse)
async def search_web(request: WebSearchRequest):
    """고속 웹검색 실행 (Monkey Patch 최적화)"""
    try:
        server_stats["total_requests"] += 1
        
        logger.info(f"웹검색 요청: '{request.query}'")
        
        # 초고속 검색 실행
        result = await execute_ultra_fast_search(
            query=request.query,
            max_results=request.max_results,
            engine=request.engine,
            use_cache=request.use_cache
        )
        
        if result["success"]:
            server_stats["successful_searches"] += 1
            perf_icon = result.get("optimization_info", {}).get("performance", "")
            logger.info(f"✅ 검색 완료: {result['total_results']}개 결과 ({result['execution_time']:.3f}초) {perf_icon}")
            if result.get("cached"):
                logger.info("⚡ 캐시된 결과 사용")
        else:
            server_stats["failed_searches"] += 1
            logger.error(f"❌ 검색 실패: {result.get('error', '알 수 없는 오류')}")
        
        return WebSearchResponse(**result)
        
    except Exception as e:
        server_stats["failed_searches"] += 1
        error_msg = f"초고속 웹검색 API 오류: {str(e)}"
        logger.error(error_msg)
        logger.error(f"상세 오류:\n{traceback.format_exc()}")
        
        raise HTTPException(status_code=500, detail=error_msg)

@app.post("/cache/clear")
async def clear_cache():
    """캐시 삭제 - 빠른 응답을 위한 백그라운드 처리"""
    try:
        # 즉시 응답하고 백그라운드에서 처리
        asyncio.create_task(background_cache_clear())
        logger.info("🗑️ 웹검색 캐시 삭제 요청 접수")
        return {"message": "캐시 삭제가 요청되었습니다", "success": True}
    except Exception as e:
        error_msg = f"캐시 삭제 요청 실패: {str(e)}"
        logger.error(error_msg)
        raise HTTPException(status_code=500, detail=error_msg)

async def background_cache_clear():
    """백그라운드에서 실제 캐시 삭제 수행"""
    try:
        if websearch_cache:
            websearch_cache.clear()
        logger.info("🗑️ 웹검색 캐시 삭제 완료 (백그라운드)")
    except Exception as e:
        logger.error(f"백그라운드 캐시 삭제 오류: {e}")

@app.get("/stats")
async def get_stats():
    """서비스 통계"""
    try:
        uptime = "0초"
        if server_stats["start_time"]:
            uptime_delta = datetime.now() - server_stats["start_time"]
            uptime = str(uptime_delta).split('.')[0]
            
        return {
            "service": "airun-websearch-server-monkeypatch",
            "version": "5.0.0",
            "uptime": uptime,
            "stats": server_stats,
            "cache_size": len(websearch_cache.cache) if websearch_cache else 0,
            "driver_pool_status": {
                "initialized": driver_pool.initialized if driver_pool else False,
                "total_drivers": len(driver_pool.drivers) if driver_pool else 0,
                "available_drivers": len(driver_pool.available_drivers) if driver_pool else 0,
                "busy_drivers": len(driver_pool.drivers) - len(driver_pool.available_drivers) if driver_pool else 0,
                "reuse_count": server_stats["driver_reuse_count"]
            },
            "optimization_features": [
                "✅ Monkey Patch utils.get_selenium_driver",
                "✅ 드라이버 풀 미리 로딩",
                "✅ 병렬 검색 최적화", 
                "✅ 초고속 메모리 캐시",
                "✅ ThreadPoolExecutor 병렬 처리",
                "🎯 AI 최적화 검색 (구글3개+네이버3개 고품질)",
                "🔧 네이버 상세설명 추출 개선"
            ]
        }
    except Exception as e:
        error_msg = f"통계 조회 실패: {str(e)}"
        logger.error(error_msg)
        raise HTTPException(status_code=500, detail=error_msg)

# =============================================================================
# 메인 함수
# =============================================================================

def main():
    """서버 시작"""
    
    # 시그널 핸들러 설정
    def signal_handler(signum, frame):
        logger.info(f"🛑 시그널 {signum} 수신 - 서버 종료 중...")
        if 'driver_pool' in globals() and driver_pool:
            driver_pool.cleanup()
        sys.exit(0)
    
    signal.signal(signal.SIGTERM, signal_handler)
    signal.signal(signal.SIGINT, signal_handler)
    
    try:
        # 환경 설정
        host = os.getenv('WEBSEARCH_HOST', '0.0.0.0')
        port = int(os.getenv('WEBSEARCH_PORT', 5610))
        
        logger.info(f"⚡ 초고속 서버 시작: http://{host}:{port}")
        
        # Uvicorn 서버 실행
        uvicorn.run(
            app,
            host=host,
            port=port,
            log_level="info",
            access_log=True,
            workers=1  # Monkey Patch 관리를 위해 단일 워커 사용
        )
        
    except Exception as e:
        logger.error(f"❌ 서버 시작 실패: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main() 