/**
 * @swagger
 * tags:
 *   - name: Task
 *     description: 태스크 관리 관련 API
 */

/**
 * @swagger
 * /api/v1/tasks:
 *   get:
 *     summary: 태스크 목록 조회
 *     description: 모든 태스크의 목록을 조회합니다.
 *     tags: [Task]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     tasks:
 *                       type: array
 *                       items:
 *                         $ref: '#/components/schemas/Task'
 *                     count:
 *                       type: integer
 *                       description: 전체 태스크 수
 *             example:
 *               success: true
 *               data: {
 *                 tasks: [{
 *                   id: "1682334455-abc123",
 *                   name: "파이썬 계산기",
 *                   prompt: "간단한 계산기 만들기",
 *                   code: "def calculator():\n    pass",
 *                   createdAt: "2025-04-22T05:30:08.520Z",
 *                   lastUsed: "2025-04-22T05:30:08.520Z",
 *                   executionCount: 1,
 *                   metadata: {
 *                     lastExecutionTime: "2025-04-22T05:30:08.520Z",
 *                     successRate: "100%"
 *                   }
 *                 }],
 *                 count: 1
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *   post:
 *     summary: 새로운 태스크 생성
 *     description: 새로운 태스크를 생성하고 저장합니다.
 *     tags: [Task]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/TaskCreateRequest'
 *           example:
 *             name: "파이썬 계산기"
 *             prompt: "간단한 계산기 만들기"
 *             code: "def calculator():\n    pass"
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     task:
 *                       $ref: '#/components/schemas/Task'
 *             example:
 *               success: true
 *               data: {
 *                 task: {
 *                   id: "1682334455-abc123",
 *                   name: "파이썬 계산기",
 *                   prompt: "간단한 계산기 만들기",
 *                   code: "def calculator():\n    pass",
 *                   createdAt: "2025-04-22T05:30:08.520Z",
 *                   lastUsed: "2025-04-22T05:30:08.520Z",
 *                   executionCount: 0,
 *                   metadata: null
 *                 }
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       400:
 *         $ref: '#/components/responses/BadRequestError'
 */

/**
 * @swagger
 * /api/v1/tasks/{id}:
 *   delete:
 *     summary: 태스크 삭제
 *     description: 특정 태스크를 삭제합니다.
 *     tags: [Task]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: 태스크 ID
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *             example:
 *               success: true
 *               message: "태스크가 삭제되었습니다."
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       404:
 *         $ref: '#/components/responses/NotFoundError'
 */

/**
 * @swagger
 * /api/v1/tasks/{id}/execute:
 *   post:
 *     summary: 태스크 실행
 *     description: 특정 태스크를 실행합니다.
 *     tags: [Task]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: 태스크 ID
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     task:
 *                       $ref: '#/components/schemas/Task'
 *                     result:
 *                       type: object
 *                       properties:
 *                         code:
 *                           type: string
 *                         output:
 *                           type: string
 *                         error:
 *                           type: string
 *             example:
 *               success: true
 *               data: {
 *                 task: {
 *                   id: "1682334455-abc123",
 *                   name: "파이썬 계산기",
 *                   prompt: "간단한 계산기 만들기",
 *                   code: "def calculator():\n    pass",
 *                   createdAt: "2025-04-22T05:30:08.520Z",
 *                   lastUsed: "2025-04-22T05:30:08.520Z",
 *                   executionCount: 2,
 *                   metadata: {
 *                     lastExecutionTime: "2025-04-22T05:35:08.520Z",
 *                     successRate: "100%"
 *                   }
 *                 },
 *                 result: {
 *                   code: "def calculator():\n    pass",
 *                   output: "실행 결과",
 *                   error: null
 *                 }
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       404:
 *         $ref: '#/components/responses/NotFoundError'
 */

export const taskSchemas = {
    Task: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: '태스크 ID'
            },
            name: {
                type: 'string',
                description: '태스크 이름'
            },
            prompt: {
                type: 'string',
                description: '태스크 프롬프트'
            },
            code: {
                type: 'string',
                description: '생성된 코드'
            },
            createdAt: {
                type: 'string',
                format: 'date-time',
                description: '생성 시간'
            },
            lastUsed: {
                type: 'string',
                format: 'date-time',
                description: '마지막 사용 시간'
            },
            executionCount: {
                type: 'integer',
                description: '실행 횟수'
            },
            metadata: {
                type: 'object',
                properties: {
                    lastExecutionTime: {
                        type: 'string',
                        format: 'date-time',
                        description: '마지막 실행 시간'
                    },
                    successRate: {
                        type: 'string',
                        description: '성공률'
                    }
                }
            }
        }
    },
    TaskCreateRequest: {
        type: 'object',
        required: ['name', 'prompt'],
        properties: {
            name: {
                type: 'string',
                description: '태스크 이름'
            },
            prompt: {
                type: 'string',
                description: '태스크 프롬프트'
            },
            code: {
                type: 'string',
                description: '생성된 코드'
            }
        }
    }
}; 