# AI Tool System

이 디렉토리는 AI 시스템에서 사용할 수 있는 도구들을 정의하고 관리하는 확장 가능한 도구 시스템입니다.

## 📁 디렉토리 구조

```
tools/
├── tools-registry.js          # 도구 레지스트리 시스템 (핵심)
├── tool-loader.js            # 도구 자동 로더
├── builtin-tools.js          # 기본 내장 도구들
├── custom-tools/             # 커스텀 도구 디렉토리
│   └── example-tools.js      # 도구 작성 예제
└── README.md                 # 이 파일
```

## 🚀 사용법

### 시스템 초기화 시 도구 로드

```javascript
import toolLoader from './tools/tool-loader.js';

// 시스템 시작 시 모든 도구 로드
await toolLoader.loadAllTools();
```

### 도구 실행

```javascript
import toolRegistry from './tools/tools-registry.js';

// 도구 실행
const result = await toolRegistry.executeTool('calculator', {
    operation: 'add',
    a: 5,
    b: 3
}, userId, job);

console.log(result); // { success: true, data: { result: 8 }, message: '계산 완료: 5 add 3 = 8' }
```

### 등록된 도구 조회

```javascript
// 모든 도구 이름 가져오기
const toolNames = toolRegistry.getTools();

// 카테고리별 도구 목록
const toolsByCategory = toolRegistry.getToolsByCategory();

// 특정 도구 정보
const toolInfo = toolRegistry.getToolInfo('calculator');
```

## 🛠️ 새로운 도구 만들기

### 1. 도구 정의 구조

```javascript
const myTool = {
    name: 'my_tool',                    // 도구 이름 (필수)
    category: 'utility',                // 카테고리 (선택, 기본값: 'general')
    description: '내 도구 설명',        // 도구 설명
    version: '1.0.0',                   // 버전 (선택, 기본값: '1.0.0')
    parameters: {                       // 매개변수 정의
        param1: { 
            type: 'string', 
            required: true, 
            description: '첫 번째 매개변수' 
        },
        param2: { 
            type: 'number', 
            default: 10, 
            description: '두 번째 매개변수' 
        }
    },
    handler: async (args, context) => { // 도구 실행 핸들러 (필수)
        const { param1, param2 = 10 } = args;
        const { userId, job, toolName, version } = context;
        
        try {
            // 도구 로직 구현
            const result = await someOperation(param1, param2);
            
            return {
                success: true,
                data: result,
                message: '작업 완료'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: '작업 실패'
            };
        }
    }
};
```

### 2. 매개변수 타입

- `string`: 문자열
- `number`: 숫자
- `boolean`: 불린값
- `array`: 배열
- `object`: 객체

### 3. 매개변수 옵션

- `required`: 필수 여부 (boolean)
- `default`: 기본값
- `description`: 설명
- `enum`: 허용되는 값들의 배열

### 4. 핸들러 컨텍스트

핸들러 함수는 두 번째 매개변수로 컨텍스트 객체를 받습니다:

```javascript
{
    userId: 'user123',           // 사용자 ID
    job: jobObject,              // 작업 객체 (진행 상황 업데이트용)
    toolName: 'my_tool',         // 도구 이름
    version: '1.0.0',            // 도구 버전
    registeredAt: '2024-01-01'   // 등록 시간
}
```

### 5. 반환 형식

모든 도구는 다음 형식으로 결과를 반환해야 합니다:

```javascript
// 성공 시
{
    success: true,
    data: { /* 결과 데이터 */ },
    message: '작업 완료'
}

// 실패 시
{
    success: false,
    error: 'error message',
    message: '사용자에게 표시할 메시지'
}
```

## 📦 도구 배포 방법

### 방법 1: 파일에 도구들 정의

```javascript
// my-tools.js
const tool1 = { /* 도구 정의 */ };
const tool2 = { /* 도구 정의 */ };

export default [tool1, tool2];
```

### 방법 2: custom-tools 디렉토리에 추가

`tools/custom-tools/` 디렉토리에 `.js` 파일을 추가하면 자동으로 로드됩니다.

### 방법 3: 사용자 정의 디렉토리

프로젝트 루트에 `user-tools/` 디렉토리를 만들고 도구 파일들을 추가하면 자동으로 로드됩니다.

### 방법 4: 런타임에 등록

```javascript
import toolRegistry from './tools/tools-registry.js';

// 개별 도구 등록
toolRegistry.registerTool(myTool);

// 파일에서 로드
await toolRegistry.loadToolsFromFile('./my-tools.js');

// 디렉토리에서 로드
await toolRegistry.loadToolsFromDirectory('./my-tools-dir');
```

## 🔧 고급 기능

### 진행 상황 업데이트

장시간 실행되는 도구에서는 진행 상황을 업데이트할 수 있습니다:

```javascript
handler: async (args, context) => {
    const { job } = context;
    
    if (job && job.progress) {
        await job.progress(30, '작업 30% 완료...');
        // 작업 수행
        await job.progress(60, '작업 60% 완료...');
        // 더 많은 작업
        await job.progress(100, '작업 완료!');
    }
    
    return { success: true, data: result, message: '완료' };
}
```

### 도구 검색

```javascript
import toolLoader from './tools/tool-loader.js';

// 키워드로 도구 검색
const searchResults = toolLoader.searchTools('계산');
```

### 도구 통계

```javascript
const stats = toolLoader.getToolStats();
console.log(stats);
// {
//   total: 15,
//   categories: 5,
//   byCategory: {
//     utility: 3,
//     search: 4,
//     communication: 2,
//     memory: 3,
//     validation: 3
//   }
// }
```

## 📝 카테고리

권장되는 도구 카테고리:

- `search`: 검색 관련
- `communication`: 통신 (이메일, 메시지 등)
- `utility`: 유틸리티 (계산, 변환 등)
- `filesystem`: 파일 시스템
- `data`: 데이터 처리
- `memory`: 메모리/저장
- `validation`: 검증/품질 확인
- `research`: 리서치/분석
- `information`: 정보 조회

## 🐛 디버깅

도구 실행 과정은 콘솔에 자세히 로깅됩니다:

```
[ToolRegistry] 도구 등록됨: calculator (utility)
[ToolRegistry] 도구 실행: calculator
[calculator] 계산 수행: 5 add 3 (사용자: user123)
[ToolRegistry] 도구 실행 완료: calculator
```

## ⚠️ 보안 고려사항

- 파일 시스템 접근 시 경로 검증 필수
- 사용자 입력 데이터 검증
- 외부 API 호출 시 API 키 보안
- 민감한 정보 로깅 금지

## 🔄 마이그레이션

기존 하드코딩된 도구에서 새 시스템으로 마이그레이션:

1. 기존 도구를 새 형식으로 변환
2. `builtin-tools.js`에 추가
3. `mainLogic.js`에서 해당 case 제거
4. 테스트 후 레거시 코드 정리 