// 공통 로그 유틸리티 임포트
import { debugLog, logger } from '../utils/logger.js';
import path from 'path';
import os from 'os';
import { URLs } from '../config/urls.js';

// 기본 내장 도구들 정의

// AI프로 리서치 도구
// const aiProTool = {
//     name: 'ai_pro',
//     category: 'research',
//     description: '웹과 문서를 통합 검색하여 고품질 응답을 생성합니다. 사용자가 요청한 정보를 검색하고 분석한 후 결과만 반환합니다. 사전 설명이나 작업 시간 안내 없이 바로 검색 결과를 제공합니다.',
//     parameters: {
//         query: { type: 'string', required: true, description: '검색할 질문' },
//         search_types: { type: 'array', default: ['both'], description: '검색 유형 (web, documents, both)' },
//         quality_threshold: { type: 'number', default: 7, description: '품질 임계값 (1-10)' },
//         enable_cross_validation: { type: 'boolean', default: false, description: '교차 검증 활성화' },
//         max_iterations: { type: 'number', default: 3, description: '최대 반복 횟수' }
//     },
//     handler: async (args, context) => {
//         const { userId, job, sessionId, updateProgress } = context;
//         const { 
//             query, 
//             search_types = ['both'], 
//             quality_threshold = 7, 
//             enable_cross_validation = false, 
//             max_iterations = 3 
//         } = args;
        
//                     logger.debug('\n🧠 === AI프로 리서치 시작 ===');
//             logger.debug(`📝 질문: ${query}`);
//             logger.debug(`🔍 검색 유형: ${search_types.join(', ')}`);
//             logger.debug(`📊 품질 임계값: ${quality_threshold}/10`);
//             logger.debug(`🔄 교차 검증: ${enable_cross_validation ? 'ON' : 'OFF'}`);
//             logger.debug(`🔁 최대 반복: ${max_iterations}회`);
        
//         let iteration = 0;
//         let finalResponse = '';
//         let allSearchResults = [];
//         let qualityScore = 0;
//         let webResultsCount = 0;
//         let docResultsCount = 0;
//         let improvementAttempts = 0; // 변수 초기화 추가
        
//         // 검색 결과 배열을 try 블록 밖에서 선언 (스코프 문제 해결)
//         let webResults = [];
//         let docResults = [];
        
//         try {
//             logger.debug(`[AI_PRO] 실행 시작 - sessionId: ${sessionId}, userId: ${userId}`);
            
//             const shouldSearchWeb = search_types.includes('web') || search_types.includes('both');
//             const shouldSearchDocuments = search_types.includes('documents') || search_types.includes('both');
            
//             logger.debug(`[AI_PRO] 검색 설정 - Web: ${shouldSearchWeb}, Documents: ${shouldSearchDocuments}`);
            
//             if (updateProgress) {
//                 await updateProgress(`🧠 AI프로 리서치 시작: ${query.substring(0, 50)}...`, 10);
//             }
            
//             logger.debug('\n📊 === 1단계: 초기 검색 ===');
//             iteration++;
            
//             // 검색 결과 배열 초기화 (이미 위에서 선언됨)
//             webResults = [];
//             docResults = [];
            
//             logger.debug(`[AI_PRO] 변수 초기화 완료 - webResults: ${Array.isArray(webResults)}, docResults: ${Array.isArray(docResults)}`);
            
//             if (shouldSearchWeb) {
//                 logger.debug('🔍 AI프로 리서치: 웹 정보 수집 중...');
//                 if (updateProgress) await updateProgress('🔍 AI프로 리서치: 웹 정보 수집 중...', 25);
//                 try {
//                     const { WebSearchService } = await import('../plugins/websearch/webSearchService.js');
//                     await WebSearchService.checkIfEnabled(true);
//                     const searchResult = await WebSearchService.searchWeb(query, 'auto', 8);
//                     webResults = Array.isArray(searchResult) ? searchResult : [];
//                     webResultsCount = webResults.length;
//                     logger.debug(`✅ 웹 검색 완료: ${webResultsCount}개 결과`);
//                     if (updateProgress && webResultsCount > 0) await updateProgress(`✅ 웹 검색 완료: ${webResultsCount}개 결과 발견`, 35);
//                 } catch (error) {
//                     console.error('❌ 웹 검색 실패:', error);
//                     webResults = [];
//                     webResultsCount = 0;
//                     if (updateProgress) await updateProgress(`❌ 웹 검색 실패: ${error.message}`, 35);
//                 }
//             }
            
//             if (shouldSearchDocuments) {
//                 logger.debug('📋 AI프로 리서치: 문서 정보 수집 중...');
//                 if (updateProgress) await updateProgress('📋 AI프로 리서치: 문서 정보 수집 중...', 40);
//                 try {
//                     const { RAGService } = await import('../plugins/rag/ragService.js');
//                     await RAGService.checkIfEnabled(true);
//                     const searchUserId = (!userId || userId === 1) ? 'all' : String(userId);
//                     const ragResults = await RAGService.searchDocuments(query, searchUserId);
                    
//                     if (ragResults && ragResults.documents && ragResults.documents[0]) {
//                         docResults = ragResults.documents[0].map((doc, i) => ({
//                             document: doc,
//                             metadata: ragResults.metadatas[0][i] || {}
//                         }));
//                     } else {
//                         docResults = [];
//                     }
//                     docResultsCount = docResults.length;
//                     logger.debug(`✅ 문서 검색 완료: ${docResultsCount}개 결과`);
//                     if (updateProgress && docResultsCount > 0) await updateProgress(`✅ 문서 검색 완료: ${docResultsCount}개 결과 발견`, 50);
//                 } catch (error) {
//                     console.error('❌ 문서 검색 실패:', error);
//                     docResults = [];
//                     docResultsCount = 0;
//                     if (updateProgress) await updateProgress(`❌ 문서 검색 실패: ${error.message}`, 50);
//                 }
//             }
            
//             // 검색 결과 통합 (안전한 배열 처리)
//             allSearchResults = [];
//             if (Array.isArray(webResults)) allSearchResults.push(...webResults);
//             if (Array.isArray(docResults)) allSearchResults.push(...docResults);
            
//             logger.debug(`📊 총 검색 결과: ${allSearchResults.length}개 (웹: ${webResults.length}, 문서: ${docResults.length})`);
            
//             logger.debug('\n🤖 === 2단계: 응답 생성 ===');
//             if (updateProgress) await updateProgress('🤖 검색 결과를 바탕으로 응답 생성 중...', 60);
            
//             if (allSearchResults.length === 0) {
//                 finalResponse = `죄송합니다. "${query}"에 대한 검색 결과를 찾을 수 없습니다. 다른 키워드로 다시 시도해보시거나, 질문을 더 구체적으로 표현해주세요.`;
//                 qualityScore = 2;
//             } else {
//                 const contextPrompt = aiProTool.buildContextPrompt(query, webResults, docResults);
                
//                 try {
//                     logger.debug(`[ai_pro] AI 응답 생성 시작 - 검색 결과: ${allSearchResults.length}개`);
//                     const { sharedContext } = await import('../aiFeatures.js');
//                     const session = sessionId ? await sharedContext.getSession(sessionId) : null;
//                     const provider = session?.getProvider() || 'ollama';
//                     logger.debug(`[ai_pro] 프로바이더: ${provider}, 세션ID: ${sessionId}`);
                    
//                     let aiChatFunction;
//                     if (provider === 'openai') {
//                         const { openaiChat } = await import('../aiFeatures.js');
//                         aiChatFunction = openaiChat;
//                     } else if (provider === 'anthropic') {
//                         const { anthropicChat } = await import('../aiFeatures.js');
//                         aiChatFunction = anthropicChat;
//                     } else if (provider === 'gemini') {
//                         const { geminiChat } = await import('../aiFeatures.js');
//                         aiChatFunction = geminiChat;
//                     } else if (provider === 'groq') {
//                         const { groqChat } = await import('../aiFeatures.js');
//                         aiChatFunction = groqChat;
//                     } else if (provider === 'vllm') {
//                         const { vllmChat } = await import('../aiFeatures.js');
//                         aiChatFunction = vllmChat;
//                     } else {
//                         const { ollamaChat } = await import('../aiFeatures.js');
//                         aiChatFunction = ollamaChat;
//                     }
                    
//                     const messages = [
//                         {
//                             role: 'system',
//                             content: `당신은 전문적인 연구 어시스턴트입니다. 아래 검색 결과를 바탕으로 사용자의 질문에 대해 정확하고 상세한 답변을 제공하세요.

// **지침:**
// - 반드시 한국어로 답변하세요
// - 검색 결과에 명시된 정보만을 사용하세요
// - 정보가 불확실할 때는 "검색 결과에 따르면" 등으로 명시하세요
// - 답변 마지막에 출처를 포함하세요
// - 구체적이고 실용적인 정보를 우선적으로 제공하세요`
//                         },
//                         {
//                             role: 'user',
//                             content: contextPrompt
//                         }
//                     ];
                    
//                     const result = await aiChatFunction(messages, { 
//                         temperature: 0.3, 
//                         max_tokens: 1500
//                     }, sessionId);
                    
//                     logger.debug(`[ai_pro] ${provider} 응답 받음 - 길이: ${result?.content?.length || 0}자`);
//                     finalResponse = result.content || '응답 생성에 실패했습니다.';
                    
//                 } catch (error) {
//                     console.error('❌ 응답 생성 실패:', error);
//                     finalResponse = `검색 결과는 찾았지만 응답 생성 중 오류가 발생했습니다: ${error.message}`;
//                 }
//             }
            
//             logger.debug('\n📊 === 3단계: 품질 평가 (필수) ===');
//             if (updateProgress) await updateProgress('📊 응답 품질 평가 중...', 80);
            
//             // 품질 평가는 항상 실행 (100% 강제)
//             qualityScore = await aiProTool.evaluateResponseQuality(finalResponse, query, allSearchResults);
//             logger.debug(`📊 초기 품질 점수: ${qualityScore}/10 (임계값: ${quality_threshold})`);
            
//             // 품질 평가 결과 상세 로깅
//             if (qualityScore < quality_threshold) {
//                 logger.debug(`⚠️  품질 점수 미달: ${qualityScore} < ${quality_threshold} - 자동 개선 필요`);
//             } else {
//                 logger.debug(`✅ 품질 점수 충족: ${qualityScore} >= ${quality_threshold}`);
//             }
//             if (updateProgress) await updateProgress(`✅ 품질 평가 완료: ${qualityScore}/10점`, 85);
            
//             if (enable_cross_validation && qualityScore >= quality_threshold) {
//                 logger.debug('\n🔍 === 4단계: 교차 검증 ===');
//                 if (updateProgress) await updateProgress('🔍 교차 검증 수행 중...', 90);
//                 try {
//                     const validationResult = await aiProTool.performCrossValidation(finalResponse, query, sessionId);
//                     if (validationResult) {
//                         logger.debug('✅ 교차 검증 완료');
//                         qualityScore = Math.min(10, qualityScore + 0.5);
//                         if (updateProgress) await updateProgress('✅ 교차 검증 완료', 95);
//                     }
//                 } catch (error) {
//                     console.error('❌ 교차 검증 실패:', error);
//                     if (updateProgress) await updateProgress(`❌ 교차 검증 실패: ${error.message}`, 95);
//                 }
//             }
            
//             // 4단계: 자동 품질 개선 시스템 (최대 3회 재시도)
//             const maxImprovementAttempts = Math.min(max_iterations, 3); // 최대 3회로 제한
            
//             while (qualityScore < quality_threshold && improvementAttempts < maxImprovementAttempts) {
//                 improvementAttempts++;
//                 logger.debug(`\n🔄 === 4단계: 자동 품질 개선 ${improvementAttempts}회차 ===`);
//                 logger.debug(`현재 점수: ${qualityScore}/${quality_threshold} - 개선 전략 수행`);
                
//                 if (updateProgress) await updateProgress(`🔄 품질 개선 ${improvementAttempts}/${maxImprovementAttempts}회차 수행 중...`, 60 + (improvementAttempts * 10));
                
//                 // 개선 전략 1: 추가 키워드 생성 및 검색
//                 const additionalKeywords = await aiProTool.generateAdditionalKeywords(query, finalResponse, sessionId);
//                 logger.debug(`🔍 추가 키워드 생성: ${additionalKeywords.join(', ')}`);
                
//                 // 기존 결과 배열 안전성 확인
//                 if (!Array.isArray(webResults)) webResults = [];
//                 if (!Array.isArray(docResults)) docResults = [];
                
//                 // 개선 전략 2: 다각도 검색 수행
//                 let newWebResults = 0;
//                 let newDocResults = 0;
                
//                 for (const keyword of additionalKeywords) {
//                     if (shouldSearchWeb) {
//                         try {
//                             const { WebSearchService } = await import('../plugins/websearch/webSearchService.js');
//                             const additionalWebResults = await WebSearchService.searchWeb(keyword, 'auto', 3);
//                             if (Array.isArray(additionalWebResults) && additionalWebResults.length > 0) {
//                                 webResults.push(...additionalWebResults);
//                                 newWebResults += additionalWebResults.length;
//                                 logger.debug(`✅ 추가 웹 검색: ${additionalWebResults.length}개 결과 (키워드: ${keyword})`);
//                             }
//                         } catch (error) {
//                             logger.debug(`❌ 추가 웹 검색 실패 (${keyword}): ${error.message}`);
//                         }
//                     }
                    
//                     if (shouldSearchDocuments) {
//                         try {
//                             const { RAGService } = await import('../plugins/rag/ragService.js');
//                             const searchUserId = (!userId || userId === 1) ? 'all' : String(userId);
//                             const additionalRagResults = await RAGService.searchDocuments(keyword, searchUserId);
                            
//                             if (additionalRagResults && additionalRagResults.documents && additionalRagResults.documents[0]) {
//                                 const additionalDocResults = additionalRagResults.documents[0].map((doc, i) => ({
//                                     document: doc,
//                                     metadata: additionalRagResults.metadatas[0][i] || {}
//                                 }));
//                                 docResults.push(...additionalDocResults);
//                                 newDocResults += additionalDocResults.length;
//                                 logger.debug(`✅ 추가 문서 검색: ${additionalDocResults.length}개 결과 (키워드: ${keyword})`);
//                             }
//                         } catch (error) {
//                             logger.debug(`❌ 추가 문서 검색 실패 (${keyword}): ${error.message}`);
//                         }
//                     }
//                 }
                
//                 // 검색 결과 업데이트
//                 webResultsCount = webResults.length;
//                 docResultsCount = docResults.length;
//                 allSearchResults = [...webResults, ...docResults];
                
//                 logger.debug(`📊 개선 후 검색 결과: 웹 +${newWebResults}개, 문서 +${newDocResults}개, 총 ${allSearchResults.length}개`);
                
//                 // 개선 전략 3: 향상된 프롬프트로 응답 재생성
//                 const improvementPrompt = aiProTool.buildImprovementPrompt(query, webResults, docResults, finalResponse, qualityScore);
                
//                 try {
//                     const { sharedContext } = await import('../aiFeatures.js');
//                     const session = sessionId ? await sharedContext.getSession(sessionId) : null;
//                     const provider = session?.getProvider() || 'ollama';

//                     let aiChatFunction;
//                     if (provider === 'openai') {
//                         const { openaiChat } = await import('../aiFeatures.js');
//                         aiChatFunction = openaiChat;
//                     } else if (provider === 'anthropic') {
//                         const { anthropicChat } = await import('../aiFeatures.js');
//                         aiChatFunction = anthropicChat;
//                     } else if (provider === 'gemini') {
//                         const { geminiChat } = await import('../aiFeatures.js');
//                         aiChatFunction = geminiChat;
//                     } else if (provider === 'groq') {
//                         const { groqChat } = await import('../aiFeatures.js');
//                         aiChatFunction = groqChat;
//                     } else if (provider === 'vllm') {
//                         const { vllmChat } = await import('../aiFeatures.js');
//                         aiChatFunction = vllmChat;
//                     } else {
//                         const { ollamaChat } = await import('../aiFeatures.js');
//                         aiChatFunction = ollamaChat;
//                     }
                    
//                     const messages = [
//                         {
//                             role: 'system',
//                             content: `이전 응답의 품질이 ${qualityScore}/10점으로 기준점 ${quality_threshold}점에 미달했습니다. 
//                             다음 사항을 개선하여 더 높은 품질의 답변을 제공하세요:
//                             1. 더 구체적이고 상세한 정보 제공
//                             2. 명확한 출처 표시 및 인용
//                             3. 질문과의 관련성 강화
//                             4. 구조화된 답변 형식 사용
//                             5. 실용적이고 actionable한 정보 포함`
//                         },
//                         {
//                             role: 'user',
//                             content: improvementPrompt
//                         }
//                     ];
                    
//                     const result = await aiChatFunction(messages, { temperature: 0.2, max_tokens: 2000 }, sessionId);
//                     const improvedResponse = result.content || finalResponse;
                    
//                     // 개선된 응답 품질 재평가
//                     const newQualityScore = await aiProTool.evaluateResponseQuality(improvedResponse, query, allSearchResults);
                    
//                     // 품질이 개선되었으면 업데이트
//                     if (newQualityScore > qualityScore) {
//                         finalResponse = improvedResponse;
//                         qualityScore = newQualityScore;
//                         logger.debug(`✅ 품질 개선 성공: ${qualityScore}/10 (이전: ${qualityScore})`);
//                     } else {
//                         logger.debug(`⚠️  품질 개선 실패: ${newQualityScore}/10 (이전: ${qualityScore})`);
//                     }
                    
//                 } catch (error) {
//                     logger.debug(`❌ 응답 개선 실패: ${error.message}`);
//                 }
                
//                 logger.debug(`📊 ${improvementAttempts}회차 개선 완료: 품질 점수 ${qualityScore}/10`);
//             }
            
//             // 최종 품질 개선 결과 로깅
//             if (qualityScore >= quality_threshold) {
//                 logger.debug(`🎉 품질 기준 달성: ${qualityScore}/10 >= ${quality_threshold} (${improvementAttempts}회 개선)`);
//             } else {
//                 logger.debug(`⚠️  품질 기준 미달: ${qualityScore}/10 < ${quality_threshold} (최대 ${maxImprovementAttempts}회 시도 완료)`);
//             }

//         } catch (error) {
//             console.error('❌ AI프로 리서치 핸들러 오류:', error);
//             finalResponse = `AI프로 리서치 도중 심각한 오류가 발생했습니다: ${error.message}`;
//             qualityScore = 1;
            
//             // 오류 발생 시 안전한 기본값 설정
//             if (!Array.isArray(webResults)) webResults = [];
//             if (!Array.isArray(docResults)) docResults = [];
//             if (!Array.isArray(allSearchResults)) allSearchResults = [];
//             webResultsCount = 0;
//             docResultsCount = 0;
//         } finally {
//             logger.debug('\n✅ === AI프로 리서치 완료 ===');
//             logger.debug(`📊 최종 품질 점수: ${qualityScore}/10`);
//             logger.debug(`🔁 총 개선 횟수: ${improvementAttempts || 0}회`);
//             logger.debug(`📂 총 검색 결과: ${allSearchResults.length}개`);
//             if (updateProgress) await updateProgress(`🎉 AI프로 분석 완료! 품질: ${qualityScore}/10점`, 100);
//         }

//         // 최종 결과 안전성 검증
//         const safeWebResults = Array.isArray(webResults) ? webResults : [];
//         const safeDocResults = Array.isArray(docResults) ? docResults : [];
//         const safeAllResults = Array.isArray(allSearchResults) ? allSearchResults : [];
        
//         // 카운터도 실제 배열 길이로 업데이트
//         const finalWebCount = safeWebResults.length;
//         const finalDocCount = safeDocResults.length;
//         const finalTotalCount = safeAllResults.length;
        
//         logger.debug(`[AI_PRO] 최종 결과 검증 - 웹: ${finalWebCount}개, 문서: ${finalDocCount}개, 전체: ${finalTotalCount}개`);
//         logger.debug(`[AI_PRO] 응답 길이: ${finalResponse ? finalResponse.length : 0}자`);
//         logger.debug(`[AI_PRO] 품질 점수: ${qualityScore}/10`);
        
//         return {
//             success: true,
//             data: {
//                 final_response: finalResponse || '',
//                 quality_score: qualityScore || 1,
//                 search_results_count: finalTotalCount,
//                 improvement_attempts: improvementAttempts || 0,
//                 web_results_count: finalWebCount,
//                 document_results_count: finalDocCount,
//                 web_results: safeWebResults,
//                 document_results: safeDocResults
//             },
//             message: `AI프로 리서치 완료 (품질: ${qualityScore}/10, 개선: ${improvementAttempts || 0}회, 검색결과: ${finalTotalCount}개)`
//         };
//     },
    
//     buildContextPrompt(query, webResults, docResults) {
//         logger.debug(`[buildContextPrompt] 시작 - 웹: ${Array.isArray(webResults) ? webResults.length : 'not array'}, 문서: ${Array.isArray(docResults) ? docResults.length : 'not array'}`);
        
//         // 안전한 배열 변환
//         const safeWebResults = Array.isArray(webResults) ? webResults : [];
//         const safeDocResults = Array.isArray(docResults) ? docResults : [];
        
//         let prompt = `### 질문: ${query}\n`;
        
//         if (safeWebResults.length > 0) {
//             prompt += '\n\n### 웹 검색 결과:\n';
//             safeWebResults.forEach((result, index) => {
//                 if (result && typeof result === 'object') {
//                     prompt += `[${index + 1}] ${result.title || '제목 없음'}\n`;
//                     prompt += `   - URL: ${result.link || result.url || '링크 없음'}\n`;
//                     prompt += `   - Snippet: ${result.snippet || result.description || '내용 없음'}\n`;
//                 }
//             });
//         }
        
//         if (safeDocResults.length > 0) {
//             prompt += '\n\n### 내부 문서 검색 결과:\n';
//             safeDocResults.forEach((result, index) => {
//                 if (result && typeof result === 'object') {
//                     const source = result.metadata?.source || '알 수 없음';
//                     const fileName = source.split('/').pop() || '알 수 없음';
//                     prompt += `[${index + 1}] ${fileName}\n`;
//                     prompt += `   - Snippet: ${result.document || result.content || '내용 없음'}\n`;
//                 }
//             });
//         }
        
//         if (safeWebResults.length === 0 && safeDocResults.length === 0) {
//              prompt += '\n\n검색 결과가 없습니다.';
//         }
        
//         logger.debug(`[buildContextPrompt] 완료 - 프롬프트 길이: ${prompt.length}자`);
//         return prompt;
//     },

//     async evaluateResponseQuality(response, originalQuery, searchResults) {
//         logger.debug('\n📊 === 다중 품질 평가 시작 ===');
        
//         if (!response || response.includes('오류가 발생했습니다') || response.includes('찾을 수 없습니다')) {
//             logger.debug('❌ 응답 오류 또는 빈 응답으로 최저 점수 반환');
//             return 1;
//         }
        
//         // 1. 내용 품질 평가 (40% 가중치)
//         const contentScore = await this.evaluateContentQuality(response, originalQuery);
//         logger.debug(`📝 내용 품질 점수: ${contentScore}/10`);
        
//         // 2. 출처 신뢰성 평가 (30% 가중치)
//         const sourceScore = await this.evaluateSourceQuality(response, searchResults);
//         logger.debug(`📚 출처 품질 점수: ${sourceScore}/10`);
        
//         // 3. 관련성 평가 (30% 가중치)
//         const relevanceScore = await this.evaluateRelevance(response, originalQuery);
//         logger.debug(`🎯 관련성 점수: ${relevanceScore}/10`);
        
//         // 가중 평균 계산
//         const finalScore = Math.round((contentScore * 0.4 + sourceScore * 0.3 + relevanceScore * 0.3) * 10) / 10;
        
//         logger.debug(`🏁 최종 품질 점수: ${finalScore}/10 (내용: ${contentScore}, 출처: ${sourceScore}, 관련성: ${relevanceScore})`);
//         logger.debug('='.repeat(50));
        
//         return Math.max(1, Math.min(10, finalScore));
//     },

//     async evaluateContentQuality(response, originalQuery) {
//         let score = 5; // 기본 점수
        
//         // 응답 길이 평가
//         if (response.length < 50) {
//             score -= 3;
//         } else if (response.length < 100) {
//             score -= 1;
//         } else if (response.length > 300) {
//             score += 1;
//         }
        
//         // 구조화된 답변 평가
//         const hasStructure = response.includes('##') || response.includes('###') || response.includes('**') || response.includes('- ');
//         if (hasStructure) score += 1;
        
//         // 구체적 정보 포함 평가
//         const hasSpecificInfo = /\d+/.test(response) || response.includes('예를 들어') || response.includes('구체적으로');
//         if (hasSpecificInfo) score += 1;
        
//         return Math.max(1, Math.min(10, score));
//     },

//     async evaluateSourceQuality(response, searchResults) {
//         let score = 5; // 기본 점수
        
//         if (!searchResults || searchResults.length === 0) {
//             return 2; // 검색 결과가 없으면 낮은 점수
//         }
        
//         // 출처 표시 확인
//         const hasCitations = /\[\d+\]/.test(response) || response.includes('출처:') || response.includes('References') || response.includes('참고');
//         if (hasCitations) {
//             score += 2;
//         } else {
//             score -= 2; // 출처 표시 없으면 감점
//         }
        
//         // 검색 결과 품질 평가
//         const hasWebSources = searchResults.some(res => res.link || res.url);
//         const hasDocSources = searchResults.some(res => res.metadata?.source || res.document);
        
//         if (hasWebSources) score += 1;
//         if (hasDocSources) score += 1;
        
//         // 검색 결과 수량 평가
//         if (searchResults.length >= 5) score += 1;
        
//         return Math.max(1, Math.min(10, score));
//     },

//     async evaluateRelevance(response, originalQuery) {
//         let score = 5; // 기본 점수
        
//         const responseLower = response.toLowerCase();
//         const queryLower = originalQuery.toLowerCase();
        
//         // 키워드 매칭 평가
//         const queryWords = queryLower.split(/\s+/).filter(word => word.length > 2);
//         const matchedWords = queryWords.filter(word => responseLower.includes(word));
//         const relevanceRatio = queryWords.length > 0 ? matchedWords.length / queryWords.length : 0;
        
//         if (relevanceRatio >= 0.8) {
//             score += 2;
//         } else if (relevanceRatio >= 0.6) {
//             score += 1;
//         } else if (relevanceRatio < 0.3) {
//             score -= 2;
//         }
        
//         // 직접적인 답변 여부 확인
//         const hasDirectAnswer = responseLower.includes(queryLower) || queryWords.some(word => responseLower.includes(word));
//         if (hasDirectAnswer) score += 1;
        
//         return Math.max(1, Math.min(10, score));
//     },

//     buildImprovementPrompt(query, webResults, docResults, previousResponse, qualityScore) {
//         let prompt = `질문: ${query}\n\n`;
        
//         // 검색 결과 추가
//         if (webResults && webResults.length > 0) {
//             prompt += `웹 검색 결과:\n`;
//             webResults.slice(0, 5).forEach((result, index) => {
//                 prompt += `${index + 1}. ${result.title || result.snippet || result.content}\n`;
//                 if (result.link) prompt += `   출처: ${result.link}\n`;
//             });
//             prompt += `\n`;
//         }
        
//         if (docResults && docResults.length > 0) {
//             prompt += `문서 검색 결과:\n`;
//             docResults.slice(0, 5).forEach((result, index) => {
//                 prompt += `${index + 1}. ${result.document || result.content}\n`;
//                 if (result.metadata?.source) prompt += `   출처: ${result.metadata.source}\n`;
//             });
//             prompt += `\n`;
//         }
        
//         prompt += `이전 응답 (품질 점수: ${qualityScore}/10):\n${previousResponse}\n\n`;
//         prompt += `위의 검색 결과를 바탕으로 질문에 대한 더 나은 답변을 제공하세요. 이전 응답의 부족한 부분을 보완하고 더 구체적이고 정확한 정보를 포함하세요.`;
        
//         return prompt;
//     },

//     async performCrossValidation(response, query, sessionId = null) {
//         if (!sessionId) return false;
        
//         logger.debug('[CrossValidation] 교차 검증 시작...');
//         const validationPrompt = `아래는 특정 질문에 대한 검색 기반 답변입니다. 이 답변이 원래 질문에 대해 논리적이고 일관성 있는지 평가해주세요. "예" 또는 "아니오"로만 답하고, 그 이유를 간략히 설명해주세요.\n\n질문: "${query}"\n\n답변:\n${response}`;
        
//         const { sharedContext } = await import('../aiFeatures.js');
//         const session = await sharedContext.getSession(sessionId);
//         const provider = session?.getProvider() || 'ollama';

//         let aiChatFunction;
//         if (provider === 'openai') {
//             const { openaiChat } = await import('../aiFeatures.js');
//             aiChatFunction = openaiChat;
//         } else if (provider === 'anthropic') {
//             const { anthropicChat } = await import('../aiFeatures.js');
//             aiChatFunction = anthropicChat;
//         } else {
//             const { ollamaChat } = await import('../aiFeatures.js');
//             aiChatFunction = ollamaChat;
//         }

//         const result = await aiChatFunction([{ role: 'user', content: validationPrompt }], { temperature: 0 }, sessionId);
//         const validationResponse = result.content.toLowerCase();
        
//         logger.debug(`[CrossValidation] 검증 응답: ${validationResponse}`);
//         return validationResponse.startsWith('예');
//     },

//     async generateAdditionalKeywords(originalQuery, currentResponse, sessionId) {
//         logger.debug('[generateAdditionalKeywords] AI를 사용하여 추가 검색어 생성 시작...');
//         const prompt = `사용자의 원래 질문과 현재까지의 불충분한 답변을 바탕으로, 더 나은 답변을 찾기 위한 새로운 검색어 3개를 생성해줘. 각 검색어는 한 줄에 하나씩, 다른 설명 없이 검색어만 제공해줘.

// - 원래 질문: "${originalQuery}"
// - 현재 답변: "${currentResponse}"

// 새로운 검색어:`;
        
//         try {
//             const { sharedContext } = await import('../aiFeatures.js');
//             const session = sessionId ? await sharedContext.getSession(sessionId) : null;
//             const provider = session?.getProvider() || 'ollama';

//             let aiChatFunction;
//             if (provider === 'openai') {
//                 const { openaiChat } = await import('../aiFeatures.js');
//                 aiChatFunction = openaiChat;
//             } else if (provider === 'anthropic') {
//                 const { anthropicChat } = await import('../aiFeatures.js');
//                 aiChatFunction = anthropicChat;
//             } else if (provider === 'gemini') {
//                 const { geminiChat } = await import('../aiFeatures.js');
//                 aiChatFunction = geminiChat;
//             } else if (provider === 'groq') {
//                 const { groqChat } = await import('../aiFeatures.js');
//                 aiChatFunction = groqChat;
//             } else if (provider === 'vllm') {
//                 const { vllmChat } = await import('../aiFeatures.js');
//                 aiChatFunction = vllmChat;
//             } else {
//                 const { ollamaChat } = await import('../aiFeatures.js');
//                 aiChatFunction = ollamaChat;
//             }

//             const result = await aiChatFunction([{ role: 'user', content: prompt }], { temperature: 0.5, max_tokens: 100 }, sessionId);
//             const newKeywords = result.content.split('\n').map(k => k.trim()).filter(Boolean);
            
//             logger.debug(`[generateAdditionalKeywords] 생성된 새 검색어: ${newKeywords.join(', ')}`);
            
//             return newKeywords.length > 0 ? newKeywords : [originalQuery]; // Fallback
//         } catch (error) {
//             console.error('[generateAdditionalKeywords] AI를 통한 키워드 생성 실패:', error);
//             // AI 호출 실패 시 기존 로직으로 대체
//             const keywords = new Set(originalQuery.split(' '));
//             const nouns = currentResponse.match(/\b[가-힣]{2,}\b/g) || [];
//             nouns.slice(0, 5).forEach(noun => keywords.add(noun));
//             return Array.from(keywords).slice(0, 3);
//         }
//     }
// };

// 웹 검색 도구
const webSearchTool = {
    name: 'web_search',
    category: 'search',
    description: '웹에서 정보를 검색합니다.',
    parameters: {
        query: { type: 'string', required: true, description: '검색할 키워드' }
    },
    handler: async (args, context) => {
        const { updateProgress } = context; // updateProgress 함수 추출
        
        // args 파라미터 안전하게 추출
        const query = args?.query;
        
        // 필수 파라미터 검증
        if (!query || typeof query !== 'string') {
            logger.error('[WebSearch] ❌ query 파라미터가 누락되거나 유효하지 않음:', { 
                args, 
                query, 
                queryType: typeof query,
                argsType: typeof args 
            });
            return {
                success: false,
                error: 'query 파라미터가 필요합니다.',
                data: { results: [], count: 0 }
            };
        }
        
        logger.debug(`[WebSearch] 검색 시작: ${query}`);
        
        try {
            // 진행 상태 업데이트 - 웹검색 시작
            // if (updateProgress) {
            //     await updateProgress(`🌐 웹에서 "${query}" 검색 중...`, 30);
            // }
            
            const { WebSearchService } = await import('../plugins/websearch/webSearchService.js');
            await WebSearchService.checkIfEnabled(true);
            
            // 새로운 웹검색 서비스를 우선 사용하도록 설정
            const results = await WebSearchService.searchWeb(
                query,
                'auto',  // 최적화된 엔진 (구글+네이버)
                5,       // 최대 결과 수
                true     // useRawQuery=true (쿼리 최적화 비활성화)
            );
            
            // 진행 상태 업데이트 - 웹검색 완료 (결과가 있을 때만)
            if (updateProgress) {
                const resultCount = results?.length || 0;
                if (resultCount > 0) {
                    await updateProgress(`✅ Searching the web... ${resultCount} results found`, 70);
                }
            }
            
            return {
                success: true,
                data: { results, count: results?.length || 0 },
                message: `웹 검색 완료: ${results?.length || 0}개 결과`
            };
        } catch (error) {
            // 진행 상태 업데이트 - 웹검색 실패
            if (updateProgress) {
                await updateProgress(`❌ Searching the web... ${error.message}`, 70);
            }
            
            return {
                success: false,
                error: error.message,
                message: `웹 검색 실패: ${error.message}`
            };
        }
    }
};

// 문서 검색 도구
const documentSearchTool = {
    name: 'doc_search',
    category: 'search',
    description: '로컬 문서에서 정보를 검색합니다.',
    parameters: {
        query: { type: 'string', required: true, description: '검색할 키워드' },
        ragSearchScope: { type: 'string', required: false, description: 'RAG 검색 범위 (personal/shared/all)', default: 'personal' }
    },
    handler: async (args, context) => {
        const { userId, job, taskId, sessionId, mentionedDocumentContent, username, updateProgress } = context;
        
        // args 파라미터 안전하게 추출 및 기본값 설정
        const query = args?.query;
        const ragSearchScope = args?.ragSearchScope || 'personal';
        
        // 필수 파라미터 검증
        if (!query || typeof query !== 'string') {
            logger.error('[DocumentSearch] ❌ query 파라미터가 누락되거나 유효하지 않음:', { 
                args, 
                query, 
                queryType: typeof query,
                argsType: typeof args 
            });
            return {
                success: false,
                error: 'query 파라미터가 필요합니다.',
                data: { results: [], count: 0 }
            };
        }
        
        logger.debug(`[DocumentSearch] 검색 시작: ${query}`);
        
        try {
            // 진행 상태 업데이트 - 문서검색 시작
            if (updateProgress) {
                await updateProgress(`📚 Searching the documents... "${query}"`, 30);
            }
            
            let combinedResults = [];
            let totalCount = 0;
            
            // 1. 멘션된 문서 내용이 있는 경우 먼저 추가
            if (mentionedDocumentContent && mentionedDocumentContent.trim().length > 0) {
                logger.debug(`[DocumentSearch] 📝 멘션된 문서 내용 추가`);
                
                // 멘션된 문서 내용을 문서별로 분할
                const documentSections = mentionedDocumentContent.split(/\n\n===\s*([^=]+)\s*===\n/);
                
                for (let i = 1; i < documentSections.length; i += 2) {
                    const documentName = documentSections[i].trim();
                    const documentContent = documentSections[i + 1];
                    
                    if (!documentContent || !documentContent.trim()) continue;
                    
                    // 페이지별로 분할
                    const pageSections = documentContent.split(/\n\n---\s*페이지\s*(\d+)\s*---\n/);
                    
                    if (pageSections.length > 1) {
                        // 페이지 정보가 있는 경우
                        for (let j = 1; j < pageSections.length; j += 2) {
                            const pageNumber = pageSections[j];
                            const pageContent = pageSections[j + 1];
                            
                            if (pageContent && pageContent.trim()) {
                                combinedResults.push({
                                    content: pageContent.trim(),
                                    metadata: {
                                        source: documentName,
                                        page_number: parseInt(pageNumber),
                                        type: 'mentioned_document',
                                        relevance_score: 1.0
                                    }
                                });
                                totalCount++;
                            }
                        }
                    } else {
                        // 페이지 정보가 없는 경우 전체 문서로 처리
                        combinedResults.push({
                            content: documentContent.trim(),
                            metadata: {
                                source: documentName,
                                type: 'mentioned_document',
                                relevance_score: 1.0
                            }
                        });
                        totalCount++;
                    }
                }
                
                logger.debug(`[DocumentSearch] ✅ 멘션된 문서 내용 추가 완료: ${totalCount}개 청크`);
            }
            
            // 2. 일반 RAG 검색도 수행
            const { RAGService } = await import('../plugins/rag/ragService.js');
            await RAGService.checkIfEnabled(true);
            
                // 사용자 정보 가져오기 - context.username을 우선 사용 (강제 도구 호출 시 전달됨)
    const { sharedContext } = await import('../aiFeatures.js');
    const actualSessionId = sessionId || taskId;
    const session = actualSessionId ? await sharedContext.getSession(actualSessionId) : null;
    
    // 사용자명 결정: args.username (강제 도구 호출), context.username, 세션 순으로 확인
    const finalUsername = args?.username || username || session?.getUsername();
    
    // 실제 사용자 ID 결정 - 검색 범위와 분리
    let actualUserId = finalUsername;
    if (!actualUserId) {
        if (userId === 1 || userId === 'admin') {
            actualUserId = 'admin';   // 기본 관리자 계정
        } else {
            actualUserId = `user_${userId}`;  // userId 기반 사용자명
        }
    }
    
    logger.debug(`[DocumentSearch] 검색 정보: userId=${userId}, sessionId=${actualSessionId}, contextUsername=${username}, finalUsername=${finalUsername}, ragSearchScope=${ragSearchScope}, actualUserId=${actualUserId}`);
            
            const ragResults = await RAGService.searchDocuments(query, actualUserId, null, ragSearchScope);
            
            // 3. RAG 검색 결과를 통합 결과에 추가
            if (ragResults?.documents?.[0]?.length > 0) {
                logger.debug(`[DocumentSearch] 🔍 일반 RAG 검색 결과: ${ragResults.documents[0].length}개 추가`);
                
                ragResults.documents[0].forEach((doc, index) => {
                    // RAG 검색 결과의 메타데이터 가져오기
                    const originalMetadata = ragResults.metadatas?.[0]?.[index] || {};
                    
                    // 페이지 정보 추출 및 정규화
                    const pageNumber = originalMetadata.page_number || originalMetadata.page || originalMetadata.chunk_index;
                    const source = originalMetadata.source || originalMetadata.filename || originalMetadata.file_name || 'unknown';
                    
                    combinedResults.push({
                        content: doc,
                        metadata: {
                            ...originalMetadata,
                            source: source,
                            page_number: pageNumber,
                            type: 'rag_search',
                            relevance_score: 0.5
                        }
                    });
                });
                
                totalCount += ragResults.documents[0].length;
            }
            
            // 4. 통합 결과 반환
            const finalResults = {
                documents: [combinedResults.map(r => r.content)],
                metadatas: [combinedResults.map(r => r.metadata)],
                distances: [combinedResults.map(r => 1 - (r.metadata.relevance_score || 0.5))]
            };
            
            logger.debug(`[DocumentSearch] ✅ 검색 완료: 총 ${totalCount}개 결과 (멘션: ${mentionedDocumentContent ? 1 : 0}개, RAG: ${ragResults?.documents?.[0]?.length || 0}개)`);
            
            // 진행 상태 업데이트 - 문서검색 완료 (결과가 있을 때만)
            if (updateProgress) {
                if (totalCount > 0) {
                    await updateProgress(`✅ Searching the documents... ${totalCount} results found`, 70);
                } else {
                    await updateProgress(`📚 Searching the documents... no results found`, 70);
                }
            }
            
            if (totalCount === 0) {
                return {
                    success: true,
                    data: { results: null, count: 0 },
                    message: '문서 검색 완료'
                };
            }
            return {
                success: true,
                data: { results: finalResults, count: totalCount },
                message: `문서 검색 완료: ${totalCount}개 결과`
            };
        } catch (error) {
            // 진행 상태 업데이트 - 문서검색 실패
            if (updateProgress) {
                await updateProgress(`❌ Searching the documents... ${error.message}`, 70);
            }
            
            return {
                success: false,
                error: error.message,
                message: `문서 검색 실패: ${error.message}`
            };
        }
    }
};

// 이메일 발송 도구
const emailTool = {
    name: 'email',
    category: 'communication',
    description: '이메일을 발송합니다. 사용자가 명시적으로 이메일 발송을 요청할 때만 사용됩니다.',
    parameters: {
        to_email: { type: 'string', required: true, description: '수신자 이메일 주소' },
        subject: { type: 'string', required: false, description: '이메일 제목 (미제공시 스마트 생성)' },
        body: { type: 'string', required: false, description: '이메일 본문 (미제공시 히스토리 요약 또는 응답 생성)' },
        html_body: { type: 'string', required: false, description: 'HTML 이메일 본문' },
        attachments: { type: 'array', required: false, description: '첨부 파일 목록' }
    },
    handler: async (args, context) => {
        const { userId, sessionId, history = [], updateProgress } = context;
        const { to_email, subject, body, html_body, attachments } = args;
        
        // 필수 파라미터 검증
        if (!to_email) {
            return {
                success: false,
                error: '수신자 이메일 주소가 필요합니다',
                message: '이메일 발송 실패: 수신자 이메일 주소가 필요합니다'
            };
        }

        // 진행 상황 업데이트 - 이메일 발송 시작
        if (updateProgress) {
            await updateProgress(`📧 이메일 발송 준비`);
        }

        let finalSubject = subject;
        let finalBody = body;
        
        // body가 도구 호출 텍스트인지만 감지 (이메일 발송 요청은 유효한 내용으로 처리)
        const isInvalidBody = body && (
            // 도구 호출 텍스트만 감지
            body.includes('email(') ||
            body.includes('to_email=') ||
            body.includes('handler:') ||
            body.includes('parameters:')
        );
        
        if (isInvalidBody) {
            logger.debug(`[Email] ⚠️ 도구 호출 텍스트 감지 (${body.length}자): "${body.substring(0, 50)}..." - 사용자 요청에서 실제 내용 찾기`);
            finalBody = null; // 도구 호출 텍스트만 무시
        }

        // 제목이나 본문이 없는 경우 스마트 생성
        if (!finalSubject || !finalBody) {
            
            // 진행 상황 업데이트 - 콘텐츠 생성
            if (updateProgress) {
                await updateProgress('🤖 이메일 제목과 본문을 생성하는 중...', 30);
            }
            
            try {
                logger.debug(`[Email] 🤖 스마트 생성 시작 - 제목: ${!!finalSubject}, 본문: ${!!finalBody}`);
                
                // 히스토리에서 사용자의 실제 요청 찾기
                const userMessages = history.filter(msg => 
                    msg.role === 'user' && 
                    msg.content && 
                    msg.content.length > 10
                );
                
                const assistantResponses = history.filter(msg => 
                    msg.role === 'assistant' && 
                    msg.content && 
                    msg.content.length > 50 &&
                    !msg.content.includes('email(') &&
                    !msg.content.includes('to_email=')
                );
                
                logger.debug(`[Email] 📊 히스토리 분석 - 사용자 메시지: ${userMessages.length}개, 어시스턴트 응답: ${assistantResponses.length}개`);
                
                // 이메일 생성을 위한 컨텍스트 구성
                let contextForEmail = '';
                
                // 사용자 요청 분석 및 적절한 콘텐츠 준비
                if (userMessages.length > 0) {
                    const recentUserMessage = userMessages[userMessages.length - 1];
                    contextForEmail += `사용자의 이메일 요청:\n${recentUserMessage.content}\n\n`;
                    
                    // 요청 의도 분석 및 관련 콘텐츠 추가
                    const requestContent = recentUserMessage.content.toLowerCase();
                    
                    // 1. 도구목록 요청인 경우
                    if (requestContent.includes('도구목록') || requestContent.includes('도구 목록') || requestContent.includes('tool')) {
                        try {
                            const { toolRegistry } = await import('./tools-registry.js');
                            const availableTools = toolRegistry.getAllTools();
                            
                            if (availableTools.length > 0) {
                                contextForEmail += `참고: 현재 사용 가능한 도구 정보\n`;
                                const categories = {};
                                availableTools.forEach(tool => {
                                    const category = tool.category || 'utility';
                                    if (!categories[category]) categories[category] = [];
                                    categories[category].push(tool);
                                });
                                
                                Object.keys(categories).sort().forEach(category => {
                                    contextForEmail += `\n[${category.toUpperCase()}]\n`;
                                    categories[category].forEach(tool => {
                                        contextForEmail += `- ${tool.name}: ${tool.description}\n`;
                                    });
                                });
                                contextForEmail += `\n총 ${availableTools.length}개의 도구가 사용 가능합니다.\n\n`;
                            }
                        } catch (error) {
                            logger.debug(`[Email] 도구 정보 가져오기 실패: ${error.message}`);
                        }
                    }
                    
                    // 2. 대화 내용 정리/요약 요청인 경우
                    else if (requestContent.includes('정리') || requestContent.includes('요약') || requestContent.includes('내용') || requestContent.includes('대화')) {
                        contextForEmail += `참고: 지금까지의 대화 내용\n`;
                        
                        // 사용자 메시지들 (이메일 요청 제외)
                        const conversationUserMessages = userMessages.slice(0, -1).filter(msg => 
                            !msg.content.toLowerCase().includes('이메일') && 
                            !msg.content.toLowerCase().includes('보내')
                        );
                        
                        if (conversationUserMessages.length > 0) {
                            contextForEmail += `\n사용자 질문/요청:\n`;
                            conversationUserMessages.forEach((msg, index) => {
                                contextForEmail += `${index + 1}. ${msg.content}\n`;
                            });
                            contextForEmail += `\n`;
                        }
                        
                        // AI 응답들
                        if (assistantResponses.length > 0) {
                            contextForEmail += `AI 응답 내용:\n`;
                            assistantResponses.forEach((msg, index) => {
                                const preview = msg.content.substring(0, 200);
                                contextForEmail += `${index + 1}. ${preview}${msg.content.length > 200 ? '...' : ''}\n\n`;
                            });
                        }
                    }
                    
                    // 3. 기타 요청인 경우 - 최근 대화 컨텍스트 제공
                    else {
                        if (userMessages.length > 1) {
                            contextForEmail += `최근 대화 맥락:\n`;
                            userMessages.slice(-3, -1).forEach(msg => {
                                contextForEmail += `- 사용자: ${msg.content}\n`;
                            });
                            contextForEmail += `\n`;
                        }
                        
                        if (assistantResponses.length > 0) {
                            contextForEmail += `관련 AI 응답:\n`;
                            const recentResponse = assistantResponses[assistantResponses.length - 1];
                            contextForEmail += `- ${recentResponse.content.substring(0, 300)}${recentResponse.content.length > 300 ? '...' : ''}\n\n`;
                        }
                    }
                }

                
                // AI를 통한 이메일 생성
                const { sharedContext } = await import('../aiFeatures.js');
                const session = sessionId ? await sharedContext.getSession(sessionId) : null;
                const provider = session?.getProvider() || 'ollama';

                let aiChatFunction;
                if (provider === 'openai') {
                    const { openaiChat } = await import('../aiFeatures.js');
                    aiChatFunction = openaiChat;
                } else if (provider === 'anthropic') {
                    const { anthropicChat } = await import('../aiFeatures.js');
                    aiChatFunction = anthropicChat;
                } else if (provider === 'gemini') {
                    const { geminiChat } = await import('../aiFeatures.js');
                    aiChatFunction = geminiChat;
                } else if (provider === 'groq') {
                    const { groqChat } = await import('../aiFeatures.js');
                    aiChatFunction = groqChat;
                } else if (provider === 'vllm') {
                    const { vllmChat } = await import('../aiFeatures.js');
                    aiChatFunction = vllmChat;
                } else {
                    const { ollamaChat } = await import('../aiFeatures.js');
                    aiChatFunction = ollamaChat;
                }

                const emailGenerationPrompt = `사용자의 이메일 발송 요청을 분석하여 적절한 이메일을 작성해주세요.

${contextForEmail}

작성 지침:
1. 사용자의 요청 의도를 정확히 파악하세요
2. 제목: ${finalSubject ? `"${finalSubject}"를 사용하되, 더 적절하다면 개선해주세요` : '사용자 요청에 맞는 명확하고 구체적인 제목을 생성해주세요'}
3. 본문: ${finalBody ? `"${finalBody}"를 기반으로 하되, 더 전문적이고 완성도 높게 작성해주세요` : '사용자가 요청한 내용을 정확히 반영한 본문을 작성해주세요'}

요청 유형별 처리:
- 도구목록 요청: 제공된 도구 정보를 활용하여 카테고리별로 정리해서 제시
- 대화 내용 정리/요약 요청: 제공된 대화 내용을 체계적으로 요약하고 핵심 포인트 정리
- 분석 결과 공유 요청: 관련 AI 응답을 정리하여 명확하게 전달
- 기타 요청: 제공된 맥락을 바탕으로 사용자 의도에 맞는 내용 작성

형식: 정중하고 전문적인 이메일 형식으로 작성하되, 내용이 명확하고 실용적이어야 합니다.

다음 형식으로 응답해주세요:
SUBJECT: [이메일 제목]
BODY: [이메일 본문]`;

                const result = await aiChatFunction([
                    { role: 'user', content: emailGenerationPrompt }
                ], { temperature: 0.7, max_tokens: 1000 }, sessionId);

                if (result && result.content) {
                    const content = result.content;
                    logger.debug(`[Email] AI 응답 원본 (${content.length}자): ${content.substring(0, 200)}...`);
                    
                    // 제목 추출
                    const subjectMatch = content.match(/SUBJECT:\s*(.+?)(?:\n|$)/);
                    if (subjectMatch && !finalSubject) {
                        finalSubject = subjectMatch[1].trim();
                        logger.debug(`[Email] 제목 추출됨: ${finalSubject}`);
                    }
                    
                    // 본문 추출 - BODY: 이후의 모든 내용을 가져옴
                    const bodyIndex = content.indexOf('BODY:');
                    if (bodyIndex !== -1 && !finalBody) {
                        finalBody = content.substring(bodyIndex + 5).trim();
                        // 마지막에 "---"가 있으면 제거
                        if (finalBody.endsWith('---')) {
                            finalBody = finalBody.substring(0, finalBody.length - 3).trim();
                        }
                        logger.debug(`[Email] 본문 추출됨 (${finalBody.length}자): ${finalBody.substring(0, 100)}...`);
                    }
                    
                    // 전체 내용을 본문으로 사용 (구조화된 형식이 아닌 경우)
                    if (!finalBody && content.length > 100) {
                        finalBody = content.replace(/SUBJECT:.*?\n/g, '').trim();
                        logger.debug(`[Email] 전체 내용을 본문으로 사용 (${finalBody.length}자): ${finalBody.substring(0, 100)}...`);
                    }
                }

                logger.debug(`[Email] 📧 최종 콘텐츠 준비 완료 - 본문 길이: ${finalBody.length}자`);
                
            } catch (error) {
                logger.debug(`[Email] ❌ 스마트 생성 실패: ${error.message}, 기본값 사용`);
                finalSubject = finalSubject || '하모나이즈 응답';
                finalBody = finalBody || '안녕하세요. 하모나이즈에서 연락드립니다.';
            }
        }

        // 여전히 제목이나 본문이 없는 경우 기본값 설정
        if (!finalSubject) finalSubject = '하모나이즈 응답';
        if (!finalBody) finalBody = '안녕하세요. 하모나이즈에서 연락드립니다.';
        
        logger.debug(`[Email] 이메일 발송: ${to_email}`);
        logger.debug(`[Email] 제목: ${finalSubject ? finalSubject.substring(0, 50) : 'undefined'}...`);
        logger.debug(`[Email] 본문 길이: ${finalBody ? finalBody.length : 0}자`);
        
        // 진행 상황 업데이트 - 이메일 발송 중
        if (updateProgress) {
            await updateProgress(`📤 이메일 발송 중`);
        }
        
        try {
            // Python의 send_email 함수 호출
            const { execInVenv } = await import('../codeExecution.js');
            
            // Python 문자열을 안전하게 이스케이프하는 함수
            const escapePythonString = (str) => {
                return str
                    .replace(/\\/g, '\\\\')  // 백슬래시 이스케이프
                    .replace(/"/g, '\\"')    // 따옴표 이스케이프
                    .replace(/\n/g, '\\n')   // 줄바꿈 이스케이프
                    .replace(/\r/g, '\\r')   // 캐리지 리턴 이스케이프
                    .replace(/\t/g, '\\t');  // 탭 이스케이프
            };

            // 첨부 파일 리스트를 Python 리스트 형식으로 변환
            const attachmentsStr = attachments && attachments.length > 0 
                ? `[${attachments.map(f => `"${f}"`).join(', ')}]`
                : 'None';
            
            // HTML 본문 처리
            const htmlBodyStr = html_body ? `"${escapePythonString(html_body)}"` : 'None';

            const pythonCode = `
import sys
sys.path.append('.')
from utils import send_email

result = send_email(
    to_email="${to_email}",
    subject="${escapePythonString(finalSubject)}",
    body="${escapePythonString(finalBody)}",
    attachments=${attachmentsStr},
    html_body=${htmlBodyStr}
)

print(f"EMAIL_RESULT: {result}")
`;
            
            const result = await execInVenv(pythonCode, false);
            
            if (result.code === 0 && result.stdout.includes('EMAIL_RESULT: True')) {
                // 진행 상황 업데이트 - 이메일 발송 완료
                if (updateProgress) {
                    await updateProgress(`✅ 이메일 발송 완료: ${to_email}`, 100);
                }
                
                return {
                    success: true,
                    data: { 
                        sent: true, 
                        to_email: to_email,
                        subject: finalSubject,
                        body_length: finalBody.length
                    },
                    message: `📧 이메일 발송 완료!\n수신자: ${to_email}\n제목: ${finalSubject}\n본문: ${finalBody.length}자`
                };
            } else {
                const errorMsg = result.stderr || result.stdout || '알 수 없는 오류';
                
                // 진행 상황 업데이트 - 이메일 발송 실패
                if (updateProgress) {
                    await updateProgress(`❌ 이메일 발송 실패: ${errorMsg}`, 100);
                }
                
                return {
                    success: false,
                    error: errorMsg,
                    message: `이메일 발송 실패: ${errorMsg}`
                };
            }
        } catch (error) {
            // 진행 상황 업데이트 - 이메일 발송 오류
            if (updateProgress) {
                await updateProgress(`❌ 이메일 발송 오류: ${error.message}`, 100);
            }
            
            return {
                success: false,
                error: error.message,
                message: `이메일 발송 실패: ${error.message}`
            };
        }
    }
};


// RAG 문서에서 환자 데이터 검색
async function getPatientDataFromRAG(userRequest, username = 'admin') {
    logger.debug('[TRACE] getPatientDataFromRAG 함수 진입');
    
    try {
        // 환자명 추출
        let patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*환자/);
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*의\s*건강/);
        }
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*건강검진/);
        }
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*검진/);
        }
        
        const patientName = patientNameMatch ? patientNameMatch[1].trim() : null;
        logger.debug('[TRACE] RAG 검색용 환자명:', patientName);
        
        if (!patientName) {
            logger.debug('[TRACE] RAG: 환자명을 추출할 수 없음');
            return null;
        }
        
        // RAG 검색 수행
        logger.debug('[TRACE] RAG 서비스 로딩 시작');
        const { RAGService } = await import('../plugins/rag/ragService.js');
        
        // 환자명으로 RAG 검색 시도
        const searchQuery = `${patientName} 환자 건강검진 데이터`;
        logger.debug('[TRACE] RAG 검색 쿼리:', searchQuery);
        
        const ragResults = await RAGService.searchDocuments(searchQuery, username, null, 'personal');
        logger.debug('[TRACE] RAG 검색 결과:', ragResults ? '성공' : '실패');
        
        if (!ragResults || !ragResults.documents || !ragResults.documents[0] || ragResults.documents[0].length === 0) {
            logger.debug('[TRACE] RAG에서 관련 문서를 찾을 수 없음');
            return null;
        }
        
        // 검색된 문서들을 분석하여 환자 데이터 추출
        const documents = ragResults.documents[0];
        const metadatas = ragResults.metadatas ? ragResults.metadatas[0] : [];
        
        logger.debug('[TRACE] RAG 문서 개수:', documents.length);
        
        // 모든 문서에서 환자 데이터를 검색 (여러 파일 처리)
        let bestPatientData = null;
        let bestScore = 0;
        let foundDocuments = [];
        
        for (let i = 0; i < documents.length; i++) {
            const doc = documents[i];
            const score = ragResults.distances ? (1 - ragResults.distances[0][i]) : 0.5;
            const metadata = metadatas[i] || {};
            
            logger.debug(`[TRACE] 문서 ${i+1}/${documents.length} 검토 중... (점수: ${score})`);
            
            // 각 문서에서 환자 데이터 파싱 시도
            const parsedData = parsePatientDataFromText(doc, patientName);
            
            if (parsedData) {
                foundDocuments.push({
                    content: doc,
                    data: parsedData,
                    score: score,
                    metadata: metadata,
                    documentIndex: i
                });
                
                logger.debug(`[TRACE] 문서 ${i+1}에서 환자 데이터 발견:`, parsedData.name);
                
                // 가장 높은 점수의 데이터를 선택
                if (score > bestScore) {
                    bestPatientData = parsedData;
                    bestScore = score;
                    // 파일명 추출 로직 개선
                    let extractedFilename = 'RAG Documents';
                    if (metadata.file_name) {
                        extractedFilename = metadata.file_name;
                    } else if (metadata.filename) {
                        extractedFilename = metadata.filename;
                    } else if (metadata.source) {
                        const pathMatch = metadata.source.match(/([^/\\]+\.(xlsx?|pdf|docx?|txt))$/i);
                        if (pathMatch) {
                            extractedFilename = pathMatch[1];
                        } else {
                            const lastPart = metadata.source.split(/[/\\]/).pop();
                            if (lastPart && lastPart !== 'source') {
                                extractedFilename = lastPart;
                            }
                        }
                    }
                    
                    bestPatientData.documentSource = `문서 ${i+1} (${extractedFilename})`;
                    bestPatientData.filename = extractedFilename;
                    bestPatientData.metadata = metadata;
                }
            } else {
                logger.debug(`[TRACE] 문서 ${i+1}에서 환자 데이터를 찾을 수 없음`);
            }
        }
        
        logger.debug(`[TRACE] 총 ${foundDocuments.length}개 문서에서 환자 데이터 발견`);
        
        if (!bestPatientData) {
            logger.debug('[TRACE] 모든 문서에서 환자 데이터를 찾을 수 없음');
            
            // 환자명이 포함된 문서가 있는지 최종 확인
            const documentsWithPatientName = documents.filter(doc => 
                doc.toLowerCase().includes(patientName.toLowerCase())
            );
            
            if (documentsWithPatientName.length > 0) {
                logger.debug(`[TRACE] 환자명이 포함된 문서 ${documentsWithPatientName.length}개 발견, 하지만 파싱 실패`);
                logger.debug('[TRACE] 첫 번째 관련 문서 내용 일부:', documentsWithPatientName[0].substring(0, 200));
            }
            
            return null;
        }
        
        logger.debug('[TRACE] 최종 선택된 환자 데이터:', bestPatientData.name, '점수:', bestScore);
        
        // 최고 점수 데이터에 메타데이터 추가
        bestPatientData.dataSource = 'RAG Documents';
        bestPatientData.timestamp = new Date().toISOString();
        bestPatientData.relevanceScore = bestScore;
        bestPatientData.totalDocumentsSearched = documents.length;
        bestPatientData.documentsWithData = foundDocuments.length;
        
        logger.debug('[TRACE] RAG에서 환자 데이터 파싱 성공');
        return bestPatientData;
        
    } catch (error) {
        logger.debug('[TRACE] RAG 검색 중 오류:', error.message);
        return null;
    }
}

// 텍스트에서 환자 데이터 파싱
function parsePatientDataFromText(text, patientName) {
    try {
        logger.debug('[TRACE] 텍스트 파싱 시작, 환자명:', patientName);
        
        // 기본 환자 데이터 구조
        const patientData = {
            name: patientName,
            gender: null,
            birthDate: null,
            checkupDate: null,
            systolicBP: { first: null, second: null, average: null },
            diastolicBP: { first: null, second: null, average: null },
            lipids: { triglycerides: null, hdl: null, ldl: null },
            bloodSugar: { general: null, fasting: null, random: null, hba1c: null },
            obesity: { bmi: null, waistCircumference: null },
            urineProtein: null
        };
        
        // 다양한 패턴으로 건강 데이터 추출 (엑셀 데이터 형식에 최적화)
        const patterns = {
            // 혈압 - 다양한 형식 지원
            systolicBP: [
                /수축기혈압[:\s]*(\d+)/gi,
                /수축기[:\s]*(\d+)/gi,
                /systolic[:\s]*(\d+)/gi,
                /\b(\d{2,3})\s*\/\s*\d{2,3}\b/gi  // 140/90 형식
            ],
            diastolicBP: [
                /확장기혈압[:\s]*(\d+)/gi,
                /확장기[:\s]*(\d+)/gi,
                /diastolic[:\s]*(\d+)/gi,
                /\b\d{2,3}\s*\/\s*(\d{2,3})\b/gi  // 140/90 형식
            ],
            
            // 지질 - 더 넓은 패턴
            triglycerides: [
                /중성지방[:\s]*(\d+)/gi,
                /triglyceride[s]?[:\s]*(\d+)/gi,
                /TG[:\s]*(\d+)/gi,
                /\b(\d{2,4})\s*mg\/dl.*중성지방/gi,
                /중성지방.*?(\d{2,4})/gi
            ],
            hdl: [
                /HDL[:\s]*(\d+)/gi,
                /HDL[-_]?콜레스테롤[:\s]*(\d+)/gi,
                /HDL[-_]?cholesterol[:\s]*(\d+)/gi,
                /\b(\d{1,3})\s*mg\/dl.*HDL/gi
            ],
            ldl: [
                /LDL[:\s]*(\d+)/gi,
                /LDL[-_]?콜레스테롤[:\s]*(\d+)/gi,
                /LDL[-_]?cholesterol[:\s]*(\d+)/gi,
                /\b(\d{2,3})\s*mg\/dl.*LDL/gi
            ],
            
            // 혈당 - 다양한 표현
            bloodSugar: [
                /혈당[:\s]*(\d+)/gi,
                /blood\s*sugar[:\s]*(\d+)/gi,
                /glucose[:\s]*(\d+)/gi,
                /\b(\d{2,3})\s*mg\/dl.*혈당/gi
            ],
            fastingBloodSugar: [
                /공복시혈당[:\s]*(\d+)/gi,
                /공복혈당[:\s]*(\d+)/gi,
                /fasting[:\s]*(\d+)/gi,
                /FBS[:\s]*(\d+)/gi
            ],
            hba1c: [
                /HbA1c[:\s]*(\d+\.?\d*)/gi,
                /당화혈색소[:\s]*(\d+\.?\d*)/gi,
                /A1C[:\s]*(\d+\.?\d*)/gi,
                /\b(\d\.\d)\s*%.*HbA1c/gi
            ],
            
            // 비만 관련
            bmi: [
                /BMI[:\s]*(\d+\.?\d*)/gi,
                /비만도[:\s]*(\d+\.?\d*)/gi,
                /체질량지수[:\s]*(\d+\.?\d*)/gi,
                /\b(\d{2}\.\d)\s*kg\/m/gi
            ],
            waist: [
                /복위[:\s]*(\d+)/gi,
                /허리둘레[:\s]*(\d+)/gi,
                /waist[:\s]*(\d+)/gi,
                /\b(\d{2,3})\s*cm.*복위/gi
            ],
            
            // 성별 - 더 다양한 표현
            gender: [
                /성별[:\s]*(남성|여성|[13MF])/gi,
                /gender[:\s]*(male|female|M|F|남|여)/gi,
                /(남성|여성)\s*\(([13])\)/gi
            ],
            
            // 요단백 - 다양한 기호
            urineProtein: [
                /요단백[:\s]*([－±＋\-\+]+)/gi,
                /protein[:\s]*([－±＋\-\+]+)/gi,
                /urine\s*protein[:\s]*([－±＋\-\+]+)/gi
            ]
        };
        
        // 헬퍼 함수: 여러 패턴으로 매칭 시도
        const tryPatterns = (patternArray, text) => {
            for (const pattern of patternArray) {
                const match = pattern.exec(text);
                if (match) {
                    pattern.lastIndex = 0; // 정규식 상태 초기화
                    return match;
                }
                pattern.lastIndex = 0; // 정규식 상태 초기화
            }
            return null;
        };
        
        // 패턴 매칭으로 데이터 추출
        let match;
        
        logger.debug('[TRACE] 텍스트 파싱 시작, 텍스트 길이:', text.length);
        logger.debug('[TRACE] 텍스트 샘플:', text.substring(0, 300));
        
        // 혈압
        if (match = tryPatterns(patterns.systolicBP, text)) {
            patientData.systolicBP.average = parseInt(match[1]);
            logger.debug('[TRACE] 수축기혈압 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.diastolicBP, text)) {
            patientData.diastolicBP.average = parseInt(match[1]);
            logger.debug('[TRACE] 확장기혈압 파싱:', match[1]);
        }
        
        // 지질
        if (match = tryPatterns(patterns.triglycerides, text)) {
            patientData.lipids.triglycerides = parseInt(match[1]);
            logger.debug('[TRACE] 중성지방 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.hdl, text)) {
            patientData.lipids.hdl = parseInt(match[1]);
            logger.debug('[TRACE] HDL 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.ldl, text)) {
            patientData.lipids.ldl = parseInt(match[1]);
            logger.debug('[TRACE] LDL 파싱:', match[1]);
        }
        
        // 혈당
        if (match = tryPatterns(patterns.bloodSugar, text)) {
            patientData.bloodSugar.general = parseInt(match[1]);
            logger.debug('[TRACE] 혈당 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.fastingBloodSugar, text)) {
            patientData.bloodSugar.fasting = parseInt(match[1]);
            logger.debug('[TRACE] 공복시혈당 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.hba1c, text)) {
            patientData.bloodSugar.hba1c = parseFloat(match[1]);
            logger.debug('[TRACE] HbA1c 파싱:', match[1]);
        }
        
        // 비만
        if (match = tryPatterns(patterns.bmi, text)) {
            patientData.obesity.bmi = parseFloat(match[1]);
            logger.debug('[TRACE] BMI 파싱:', match[1]);
        }
        if (match = tryPatterns(patterns.waist, text)) {
            patientData.obesity.waistCircumference = parseInt(match[1]);
            logger.debug('[TRACE] 복위 파싱:', match[1]);
        }
        
        // 성별
        if (match = tryPatterns(patterns.gender, text)) {
            const genderValue = match[1] || match[2]; // 두 번째 그룹도 확인
            logger.debug('[TRACE] 성별 파싱:', genderValue);
            if (genderValue === '남성' || genderValue === '1' || genderValue === 'M' || genderValue === 'male' || genderValue === '남') {
                patientData.gender = 1;
            } else if (genderValue === '여성' || genderValue === '3' || genderValue === 'F' || genderValue === 'female' || genderValue === '여') {
                patientData.gender = 3;
            }
        }
        
        // 요단백
        if (match = tryPatterns(patterns.urineProtein, text)) {
            patientData.urineProtein = match[1];
            logger.debug('[TRACE] 요단백 파싱:', match[1]);
        }
        
        // 추가: 테이블 형식 데이터 파싱 시도
        const tryTableParsing = () => {
            // 환자명 기준으로 해당 행 찾기
            const lines = text.split(/\r?\n/);
            let patientLine = null;
            
            for (const line of lines) {
                if (line.includes(patientName)) {
                    patientLine = line;
                    break;
                }
            }
            
            if (patientLine) {
                logger.debug('[TRACE] 환자 라인 발견:', patientLine);
                
                // 탭이나 공백으로 분리된 데이터 처리
                const cells = patientLine.split(/[\t\s]+/).filter(cell => cell.trim());
                logger.debug('[TRACE] 셀 데이터:', cells);
                
                // 숫자 데이터를 순서대로 찾아서 매핑
                const numbers = [];
                for (const cell of cells) {
                    const num = parseFloat(cell);
                    if (!isNaN(num) && num > 0) {
                        numbers.push(num);
                    }
                }
                
                logger.debug('[TRACE] 추출된 숫자들:', numbers);
                
                // 일반적인 건강검진 순서로 매핑 시도
                if (numbers.length >= 6) {
                    let idx = 0;
                    
                    // 성별이 숫자로 표현된 경우
                    if (numbers[idx] === 1 || numbers[idx] === 3) {
                        patientData.gender = numbers[idx];
                        idx++;
                    }
                    
                    // 혈압 (수축기, 확장기)
                    if (numbers[idx] >= 80 && numbers[idx] <= 200) {
                        patientData.systolicBP.average = numbers[idx];
                        idx++;
                    }
                    if (numbers[idx] >= 50 && numbers[idx] <= 120) {
                        patientData.diastolicBP.average = numbers[idx];
                        idx++;
                    }
                    
                    // 지질 데이터들
                    for (let i = idx; i < numbers.length; i++) {
                        const num = numbers[i];
                        if (num >= 100 && num <= 500 && !patientData.lipids.triglycerides) {
                            patientData.lipids.triglycerides = num;
                        } else if (num >= 20 && num <= 80 && !patientData.lipids.hdl) {
                            patientData.lipids.hdl = num;
                        } else if (num >= 60 && num <= 200 && !patientData.lipids.ldl) {
                            patientData.lipids.ldl = num;
                        } else if (num >= 70 && num <= 200 && !patientData.bloodSugar.fasting) {
                            patientData.bloodSugar.fasting = num;
                        } else if (num >= 4 && num <= 15 && num.toString().includes('.') && !patientData.bloodSugar.hba1c) {
                            patientData.bloodSugar.hba1c = num;
                        } else if (num >= 15 && num <= 40 && !patientData.obesity.bmi) {
                            patientData.obesity.bmi = num;
                        } else if (num >= 60 && num <= 120 && !patientData.obesity.waistCircumference) {
                            patientData.obesity.waistCircumference = num;
                        }
                    }
                }
            }
        };
        
        tryTableParsing();
        
        // 최소한의 데이터가 있는지 확인
        const hasData = patientData.systolicBP.average || patientData.diastolicBP.average ||
                       patientData.lipids.triglycerides || patientData.lipids.hdl || patientData.lipids.ldl ||
                       patientData.bloodSugar.general || patientData.bloodSugar.fasting || patientData.bloodSugar.hba1c ||
                       patientData.obesity.bmi || patientData.obesity.waistCircumference;
        
        if (hasData) {
            logger.debug('[TRACE] 텍스트에서 환자 데이터 파싱 성공');
            return patientData;
        } else {
            logger.debug('[TRACE] 텍스트에서 유효한 건강 데이터를 찾을 수 없음');
            return null;
        }
        
    } catch (error) {
        logger.debug('[TRACE] 텍스트 파싱 중 오류:', error.message);
        return null;
    }
}

// PostgreSQL에서 환자 데이터 조회 (RAG 백업 포함)
async function getPatientDataFromDatabase(userRequest, username = 'admin') {
    logger.debug('[TRACE] getPatientDataFromDatabase 함수 진입 - 첫번째 라인');
    
    try {
        logger.debug('[TRACE] try 블록 진입');
        logger.debug('[TRACE] userRequest 타입:', typeof userRequest);
        logger.debug('[TRACE] userRequest 값:', JSON.stringify(userRequest));
        
        logger.debug('[TRACE] PostgreSQL 모듈 로딩 시작');
        
        // 공통 DB 매니저 로딩
        let dbManager;
        try {
            logger.debug('[TRACE] DB 매니저 import 시작');
            const dbModule = await import('../services/database/index.js');
            dbManager = dbModule.default;
            logger.debug('[TRACE] DB 매니저 로드 성공');
        } catch (moduleError) {
            logger.debug('[TRACE] DB 매니저 로딩 오류:', moduleError.message);
            logger.debug('[TRACE] DB 매니저 로딩 스택:', moduleError.stack);
            
            // DB 실패 시 바로 RAG 시도
            logger.debug('[TRACE] DB 실패로 RAG 백업 검색 시작');
            return await getPatientDataFromRAG(userRequest, username);
        }
        
        logger.debug('[TRACE] 데이터베이스 조회 시작');
        
        // 환자명 추출 - 더 넓은 범위의 패턴 매칭
        logger.debug('[getPatientDataFromDatabase] 원본 요청:', userRequest);
        
        // 다양한 패턴으로 환자명 추출 시도
        let patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*환자/);
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*의\s*건강/);
        }
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*건강검진/);
        }
        if (!patientNameMatch) {
            patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*검진/);
        }
        
        const patientName = patientNameMatch ? patientNameMatch[1].trim() : null;
        logger.debug('[getPatientDataFromDatabase] 매치 결과:', patientNameMatch);
        logger.debug('[getPatientDataFromDatabase] 추출된 환자명 (trim 전):', patientNameMatch?.[1]);
        logger.debug('[getPatientDataFromDatabase] 추출된 환자명 (trim 후):', patientName);
        
        if (!patientName) {
            logger.debug('[getPatientDataFromDatabase] 환자명을 추출할 수 없음');
            // 환자명 추출 실패 시 RAG 시도
            logger.debug('[TRACE] 환자명 추출 실패로 RAG 백업 검색 시작');
            return await getPatientDataFromRAG(userRequest, username);
        }
        
        logger.debug('[getPatientDataFromDatabase] 추출된 환자명:', patientName);
        
        // 테이블 존재 여부 확인
        logger.debug('[getPatientDataFromDatabase] patient_health_data 테이블 존재 여부 확인');
        const tableExistsQuery = `
            SELECT EXISTS (
                SELECT FROM information_schema.tables 
                WHERE table_schema = 'public' 
                AND table_name = 'patient_health_data'
            );
        `;
        
        const tableExistsResult = await dbManager.query(tableExistsQuery);
        const tableExists = tableExistsResult.rows[0].exists;
        
        logger.debug('[getPatientDataFromDatabase] 테이블 존재 여부:', tableExists);
        
        if (!tableExists) {
            logger.debug('[getPatientDataFromDatabase] patient_health_data 테이블이 존재하지 않음 - RAG 백업 검색으로 이동');
            return await getPatientDataFromRAG(userRequest, username);
        }
        
        // 환자명으로 검색 (LIKE 사용)
        const searchPattern = `%${patientName}%`;
        const query = `
            SELECT * FROM patient_health_data 
            WHERE patient_name LIKE $1 
            ORDER BY id LIMIT 1
        `;
        
        logger.debug('[getPatientDataFromDatabase] SQL 쿼리:', query);
        logger.debug('[getPatientDataFromDatabase] 검색 패턴:', searchPattern);
        
        const result = await dbManager.query(query, [searchPattern]);
        
        logger.debug('[getPatientDataFromDatabase] 쿼리 결과 개수:', result.rows.length);
        
        if (result.rows.length === 0) {
            logger.debug('[getPatientDataFromDatabase] 데이터베이스에서 환자를 찾을 수 없음:', patientName);
            
            // 데이터베이스에서 찾지 못한 경우 RAG 검색 시도
            logger.debug('[TRACE] DB에서 환자 없음, RAG 백업 검색 시작');
            return await getPatientDataFromRAG(userRequest, username);
        }
        
        const patient = result.rows[0];
        logger.debug('[getPatientDataFromDatabase] 환자 데이터 조회 성공:', patient.patient_name);
        
        // 구조화된 환자 데이터 반환
        return {
            name: patient.patient_name,
            gender: patient.gender,
            birthDate: patient.birth_date,
            checkupDate: patient.checkup_date,
            systolicBP: {
                first: patient.systolic_bp_1,
                second: patient.systolic_bp_2,
                average: patient.systolic_bp_avg
            },
            diastolicBP: {
                first: patient.diastolic_bp_1,
                second: patient.diastolic_bp_2,
                average: patient.diastolic_bp_avg
            },
            lipids: {
                triglycerides: patient.triglycerides,
                hdl: patient.hdl,
                ldl: patient.ldl
            },
            bloodSugar: {
                general: patient.blood_sugar,
                fasting: patient.fasting_blood_sugar,
                random: patient.random_blood_sugar,
                hba1c: patient.hba1c
            },
            obesity: {
                bmi: patient.bmi,
                waistCircumference: patient.waist_circumference
            },
            urineProtein: patient.urine_protein,
            dataSource: 'PostgreSQL Database',
            sourceFile: patient.source_file || '건강검진데이터.xlsx',
            timestamp: new Date().toISOString(),
            relevanceScore: 1.0  // 정확한 매치이므로 최고점
        };
        
    } catch (error) {
        logger.debug('[TRACE] 최상위 catch 블록 진입');
        logger.debug('[TRACE] 오류 타입:', typeof error);
        logger.debug('[TRACE] 오류 메시지:', error.message || 'No message');
        logger.debug('[TRACE] 오류 스택:', error.stack || 'No stack');
        logger.debug('[TRACE] 오류 생성자:', error.constructor?.name || 'No constructor');
        
        // 예외 발생 시에도 RAG 백업 시도
        logger.debug('[TRACE] 예외 발생으로 RAG 백업 검색 시작');
        try {
            return await getPatientDataFromRAG(userRequest, username);
        } catch (ragError) {
            logger.debug('[TRACE] RAG 백업도 실패:', ragError.message);
            return null;
        }
    }
}

// 환자 데이터 자동 임포트 함수
async function attemptPatientDataImport(ragResults, userRequest, username = 'admin') {
    const logger = console;
    logger.debug('[AUTO-IMPORT] 환자 데이터 자동 임포트 시도 시작');
    
    try {
        // 환자 데이터 파일 패턴 정의
        const PATIENT_DATA_FILE_PATTERNS = [
            /patient_?data\.xlsx?$/i,
            /patients\.xlsx?$/i,
            /health_?checkup\.xlsx?$/i,
            /건강검진데이터\.xlsx?$/i,
            /환자데이터\.xlsx?$/i,
            /sample_patients_\d+\.xlsx?$/i,
            /jp-data\.xlsx?$/i
        ];
        
        logger.debug('[AUTO-IMPORT] RAG 시스템에서 사용자 파일 목록 조회 중...');
        
        // RAG 시스템에서 직접 사용자 파일 목록 가져오기
        const response = await fetch(`${URLs.RAG_SERVER()}/list?user_id=${username}`);
        if (!response.ok) {
            throw new Error('RAG 파일 목록 조회 실패: ' + response.statusText);
        }
        
        const fileList = await response.json();
        logger.debug('[AUTO-IMPORT] RAG 파일 목록 조회 성공 - 총', fileList.documents?.length || 0, '개 파일');
        
        let patientDataFiles = [];
        
        // 파일 목록에서 모든 환자 데이터 파일 찾기
        if (fileList.documents && Array.isArray(fileList.documents)) {
            for (let i = 0; i < fileList.documents.length; i++) {
                const doc = fileList.documents[i];
                const filename = doc.file_name;
                logger.debug('[AUTO-IMPORT] 파일', i+1, '검사:', filename);
            
                if (filename && PATIENT_DATA_FILE_PATTERNS.some(pattern => pattern.test(filename))) {
                    logger.debug('[AUTO-IMPORT] 환자 데이터 파일 발견:', filename);
                    
                    // 파일 경로 찾기
                    const filePath = doc.source || doc.metadata?.full_path || `/home/hamonikr/문서/rag-docs/${username}/${filename}`;
                    
                    patientDataFiles.push({
                        filename: filename,
                        path: filePath,
                        doc: doc
                    });
                    
                    logger.debug('[AUTO-IMPORT] 파일 경로:', filePath);
                }
            }
        }
        
        if (patientDataFiles.length === 0) {
            logger.debug('[AUTO-IMPORT] 환자 데이터 파일을 찾을 수 없음');
            return { success: false, error: '환자 데이터 파일을 찾을 수 없음' };
        }
        
        logger.debug(`[AUTO-IMPORT] 총 ${patientDataFiles.length}개의 환자 데이터 파일 발견`);
        
        const fs = await import('fs/promises');
        const patientProcessor = await import('./patient-data-processor.js');
        
        let totalInsertCount = 0;
        let successfulImports = [];
        let failedImports = [];
        
        // 모든 환자 데이터 파일을 순차적으로 처리
        for (const fileInfo of patientDataFiles) {
            try {
                // 파일이 존재하는지 확인
                try {
                    await fs.default.access(fileInfo.path);
                } catch (accessError) {
                    logger.debug('[AUTO-IMPORT] 파일에 접근할 수 없음:', fileInfo.path);
                    failedImports.push({
                        filename: fileInfo.filename,
                        error: '파일에 접근할 수 없음'
                    });
                    continue;
                }
                
                logger.debug('[AUTO-IMPORT] 환자 데이터 파일 자동 임포트 실행:', fileInfo.filename);
                
                const result = await patientProcessor.processPatientDataFile(fileInfo.path, fileInfo.filename);
                
                if (result.success) {
                    logger.debug(`[AUTO-IMPORT] ${fileInfo.filename} 임포트 성공: ${result.insertCount}명의 환자 데이터`);
                    totalInsertCount += result.insertCount;
                    successfulImports.push({
                        filename: fileInfo.filename,
                        insertCount: result.insertCount
                    });
                } else {
                    logger.debug(`[AUTO-IMPORT] ${fileInfo.filename} 임포트 실패:`, result.error);
                    failedImports.push({
                        filename: fileInfo.filename,
                        error: result.error
                    });
                }
                
            } catch (error) {
                logger.debug(`[AUTO-IMPORT] ${fileInfo.filename} 처리 중 예외:`, error.message);
                failedImports.push({
                    filename: fileInfo.filename,
                    error: error.message
                });
            }
        }
        
        // 결과 보고
        if (successfulImports.length > 0) {
            const fileNames = successfulImports.map(f => `${f.filename}(${f.insertCount}명)`).join(', ');
            const message = `환자 데이터 파일 ${successfulImports.length}개에서 총 ${totalInsertCount}명의 데이터를 자동으로 임포트했습니다: ${fileNames}`;
            
            if (failedImports.length > 0) {
                const failedNames = failedImports.map(f => f.filename).join(', ');
                logger.debug(`[AUTO-IMPORT] 실패한 파일들: ${failedNames}`);
            }
            
            return {
                success: true,
                message: message,
                insertCount: totalInsertCount,
                successfulFiles: successfulImports,
                failedFiles: failedImports
            };
        } else {
            return { 
                success: false, 
                error: `모든 환자 데이터 파일 임포트 실패. 실패한 파일: ${failedImports.map(f => f.filename).join(', ')}` 
            };
        }
        
    } catch (error) {
        logger.debug('[AUTO-IMPORT] 자동 임포트 중 오류:', error.message);
        return { success: false, error: error.message };
    }
}

// 건강검진 분석 헬퍼 함수들 (RAG fallback)
async function parsePatientDataFromRag(ragResults, userRequest) {
    const logger = console;
    
    try {
        logger.debug('[parsePatientDataFromRag] RAG 결과 구조 분석 시작');
        logger.debug('[parsePatientDataFromRag] ragResults 키들:', Object.keys(ragResults));
        
        // ChromaDB 형식의 RAG 결과를 파싱
        if (!ragResults.documents || !ragResults.documents[0] || ragResults.documents[0].length === 0) {
            logger.debug('[parsePatientDataFromRag] 문서가 없음');
            return null;
        }
        
        const documents = ragResults.documents[0];
        const metadatas = ragResults.metadatas ? ragResults.metadatas[0] : [];
        const distances = ragResults.distances ? ragResults.distances[0] : [];
        
        logger.debug('[parsePatientDataFromRag] 문서 수:', documents.length);
        
        // 가장 관련성 높은 문서들을 결합하여 환자 데이터 추출
        const combinedContent = documents.slice(0, 3).join('\n\n');
        
        // 환자명 추출 (사용자 요청에서)
        const patientNameMatch = userRequest.match(/([가-힣]{2,4}|[A-Za-z\s]{2,20}|[\u4e00-\u9fff]{2,4})\s*환자|([가-힣]{2,4}|[A-Za-z\s]{2,20}|[\u4e00-\u9fff]{2,4})\s*의\s*건강|([가-힣]{2,4}|[A-Za-z\s]{2,20}|[\u4e00-\u9fff]{2,4})\s*검진/);
        const patientName = patientNameMatch ? (patientNameMatch[1] || patientNameMatch[2] || patientNameMatch[3]) : '환자';
        
        logger.debug('[parsePatientDataFromRag] 추출된 환자명:', patientName);
        
        // 🚨 중요: 실제 건강검진 데이터가 포함되어 있는지 검증
        const healthDataKeywords = [
            '혈압', '수축기', '이완기', 'mmHg',
            '혈당', '공복', 'mg/dl', 'HbA1c',
            '콜레스테롤', 'HDL', 'LDL', '중성지방',
            'BMI', '복위', '비만',
            '요단백', '소변검사',
            '건강검진', '검진결과', '판정결과'
        ];
        
        const hasHealthData = healthDataKeywords.some(keyword => 
            combinedContent.includes(keyword)
        );
        
        // 환자명이 실제로 문서에 포함되어 있는지 확인
        const patientNameInContent = combinedContent.includes(patientName);
        
        logger.debug('[parsePatientDataFromRag] 건강검진 데이터 포함 여부:', hasHealthData);
        logger.debug('[parsePatientDataFromRag] 환자명 포함 여부:', patientNameInContent);
        logger.debug('[parsePatientDataFromRag] 문서 내용 미리보기:', combinedContent.substring(0, 200));
        
        // 🛑 건강검진 관련 데이터가 없거나 환자명이 문서에 없으면 null 반환
        if (!hasHealthData || !patientNameInContent) {
            logger.debug('[parsePatientDataFromRag] 건강검진 데이터 또는 환자명이 없어 null 반환');
            return null;
        }
        
        // 구조화된 환자 데이터 생성 - dataSource와 filename 정보 추가
        const firstMetadata = metadatas[0] || {};
        
        // 파일명 추출 로직 개선
        let extractedFilename = 'RAG Documents';
        
        // 우선순위: file_name → filename → source (파일명만 추출) → RAG Documents
        if (firstMetadata.file_name) {
            extractedFilename = firstMetadata.file_name;
        } else if (firstMetadata.filename) {
            extractedFilename = firstMetadata.filename;
        } else if (firstMetadata.source) {
            // source에서 파일명만 추출 (경로 제거)
            const pathMatch = firstMetadata.source.match(/([^/\\]+\.(xlsx?|pdf|docx?|txt))$/i);
            if (pathMatch) {
                extractedFilename = pathMatch[1];
            } else {
                // 확장자가 없어도 마지막 경로 부분을 파일명으로 사용
                const lastPart = firstMetadata.source.split(/[/\\]/).pop();
                if (lastPart && lastPart !== 'source') {
                    extractedFilename = lastPart;
                }
            }
        }
        
        logger.debug('[parsePatientDataFromRag] 추출된 파일명:', extractedFilename);
        logger.debug('[parsePatientDataFromRag] 첫 번째 메타데이터:', firstMetadata);
        
        const patientData = {
            name: patientName,
            content: combinedContent,
            sources: metadatas.map(m => m.source || '문서').slice(0, 3),
            relevanceScore: distances.length > 0 ? (1 - distances[0]) : 0.5,
            timestamp: new Date().toISOString(),
            rawDocuments: documents.slice(0, 3),
            searchQuery: userRequest,
            dataSource: 'RAG Documents',
            filename: extractedFilename,
            metadata: firstMetadata
        };
        
        logger.debug('[parsePatientDataFromRag] 파싱된 환자 데이터 구조:', {
            name: patientData.name,
            contentLength: patientData.content.length,
            sourcesCount: patientData.sources.length,
            relevanceScore: patientData.relevanceScore
        });
        
        return patientData;
        
    } catch (error) {
        logger.debug('[parsePatientDataFromRag] 파싱 오류:', error.message);
        return null;
    }
}

// 환자 데이터로부터 실제 판정값 계산 함수
function calculateActualAssessments(patientData, criteria) {
    const assessments = {};
    
    // ①혈압해당 판정
    const systolicBP = patientData.systolicBP?.average;
    const diastolicBP = patientData.diastolicBP?.average;
    assessments.bloodPressure = (
        (systolicBP && systolicBP >= criteria.bloodPressure.systolic) ||
        (diastolicBP && diastolicBP >= criteria.bloodPressure.diastolic)
    ) ? 1 : 3;
    
    // ②지질해당 판정
    const triglycerides = patientData.lipids?.triglycerides;
    const hdl = patientData.lipids?.hdl;
    const ldl = patientData.lipids?.ldl;
    assessments.lipid = (
        (triglycerides && triglycerides >= criteria.lipid.triglycerides) ||
        (hdl && hdl < criteria.lipid.hdl) ||
        (ldl && ldl >= criteria.lipid.ldl)
    ) ? 1 : 3;
    
    // ③HbA1c수진유무
    assessments.hba1cTested = patientData.bloodSugar?.hba1c ? 3 : 1;
    
    // ④혈당해당 판정 (특별 규칙 포함)
    const bloodSugar = patientData.bloodSugar?.general;
    const fastingBS = patientData.bloodSugar?.fasting;
    const randomBS = patientData.bloodSugar?.random;
    const hba1c = patientData.bloodSugar?.hba1c;
    
    // 특별 규칙: 수시혈당만 있고 HbA1c 없는 경우 비해당
    if (randomBS && !hba1c && !bloodSugar && !fastingBS) {
        assessments.bloodSugar = 3;
    } else {
        assessments.bloodSugar = (
            (bloodSugar && bloodSugar >= criteria.bloodSugar.general) ||
            (fastingBS && fastingBS >= criteria.bloodSugar.fasting) ||
            (hba1c && hba1c >= criteria.bloodSugar.hba1c)
        ) ? 1 : 3;
    }
    
    // ⑤비만해당 판정
    const bmi = patientData.obesity?.bmi;
    const waist = patientData.obesity?.waistCircumference;
    const gender = patientData.gender;
    const waistThreshold = gender === 1 ? criteria.obesity.waist.male : criteria.obesity.waist.female;
    
    assessments.obesity = (
        (bmi && bmi >= criteria.obesity.bmi) ||
        (waist && waist >= waistThreshold)
    ) ? 1 : 3;
    
    // ⑥요단백 판정
    const urineProtein = patientData.urineProtein || '－';
    assessments.urineProtein = criteria.urineProtein.codes[urineProtein] || 1;
    
    // ⑦최종 판정
    const riskFactors = [
        assessments.bloodPressure,
        assessments.lipid,
        assessments.bloodSugar,
        assessments.obesity
    ].filter(v => v === 1).length;
    
    if (riskFactors === 4) assessments.finalRisk = 1; // 고위험군
    else if (riskFactors >= 1) assessments.finalRisk = 2; // 중위험군
    else assessments.finalRisk = 3; // 저위험군
    
    return assessments;
}

// AI 응답에서 판정값 추출 함수
function extractAssessmentsFromResponse(response) {
    const assessments = {};
    
    // 정규식으로 각 판정값 추출
    const bloodPressureMatch = response.match(/①혈압해당:\s*([13])/);
    if (bloodPressureMatch) assessments.bloodPressure = parseInt(bloodPressureMatch[1]);
    
    const lipidMatch = response.match(/②지질해당:\s*([13])/);
    if (lipidMatch) assessments.lipid = parseInt(lipidMatch[1]);
    
    const hba1cMatch = response.match(/③HbA1c수진유무:\s*([13])/);
    if (hba1cMatch) assessments.hba1cTested = parseInt(hba1cMatch[1]);
    
    const bloodSugarMatch = response.match(/④혈당해당:\s*([13])/);
    if (bloodSugarMatch) assessments.bloodSugar = parseInt(bloodSugarMatch[1]);
    
    const obesityMatch = response.match(/⑤비만해당:\s*([13])/);
    if (obesityMatch) assessments.obesity = parseInt(obesityMatch[1]);
    
    const urineMatch = response.match(/⑥요단백:\s*([13579])/);
    if (urineMatch) assessments.urineProtein = parseInt(urineMatch[1]);
    
    const finalMatch = response.match(/⑦판정:\s*([1234])/);
    if (finalMatch) assessments.finalRisk = parseInt(finalMatch[1]);
    
    return assessments;
}

// 검증 함수
function validateHealthAssessment(aiResponse, patientData, criteria) {
    const actualAssessments = calculateActualAssessments(patientData, criteria);
    const aiAssessments = extractAssessmentsFromResponse(aiResponse);
    
    const validationResult = {
        isValid: true,
        errors: [],
        actual: actualAssessments,
        ai: aiAssessments
    };
    
    // 각 판정값 비교
    for (const [key, actualValue] of Object.entries(actualAssessments)) {
        const aiValue = aiAssessments[key];
        if (aiValue !== actualValue) {
            validationResult.isValid = false;
            validationResult.errors.push(
                `${key}: AI 판정(${aiValue}) ≠ 실제 계산값(${actualValue})`
            );
        }
    }
    
    return validationResult;
}

async function performHealthAnalysis(patientData, context, attempt = 1) {
    const logger = console;

    logger.debug('[performHealthAnalysis] 🚀 FUNCTION START:', {
        patientDataType: typeof patientData,
        contextType: typeof context,
        contextKeys: context ? Object.keys(context) : null,
        attempt
    });

    // context에서 sessionId 추출
    const sessionId = context?.sessionId || 'health-analysis-' + Date.now();

    logger.debug('[performHealthAnalysis] 📝 Session ID:', sessionId);
    
    // 건강검진 평가 기준 정의 (output.md, metric.md 기준)
    const HEALTH_EVALUATION_CRITERIA = {
        bloodPressure: {
            systolic: 140,    // mmHg 이상 시 해당(1)
            diastolic: 90     // mmHg 이상 시 해당(1)
        },
        lipid: {
            triglycerides: 150,  // mg/dl 이상 시 해당(1)
            hdl: 40,            // mg/dl 미만 시 해당(1)
            ldl: 140            // mg/dl 이상 시 해당(1)
        },
        bloodSugar: {
            general: 100,       // mg/dl 이상 시 해당(1)
            fasting: 100,       // mg/dl 이상 시 해당(1)
            hba1c: 5.6          // % 이상 시 해당(1)
        },
        obesity: {
            bmi: 25,           // kg/m² 이상 시 해당(1)
            waist: { male: 85, female: 90 }  // cm 이상 시 해당(1)
        },
        urineProtein: {
            // ー(음성)→1, ±→3, ＋→5, ＋＋→7, ＋＋＋→9
            codes: { '－': 1, '±': 3, '＋': 5, '＋＋': 7, '＋＋＋': 9 }
        }
    };
    
    try {        
        // 데이터베이스 구조화 데이터를 텍스트로 포맷팅
        let content;
        if (patientData.dataSource === 'PostgreSQL Database') {
            // 구조화된 데이터베이스 데이터를 포맷팅
            content = `**기본정보**
- 성명: ${patientData.name}
- 성별: ${patientData.gender === 1 ? '남성' : '여성'} (${patientData.gender})
- 생년월일: ${patientData.birthDate || '미확인'}
- 검진일: ${patientData.checkupDate || '미확인'}

**혈압측정값**
- 수축기혈압: ${patientData.systolicBP?.first || '-'}mmHg / ${patientData.systolicBP?.second || '-'}mmHg (평균: ${patientData.systolicBP?.average || '-'}mmHg)
- 확장기혈압: ${patientData.diastolicBP?.first || '-'}mmHg / ${patientData.diastolicBP?.second || '-'}mmHg (평균: ${patientData.diastolicBP?.average || '-'}mmHg)

**혈중지질**
- 중성지방: ${patientData.lipids?.triglycerides || '-'}mg/dl
- HDL콜레스테롤: ${patientData.lipids?.hdl || '-'}mg/dl
- LDL콜레스테롤: ${patientData.lipids?.ldl || '-'}mg/dl

**혈당**
- 일반혈당: ${patientData.bloodSugar?.general || '-'}mg/dl
- 공복시혈당: ${patientData.bloodSugar?.fasting || '-'}mg/dl
- 수시혈당: ${patientData.bloodSugar?.random || '-'}mg/dl
- HbA1c: ${patientData.bloodSugar?.hba1c || '-'}%

**비만도**
- BMI: ${patientData.obesity?.bmi || '-'}kg/m²
- 복위: ${patientData.obesity?.waistCircumference || '-'}cm

**요단백**: ${patientData.urineProtein || '-'}

**데이터 출처**: ${patientData.dataSource} (관련도: ${patientData.relevanceScore || 1.0})`;
        } else {
            // RAG 데이터의 경우 기존 방식 사용 (이미 handler에서 검증됨)
            content = patientData.content || patientData.rawContent || '데이터를 찾을 수 없습니다.';
            
            // RAG 데이터에 대한 추가 검증 - performHealthAnalysis 레벨에서도 확인
            const noDataIndicators = [
                '데이터를 찾을 수 없습니다',
                '찾을 수 없습니다',
                '검색 결과가 없습니다',
                '데이터가 없습니다',
                '정보가 없습니다'
            ];
            
            const hasNoDataIndicator = noDataIndicators.some(indicator => 
                content.toLowerCase().includes(indicator.toLowerCase())
            );
            
            if (!content || content.trim() === '' || hasNoDataIndicator || content.length < 100) {
                throw new Error(`${patientData.name} 환자의 건강검진 데이터가 부족하여 분석할 수 없습니다.`);
            }
        }
        
        const analysisPrompt = `다음은 ${patientData.name} 환자의 건강검진 데이터입니다.

건강검진 데이터:
${content}

**중요 지침:**
1. 제공된 데이터만을 기반으로 분석해야 합니다.
2. 데이터가 부족하거나 없는 항목은 "데이터 없음" 또는 "확인 불가"로 표시해야 합니다.
3. 추측하거나 가정하여 임의의 수치나 판정을 생성하지 마세요.
4. 실제 데이터가 없는 경우 허구의 분석 결과를 제공하지 마세요.

**반드시 다음 표준 형식으로 분석 결과를 제공해야 합니다:**

=== 건강검진 판정 결과 ===

【기본정보】
성명: ${patientData.name}
성별: ${patientData.gender === 1 ? '남성' : '여성'} (${patientData.gender})
검진일: ${patientData.checkupDate || '미확인'}

【항목별 판정】
①혈압해당: [1 또는 3] ([해당/비해당])
  - 수축기혈압 평균: ${patientData.systolicBP?.average || '-'}mmHg (기준: 140mmHg 이상)
  - 확장기혈압 평균: ${patientData.diastolicBP?.average || '-'}mmHg (기준: 90mmHg 이상)

②지질해당: [1 또는 3] ([해당/비해당])
  - 중성지방: ${patientData.lipids?.triglycerides || '-'}mg/dl (기준: 150mg/dl 이상)
  - HDL: ${patientData.lipids?.hdl || '-'}mg/dl (기준: 40mg/dl 미만)
  - LDL: ${patientData.lipids?.ldl || '-'}mg/dl (기준: 140mg/dl 이상)

③HbA1c수진유무: [1 또는 3] ([미수진/수진])

④혈당해당: [1 또는 3] ([해당/비해당])
  - 관련 수치와 기준값 명시

⑤비만해당: [1 또는 3] ([해당/비해당])
  - BMI: ${patientData.obesity?.bmi || '-'}kg/m² (기준: 25kg/m² 이상)
  - 복위: ${patientData.obesity?.waistCircumference || '-'}cm (남성 기준: 85cm 이상, 여성 기준: 90cm 이상)

⑥요단백: [1, 3, 5, 7, 또는 9] (변환값)

【최종 판정】
⑦판정: [1, 2, 3, 또는 4] ([고위험군/중위험군/저위험군/대상외])
- 판정 근거 설명

**평가기준**:
- 혈압: 수축기 ≥140mmHg 또는 확장기 ≥90mmHg 시 해당(1)
- 지질: 중성지방 ≥150, HDL <40, LDL ≥140 중 하나라도 해당 시 해당(1)  
- 혈당: 일반/공복 ≥100mg/dl, HbA1c ≥5.6% 시 해당(1)
- 비만: BMI ≥25 또는 복위(남≥85cm, 여≥90cm) 시 해당(1)
- 최종: 4개 항목 모두 해당(1), 1~3개 해당(2), 0개 해당(3), 중증질환 대상외(4)

위 형식을 정확히 따라 분석해주세요.`;

        // 새로운 워크플로우: 환자데이터검색 → 정규화 → AI분석 → 분석결과검증 → 최종 AI응답
        
        // 3단계: AI가 자유롭게 의학적 분석 수행
        const freeAnalysisPrompt = `다음은 ${patientData.name} 환자의 건강검진 데이터입니다.

건강검진 데이터:
${content}

**중요 지침:**
- 오직 제공된 실제 데이터만을 기반으로 분석해야 합니다.
- 데이터가 부족하거나 누락된 항목에 대해서는 추측하지 말고 "데이터 부족으로 평가 불가" 등으로 명시해야 합니다.
- 가상의 수치나 허구의 분석 결과를 생성하지 마세요.
- 환자 데이터가 전혀 없거나 불충분한 경우, "충분한 건강검진 데이터가 없어 분석을 수행할 수 없습니다"라고 명시해야 합니다.

위 데이터를 바탕으로 전문적인 건강검진 분석을 해주세요. 다음 내용을 포함해서 상세히 분석해주세요:

1. **전체적인 건강 상태 평가**
   - 전반적인 상태 요약
   - 위험군 분류 (고/중/저위험군)
   - 주요 건강 이슈

2. **주요 검진 지표별 분석**
   - 혈압, 혈중지질, 혈당, HbA1c, BMI/복위, 요단백 등
   - 각 지표의 정상/이상 여부와 의학적 의미

3. **위험 요소 및 주의사항**
   - 현재 발견된 위험 요소들
   - 향후 주의해야 할 질병들
   - 기왕력/현병력과의 연관성

4. **개선 권장사항**
   - 생활습관 개선 방안
   - 운동 및 영양 관리
   - 정기 검진 권장사항

5. **추가 검사 필요 여부**
   - 권장하는 추가 검사들
   - 검사가 필요한 이유

전문의의 관점에서 환자에게 도움이 되는 실용적인 조언을 포함해 주세요.`;

        let aiAnalysis;
        let aiChatFunction; // AI 함수를 함수 상단에서 선언
        // AI 자유 분석을 위한 provider 결정
        let provider = 'ollama'; // 기본값 설정
        let session = null; // 세션 변수를 함수 상단에서 선언

        try {
            // AI 자유 분석 호출
            const { sharedContext } = await import('../aiFeatures.js');

            // 세션 처리 강화

            if (sessionId) {
                try {
                    session = await sharedContext.getSession(sessionId);
                    logger.debug('[health_analysis] 세션 조회 결과:', {
                        hasSession: !!session,
                        sessionId,
                        sessionProvider: session?.provider,
                        sessionGetProviderType: typeof session?.getProvider,
                        contextSessionId: context?.sessionId
                    });
                } catch (sessionError) {
                    logger.error('[health_analysis] 세션 조회 오류:', sessionError);
                    session = null;
                }
            }

            // provider 결정 로직 강화
            if (session) {
                // 세션에서 provider 가져오기 시도
                if (session.provider && typeof session.provider === 'string') {
                    provider = session.provider;
                    logger.debug('[health_analysis] Provider from session.provider:', provider);
                } else if (typeof session.getProvider === 'function') {
                    const sessionProvider = session.getProvider();
                    if (sessionProvider && typeof sessionProvider === 'string') {
                        provider = sessionProvider;
                        logger.debug('[health_analysis] Provider from session.getProvider():', provider);
                    }
                }
            }

            // context에서도 provider 정보 시도
            if (!provider || provider === 'undefined' || provider === 'null') {
                if (context?.session?.provider) {
                    provider = context.session.provider;
                    logger.debug('[health_analysis] Provider from context.session:', provider);
                } else if (context?.session?.getProvider && typeof context.session.getProvider === 'function') {
                    provider = context.session.getProvider();
                    logger.debug('[health_analysis] Provider from context.session.getProvider():', provider);
                }
            }

            // 여전히 유효한 provider가 없으면 기본값 사용
            if (!provider || provider === 'undefined' || provider === 'null') {
                provider = 'ollama';
                logger.debug('[health_analysis] Using default provider:', provider);
            }

            logger.debug('[health_analysis] 최종 Provider 확인:', {
                finalProvider: provider,
                providerType: typeof provider,
                hasSession: !!session,
                sessionId,
                contextKeys: context ? Object.keys(context) : []
            });

            // 세션이 없거나 provider가 여전히 문제인 경우 새로운 세션 생성
            if (!session) {
                const taskId = `health_analysis_${Date.now()}`;
                session = await sharedContext.createSession(taskId, provider, null, 'chat');
                logger.debug('[health_analysis] 새 세션 생성:', { taskId, provider });
            }

            if (provider === 'openai') {
                const { openaiChat } = await import('../aiFeatures.js');
                aiChatFunction = openaiChat;
            } else if (provider === 'anthropic') {
                const { anthropicChat } = await import('../aiFeatures.js');
                aiChatFunction = anthropicChat;
            } else if (provider === 'gemini') {
                const { geminiChat } = await import('../aiFeatures.js');
                aiChatFunction = geminiChat;
            } else if (provider === 'groq') {
                const { groqChat } = await import('../aiFeatures.js');
                aiChatFunction = groqChat;
            } else {
                const { ollamaChat } = await import('../aiFeatures.js');
                aiChatFunction = ollamaChat;
            }

            // AI 호출을 위한 메시지 배열 구성
            const messages = [
                { role: 'user', content: freeAnalysisPrompt }
            ];

            // 올바른 매개변수 순서로 AI 함수 호출
            const taskId = session.getId() || `health_analysis_${Date.now()}`;

            logger.debug('[health_analysis] AI 함수 호출 시작:', {
                provider,
                taskId,
                messageCount: messages.length,
                aiChatFunctionType: typeof aiChatFunction,
                aiChatFunctionName: aiChatFunction?.name || 'unknown'
            });

            // 세션 메소드들 디버그
            const sessionModel = session?.getModel ? session.getModel() : 'hamonize:latest';
            const sessionUserId = session?.getUserId ? session.getUserId() : context?.userId;
            const contextUsername = context?.username;

            logger.debug('[health_analysis] AI 함수 파라미터 준비:', {
                sessionModel,
                sessionUserId,
                contextUsername,
                sessionHasGetModel: !!(session?.getModel),
                sessionHasGetUserId: !!(session?.getUserId),
                contextHasUserId: !!(context?.userId),
                contextHasUsername: !!(context?.username)
            });

            const aiResult = await aiChatFunction(
                messages,                                    // messages array
                { temperature: 0.7 },                       // parameters object
                taskId,                                      // task ID
                null,                                        // images (null for text-only)
                sessionModel,                               // model name
                sessionUserId,                              // user ID
                contextUsername,                            // username
                session                                      // existing session
            );

            logger.debug('[health_analysis] AI 함수 호출 결과:', {
                hasResult: !!aiResult,
                resultType: typeof aiResult,
                hasContent: !!(aiResult && aiResult.content),
                contentLength: aiResult && aiResult.content ? aiResult.content.length : 0,
                resultKeys: aiResult ? Object.keys(aiResult) : [],
                resultPreview: aiResult ? JSON.stringify(aiResult).substring(0, 100) + '...' : 'null'
            });

            if (aiResult && aiResult.content && !aiResult.isErrorResponse) {
                aiAnalysis = aiResult.content;
                logger.debug('[health_analysis] AI 자유 분석 완료');
            } else {
                logger.error('[health_analysis] AI 결과가 예상과 다름:', {
                    aiResult,
                    hasResult: !!aiResult,
                    hasContent: !!(aiResult && aiResult.content),
                    isErrorResponse: aiResult?.isErrorResponse,
                    provider,
                    taskId
                });
                throw new Error(`AI 자유 분석 실패: ${aiResult?.isErrorResponse ? 'AI 서비스 오류' : (aiResult ? '내용 없음' : '결과 없음')}`);
            }
            
        } catch (aiError) {
            logger.error('[health_analysis] AI 분석 오류 발생:', aiError);
            logger.error('[health_analysis] AI 분석 오류 스택:', aiError.stack);
            logger.error('[health_analysis] AI 분석 오류 메시지:', aiError.message);
            logger.error('[health_analysis] AI 분석 시도한 프로바이더:', provider || 'unknown');
            logger.error('[health_analysis] AI 분석 사용한 함수:', typeof aiChatFunction !== 'undefined' ? (aiChatFunction?.name || 'unknown') : 'unknown');
            logger.error('[health_analysis] AI 분석 세션:', session ? 'exists' : 'null');
            
            aiAnalysis = `${patientData.name} 환자의 건강검진 데이터 분석을 수행할 수 없습니다. (AI 분석 서비스 오류: ${aiError.message})`;
        }
        
        // 4단계: 표준 판정 계산 (검증 단계)
        const standardAssessments = calculateActualAssessments(patientData, HEALTH_EVALUATION_CRITERIA);
        
        // 5단계: AI 분석 + 표준 판정 결합
        const standardAssessmentText = `

=== 표준 판정 결과 ===

【기본정보】
성명: ${patientData.name}
성별: ${patientData.gender === 1 ? '남성' : '여성'} (${patientData.gender})
검진일: ${patientData.checkupDate || '미확인'}

【항목별 표준 판정】
①혈압해당: ${standardAssessments.bloodPressure} (${standardAssessments.bloodPressure === 1 ? '해당' : '비해당'})
  - 수축기혈압 평균: ${patientData.systolicBP?.average || '-'}mmHg (기준: 140mmHg 이상)
  - 확장기혈압 평균: ${patientData.diastolicBP?.average || '-'}mmHg (기준: 90mmHg 이상)

②지질해당: ${standardAssessments.lipid} (${standardAssessments.lipid === 1 ? '해당' : '비해당'})
  - 중성지방: ${patientData.lipids?.triglycerides || '-'}mg/dl (기준: 150mg/dl 이상)
  - HDL: ${patientData.lipids?.hdl || '-'}mg/dl (기준: 40mg/dl 미만)
  - LDL: ${patientData.lipids?.ldl || '-'}mg/dl (기준: 140mg/dl 이상)

③HbA1c수진유무: ${standardAssessments.hba1cTested} (${standardAssessments.hba1cTested === 3 ? '수진' : '미수진'})

④혈당해당: ${standardAssessments.bloodSugar} (${standardAssessments.bloodSugar === 1 ? '해당' : '비해당'})
  - 관련 수치: 일반혈당 ${patientData.bloodSugar?.general || '-'}, 공복시혈당 ${patientData.bloodSugar?.fasting || '-'}, HbA1c ${patientData.bloodSugar?.hba1c || '-'}%

⑤비만해당: ${standardAssessments.obesity} (${standardAssessments.obesity === 1 ? '해당' : '비해당'})
  - BMI: ${patientData.obesity?.bmi || '-'}kg/m² (기준: 25kg/m² 이상)
  - 복위: ${patientData.obesity?.waistCircumference || '-'}cm (남성 기준: 85cm 이상, 여성 기준: 90cm 이상)

⑥요단백: ${standardAssessments.urineProtein} (변환값)

【최종 표준 판정】
⑦판정: ${standardAssessments.finalRisk} (${
    standardAssessments.finalRisk === 1 ? '고위험군' : 
    standardAssessments.finalRisk === 2 ? '중위험군' : 
    standardAssessments.finalRisk === 3 ? '저위험군' : 
    '대상외'
})
- 4개 항목 중 ${[standardAssessments.bloodPressure, standardAssessments.lipid, standardAssessments.bloodSugar, standardAssessments.obesity].filter(v => v === 1).length}개 항목 해당`;

        // 참조 문서 정보 추가 - 실제 사용된 문서명 표시
        const referencedDocuments = [];
        if (patientData.dataSource === 'RAG Documents') {
            // RAG에서 검색된 경우 실제 문서 파일명을 추가
            logger.debug('[DEBUG] patientData.filename:', patientData.filename);
            logger.debug('[DEBUG] patientData.documentSource:', patientData.documentSource);
            logger.debug('[DEBUG] patientData.metadata:', patientData.metadata);
            
            if (patientData.filename && patientData.filename !== 'RAG Documents') {
                referencedDocuments.push(patientData.filename);
            } else if (patientData.metadata && patientData.metadata.filename) {
                referencedDocuments.push(patientData.metadata.filename);
            } else if (patientData.documentSource) {
                // documentSource에서 실제 파일명 추출
                const filename = patientData.documentSource.match(/\(([^)]+)\)$/);
                if (filename && filename[1] !== 'unknown') {
                    referencedDocuments.push(filename[1]);
                } else {
                    referencedDocuments.push('RAG Documents');
                }
            } else {
                referencedDocuments.push('RAG Documents');
            }
        } else if (patientData.dataSource === 'PostgreSQL Database') {
            // 데이터베이스의 경우 원본 Excel 파일명 표시 (자동 임포트된 경우)
            const sourceFile = patientData.sourceFile || '건강검진데이터.xlsx';
            referencedDocuments.push(sourceFile);
        }
        
        // AI 응답에 참조 정보 포함
        const citationText = referencedDocuments.length > 0 
            ? `\n\n{"cited_documents": [${referencedDocuments.map(doc => `"${doc}"`).join(', ')}]}`
            : '';
        
        // 최종 응답: AI 분석 + 표준 판정 + 참조 정보
        const aiResponse = aiAnalysis + standardAssessmentText + citationText;

        const analysisResult = {
            patientName: patientData.name,
            analysisType: '건강검진 종합분석',
            dataSource: patientData.dataSource,
            relevanceScore: patientData.relevanceScore,
            timestamp: new Date().toISOString(),
            freeAnalysisPrompt: freeAnalysisPrompt,
            aiAnalysis: aiAnalysis,
            aiResponse: aiResponse,
            rawContent: content,
            standardAssessments: standardAssessments,
            status: 'analysis_complete',
            message: `${patientData.name} 환자의 건강검진 분석이 완료되었습니다. (새로운 워크플로우 적용)`
        };
        
        return analysisResult;
        
    } catch (error) {
        logger.debug('[performHealthAnalysis] 분석 오류:', error.message);
        throw new Error(`건강 분석 실행 중 오류: ${error.message}`);
    }
}

// 건강검진 분석 도구
const healthAnalysisTool = {
    name: 'health_analysis',
    category: 'healthcare',
    description: '특정 환자의 건강검진 분석을 요청받았을 때 사용하는 도구입니다. 환자 이름이 포함된 건강검진 분석, 건강 상태 확인, 검진 결과 판정 요청 시 자동으로 호출됩니다. 예: "김철수 환자 건강검진 분석", "前田実 검진 결과 분석" 등',
    version: '2.0.0',
    parameters: {
        request: { 
            type: 'string', 
            required: true, 
            description: '사용자의 건강검진 분석 요청 전문 (환자명과 요청사항 포함)' 
        },
        username: { 
            type: 'string', 
            required: false, 
            default: 'admin',
            description: '사용자 이름 (RAG 검색용)' 
        },
        validate_response: {
            type: 'boolean',
            default: true,
            description: '3번 시트 평가기준 검증 수행 여부'
        }
    },
    handler: async (args, context) => {
            logger.debug('===== HANDLER ENTRY POINT =====');
            logger.debug('health_analysis 도구 핸들러 진입!');
            logger.debug('받은 인자:', JSON.stringify(args));
            logger.debug('컨텍스트 키들:', Object.keys(context || {}));
            logger.debug('===============================');

            try {
                // 파라미터 유효성 검증
                if (!args || !args.request || typeof args.request !== 'string') {
                    logger.debug('[DEBUG] 파라미터 검증 실패 - request가 없거나 잘못됨');
                    return {
                        success: false,
                        error: '환자 분석 요청이 제공되지 않았습니다.',
                        message: '요청 내용을 확인해주세요.'
                    };
                }

                const userRequest = args.request;
                const username = args.username || context?.username || 'admin';
                
                logger.debug('[DEBUG] 파라미터 검증 통과');
                logger.debug('[DEBUG] userRequest:', userRequest);
                logger.debug('[DEBUG] username:', username);

                // 환자 데이터 파싱 - 데이터베이스 우선 시도
                logger.debug('[DEBUG] 환자 데이터 파싱 시작...');
                
                // 1. PostgreSQL 데이터베이스에서 먼저 조회 시도
                logger.debug('[DEBUG] getPatientDataFromDatabase 호출 시작');
                let patientData;
                try {
                    patientData = await getPatientDataFromDatabase(userRequest);
                    logger.debug('[DEBUG] getPatientDataFromDatabase 호출 완료:', patientData ? '성공' : '실패');
                } catch (dbError) {
                    logger.debug('[DEBUG] getPatientDataFromDatabase 호출 중 예외 발생:', dbError.message);
                    logger.debug('[DEBUG] getPatientDataFromDatabase 스택 트레이스:', dbError.stack);
                    patientData = null;
                }

                // 2. 데이터베이스에서 찾지 못한 경우에만 RAG 검색 실행
                let ragResults = null;
                if (!patientData) {
                    logger.debug('[DEBUG] 데이터베이스에서 환자 데이터 없음 - RAG 검색 시작...');
                    
                    // RAG 서비스 동적 로드
                    logger.debug('[DEBUG] RAGService 동적 로드 시작...');
                    try {
                        const { RAGService } = await import('../plugins/rag/ragService.js');
                        await RAGService.checkIfEnabled(true);
                        logger.debug('[DEBUG] RAGService 로드 완료');

                        ragResults = await RAGService.searchDocuments(userRequest, username, null, 'personal');
                        logger.debug('[DEBUG] RAG 검색 완료');
                        
                        if (ragResults && ragResults.documents && ragResults.documents[0] && ragResults.documents[0].length > 0) {
                            logger.debug('[DEBUG] RAG 검색 결과 발견:', ragResults.documents[0].length + '개 문서');
                        } else {
                            logger.debug('[DEBUG] RAG 검색 결과 없음');
                            ragResults = null;
                        }
                    } catch (ragError) {
                        logger.debug('[DEBUG] RAG 검색 오류:', ragError.message);
                        ragResults = null;
                    }
                }
                
                // 1.5. 데이터베이스에서 찾지 못한 경우 자동 임포트 시도
                if (!patientData) {
                    logger.debug('[DEBUG] 환자 데이터 자동 임포트 시도...');
                    const importResult = await attemptPatientDataImport(null, userRequest, username);
                    if (importResult.success) {
                        logger.debug('[DEBUG] 환자 데이터 자동 임포트 성공 - 다시 데이터베이스 조회');
                        // 임포트 성공 시 다시 데이터베이스에서 조회
                        try {
                            patientData = await getPatientDataFromDatabase(userRequest);
                            logger.debug('[DEBUG] 자동 임포트 후 데이터베이스 재조회:', patientData ? '성공' : '실패');
                        } catch (retryError) {
                            logger.debug('[DEBUG] 자동 임포트 후 데이터베이스 재조회 실패:', retryError.message);
                        }
                    }
                }
                
                // 3. RAG fallback 시도 (RAG 결과가 있는 경우에만)
                if (!patientData && ragResults) {
                    logger.debug('[DEBUG] RAG fallback 시도...');
                    patientData = await parsePatientDataFromRag(ragResults, userRequest);
                    logger.debug('[DEBUG] parsePatientDataFromRag 완료:', patientData ? '성공' : '실패');
                }
                
                logger.debug('[DEBUG] 최종 patientData:', patientData ? `출처: ${patientData.dataSource}` : 'null');

                if (!patientData) {
                    logger.debug('[DEBUG] 환자 데이터 파싱 실패 - 데이터베이스와 RAG 모두에서 데이터를 찾을 수 없음');
                    
                    // 환자명 추출 시도
                    const patientNameMatch = userRequest.match(/([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*환자|([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*의\s*건강|([가-힣\u4e00-\u9fff\u3000\s]{2,15})\s*건강검진/);
                    const extractedName = patientNameMatch ? (patientNameMatch[1] || patientNameMatch[2] || patientNameMatch[3])?.trim() : '요청하신 환자';
                    
                    return {
                        success: true,
                        data: {
                            analysis: `죄송합니다. ${extractedName} 환자의 건강검진 데이터를 찾을 수 없습니다.\n\n현재 시스템에 등록된 환자 데이터가 없거나, 요청하신 환자명이 정확하지 않을 수 있습니다.\n\n**해결 방법:**\n1. 📁 환자 데이터 파일(Excel)을 업로드해주세요\n2. ✏️ 환자명을 정확히 입력했는지 확인해주세요\n3. 🔄 다른 환자명으로 다시 시도해보세요\n\n환자 데이터를 업로드하신 후 다시 건강검진 분석을 요청해주시면 정확한 분석 결과를 제공해드리겠습니다.`,
                            message: `죄송합니다. ${extractedName} 환자의 건강검진 데이터를 찾을 수 없습니다.`,
                            patientName: extractedName,
                            dataStatus: 'not_found'
                        }
                    };
                }

                // 강화된 데이터 검증 - RAG와 Database 모든 경우에 적용
                const content = patientData.content || patientData.rawContent || '';
                logger.debug('[DEBUG] 강화된 데이터 검증 시작');
                logger.debug('[DEBUG] 데이터 소스:', patientData.dataSource);
                logger.debug('[DEBUG] 환자명:', patientData.name);
                logger.debug('[DEBUG] 콘텐츠 길이:', content.length);
                logger.debug('[DEBUG] 콘텐츠 내용:', content.substring(0, 200));
                
                // RAG 데이터 검증
                if (patientData.dataSource === 'RAG Documents') {
                    // RAG 결과에서 "데이터를 찾을 수 없습니다" 또는 유사한 메시지 확인
                    const noDataIndicators = [
                        '데이터를 찾을 수 없습니다',
                        '찾을 수 없습니다',
                        '검색 결과가 없습니다',
                        '데이터가 없습니다',
                        '정보가 없습니다',
                        'no data found',
                        'not found'
                    ];
                    
                    const hasNoDataIndicator = noDataIndicators.some(indicator => 
                        content.toLowerCase().includes(indicator.toLowerCase())
                    );
                    
                    if (!content || content.trim() === '' || hasNoDataIndicator || content.length < 100) {
                        logger.debug('[DEBUG] RAG 데이터 부족으로 분석 불가 - 즉시 반환');
                        const patientName = patientData.name || '요청하신 환자';
                        return {
                            success: true,
                            data: {
                                analysis: `죄송합니다. ${patientName}의 건강검진 데이터를 찾을 수 없습니다.\n\n문서에서 해당 환자의 건강검진 정보를 찾지 못했습니다.\n\n**해결 방법:**\n1. 📁 환자 데이터 파일(Excel)을 업로드해주세요\n2. ✏️ 환자명을 정확히 입력했는지 확인해주세요\n3. 📋 업로드된 문서에 해당 환자 정보가 포함되어 있는지 확인해주세요\n\n정확한 환자 데이터가 준비되면 건강검진 분석을 다시 진행해드리겠습니다.`,
                                message: `${patientName}의 건강검진 데이터를 찾을 수 없습니다.`,
                                patientName: patientName,
                                dataStatus: 'not_found_in_documents'
                            }
                        };
                    }
                }
                
                // Database 데이터 검증 - 필수 필드 확인
                if (patientData.dataSource === 'PostgreSQL Database') {
                    const hasEssentialData = (
                        patientData.systolicBP?.average || 
                        patientData.diastolicBP?.average ||
                        patientData.lipids?.triglycerides ||
                        patientData.lipids?.hdl ||
                        patientData.bloodSugar?.fasting ||
                        patientData.obesity?.bmi
                    );
                    
                    if (!hasEssentialData) {
                        logger.debug('[DEBUG] Database 데이터에 필수 건강 지표 없음 - 분석 불가');
                        return {
                            success: false,
                            error: '환자 데이터가 불충분합니다',
                            message: `${patientData.name} 환자의 건강검진 데이터가 불충분합니다. 혈압, 혈중지질, 혈당, BMI 등의 기본 검진 데이터가 필요합니다.`,
                            aiResponse: `죄송합니다. ${patientData.name} 환자의 건강검진 데이터가 불충분합니다.\n\n데이터베이스에서 환자 정보는 찾았지만, 건강 분석에 필요한 기본 검진 데이터가 부족합니다.\n\n**필요한 검진 데이터:**\n- 🩺 혈압 (수축기/이완기)\n- 🔬 혈중지질 (중성지방, HDL, LDL)\n- 🍭 혈당 (공복혈당, HbA1c)\n- ⚖️ BMI (체질량지수)\n\n더 완전한 건강검진 데이터 파일을 업로드해주시면 정확한 분석을 제공해드리겠습니다.`
                        };
                    }
                }

                // 건강 분석 실행 (검증 로직 포함)
                logger.debug('[DEBUG] 건강 분석 실행 중...');
                const validateResponse = args.validate_response !== false; // 기본값 true
                let analysisResult;
                let attempt = 1;
                const maxAttempts = 3;
                
                while (attempt <= maxAttempts) {
                    logger.debug(`[DEBUG] 건강 분석 시도 ${attempt}/${maxAttempts}`);
                    
                    analysisResult = await performHealthAnalysis(patientData, context, attempt);
                    
                    if (validateResponse && analysisResult.aiResponse) {
                        // AI 응답 검증
                        const criteria = {
                            bloodPressure: { systolic: 140, diastolic: 90 },
                            lipid: { triglycerides: 150, hdl: 40, ldl: 140 },
                            bloodSugar: { general: 100, fasting: 100, hba1c: 5.6 },
                            obesity: { bmi: 25, waist: { male: 85, female: 90 } },
                            urineProtein: { codes: { '－': 1, '±': 3, '＋': 5, '＋＋': 7, '＋＋＋': 9 } }
                        };
                        
                        const validation = validateHealthAssessment(analysisResult.aiResponse, patientData, criteria);
                        
                        if (validation.isValid) {
                            logger.debug(`[DEBUG] 검증 성공: 시도 ${attempt}에서 올바른 판정`);
                            analysisResult.validation = {
                                success: true,
                                attempt: attempt,
                                message: '판정 기준에 따른 검증 통과'
                            };
                            break;
                        } else {
                            logger.debug(`[DEBUG] 검증 실패: 시도 ${attempt} - 오류: ${validation.errors.join(', ')}`);
                            analysisResult.validation = {
                                success: false,
                                attempt: attempt,
                                errors: validation.errors,
                                actual: validation.actual,
                                ai: validation.ai
                            };
                            
                            if (attempt === maxAttempts) {
                                logger.debug('[DEBUG] 최대 시도 횟수 도달 - 마지막 결과 반환');
                                break;
                            }
                            
                            attempt++;
                            continue;
                        }
                    } else {
                        // 검증하지 않는 경우 첫 번째 시도 결과 사용
                        break;
                    }
                }
                
                logger.debug('[DEBUG] 건강 분석 완료');

                return {
                    success: true,
                    data: analysisResult,
                    patientName: patientData.name,
                    message: `건강검진 분석이 완료되었습니다.${validateResponse ? ` (검증: ${analysisResult.validation?.success ? '성공' : '실패'})` : ''}`
                };

            } catch (error) {
                logger.debug('[Health Checkup] 처리 오류:', error);
                logger.debug('[DEBUG] 오류 메시지:', error.message);
                logger.debug('[DEBUG] 스택 트레이스:', error.stack);
                return {
                    success: false,
                    error: '건강검진 분석 중 오류가 발생했습니다',
                    message: error.message || '알 수 없는 오류'
                };
            }
        }
};

// 품질 검증 도구
const qualityCheckTool = {
    name: 'quality_check',
    category: 'validation',
    description: '응답의 품질을 검증합니다.',
    parameters: {
        response: { type: 'string', required: true, description: '검증할 응답' },
        original_query: { type: 'string', required: true, description: '원본 질문' },
        quality_issues: { type: 'array', default: [], description: '확인할 품질 이슈들' }
    },
    handler: async (args, context) => {
        const { response, original_query, quality_issues = [] } = args;
        
        try {
            const { evaluateResponseQuality } = await import('../mainLogic.js');
            const qualityScore = await evaluateResponseQuality(response, original_query, quality_issues);
            
            return {
                success: true,
                data: {
                    quality_score: qualityScore,
                    response: response,
                    original_query: original_query,
                    issues_checked: quality_issues
                },
                message: `품질 검증 완료: ${qualityScore}/10점`
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: `품질 검증 실패: ${error.message}`
            };
        }
    }
};

// ContextAI 메모리 검색 도구
const searchMemoryTool = {
    name: 'search_memory',
    category: 'memory',
    description: '사용자의 기억된 정보(선호도, 기술 스택, 프로젝트 정보 등)를 검색합니다. 개인화된 답변을 위해 필요한 컨텍스트 정보를 조회할 때 사용하세요.',
    parameters: {
        query: { type: 'string', required: false, description: '검색할 메모리 내용 (선택사항)' },
        type: { type: 'string', required: false, description: '메모리 타입 (entity, preference, skill, project, fact)' },
        category: { type: 'string', required: false, description: '특정 카테고리 (언어, 프레임워크, 데이터베이스, IDE 등)' }
    },
    handler: async (args, context) => {
        const { userId } = context;

        try {
            logger.debug('[SearchMemory] 🔍 사용자 메모리 검색 시작', { userId, args });

            // API 키 가져오기
            const dbManager = (await import('../services/database/index.js')).default;
            const apiKeyQuery = 'SELECT key_value FROM api_keys WHERE user_id = $1 AND status = $2 LIMIT 1';
            const apiKeyResult = await dbManager.query(apiKeyQuery, [userId, 'active']);

            if (!apiKeyResult.rows.length) {
                throw new Error('사용자 API 키를 찾을 수 없습니다');
            }

            const apiKey = apiKeyResult.rows[0].key_value;

            // ContextAI 메모리 API 호출
            const axios = (await import('axios')).default;
            const API_BASE = process.env.API_BASE_URL || `${URLs.API_SERVER()}/api/v1`;

            let url = `${API_BASE}/context/memories`;
            const params = new URLSearchParams();

            if (args.type) params.append('type', args.type);
            if (args.query) params.append('search', args.query);

            if (params.toString()) {
                url += '?' + params.toString();
            }

            const response = await axios.get(url, {
                headers: {
                    'X-API-Key': apiKey,
                    'Content-Type': 'application/json'
                }
            });

            const memories = response.data.data.memories || [];
            const total = response.data.data.total || 0;

            // 결과 포맷팅
            let result = `📋 저장된 메모리 정보 (총 ${total}개):\n\n`;

            if (memories.length === 0) {
                result += '저장된 정보가 없습니다.';
            } else {
                // 타입별로 그룹화
                const groupedMemories = {};
                memories.forEach(memory => {
                    const type = memory.type || 'other';
                    if (!groupedMemories[type]) groupedMemories[type] = [];
                    groupedMemories[type].push(memory);
                });

                // 각 타입별로 출력
                for (const [type, items] of Object.entries(groupedMemories)) {
                    const typeEmoji = {
                        'preference': '⚙️',
                        'entity': '🏷️',
                        'skill': '💡',
                        'project': '📁',
                        'fact': '📝'
                    }[type] || '📌';

                    result += `${typeEmoji} **${type.toUpperCase()}**:\n`;
                    items.forEach(item => {
                        result += `  • ${item.content}`;
                        if (item.summary && item.summary !== item.content) {
                            result += ` (${item.summary})`;
                        }
                        result += '\n';
                    });
                    result += '\n';
                }
            }

            logger.debug('[SearchMemory] ✅ 메모리 검색 완료', {
                userId,
                foundMemories: memories.length,
                totalMemories: total
            });

            return {
                success: true,
                data: {
                    memories,
                    total,
                    formatted_result: result
                },
                message: `${total}개의 메모리 정보를 찾았습니다`
            };

        } catch (error) {
            logger.error('[SearchMemory] ❌ 메모리 검색 실패:', error);
            
            // 메모리 검색 실패 시 부분적 성공으로 처리하여 AI가 계속 동작할 수 있도록 함
            return {
                success: true, // 부분적 성공으로 처리
                data: {
                    memories: [],
                    total: 0,
                    formatted_result: '저장된 메모리 정보를 찾을 수 없습니다. 일반적인 답변을 제공하겠습니다.'
                },
                message: `메모리 검색 실패 (${error.message}), 하지만 일반 답변 가능`
            };
        }
    }
};

// ContextAI 메모리 저장 도구
const saveMemoryTool = {
    name: 'save_memory',
    category: 'memory',
    description: '중요한 정보를 사용자 메모리에 저장합니다. 사용자가 언급한 선호도, 기술 스택, 프로젝트 정보 등을 기억해두고 싶을 때 사용하세요. 사용자가 특정 타입을 명시적으로 요청하면 반드시 해당 타입을 사용하세요.',
    parameters: {
        content: { type: 'string', required: true, description: '저장할 내용' },
        type: { type: 'string', required: true, description: '메모리 타입: entity(개체명), preference(선호도), skill(기술/능력), project(프로젝트), fact(사실정보), pattern(패턴/규칙), relationship(관계정보). 사용자가 특정 타입을 명시하면 반드시 해당 타입 사용.' },
        summary: { type: 'string', required: false, description: '요약 정보 (선택사항)' },
        category: { type: 'string', required: false, description: '카테고리 (언어, 프레임워크, 데이터베이스 등)' }
    },
    handler: async (args, context) => {
        const { userId } = context;
        const { content, type, summary, category } = args;

        try {
            logger.debug('[SaveMemory] 💾 사용자 메모리 저장 시작', { userId, content, type });

            // 데이터베이스에 실제 저장
            const dbManager = (await import('../services/database/index.js')).default;

            const query = `
                INSERT INTO context_memories (
                    user_id, memory_type, content, summary, entities, keywords,
                    importance_score, confidence_score, source_context
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
                ON CONFLICT (user_id, memory_type, content)
                DO UPDATE SET
                    summary = EXCLUDED.summary,
                    updated_at = CURRENT_TIMESTAMP,
                    reference_count = context_memories.reference_count + 1
                RETURNING id, created_at, updated_at
            `;

            const values = [
                String(userId), // user_id를 문자열로 변환
                type,
                content,
                summary || `${type}: ${content}`,
                JSON.stringify(category ? [{ type: 'category', value: category }] : []),
                category ? [category] : [],
                0.7, // importance_score
                0.8, // confidence_score
                `도구를 통한 직접 저장`
            ];

            const result = await dbManager.query(query, values);
            const savedMemory = result.rows[0];

            logger.debug('[SaveMemory] ✅ 데이터베이스에 메모리 저장 완료', {
                userId,
                memoryId: savedMemory?.id,
                content,
                type
            });

            const responseData = {
                content,
                type,
                summary: summary || `${type}: ${content}`,
                category,
                saved_at: savedMemory?.created_at || new Date().toISOString(),
                memory_id: savedMemory?.id
            };

            return {
                success: true,
                data: responseData,
                message: `정보가 메모리에 저장되었습니다: ${content}`
            };

        } catch (error) {
            logger.error('[SaveMemory] ❌ 메모리 저장 실패:', error);

            // 메모리 저장 실패 시에도 부분적 성공으로 처리하여 AI가 계속 동작할 수 있도록 함
            return {
                success: true, // 부분적 성공으로 처리
                data: {
                    content,
                    type: type || 'unknown',
                    summary: summary || `저장 실패: ${content}`,
                    category: category || 'general',
                    saved_at: new Date().toISOString(),
                    error: error.message
                },
                message: `메모리 저장 실패 (${error.message}), 하지만 정보는 기억하겠습니다`
            };
        }
    }
};

// 문서 생성 도구
const createReportTool = {
    name: 'create_report',
    category: 'document',
    description: '사용자 요청에 맞는 문서를 생성합니다. 문서 타입 선택 기준: **default**(메모, 보고서, 기록, 일반문서), petition(민원신청, 허가신청), official(공문, 발송문, 통보), license(허가신청서). **중요**: 사용자가 "메모", "보고서", "기록", "문서" 등을 요청하면 반드시 default 타입을 선택하세요. PDF, DOCX, HWPX 형식 지원. **응답 규칙**: 문서 생성 후 간단히 "문서가 생성되었습니다" 또는 "완료되었습니다"라고만 답변하세요. 다운로드는 하단의 파일 섹션에서 자동 제공됩니다.',
    parameters: {
        document_type: {
            type: 'string',
            required: true,
            description: '요청 분석 후 선택: default(메모/보고서/기록/일반문서-기본값), petition(민원신청), official(공문), license(허가신청서). "메모", "보고서", "기록", "문서"라는 단어가 있으면 반드시 default 선택!'
        },
        format: {
            type: 'string',
            required: false,
            default: 'pdf',
            description: '문서 형식 (pdf, docx, hwpx)'
        },
        applicant: {
            type: 'object',
            required: true,
            properties: {
                name: { type: 'string', required: true, description: '신청자 이름' },
                address: { type: 'string', required: true, description: '신청자 주소' },
                phone: { type: 'string', required: true, description: '연락처' },
                email: { type: 'string', required: false, description: '이메일 주소' }
            },
            description: '신청자 정보'
        },
        application: {
            type: 'object',
            required: true,
            properties: {
                title: { type: 'string', required: true, description: '신청 제목' },
                purpose: { type: 'string', required: true, description: '신청 목적' },
                target: { type: 'string', required: true, description: '신청 대상' },
                period: { type: 'string', required: false, description: '신청 기간' },
                details: { type: 'string', required: true, description: '상세 내용' }
            },
            description: '신청 내용'
        },
        attachments: {
            type: 'array',
            required: false,
            items: { type: 'string' },
            description: '첨부서류 목록'
        }
    },
    handler: async (args, context) => {
        const { userId, job, taskId, sessionId, username, updateProgress } = context;

        console.log(`🔍 [CreateReport] AI가 전달한 매개변수: ${JSON.stringify(args)}`);
        logger.debug(`[CreateReport] 문서 생성 시작: ${JSON.stringify(args)}`);
        logger.debug(`[CreateReport] Context 정보:`, context);

        // 자연어 내용에서 필드 자동 추출 함수
        const extractInfoFromText = (text) => {
            const result = {
                applicant: {},
                application: {},
                document_type: 'default', // 기본값을 default로 변경
                format: 'pdf'
            };

            // 텍스트에서 문서 유형 감지
            if (text.includes('민원') || text.includes('신청서') || text.includes('허가')) {
                result.document_type = 'petition';
            } else if (text.includes('공문') || text.includes('공식문서') || text.includes('발송') || text.includes('통보')) {
                result.document_type = 'official';
            } else if (text.includes('허가신청') || text.includes('라이선스')) {
                result.document_type = 'license';
            } else if (text.includes('기본문서') || text.includes('default') ||
                      text.includes('메모') || text.includes('보고서') || text.includes('일반문서')) {
                result.document_type = 'default';
            }
            // 그 외의 경우 기본값 'default' 유지

            // 이름 추출 (홍길동, 김철수 등)
            const nameMatch = text.match(/(신청자[:\s]*)?([\u4e00-\u9fff\uac00-\ud7ff]{2,4})/i);
            if (nameMatch) result.applicant.name = nameMatch[2];

            // 주소 추출
            const addressMatch = text.match(/(주소[:\s]*)?([\uc11c울시\ubd80산\ub300구\uc778천\uad11주\ub300전\uc6b8산\uc138종][^,\n]{5,50})/i);
            if (addressMatch) result.applicant.address = addressMatch[2];

            // 전화번호 추출
            const phoneMatch = text.match(/(\d{2,3}[-\s]?\d{3,4}[-\s]?\d{4})/i);
            if (phoneMatch) result.applicant.phone = phoneMatch[1];

            // 신청 내용 추출
            const titleMatches = [
                /(음식점.{0,10}허가)/i,
                /(사업.{0,10}신청)/i,
                /(허가.{0,10}신청)/i,
                /(민원.{0,10}신청)/i
            ];

            for (const pattern of titleMatches) {
                const match = text.match(pattern);
                if (match) {
                    result.application.title = match[1];
                    break;
                }
            }

            if (!result.application.title) {
                result.application.title = '민원 신청';
            }

            // 목적 추출
            const purposeMatches = [
                /(위생업소.{0,20}영업.{0,10}허가)/i,
                /(개업.{0,20}허가)/i,
                /(영업.{0,10}허가)/i
            ];

            for (const pattern of purposeMatches) {
                const match = text.match(pattern);
                if (match) {
                    result.application.purpose = match[1];
                    break;
                }
            }

            if (!result.application.purpose) {
                result.application.purpose = '허가 신청';
            }

            // 대상 추출
            const targetMatches = [
                /(일반음식점)/i,
                /(한식당)/i,
                /(음식점)/i,
                /(사업체)/i
            ];

            for (const pattern of targetMatches) {
                const match = text.match(pattern);
                if (match) {
                    result.application.target = match[1];
                    break;
                }
            }

            if (!result.application.target) {
                result.application.target = '일반 사업';
            }

            // 상세 내용
            result.application.details = text.length > 100 ? text.substring(0, 200) + '...' : text;

            return result;
        };

        // 전달된 인수를 처리하여 적절한 형태로 변환
        let processedArgs = { ...args };

        // document_type이 default인 경우 별도 처리 (사용자 지정 제목/내용을 그대로 사용)
        if (processedArgs.document_type === 'default') {
            logger.debug('[CreateReport] 기본 문서 타입 감지 - 사용자 지정 내용으로 자유로운 문서 생성');

            // 기본 문서의 경우 최소한의 정보만 사용
            processedArgs.applicant = processedArgs.applicant || {
                name: '홍길동',
                address: '서울특별시',
                phone: '010-0000-0000'
            };

            // 사용자가 지정한 title과 content를 직접 사용 (양식 없이 자유로운 문서)
            processedArgs.application = {
                title: processedArgs.application?.title || processedArgs.title || '문서',
                details: processedArgs.application?.content || processedArgs.content || '내용을 입력해주세요.',
                purpose: processedArgs.application?.content || processedArgs.content || '문서 작성',
                target: '자유 양식 문서'
            };

            logger.debug('[CreateReport] 기본 문서 처리 완료:', {
                title: processedArgs.application.title,
                content: processedArgs.application.details
            });

        } else {
            // AI가 전달한 데이터 구조 정규화 (default 타입이 아닌 경우만)
            if (processedArgs.application) {
                // application.content가 있는 경우 적절한 필드로 매핑
                if (processedArgs.application.content) {
                    const content = processedArgs.application.content;
                    processedArgs.application.title = processedArgs.application.title || (content.includes('허가') ? `${content} 신청` : '민원 신청');
                    processedArgs.application.purpose = processedArgs.application.purpose || content;
                    processedArgs.application.target = processedArgs.application.target || (content.includes('카페') ? '카페' :
                                                       content.includes('베이커리') ? '베이커리' :
                                                       content.includes('음식점') ? '음식점' : '일반 사업');
                    processedArgs.application.details = processedArgs.application.details || `${content}에 대한 신청입니다. 관련 법령 및 규정에 따라 신청하오니 심사 후 허가를 발급해 주시기 바랍니다.`;
                    delete processedArgs.application.content; // 원본 content 제거
                }

                // 빠진 필드에 기본값 설정
                processedArgs.application.title = processedArgs.application.title || '민원 신청';
                processedArgs.application.purpose = processedArgs.application.purpose || '허가 신청';
                processedArgs.application.target = processedArgs.application.target || '일반 사업';
                processedArgs.application.details = processedArgs.application.details || '민원 신청에 대한 상세 내용입니다.';
            }
        }

        logger.debug('[CreateReport] 데이터 정규화 후:', processedArgs);

        // 기존 로직: default가 아닌 다른 타입들(petition, official, license)에 대한 처리
        if (processedArgs.document_type !== 'default') {
            // 만약 인수가 여전히 불완전하거나 누락되었다면, context나 기본값으로 보완
            if (!processedArgs.applicant || !processedArgs.application ||
                !processedArgs.applicant.name || !processedArgs.applicant.address || !processedArgs.applicant.phone ||
                !processedArgs.application.title || !processedArgs.application.purpose || !processedArgs.application.target || !processedArgs.application.details) {

                logger.debug('[CreateReport] 불완전한 인수 감지, 자동 추출 시도');

            // context에서 원본 텍스트 찾기
            let contextText = '';
            if (context.originalPrompt) {
                contextText = context.originalPrompt;
            } else if (typeof context === 'string') {
                contextText = context;
            } else {
                // 기본적인 값들로 초기화
                contextText = `신청자: 홍길동, 주소: 서울시 강남구 테헤란로 123, 전화: 010-1234-5678, 신청내용: 음식점 개업 허가`;
            }

            // 자동 추출 시도
            const extracted = extractInfoFromText(contextText);
            logger.debug(`[CreateReport] 추출된 데이터:`, extracted);

            // 기존 인수와 병합
            processedArgs = {
                document_type: processedArgs.document_type || extracted.document_type,
                format: processedArgs.format || extracted.format,
                applicant: {
                    name: processedArgs.applicant?.name || extracted.applicant.name || '미입력',
                    address: processedArgs.applicant?.address || extracted.applicant.address || '미입력',
                    phone: processedArgs.applicant?.phone || extracted.applicant.phone || '미입력',
                    email: processedArgs.applicant?.email || extracted.applicant.email
                },
                application: {
                    title: processedArgs.application?.title || extracted.application.title || '미입력',
                    purpose: processedArgs.application?.purpose || extracted.application.purpose || '미입력',
                    target: processedArgs.application?.target || extracted.application.target || '미입력',
                    details: processedArgs.application?.details || extracted.application.details || '미입력',
                    period: processedArgs.application?.period || extracted.application.period
                },
                attachments: processedArgs.attachments || []
            };

                logger.debug('[CreateReport] 자동 추출 결과:', processedArgs);
            }
        }

        // 파라미터 검증 및 기본값 설정
        if (!processedArgs.document_type) {
            processedArgs.document_type = 'default';
        }

        // AI가 잘못된 타입을 보낸 경우 매핑
        const typeMapping = {
            'memo': 'default',
            '메모': 'default',
            'report': 'default',
            '보고서': 'default',
            'document': 'default',
            '문서': 'default',
            'record': 'default',
            '기록': 'default'
        };

        if (typeMapping[processedArgs.document_type]) {
            console.log(`📝 문서 타입 매핑: ${processedArgs.document_type} → ${typeMapping[processedArgs.document_type]}`);
            processedArgs.document_type = typeMapping[processedArgs.document_type];
        }

        // 신청자 정보 기본값 설정
        if (!processedArgs.applicant) {
            processedArgs.applicant = {};
        }
        processedArgs.applicant.name = processedArgs.applicant.name || '미입력';
        processedArgs.applicant.address = processedArgs.applicant.address || '미입력';
        processedArgs.applicant.phone = processedArgs.applicant.phone || '미입력';

        // 신청 내용 기본값 설정
        if (!processedArgs.application) {
            processedArgs.application = {};
        }
        processedArgs.application.title = processedArgs.application.title || '민원 신청';
        processedArgs.application.purpose = processedArgs.application.purpose || '허가 신청';
        processedArgs.application.target = processedArgs.application.target || '일반 사업';
        processedArgs.application.details = processedArgs.application.details || '민원 신청에 대한 상세 내용입니다.';

        // 기타 설정
        processedArgs.format = processedArgs.format || 'pdf';
        processedArgs.attachments = processedArgs.attachments || [];

        logger.debug('[CreateReport] 최종 처리된 데이터:', processedArgs);

        try {
            // 진행 상태 업데이트
            if (updateProgress) {
                await updateProgress('📄 문서 생성을 시작합니다...', 10);
            }

            // Python 스크립트를 통한 문서 생성
            const { spawn } = await import('child_process');

            // 문서 생성 데이터 준비
            const documentData = {
                document_type: processedArgs.document_type || 'default',
                format: processedArgs.format || 'pdf',
                applicant: processedArgs.applicant,
                application: processedArgs.application,
                attachments: processedArgs.attachments || [],
                created_at: new Date().toISOString(),
                session_id: sessionId,
                user_id: userId
            };

            if (updateProgress) {
                await updateProgress('🔨 문서를 생성하고 있습니다...', 60);
            }

            // 임시 파일 경로 생성
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
            const filename = `${processedArgs.document_type}_${timestamp}.${processedArgs.format || 'pdf'}`;
            const outputPath = `/tmp/${filename}`;

            // Python 스크립트 실행을 위한 Promise wrapper (가상환경 사용)
            const generateDocument = () => {
                return new Promise((resolve, reject) => {
                    // 가상환경에서 Python 스크립트 실행
                    const venvPath = path.join(os.homedir(), '.airun_venv', 'bin', 'python3');
                    const scriptPath = path.join(process.cwd(), 'generate_document.py');

                    const pythonProcess = spawn(venvPath, [
                        scriptPath,
                        JSON.stringify(documentData),
                        outputPath
                    ]);

                    let stdout = '';
                    let stderr = '';

                    pythonProcess.stdout.on('data', (data) => {
                        stdout += data.toString();
                    });

                    pythonProcess.stderr.on('data', (data) => {
                        stderr += data.toString();
                    });

                    pythonProcess.on('close', (code) => {
                        if (code === 0) {
                            resolve({ stdout, outputPath });
                        } else {
                            reject(new Error(`Python 스크립트 실행 실패 (code: ${code}): ${stderr}`));
                        }
                    });

                    pythonProcess.on('error', (error) => {
                        reject(new Error(`Python 프로세스 오류: ${error.message}`));
                    });
                });
            };

            // 문서 생성 실행
            const { stdout } = await generateDocument();

            if (updateProgress) {
                await updateProgress('✅ 문서 생성이 완료되었습니다!', 100);
            }

            // 성공 응답
            const enhancedMessage = `${processedArgs.document_type === 'petition' ? '민원신청서' :
                          processedArgs.document_type === 'business' ? '사업신청서' :
                          processedArgs.document_type === 'license' ? '허가신청서' :
                          processedArgs.document_type === 'official' ? '공문' :
                          processedArgs.document_type === 'default' ? '기본문서' : '문서'}가 성공적으로 생성되었습니다.`;

            const documentType = processedArgs.document_type === 'petition' ? '민원신청서' :
                               processedArgs.document_type === 'business' ? '사업신청서' :
                               processedArgs.document_type === 'license' ? '허가신청서' :
                               processedArgs.document_type === 'official' ? '공문' :
                               processedArgs.document_type === 'default' ? '기본문서' : '문서';

            return {
                success: true,
                data: {
                    message: `${documentType}가 성공적으로 생성되었습니다. 시스템이 자동으로 다운로드 버튼을 제공합니다.`,
                    document: {
                        filename: filename,
                        type: processedArgs.document_type,
                        applicant: processedArgs.applicant.name,
                        title: processedArgs.application.title,
                        created_at: documentData.created_at,
                        download_url: `${(process.env.NEXT_PUBLIC_API_SERVER_URL || URLs.API_SERVER()).replace(/\/api\/v1$/, '')}/api/v1/download/${filename}`
                    },
                    downloadInfo: [{
                        filename: filename,
                        url: `${(process.env.NEXT_PUBLIC_API_SERVER_URL || URLs.API_SERVER()).replace(/\/api\/v1$/, '')}/api/v1/download/${filename}`,
                        source: 'generated',
                        title: `${documentType} - ${processedArgs.applicant.name}`,
                        size: 0
                    }]
                },
                message: `${documentType}가 성공적으로 생성되었습니다. 파일명: ${filename}. 사용자가 다운로드할 수 있도록 다운로드 정보를 제공해주세요.`
            };

        } catch (error) {
            logger.error('[CreateReport] 문서 생성 실패:', error);

            if (updateProgress) {
                await updateProgress(`❌ 문서 생성 실패: ${error.message}`, 100);
            }

            return {
                success: false,
                error: `문서 생성 중 오류가 발생했습니다: ${error.message}`,
                data: {}
            };
        }
    }
};

const builtinTools = [
    webSearchTool,
    documentSearchTool,
    healthAnalysisTool,
    searchMemoryTool, // ContextAI 메모리 검색 도구 활성화
    saveMemoryTool,   // ContextAI 메모리 저장 도구 활성화
    createReportTool  // 문서생성 도구 활성화
];

export default builtinTools;
export {
    webSearchTool,
    documentSearchTool,
    healthAnalysisTool,
    searchMemoryTool, // ContextAI 메모리 검색 도구 활성화
    saveMemoryTool,   // ContextAI 메모리 저장 도구 활성화
    createReportTool  // 문서생성 도구 활성화
}; 