import { logger } from './logger.js';

/**
 * Bull 큐의 진행 상황을 업데이트하는 공통 헬퍼 함수
 * @param {Object} job - Bull 큐 작업 객체
 * @param {number} percent - 진행률 (0-100)
 * @param {string} message - 진행 상황 메시지
 * @param {string} [context] - 로그용 컨텍스트 정보
 */
export async function updateJobProgress(job, percent, message, context = '') {
    // logger.debug('DEBUG-PROGRESS', `updateJobProgress 호출됨:`, {
    //     hasJob: !!job,
    //     jobType: typeof job,
    //     jobId: job?.id,
    //     hasProgress: !!(job && job.progress),
    //     progressType: job ? typeof job.progress : 'no-job',
    //     message: message,
    //     context: context
    // });
    
    if (!job || !job.progress || typeof job.progress !== 'function') {
        logger.debug('DEBUG-PROGRESS', `job.progress 함수 없음 ${context ? `(${context})` : ''}`, {
            hasJob: !!job,
            hasProgress: !!(job && job.progress),
            progressType: job ? typeof job.progress : 'no-job',
            jobKeys: job ? Object.keys(job) : []
        });
        return;
    }

    try {
        logger.debug('DEBUG-PROGRESS', `진행 상황 업데이트 ${context ? `(${context})` : ''}: ${percent}% - ${message}`);
        
        // Bull 큐 진행률 업데이트
        await job.progress(percent);
        
        // 웹소켓 스트리밍에서 진행 상황 메시지 전송 (새로 추가)
        if (job.sendProgressMessage && typeof job.sendProgressMessage === 'function') {
            await job.sendProgressMessage(message, percent, context);
            logger.debug('DEBUG-PROGRESS', `웹소켓 스트리밍 메시지 전송 완료: ${message}`);
        }
        
        // 진행 상황 메시지를 job.progressMessages 배열에 추가 (우선 순위: 항상 저장)
        await addProgressMessage(job, message);
        
        // job.data.progressMessages의 최신 항목에 percent 값 설정
        if (job.data && Array.isArray(job.data.progressMessages) && job.data.progressMessages.length > 0) {
            const lastIndex = job.data.progressMessages.length - 1;
            job.data.progressMessages[lastIndex].percent = percent;
        }
        
        // Redis에 진행 상황 메시지 저장 (Redis 실패해도 job.progressMessages는 유지)
        try {
            // 공유 Redis 클라이언트 사용
            const { redisClient } = await import('../queues/index.js');
            
            const progressKey = `progress-messages:${job.id}:${Date.now()}`;
            const progressData = {
                message: message,
                percent: percent,
                timestamp: new Date().toISOString(),
                context: context
            };
            
            logger.debug('DEBUG-PROGRESS', `Redis 저장 시도 - 키: ${progressKey}, 데이터:`, progressData);
            
            await redisClient.lpush(progressKey, JSON.stringify(progressData));
            await redisClient.expire(progressKey, 3600); // 1시간 후 만료
            
            // 저장 확인
            const savedMessages = await redisClient.lrange(progressKey, 0, -1);
            logger.debug('DEBUG-PROGRESS', `Redis 저장 후 확인 - 총 ${savedMessages.length}개 메시지 저장됨`);
            
            // 공유 Redis 클라이언트이므로 연결을 끊지 않음
            
            logger.debug('DEBUG-PROGRESS', `Redis 저장 완료 ${context ? `(${context})` : ''}: ${message}`);
        } catch (redisError) {
            logger.debug('DEBUG-PROGRESS', `Redis 저장 실패하지만 job.progressMessages에는 저장됨 ${context ? `(${context})` : ''}:`, redisError.message);
        }
        
    } catch (error) {
        logger.debug('DEBUG-PROGRESS', `진행 상황 업데이트 실패 ${context ? `(${context})` : ''}:`, error);
    }
}

/**
 * 진행 상황 메시지를 job.progressMessages 배열에 추가하는 헬퍼 함수
 * @param {Object} job - Bull 큐 작업 객체
 * @param {string} message - 추가할 메시지
 */
export async function addProgressMessage(job, message) {
    if (!job) {
        logger.debug('DEBUG-PROGRESS', `job 객체가 없어서 메시지 추가 불가: ${message}`);
        return;
    }
    
    logger.debug('DEBUG-PROGRESS', `메시지 추가 시도:`, {
        jobId: job.id,
        message: message,
        hasProgressMessages: !!(job.progressMessages),
        isProgressMessagesArray: Array.isArray(job.progressMessages),
        currentLength: job.progressMessages ? job.progressMessages.length : 0
    });
    
    // 1. 메모리의 job.progressMessages에 추가 (기존 로직)
    if (!Array.isArray(job.progressMessages)) {
        job.progressMessages = [];
        logger.debug('DEBUG-PROGRESS', `job.progressMessages 배열 초기화됨 (jobId: ${job.id})`);
    }
    
    job.progressMessages.push(message);
    
    // 2. Redis에 지속되는 job.data.progressMessages에도 추가 (새로운 로직)
    if (!job.data) {
        job.data = {};
    }
    if (!Array.isArray(job.data.progressMessages)) {
        job.data.progressMessages = [];
    }
    
    job.data.progressMessages.push({
        message: message,
        timestamp: new Date().toISOString(),
        percent: null // updateJobProgress에서 설정될 예정
    });
    
    // 3. Redis에 job.data 변경사항 저장
    try {
        if (job.update && typeof job.update === 'function') {
            await job.update(job.data);
            logger.debug('DEBUG-PROGRESS', `Redis에 job.data 업데이트 완료 (jobId: ${job.id})`);
        } else {
            logger.debug('DEBUG-PROGRESS', `job.update 함수가 없음 - Redis 저장 실패 (jobId: ${job.id})`);
        }
    } catch (error) {
        logger.debug('DEBUG-PROGRESS', `Redis 저장 중 오류 (jobId: ${job.id}):`, error.message);
    }
    
    logger.debug('DEBUG-PROGRESS', `메시지 추가 완료:`, {
        jobId: job.id,
        message: message,
        memoryLength: job.progressMessages.length,
        dataLength: job.data.progressMessages.length,
        allMessages: job.progressMessages
    });
}

/**
 * 응답 객체에 진행 상황 정보를 포함시키는 헬퍼 함수
 * @param {Object} job - Bull 큐 작업 객체
 * @returns {Array|undefined} 진행 상황 메시지 배열 또는 undefined
 */
export function getProgressMessages(job) {
    return job && job.progressMessages ? job.progressMessages : undefined;
}

/**
 * Tools Registry에서 사용할 수 있는 간단한 진행 상황 업데이트 함수
 * @param {Object} job - Bull 큐 작업 객체 또는 스트리밍 job 객체
 * @param {string} message - 진행 상황 메시지
 * @param {number} percent - 진행률 (0-100)
 */
export async function updateToolProgress(job, message, percent = null) {
    if (!job) {
        console.log(`[DEBUG-TOOLS] job 객체가 없어서 진행 상황 업데이트 불가`);
        return;
    }

    try {
        // 웹소켓 스트리밍 모드인 경우 (job.sendProgressMessage가 있는 경우)
        if (job.sendProgressMessage && typeof job.sendProgressMessage === 'function') {
            console.log(`[DEBUG-TOOLS] 웹소켓 스트리밍 모드 진행 상황 업데이트: ${percent}% - ${message}`);
            
            // 웹소켓으로 진행 상황 전송
            await job.sendProgressMessage(message, percent, 'tool-execution');
            
            console.log(`[DEBUG-TOOLS] 웹소켓 진행 상황 전송 완료: ${percent}% - ${message}`);
            return;
        }
        
        // 기존 Bull 큐 모드인 경우
        if (job.progress && typeof job.progress === 'function' && percent !== null) {
            console.log(`[DEBUG-TOOLS] Bull 큐 모드 진행 상황 업데이트: ${percent}% - ${message}`);
            
            // Bull 큐 진행 상황 업데이트
            await job.progress(percent);
            
            // 공유 Redis 클라이언트 사용
            try {
                const { redisClient } = await import('../queues/index.js');
                
                const progressData = {
                    message: message,
                    percent: percent,
                    timestamp: new Date().toISOString(),
                    context: 'tool-execution'
                };
                
                const progressKey = `progress-messages:${job.id}`;
                await redisClient.lpush(progressKey, JSON.stringify(progressData));
                await redisClient.expire(progressKey, 3600); // 1시간 후 만료
                
                console.log(`[DEBUG-TOOLS] Redis 저장 완료: ${percent}% - ${message}`);
            } catch (redisError) {
                console.log(`[DEBUG-TOOLS] Redis 저장 실패:`, redisError.message);
            }
        } else {
            console.log(`[DEBUG-TOOLS] 진행 상황 업데이트 조건 불만족 - progress함수: ${!!(job.progress && typeof job.progress === 'function')}, sendProgressMessage: ${!!(job.sendProgressMessage && typeof job.sendProgressMessage === 'function')}, percent: ${percent}`);
        }
    } catch (error) {
        console.log(`[DEBUG-TOOLS] 진행 상황 업데이트 실패:`, error.message);
    }
} 