import apiClient, { ApiResponse } from './apiClient';

// 에이전트 옵션 인터페이스
export interface AgentOptions {
  agent_id?: string;
  user_id?: string;
  rag?: boolean;
  web?: boolean;
  temperature?: number;
  model?: string;
  messages?: any[];
  first?: boolean;
}

// 에이전트 응답 인터페이스
export interface AgentResponse {
  success: boolean;
  mode: string;
  code: string;
  output: string;
  error: string | null;
  timestamp: string;
}

// 에이전트 작업 상태 인터페이스
export interface AgentJobStatus {
  jobId: string;
  state: string;
  message: string;
  data?: AgentResponse;
  timestamp: string;
}

// 태스크 인터페이스
export interface Task {
  id: string;
  name: string;
  prompt: string;
  code: string | null;
  createdAt: string;
  lastUsed: string;
  executionCount: number;
  metadata: {
    lastExecutionTime?: string;
    lastExecutionStatus?: string;
    successCount?: number;
    successRate?: string;
  } | null;
}

// 자동화 서비스 클래스
class AutomationService {
  // 에이전트 실행 요청 (비동기)
  async executeAgent(prompt: string, options: AgentOptions = {}): Promise<ApiResponse<{ jobId: string }>> {
    return apiClient.post('/agent', {
      prompt,
      options,
    });
  }

  // 에이전트 작업 상태 확인
  async checkAgentStatus(jobId: string): Promise<ApiResponse<AgentJobStatus>> {
    return apiClient.get(`/agent/status/${jobId}`);
  }

  // 작업 목록 조회
  async getTasks(): Promise<ApiResponse<Task[]>> {
    return apiClient.get('/tasks');
  }

  // 작업 저장
  async saveTask(task: { name: string; prompt: string; code?: string }): Promise<ApiResponse<Task>> {
    return apiClient.post('/tasks', task);
  }

  // 작업 삭제
  async deleteTask(taskId: string): Promise<ApiResponse<{ success: boolean; message: string }>> {
    return apiClient.delete(`/tasks/${taskId}`);
  }

  // 작업 실행
  async executeTask(taskId: string): Promise<ApiResponse<{
    task: Task;
    result: {
      success: boolean;
      code: string;
      output: string;
      error: string | null;
    };
  }>> {
    return apiClient.post(`/tasks/${taskId}/execute`);
  }
}

// 자동화 서비스 인스턴스 생성
const automationService = new AutomationService();

export default automationService; 