import apiClient, { ApiResponse } from './apiClient';

// 사용자 인터페이스
export interface User {
  id: number;
  username: string;
  email: string;
  name: string;
  role: string;
  status: string;
  created_at: string;
  last_login?: string;
}

// 사용자 생성 인터페이스
export interface CreateUserData {
  username: string;
  email: string;
  password: string;
  name?: string;
  role?: string;
}

// 사용자 업데이트 인터페이스
export interface UpdateUserData {
  email?: string;
  name?: string;
  role?: string;
  password?: string;
  status?: string;
}

// API 키 인터페이스
export interface ApiKey {
  id: string;
  name: string;
  key: string;
  userId: number;
  expiration?: string;
  lastUsed?: string;
  status: 'active' | 'revoked';
  createdAt: string;
  restrictions?: {
    allowedIps?: string[];
    allowedEndpoints?: string[];
    rateLimit?: number;
  };
}

// API 키 생성 인터페이스
export interface CreateApiKeyData {
  name: string;
  expiration?: string;
  restrictions?: {
    allowedIps?: string[];
    allowedEndpoints?: string[];
    rateLimit?: number;
  };
}

// 설정 서비스 클래스
class SettingsService {
  // 설정 가져오기
  async getConfig(key: string): Promise<ApiResponse<{ value: any }>> {
    return apiClient.get(`/config/${key}`);
  }

  // 설정 저장하기
  async setConfig(key: string, value: any): Promise<ApiResponse<{ key: string; value: any }>> {
    return apiClient.post('/config', { key, value });
  }

  // 캐시 초기화
  async clearCache(options: { type?: string } = {}): Promise<ApiResponse<any>> {
    return apiClient.post('/cache/clear', options);
  }

  // 라이선스 검증
  async validateLicense(licenseKey: string, machineId: string): Promise<ApiResponse<{
    valid: boolean;
    message: string;
    expiresAt?: string;
    details?: any;
  }>> {
    return apiClient.post('/license/validate', { licenseKey, machineId });
  }

  // 사용자 인증 (로그인)
  async login(username: string, password: string): Promise<ApiResponse<{
    token: string;
    user: User;
  }>> {
    return apiClient.post('/auth/login', { username, password });
  }

  // 사용자 등록
  async register(userData: CreateUserData): Promise<ApiResponse<User>> {
    return apiClient.post('/auth/register', userData);
  }

  // 현재 사용자 정보 조회
  async getCurrentUser(): Promise<ApiResponse<User>> {
    return apiClient.get('/auth/me');
  }

  // 현재 사용자 정보 업데이트
  async updateCurrentUser(userData: { name?: string; email?: string }): Promise<ApiResponse<User>> {
    return apiClient.put('/auth/me', userData);
  }

  // 비밀번호 변경
  async changePassword(currentPassword: string, newPassword: string): Promise<ApiResponse<{ message: string }>> {
    return apiClient.post('/auth/change-password', { currentPassword, newPassword });
  }

  // 비밀번호 초기화 (관리자용)
  async resetPassword(username: string): Promise<ApiResponse<{ message: string; tempPassword: string }>> {
    return apiClient.post('/auth/reset-password', { username });
  }

  // 사용자 목록 조회 (관리자용)
  async getUsers(options: {
    page?: number;
    limit?: number;
    sortBy?: string;
    sortOrder?: string;
    status?: string;
    role?: string;
    search?: string;
  } = {}): Promise<ApiResponse<{
    users: User[];
    total: number;
    page: number;
    limit: number;
    totalPages: number;
  }>> {
    return apiClient.get('/admin/users', options as any);
  }

  // 사용자 상세 정보 조회 (관리자용)
  async getUser(userId: number | string): Promise<ApiResponse<User>> {
    return apiClient.get(`/admin/users/${userId}`);
  }

  // 사용자 정보 업데이트 (관리자용)
  async updateUser(userId: number | string, userData: UpdateUserData): Promise<ApiResponse<User>> {
    return apiClient.put(`/admin/users/${userId}`, userData);
  }

  // 사용자 삭제 (관리자용)
  async deleteUser(userId: number): Promise<ApiResponse<{ success: boolean; message: string }>> {
    return apiClient.delete(`/admin/users/${userId}`);
  }

  // 사용자 상태 변경 (관리자용)
  async updateUserStatus(userId: number | string, status: string): Promise<ApiResponse<User>> {
    return apiClient.put(`/admin/users/${userId}/status`, { status });
  }

  // 사용자 역할 변경 (관리자용)
  async assignRole(userId: number | string, role: string): Promise<ApiResponse<User>> {
    return apiClient.post(`/admin/users/${userId}/role`, { role });
  }

  // API 키 생성
  async createApiKey(data: CreateApiKeyData): Promise<ApiResponse<ApiKey>> {
    return apiClient.post('/api-keys', data);
  }

  // API 키 목록 조회
  async listApiKeys(): Promise<ApiResponse<ApiKey[]>> {
    return apiClient.get('/api-keys');
  }

  // API 키 상태 변경
  async updateApiKeyStatus(keyId: string, status: 'active' | 'revoked'): Promise<ApiResponse<ApiKey>> {
    return apiClient.put(`/api-keys/${keyId}/status`, { status });
  }

  // API 키 삭제
  async deleteApiKey(keyId: string): Promise<ApiResponse<{ success: boolean; message: string }>> {
    return apiClient.delete(`/api-keys/${keyId}`);
  }

  // API 키 업데이트
  async updateApiKey(keyId: string, data: Partial<CreateApiKeyData>): Promise<ApiResponse<ApiKey>> {
    return apiClient.put(`/api-keys/${keyId}`, data);
  }
}

// 설정 서비스 인스턴스 생성
const settingsService = new SettingsService();

export default settingsService; 