#!/usr/bin/env node

import axios from 'axios';

class SimpleMCPServer {
  constructor() {
    this.name = "complaint-manager";
    this.version = "1.0.0";
    this.dummyServerUrl = "http://localhost:4000";
    
    // 한국어 상태 매핑
    this.statusMapping = {
      '대기': 'pending',
      '대기중': 'pending',
      '접수': 'pending',
      '처리중': 'in_progress',
      '진행중': 'in_progress',
      '검토중': 'in_progress',
      '완료': 'resolved',
      '해결': 'resolved',
      '처리완료': 'resolved',
      '종료': 'resolved',
      '보류': 'on_hold',
      '취소': 'cancelled',
      '반려': 'rejected'
    };
  }

  // MCP 서버 초기화
  async initialize() {
    console.log(`🚀 ${this.name} v${this.version} 시작`);
    console.log(`📡 더미 서버 연결 테스트: ${this.dummyServerUrl}`);
    
    try {
      await this.testConnection();
      console.log("✅ 더미 서버 연결 성공");
      return true;
    } catch (error) {
      console.error("❌ 더미 서버 연결 실패:", error.message);
      console.log("⚠️ 더미 서버가 실행되지 않았을 수 있습니다. 계속 진행합니다...");
      return true; // 연결 실패해도 계속 진행
    }
  }

  // 더미 서버 연결 테스트
  async testConnection() {
    const response = await axios.get(`${this.dummyServerUrl}/api/complaints`);
    return response.data;
  }

  // 한국어 상태를 영어 상태로 변환
  convertKoreanStatus(koreanStatus) {
    const status = this.statusMapping[koreanStatus];
    if (status) {
      console.log(`🔄 상태 변환: "${koreanStatus}" → "${status}"`);
      return status;
    }
    
    // 매핑되지 않으면 원래 값 반환 (영어 상태일 수도 있음)
    console.log(`⚠️ 상태 변환 실패: "${koreanStatus}" (원래 값 사용)`);
    return koreanStatus;
  }

  // 영어 상태를 한국어로 변환 (응답용)
  convertStatusToKorean(englishStatus) {
    const koreanStatusMap = {
      'pending': '대기중',
      'in_progress': '처리중',
      'resolved': '완료',
      'on_hold': '보류',
      'cancelled': '취소',
      'rejected': '반려'
    };
    
    return koreanStatusMap[englishStatus] || englishStatus;
  }

  // 민원 목록 조회 (검색 옵션 지원)
  async getComplaints(options = {}) {
    try {
      console.log("📋 민원 목록 조회 중...", options);
      
      // 쿼리 파라미터 구성
      const params = new URLSearchParams();
      
      if (options.date_from) params.append('date_from', options.date_from);
      if (options.date_to) params.append('date_to', options.date_to);
      if (options.category) params.append('category', options.category);
      if (options.status) params.append('status', options.status);
      if (options.priority) params.append('priority', options.priority);
      if (options.location) params.append('location', options.location);
      if (options.keyword) params.append('keyword', options.keyword);
      if (options.sort_by) params.append('sort_by', options.sort_by);
      if (options.sort_order) params.append('sort_order', options.sort_order);
      
      const url = `${this.dummyServerUrl}/api/complaints${params.toString() ? '?' + params.toString() : ''}`;
      console.log("🔗 요청 URL:", url);
      
      const response = await axios.get(url);
      console.log(`✅ 민원 목록 조회 완료: ${response.data.length}건`);
      return response.data;
    } catch (error) {
      console.error("❌ 민원 목록 조회 실패:", error.message);
      throw error;
    }
  }

  // 특정 민원 조회
  async getComplaint(id) {
    try {
      console.log(`📄 민원 ID ${id} 조회 중...`);
      const response = await axios.get(`${this.dummyServerUrl}/api/complaints/${id}`);
      console.log("✅ 민원 조회 완료:", response.data);
      return response.data;
    } catch (error) {
      console.error(`❌ 민원 ID ${id} 조회 실패:`, error.message);
      throw error;
    }
  }

  // 민원에 답변 추가
  async replyToComplaint(id, officerId, replyContent) {
    try {
      console.log(`💬 민원 ID ${id}에 답변 추가 중...`);
      const response = await axios.post(`${this.dummyServerUrl}/api/complaints/${id}/reply`, {
        officer_id: officerId,
        reply_content: replyContent
      });
      console.log("✅ 답변 추가 완료:", response.data);
      return response.data;
    } catch (error) {
      console.error(`❌ 민원 ID ${id} 답변 추가 실패:`, error.message);
      throw error;
    }
  }

  // 민원 상태 업데이트
  async updateComplaintStatus(id, status) {
    try {
      console.log(`🔄 민원 ID ${id} 상태 업데이트 요청: "${status}"`);
      
      // 한국어 상태를 영어로 변환
      const englishStatus = this.convertKoreanStatus(status);
      
      const response = await axios.put(`${this.dummyServerUrl}/api/complaints/${id}/status`, {
        status: englishStatus
      });
      
      // 응답에서 한국어 상태로 변환해서 표시
      const koreanStatus = this.convertStatusToKorean(englishStatus);
      console.log(`✅ 민원 ID ${id} 상태 업데이트 성공: "${koreanStatus}"`);
      
      return {
        ...response.data,
        status: koreanStatus,
        originalStatus: englishStatus
      };
    } catch (error) {
      console.error(`❌ 민원 ID ${id} 상태 변경 실패:`, error.message);
      throw error;
    }
  }

  // 민원 처리 히스토리 조회
  async getComplaintHistory(id) {
    try {
      console.log(`📜 민원 ID ${id} 히스토리 조회 중...`);
      const response = await axios.get(`${this.dummyServerUrl}/api/complaints/${id}/history`);
      console.log("✅ 히스토리 조회 완료:", response.data);
      return response.data;
    } catch (error) {
      console.error(`❌ 민원 ID ${id} 히스토리 조회 실패:`, error.message);
      throw error;
    }
  }

  // 자동 민원 처리 워크플로우
  async processComplaintWorkflow(complaintId) {
    console.log(`\n🔄 민원 ID ${complaintId} 자동 처리 워크플로우 시작`);
    
    try {
      // 1. 민원 조회
      const complaint = await this.getComplaint(complaintId);
      
      // 2. 간단한 AI 응답 생성 (실제로는 LLM API 호출)
      const aiResponse = this.generateAIResponse(complaint.content);
      console.log(`🤖 AI 생성 응답: ${aiResponse}`);
      
      // 3. 답변 추가
      await this.replyToComplaint(complaintId, 'mcp_agent', aiResponse);
      
      // 4. 상태를 처리중으로 변경
      await this.updateComplaintStatus(complaintId, 'in_progress');
      
      // 5. 처리 완료 후 상태 변경
      setTimeout(async () => {
        await this.updateComplaintStatus(complaintId, 'completed');
        console.log(`✅ 민원 ID ${complaintId} 자동 처리 완료`);
      }, 2000);
      
      console.log(`🎉 민원 ID ${complaintId} 워크플로우 완료`);
      return true;
      
    } catch (error) {
      console.error(`❌ 민원 ID ${complaintId} 워크플로우 실패:`, error.message);
      return false;
    }
  }

  // 간단한 AI 응답 생성 (실제로는 LLM API 호출로 대체)
  generateAIResponse(content) {
    const responses = [
      `"${content}" 민원에 대해 확인했습니다. 신속히 처리하겠습니다.`,
      `접수된 "${content}" 관련 민원을 검토 중입니다. 곧 조치를 취하겠습니다.`,
      `"${content}" 문제를 인지했습니다. 관련 부서에 전달하여 해결하겠습니다.`
    ];
    return responses[Math.floor(Math.random() * responses.length)];
  }

  // MCP 도구들을 정의 - 위치 기반 파라미터 지원
  getAvailableTools() {
    return [
      {
        name: "get_complaints",
        description: "민원 목록을 조회합니다. 검색 옵션 지원. 사용법: get_complaints() 또는 get_complaints({keyword: '소음', category: '소음/진동', date_from: '2025-01-10'})",
        inputSchema: {
          type: "object",
          properties: {
            keyword: { type: "string", description: "제목이나 내용에서 검색할 키워드" },
            category: { type: "string", description: "카테고리 필터 (환경/위생, 소음/진동, 도로/교통, 공원/녹지, 주차/교통)" },
            status: { type: "string", description: "상태 필터 (pending, in_progress, resolved)" },
            priority: { type: "string", description: "우선순위 필터 (low, medium, high, urgent)" },
            location: { type: "string", description: "위치 검색 키워드" },
            date_from: { type: "string", description: "시작 날짜 (YYYY-MM-DD 형식)" },
            date_to: { type: "string", description: "종료 날짜 (YYYY-MM-DD 형식)" },
            sort_by: { type: "string", description: "정렬 기준 (created_at, updated_at, priority)" },
            sort_order: { type: "string", description: "정렬 순서 (asc, desc)" }
          },
          required: []
        }
      },
      {
        name: "get_complaint",
        description: "특정 ID의 민원을 조회합니다. 사용법: get_complaint(id)",
        inputSchema: {
          type: "object",
          properties: {
            id: { 
              type: ["string", "number"], 
              description: "민원 ID (숫자 또는 문자열)" 
            }
          },
          required: ["id"]
        }
      },
      {
        name: "reply_to_complaint",
        description: "민원에 답변을 추가합니다. 사용법: reply_to_complaint(id, officer_id, reply_content)",
        inputSchema: {
          type: "object",
          properties: {
            id: { 
              type: ["string", "number"], 
              description: "민원 ID" 
            },
            officer_id: { 
              type: "string", 
              description: "담당자 ID" 
            },
            reply_content: { 
              type: "string", 
              description: "답변 내용" 
            }
          },
          required: ["id", "officer_id", "reply_content"]
        }
      },
      {
        name: "update_complaint_status",
        description: "민원 상태를 업데이트합니다. 사용법: update_complaint_status(id, status). 한국어 상태 지원: 대기/처리중/완료/보류/취소/반려",
        inputSchema: {
          type: "object",
          properties: {
            id: { 
              type: ["string", "number"], 
              description: "민원 ID" 
            },
            status: { 
              type: "string", 
              description: "새로운 상태 (한국어: 대기/처리중/완료/보류/취소/반려 또는 영어: pending/in_progress/resolved/on_hold/cancelled/rejected)" 
            }
          },
          required: ["id", "status"]
        }
      },
      {
        name: "process_complaint_workflow",
        description: "민원을 자동으로 처리하는 워크플로우를 실행합니다. 사용법: process_complaint_workflow(complaint_id)",
        inputSchema: {
          type: "object",
          properties: {
            complaint_id: { 
              type: ["string", "number"], 
              description: "처리할 민원 ID" 
            }
          },
          required: ["complaint_id"]
        }
      }
    ];
  }

  // 데모 실행
  async runDemo() {
    console.log("\n🎬 MCP 서버 데모 시작");
    
    // 초기화
    const initialized = await this.initialize();
    if (!initialized) {
      console.log("❌ 초기화 실패. 더미 서버가 실행 중인지 확인하세요.");
      return;
    }

    // 사용 가능한 도구 출력
    console.log("\n🛠️ 사용 가능한 MCP 도구들:");
    this.getAvailableTools().forEach((tool, index) => {
      console.log(`${index + 1}. ${tool.name}: ${tool.description}`);
    });

    // 민원 처리 워크플로우 실행
    console.log("\n🚀 민원 자동 처리 워크플로우 데모 실행");
    await this.processComplaintWorkflow("1");
    
    console.log("\n🎉 데모 완료!");
  }
}

// 스크립트가 직접 실행될 때만 데모 실행
if (import.meta.url === `file://${process.argv[1]}`) {
  const server = new SimpleMCPServer();
  server.runDemo().catch(console.error);
}

export { SimpleMCPServer }; 