"use client";
import { useState } from "react";
import type { User } from "../../../types/user";

interface UserFormProps {
  initialData?: Partial<User>;
  onSubmit: (data: Partial<User>) => Promise<void>;
  onCancel: () => void;
  loading?: boolean;
}

export default function UserForm({ initialData = {}, onSubmit, onCancel, loading }: UserFormProps) {
  const [form, setForm] = useState<Partial<User> & { password?: string }>({
    role: "user",
    status: initialData.status || "active",
    ...initialData,
  });
  const [error, setError] = useState<string | null>(null);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const target = e.target as HTMLInputElement | HTMLSelectElement;
    const { name, value, type } = target;
    setForm((prev) => ({
      ...prev,
      [name]: type === "checkbox" ? (target as HTMLInputElement).checked : value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);
    try {
      const cleanData = {
        ...form,
        username: form.username?.trim() || undefined,
        email: form.email?.trim() || undefined,
        role: form.role || undefined,
        name: form.name?.trim() || undefined,
        status: form.status || undefined,
        password: form.password || undefined,
      };
      await onSubmit(cleanData);
    } catch (err: any) {
      setError(err.message || "오류가 발생했습니다.");
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <label className="block text-sm font-medium mb-1">아이디(Username)</label>
        <input
          name="username"
          value={form.username || ""}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
          required
          disabled={!!initialData.id}
        />
      </div>
      <div>
        <label className="block text-sm font-medium mb-1">비밀번호</label>
        <input
          name="password"
          type="password"
          value={form.password || ""}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
          placeholder={initialData.id ? "비워두면 변경되지 않습니다" : ""}
          required={!initialData.id}
        />
      </div>
      <div>
        <label className="block text-sm font-medium mb-1">이름</label>
        <input
          name="name"
          value={form.name || ""}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
          required
        />
      </div>
      <div>
        <label className="block text-sm font-medium mb-1">이메일</label>
        <input
          name="email"
          type="email"
          value={form.email || ""}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
        />
      </div>
      <div>
        <label className="block text-sm font-medium mb-1">역할(Role)</label>
        <select
          name="role"
          value={form.role || "user"}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
        >
          <option value="user">user</option>
          <option value="manager">manager</option>
          <option value="admin">admin</option>
        </select>
      </div>
      <div>
        <label className="block text-sm font-medium mb-1">상태(Status)</label>
        <select
          name="status"
          value={form.status || "active"}
          onChange={handleChange}
          className="w-full border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
        >
          <option value="active">active</option>
          <option value="inactive">inactive</option>
        </select>
      </div>
      {error && <div className="text-red-600 text-sm">{error}</div>}
      <div className="flex gap-2 justify-end">
        <button type="button" onClick={onCancel} className="px-4 py-2 bg-gray-200 rounded">취소</button>
        <button type="submit" className="px-4 py-2 bg-blue-600 text-white rounded" disabled={loading}>
          저장
        </button>
      </div>
    </form>
  );
} 