import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

// 세션 디렉토리 경로 가져오기 (MCP 채팅 API와 동일한 경로 사용)
const getSessionsDir = () => {
  const baseDir = path.join(os.homedir(), '.airun', 'sessions');
  if (!fs.existsSync(baseDir)) {
    fs.mkdirSync(baseDir, { recursive: true });
  }
  return baseDir;
};

// 로컬 세션 파일에서 세션 로드
async function loadSessionFromLocal(sessionId: string) {
  const sessionsDir = getSessionsDir();
  const sessionFile = path.join(sessionsDir, `${sessionId}.json`);
  
  if (fs.existsSync(sessionFile)) {
    try {
      const sessionData = JSON.parse(fs.readFileSync(sessionFile, 'utf-8'));
      return sessionData;
    } catch (error) {
      console.error(`로컬 세션 로드 실패: ${sessionId}`, error);
      return null;
    }
  }
  
  return null;
}

// 로컬 세션 파일 삭제
async function deleteSessionFromLocal(sessionId: string) {
  const sessionsDir = getSessionsDir();
  const sessionFile = path.join(sessionsDir, `${sessionId}.json`);
  
  console.log(`🗑️ 세션 파일 삭제 시도: ${sessionFile}`);
  console.log(`📁 세션 디렉토리: ${sessionsDir}`);
  console.log(`📄 파일 존재 여부: ${fs.existsSync(sessionFile)}`);
  
  if (fs.existsSync(sessionFile)) {
    try {
      fs.unlinkSync(sessionFile);
      console.log(`✅ 로컬 세션 파일 삭제 완료: ${sessionFile}`);
      return true;
    } catch (error) {
      console.error(`❌ 로컬 세션 삭제 실패: ${sessionId}`, error);
      return false;
    }
  }
  
  console.log(`⚠️ 세션 파일이 존재하지 않음: ${sessionFile}`);
  return false;
}

// GET: 모든 assistant 세션 목록 조회
export async function GET(request: NextRequest) {
  try {
    const sessionsDir = getSessionsDir();
    const sessionFiles = fs.readdirSync(sessionsDir)
      .filter(file => file.endsWith('.json') && file.startsWith('assistant-'))
      .sort((a, b) => {
        // 파일 수정 시간 기준 내림차순 정렬
        const aPath = path.join(sessionsDir, a);
        const bPath = path.join(sessionsDir, b);
        const aStat = fs.statSync(aPath);
        const bStat = fs.statSync(bPath);
        return bStat.mtime.getTime() - aStat.mtime.getTime();
      });

    const sessions: any = {};
    
    for (const file of sessionFiles) {
      const sessionId = file.replace('.json', '');
      const sessionData = await loadSessionFromLocal(sessionId);
      
      if (sessionData) {
        sessions[sessionId] = sessionData;
      }
    }
    
    return NextResponse.json({
      success: true,
      data: sessions
    });
  } catch (error) {
    console.error('세션 목록 조회 실패:', error);
    return NextResponse.json(
      { error: 'Failed to load sessions' },
      { status: 500 }
    );
  }
}

// DELETE: 특정 assistant 세션 삭제
export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const sessionId = searchParams.get('sessionId');
    
    console.log('🗑️ Assistant 세션 삭제 API 호출:', { sessionId });
    
    if (!sessionId) {
      return NextResponse.json(
        { error: 'Session ID is required' },
        { status: 400 }
      );
    }
    
    // assistant- 접두어 검증
    if (!sessionId.startsWith('assistant-')) {
      return NextResponse.json(
        { error: 'Invalid session ID format' },
        { status: 400 }
      );
    }

    // 로컬 세션만 삭제 (외부 API 서버 호출 안 함)
    const localDeleted = await deleteSessionFromLocal(sessionId);
    
    if (localDeleted) {
      return NextResponse.json({
        success: true,
        message: 'Session deleted successfully'
      });
    } else {
      return NextResponse.json(
        { error: 'Session not found' },
        { status: 404 }
      );
    }
  } catch (error) {
    console.error('🗑️ Assistant 세션 삭제 실패:', error);
    console.error('🗑️ 오류 상세:', {
      name: (error as Error).name,
      message: (error as Error).message,
      stack: (error as Error).stack
    });
    return NextResponse.json(
      { error: 'Failed to delete session' },
      { status: 500 }
    );
  }
} 