import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';

const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

// 동기식 채팅 엔드포인트
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      // JSON 요청 처리
      const body = await req.json();
      const { prompt, options } = body;

      // 입력 유효성 검사
      if (!prompt || typeof prompt !== 'string') {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '메시지가 필요합니다.'
            }
          }, 
          { status: 400 }
        );
      }

      // 토큰 또는 API 키 설정
      const headers: Record<string, string> = {
        'Content-Type': 'application/json'
      };

      // JWT 토큰 인증 시 사용자의 API Key 가져와서 사용
      if (authType === 'token') {
        const token = req.cookies.get('auth_token')?.value;
        if (!token) {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: '인증 토큰이 필요합니다.'
              }
            },
            { status: 401 }
          );
        }

        // 사용자의 API Key 목록 가져오기
        try {
          const apiKeysResponse = await fetch(`${API_SERVER}/api-keys`, {
            method: 'GET',
            headers: {
              'Authorization': `Bearer ${token}`
            }
          });

          if (apiKeysResponse.ok) {
            const apiKeysData = await apiKeysResponse.json();
            if (apiKeysData.success && apiKeysData.data && apiKeysData.data.length > 0) {
              // 첫 번째 활성 API Key 사용
              const activeKey = apiKeysData.data.find((key: any) => key.status === 'active');
              if (activeKey) {
                headers['X-API-Key'] = activeKey.key_value;
                console.log('JWT 토큰 사용자의 API Key로 변환 완료');
              } else {
                return NextResponse.json(
                  {
                    success: false,
                    error: {
                      message: '활성화된 API 키가 없습니다. 프로필에서 API 키를 생성해주세요.'
                    }
                  },
                  { status: 401 }
                );
              }
            } else {
              return NextResponse.json(
                {
                  success: false,
                  error: {
                    message: 'API 키가 없습니다. 프로필에서 API 키를 생성해주세요.'
                  }
                },
                { status: 401 }
              );
            }
          } else {
            return NextResponse.json(
              {
                success: false,
                error: {
                  message: 'API 키를 가져올 수 없습니다.'
                }
              },
              { status: 401 }
            );
          }
        } catch (error) {
          console.error('API 키 조회 오류:', error);
          return NextResponse.json(
            {
              success: false,
              error: {
                message: 'API 키 조회 중 오류가 발생했습니다.'
              }
            },
            { status: 500 }
          );
        }
      } else if (authType === 'apikey') {
        const apiKey = req.headers.get('x-api-key');
        if (apiKey) {
          headers['X-API-Key'] = apiKey;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: 'API 키가 필요합니다.'
              }
            },
            { status: 401 }
          );
        }
      } else {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '인증이 필요합니다.'
            }
          },
          { status: 401 }
        );
      }

      // 디버깅: user, options.userId, 프론트에서 받은 값, API 서버로 전달하는 값 모두 출력
      console.log('[chat/sync] 디버그:', {
        user,
        optionsFromClient: options,
        userIdFromClient: options?.userId,
        userIdFromAuth: user?.id,
        userIdToApiServer: options?.userId || user?.id,
      });

      // API 서버에 동기식 채팅 요청 생성
      console.log('API 서버에 채팅 요청 전송:', {
        url: `${API_SERVER}/chat/sync`,
        method: 'POST',
        headers: {
          ...headers,
          'Authorization': headers['Authorization'] ? '[인증 토큰 포함]' : '없음',
          'X-API-Key': headers['X-API-Key'] ? '[API 키 포함]' : '없음'
        },
        body: {
          prompt,
          options: {
            rag: !!options?.rag,
            web: !!options?.web,
            first: options?.first !== false,
            files: options?.files || [],
            image: options?.image || null,
            ...options
          }
        }
      });

      const apiResponse = await fetch(`${API_SERVER}/chat/sync`, {
        method: 'POST',
        headers,
        body: JSON.stringify({
          prompt,
          options: {
            rag: !!options?.rag,
            web: !!options?.web,
            first: options?.first !== false,
            files: options?.files || [],
            image: options?.image || null,
            userId: options?.userId || user?.id, // userId 우선순위 명확히
            ...options
          }
        })
      });

      if (!apiResponse.ok) {
        let errorMessage = '';
        try {
          const errorData = await apiResponse.json();
          errorMessage = errorData.error?.message || '채팅 요청에 실패했습니다.';
        } catch (e) {
          errorMessage = `채팅 요청 실패 (${apiResponse.status}): ${apiResponse.statusText}`;
        }
        
        console.error('API 서버 응답 오류:', {
          status: apiResponse.status,
          statusText: apiResponse.statusText,
          message: errorMessage
        });
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: apiResponse.status }
        );
      }

      const apiData = await apiResponse.json();
      console.log('API 서버 채팅 응답 성공:', {
        success: apiData?.success,
        hasData: !!apiData?.data,
        dataKeys: apiData?.data ? Object.keys(apiData.data) : [],
        hasResponse: !!apiData?.data?.response,
        hasHistory: !!apiData?.data?.history,
        historyLength: apiData?.data?.history ? apiData.data.history.length : 0,
        responsePreview: apiData?.data?.response ? `${apiData.data.response.substring(0, 100)}...` : 'no response'
      });
      
      // API 서버 응답이 없거나 유효하지 않은 경우 처리
      if (!apiData || !apiData.data) {
        console.log('API 서버 응답이 없거나 유효하지 않음:', apiData);
        return NextResponse.json({
          success: true,
          data: {
            response: "응답을 처리할 수 없습니다.",
            sessionId: `session-${Date.now()}`,
            timestamp: new Date().toISOString()
          }
        });
      }
      
      // API 서버 응답 반환
      console.log('웹 프론트엔드에서 반환할 데이터:', {
        success: true,
        data: {
          ...apiData.data,
          sessionId: apiData.data.sessionId || apiData.data.id || `session-${Date.now()}`
        }
      });
      
      return NextResponse.json({
        success: true,
        data: {
          ...apiData.data,
          sessionId: apiData.data.sessionId || apiData.data.id || `session-${Date.now()}`
        }
      });
    } catch (error) {
      console.error('채팅 요청 처리 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '채팅 요청 처리 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 