import { createClient } from 'redis';

// Redis 클라이언트 생성 (중앙 설정에서 가져오기)
import { getRedisUrl } from '@/config/serverConfig';

const client = createClient({
  url: getRedisUrl()
});

// Redis 클라이언트 초기화 및 에러 처리
client.on('error', (err) => {
  console.error('Redis client error:', err);
});

// 자동 재연결 활성화
client.on('reconnecting', () => {
  console.log('Redis client reconnecting...');
});

client.on('connect', () => {
  console.log('Redis client connected');
});

// 초기 연결
let isConnected = false;
async function connect() {
  if (!isConnected) {
    await client.connect();
    isConnected = true;
  }
  return client;
}

/**
 * 키-값 저장
 * @param key 키
 * @param value 값
 * @param ttl 만료 시간(초)
 */
export async function set(key: string, value: string, ttl?: number) {
  try {
    const redis = await connect();
    if (ttl) {
      await redis.set(key, value, { EX: ttl });
    } else {
      await redis.set(key, value);
    }
  } catch (error) {
    console.error('Redis set error:', error);
    throw error;
  }
}

/**
 * 키-값 조회
 * @param key 키
 * @returns 값 또는 null
 */
export async function get(key: string) {
  try {
    const redis = await connect();
    return await redis.get(key);
  } catch (error) {
    console.error('Redis get error:', error);
    throw error;
  }
}

/**
 * 키 삭제
 * @param key 키
 * @returns 삭제 성공 여부
 */
export async function del(key: string) {
  try {
    const redis = await connect();
    return await redis.del(key) === 1;
  } catch (error) {
    console.error('Redis del error:', error);
    throw error;
  }
}

/**
 * 연결 테스트
 */
export async function testConnection() {
  try {
    const redis = await connect();
    await redis.ping();
    console.log('Redis connection test successful');
    return true;
  } catch (error) {
    console.error('Redis connection test failed:', error);
    return false;
  }
}

/**
 * 클라이언트 종료 (서버 종료 시 호출)
 */
export async function quit() {
  if (isConnected) {
    await client.quit();
    isConnected = false;
  }
}

export default {
  set,
  get,
  del,
  testConnection,
  quit,
}; 