import { NextRequest, NextResponse } from 'next/server';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { nodeId, type, data } = body;

    // 노드 타입별 테스트 실행
    let result: any = null;
    
    switch (type) {
      case 'process':
        result = await testProcessNode(data);
        break;
      
      case 'input':
        result = await testInputNode(data);
        break;
      
      case 'output':
        result = await testOutputNode(data);
        break;
        
      case 'condition':
        result = await testConditionNode(data);
        break;
        
      case 'loop':
        result = await testLoopNode(data);
        break;
        
      default:
        throw new Error(`Unsupported node type: ${type}`);
    }

    return NextResponse.json({
      success: true,
      data: result,
      timestamp: new Date().toISOString()
    });
  } catch (error: any) {
    console.error('Node test error:', error);
    return NextResponse.json({
      success: false,
      error: {
        code: 'TEST_ERROR',
        message: error.message || 'Failed to test node'
      }
    }, { status: 500 });
  }
}

async function testProcessNode(data: any) {
  const { processType, prompt, testInput, chatConfig, codeConfig, reportConfig, ragConfig, webSearchConfig } = data;
  
  if (!processType) {
    throw new Error('Process type is required');
  }

  const apiServerUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500';
  
  try {
    switch (processType) {
      case 'chat':
        return await testChatEndpoint(apiServerUrl, prompt, testInput, chatConfig);
      
      case 'code':
        return await testCodeEndpoint(apiServerUrl, prompt, testInput, codeConfig);
      
      case 'report':
        return await testReportEndpoint(apiServerUrl, prompt, testInput, reportConfig);
      
      case 'rag-search':
        return await testRagSearchEndpoint(apiServerUrl, testInput, ragConfig);
      
      case 'rag-add':
        return await testRagAddEndpoint(apiServerUrl, testInput, ragConfig);
      
      case 'rag-list':
        return await testRagListEndpoint(apiServerUrl);
      
      case 'rag-init':
        return await testRagInitEndpoint(apiServerUrl);
      
      case 'web-search':
        return await testWebSearchEndpoint(apiServerUrl, testInput, webSearchConfig);
      
      case 'data-transform':
      case 'condition':
      case 'loop':
        return await testLegacyProcessing(data);
      
      default:
        throw new Error(`Unsupported process type: ${processType}`);
    }
  } catch (error: any) {
    return {
      success: false,
      error: error.message,
      processType,
      timestamp: new Date().toISOString()
    };
  }
}

// Test individual API endpoints directly

async function testChatEndpoint(apiServerUrl: string, prompt: string, testInput: string, chatConfig: any = {}) {
  const config = {
    model: 'gpt-4',
    temperature: 0.7,
    maxTokens: 2000,
    systemPrompt: '',
    ...chatConfig
  };

  // testInput만 사용 (prompt는 systemPrompt로 대체됨)
  const userMessage = testInput;
  
  const response = await fetch(`${apiServerUrl}/api/v1/chat`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      prompt: userMessage,
      model: config.model,
      temperature: config.temperature,
      max_tokens: config.maxTokens,
      system_prompt: config.systemPrompt
    })
  });

  if (!response.ok) {
    throw new Error(`Chat API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/chat',
    config,
    data: result
  };
}

async function testCodeEndpoint(apiServerUrl: string, prompt: string, testInput: string, codeConfig: any = {}) {
  const config = {
    model: 'gpt-4',
    language: 'javascript',
    framework: '',
    style: 'clean',
    systemPrompt: '',
    ...codeConfig
  };

  // testInput만 사용 (prompt는 systemPrompt로 대체됨)
  const userMessage = testInput;
  
  const response = await fetch(`${apiServerUrl}/api/v1/code`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      prompt: userMessage,
      language: config.language,
      framework: config.framework,
      model: config.model,
      system_prompt: config.systemPrompt
    })
  });

  if (!response.ok) {
    throw new Error(`Code API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/code',
    config,
    data: result
  };
}

async function testReportEndpoint(apiServerUrl: string, prompt: string, testInput: string, reportConfig: any = {}) {
  const config = {
    template: 'simple',
    format: 'pdf',
    includeCharts: true,
    pageSize: 'A4',
    ...reportConfig
  };

  const content = prompt ? `${prompt}

${testInput}` : testInput;
  
  const response = await fetch(`${apiServerUrl}/api/v1/report`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      task: content,
      template: config.template,
      format: config.format,
      include_charts: config.includeCharts,
      page_size: config.pageSize
    })
  });

  if (!response.ok) {
    throw new Error(`Report API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/report',
    config,
    data: result
  };
}

async function testRagSearchEndpoint(apiServerUrl: string, testInput: string, ragConfig: any = {}) {
  const config = {
    query: testInput,
    max_results: 5,
    similarity_threshold: 0.7,
    collection: 'default',
    ...ragConfig
  };

  const response = await fetch(`${apiServerUrl}/api/v1/rag/search`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      query: config.query || testInput,
      max_results: config.max_results,
      similarity_threshold: config.similarity_threshold,
      collection: config.collection
    })
  });

  if (!response.ok) {
    throw new Error(`RAG Search API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/rag/search',
    config,
    data: result
  };
}

async function testRagAddEndpoint(apiServerUrl: string, testInput: string, ragConfig: any = {}) {
  const response = await fetch(`${apiServerUrl}/api/v1/rag/add`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      path: testInput || ragConfig.path || '/sample/path',
      user_id: ragConfig.user_id || 'test-user',
      base_path: ragConfig.base_path
    })
  });

  if (!response.ok) {
    throw new Error(`RAG Add API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/rag/add',
    data: result
  };
}

async function testRagListEndpoint(apiServerUrl: string) {
  const response = await fetch(`${apiServerUrl}/api/v1/rag/list`, {
    method: 'GET',
    headers: { 'Content-Type': 'application/json' }
  });

  if (!response.ok) {
    throw new Error(`RAG List API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/rag/list',
    data: result
  };
}

async function testRagInitEndpoint(apiServerUrl: string) {
  const response = await fetch(`${apiServerUrl}/api/v1/rag/init`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' }
  });

  if (!response.ok) {
    throw new Error(`RAG Init API error: ${response.statusText}`);
  }

  const result = await response.json();
  return {
    success: true,
    endpoint: '/api/v1/rag/init',
    data: result
  };
}









// 레거시 처리 (기존 호환성)
async function testLegacyProcessing(data: any) {
  const { processType, testInput } = data;
  
  switch (processType) {
    case 'data-transform':
      return await testDataTransform(data);
    
    case 'condition':
      return await testConditionLogic(data);
    
    case 'loop':
      return await testLoopLogic(data);
    
    default:
      throw new Error(`Unsupported legacy process type: ${processType}`);
  }
}







async function testAPICall(data: any) {
  const { apiEndpoint, method = 'GET', headers = {}, body } = data.config || {};
  
  if (!apiEndpoint) {
    return {
      warning: 'API endpoint not configured',
      mockResponse: { message: 'Configure API endpoint to test real API calls' }
    };
  }
  
  try {
    const response = await fetch(apiEndpoint, {
      method,
      headers: {
        'Content-Type': 'application/json',
        ...headers
      },
      body: method !== 'GET' ? JSON.stringify(body) : undefined
    });
    
    const responseData = await response.json();
    
    return {
      success: response.ok,
      status: response.status,
      data: responseData,
      headers: Object.fromEntries(response.headers.entries())
    };
  } catch (error: any) {
    return {
      success: false,
      error: error.message,
      endpoint: apiEndpoint,
      method
    };
  }
}

async function testDataTransform(data: any) {
  const { testInput, transformType = 'json' } = data;
  
  try {
    let transformed: any;
    
    switch (transformType) {
      case 'json':
        transformed = JSON.parse(testInput);
        break;
      
      case 'csv':
        // Simple CSV parsing
        const lines = testInput.split('\n');
        const headers = lines[0]?.split(',') || [];
        transformed = lines.slice(1).map(line => {
          const values = line.split(',');
          return headers.reduce((obj: any, header: string, index: number) => {
            obj[header.trim()] = values[index]?.trim();
            return obj;
          }, {});
        });
        break;
      
      case 'xml':
        // Mock XML transformation
        transformed = { message: 'XML transformation not yet implemented' };
        break;
      
      default:
        transformed = testInput;
    }
    
    return {
      success: true,
      input: testInput,
      output: transformed,
      transformType
    };
  } catch (error: any) {
    return {
      success: false,
      error: error.message,
      input: testInput,
      transformType
    };
  }
}

async function testInputNode(data: any) {
  const { config = {} } = data;
  return {
    success: true,
    type: 'input',
    value: config.value || data.testInput || 'Sample input data',
    timestamp: new Date().toISOString()
  };
}

async function testOutputNode(data: any) {
  const { testInput } = data;
  return {
    success: true,
    type: 'output',
    received: testInput || 'No input received',
    processed: true,
    timestamp: new Date().toISOString()
  };
}

async function testConditionLogic(data: any) {
  const { config = {}, testInput } = data;
  const { condition, operator = 'equals', value } = config;
  
  let result = false;
  
  switch (operator) {
    case 'equals':
      result = testInput === value;
      break;
    case 'contains':
      result = String(testInput).includes(String(value));
      break;
    case 'greater':
      result = Number(testInput) > Number(value);
      break;
    case 'less':
      result = Number(testInput) < Number(value);
      break;
    default:
      result = false;
  }
  
  return {
    success: true,
    type: 'condition',
    input: testInput,
    condition: { operator, value },
    result: result,
    branch: result ? 'true' : 'false'
  };
}

async function testLoopLogic(data: any) {
  const { config = {}, testInput } = data;
  const { iterations = 3, delay = 0 } = config;
  
  const results = [];
  const inputArray = Array.isArray(testInput) ? testInput : [testInput];
  
  for (let i = 0; i < Math.min(iterations, 10); i++) {
    results.push({
      iteration: i + 1,
      input: inputArray[i % inputArray.length],
      processed: true
    });
  }
  
  return {
    success: true,
    type: 'loop',
    totalIterations: iterations,
    completedIterations: results.length,
    results: results
  };
}