import { NextRequest, NextResponse } from 'next/server';
import { getNextjsServerUrl } from '@/config/serverConfig';
import { HumanMessage, SystemMessage, AIMessage } from '@langchain/core/messages';
import { ChatOpenAI } from '@langchain/openai';
import { createReactAgent } from '@langchain/langgraph/prebuilt';
import { MemorySaver } from '@langchain/langgraph';
import { DynamicStructuredTool, DynamicTool } from '@langchain/core/tools';
import { z } from 'zod';
import { spawn, ChildProcess } from 'child_process';
import * as fs from 'fs';
import * as path from 'path';
import * as os from 'os';
import { ChatAnthropic } from '@langchain/anthropic';
import { tool } from '@langchain/core/tools';
import { getGlobalMCPClient } from '@/lib/mcpClient';

// Ollama 직접 API 호출을 위한 인터페이스들
interface OllamaMessage {
  role: 'user' | 'assistant' | 'system';
  content: string | any[];
}

interface OllamaTool {
  type: 'function';
  function: {
    name: string;
    description: string;
    parameters: any;
  };
}

interface OllamaResponse {
  message: {
    role: string;
    content: string;
    tool_calls?: Array<{
      id?: string;
      type: 'function';
      function: {
        name: string;
        arguments: string | any;
      };
    }>;
  };
  done: boolean;
}

// 전역 MCP 프로세스 관리
const mcpProcesses = new Map<string, ChildProcess>();

// 도구-서버 매핑 (Ollama 직접 API 호출용) - 개선된 매핑 관리
const toolToServerMapping = new Map<string, string>();

// 서버별 도구 정보 캐시
const serverToolsCache = new Map<string, any[]>();

// 도구-서버 매핑 초기화 함수
async function initializeToolServerMapping() {
  console.log('🔄 도구-서버 매핑 초기화 시작...');
  
  try {
    // 전역 MCP 클라이언트 사용 (중복 초기화 방지)
    const client = await getGlobalMCPClient();
    
    if (!client) {
      console.warn('⚠️ 전역 MCP 클라이언트를 가져올 수 없습니다 - 기본 매핑 사용');
      return;
    }
    
    console.log(`🔄 기존 전역 MCP 클라이언트 재사용`);
    
    // 각 서버별로 도구 목록 가져오기
    const allTools = await client.getAllTools();
    
    // 도구-서버 매핑 업데이트
    toolToServerMapping.clear();
    for (const tool of allTools) {
      // 도구 이름으로 서버 찾기 (getAllTools에서 이미 매핑됨)
      const serverName = tool.serverName || 'unknown';
      toolToServerMapping.set(tool.name, serverName);
      console.log(`📍 도구 매핑 확인: ${tool.name} -> ${serverName}`);
    }
    
    console.log(`✅ 도구-서버 매핑 완료: ${toolToServerMapping.size}개 도구`);
    console.log('📋 매핑된 도구 목록:', Array.from(toolToServerMapping.entries()));
    
  } catch (error) {
    console.error('❌ 도구-서버 매핑 초기화 실패:', error);
  }
}

// MCP 도구들로부터 동적 시스템 프롬프트 생성
function generateDynamicSystemPrompt(mcpTools: any[]): string {
  console.log(`📝 ${mcpTools.length}개 MCP 도구로부터 동적 시스템 프롬프트 생성 중...`);
  
  // 현재 일시 정보 생성 (한국 시간대)
  const now = new Date();
  const kstTime = new Date(now.getTime() + (9 * 60 * 60 * 1000)); // UTC+9 (KST)
  const currentDateTime = kstTime.toLocaleString('ko-KR', {
    year: 'numeric',
    month: '2-digit',
    day: '2-digit',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
    weekday: 'long',
    timeZone: 'Asia/Seoul'
  });
  
  // 핵심 도구들 우선 식별
  const webSearchTool = mcpTools.find(t => t.name === 'web_search');
  const ragSearchTool = mcpTools.find(t => t.name === 'rag_search');
  const queryTool = mcpTools.find(t => t.name === 'query');
  const readFileTool = mcpTools.find(t => t.name === 'read_file');
  const writeFileTool = mcpTools.find(t => t.name === 'write_file');
  const executeCommandTool = mcpTools.find(t => t.name === 'execute_command');
  
  // prompt.template.chat.ollama 기반 강력한 도구 호출 시스템 프롬프트 생성
  let systemPrompt = `SYSTEM: 당신은 검색 증강 생성(RAG), 실시간 웹 검색, 도구 기반 작업 실행이 가능한 지능형 AI 어시스턴트입니다.

📅 **현재 일시**: ${currentDateTime} (한국표준시)

**중요**: 사용자가 "오늘", "현재", "지금", "최근" 등의 시간 관련 표현을 사용할 때는 위의 현재 일시를 기준으로 판단하세요.

🚨 **중요: Ollama Structured Outputs 형식**
당신은 반드시 다음 JSON 형식으로만 응답해야 합니다:

**도구 호출이 필요한 경우:**
{
  "content": "1번 민원을 조회하겠습니다.",
  "tool_calls": [
    {
      "id": "call_1",
      "type": "function",
      "function": {
        "name": "get_complaints",
        "arguments": {}
      }
    }
  ]
}

**일반 응답인 경우:**
{
  "content": "안녕하세요! 무엇을 도와드릴까요?",
  "tool_calls": []
}

**필수 규칙:**
1. 항상 "content" 필드에 사용자에게 보여줄 텍스트를 포함하세요
2. 도구 호출이 필요하면 "tool_calls" 배열에 추가하세요
3. 도구 호출이 불필요하면 "tool_calls": [] 로 설정하세요
4. 절대 content 필드를 비우지 마세요

## 🎯 기본 원칙

1. **언어**: 항상 자연스럽고 유창한 한국어로 응답
2. **사고 과정**: 복잡한 질문에는 단계별 접근 (문제 이해 → 정보 분석 → 해결 전략 → 결론)
3. **정확성**: 검색 결과에 없는 정보는 절대 창작하지 않음

## 🔥 도구 호출 규칙 (최우선)

### 필수 도구 호출 시나리오
| 요청 유형 | 키워드 예시 | 도구 |
|----------|------------|------|
| 웹 검색 | "웹에서 검색", "인터넷에서", "최신 정보", "뉴스" | web_search |
| 문서 검색 | "문서에서 찾아봐", "파일에서", "업로드된 문서" | rag_search |
| 파일 읽기 | "파일 읽어줘", "파일 내용", "파일 확인" | read_file |
| 파일 쓰기 | "파일 저장", "파일 생성", "파일 쓰기" | write_file |
| 명령 실행 | "명령어 실행", "시스템 명령", "터미널" | execute_command |
| 데이터베이스 | "데이터 조회", "SQL", "쿼리" | query |

### 🧠 도구 호출 판단 기준
**중요**: 사용자가 위 키워드를 언급하면 **반드시** 해당 도구를 호출하세요.

**예시:**
- "웹에서 오늘 'AI 관련 최신 뉴스'를 검색해주세요" → **즉시 web_search 호출**
- "문서에서 신혼부부 건강검진 정보를 찾아주세요" → **즉시 rag_search 호출**

## 📋 사용 가능한 도구 목록

### 🌐 web_search (웹 검색) - 최우선 도구
- **목적**: 실시간 웹 검색으로 최신 정보 수집
- **필수 파라미터**: {"query": "검색어"} - query는 절대 빈 문자열이면 안됨!
- **올바른 예시**: 
  * web_search({"query": "AI 관련 최신 뉴스"})
  * web_search({"query": "2024년 인공지능 동향"})
  * web_search({"query": "오늘 기술 뉴스"})
- **잘못된 예시**: web_search({}) ❌, web_search({"query": ""}) ❌

### 📚 rag_search (문서 검색) - 문서 전용 도구
- **목적**: 업로드된 문서에서 정보 검색
- **필수 파라미터**: {"query": "검색어"} - query는 절대 빈 문자열이면 안됨!
- **올바른 예시**: 
  * rag_search({"query": "신혼부부 건강검진비 지원"})
  * rag_search({"query": "문서 요약"})
  * rag_search({"query": "주요 내용"})
- **잘못된 예시**: rag_search({}) ❌, rag_search({"query": ""}) ❌

### 📋 get_complaints (민원 검색) - 고급 검색 도구
- **목적**: 민원 목록 조회 및 검색
- **필수 호출 형식**: 반드시 JSON 객체로 파라미터 전달!

🚨 **중요**: 사용자가 "2025년 1월 14일 이후" 같은 날짜 조건을 말하면:
→ 반드시 get_complaints({"date_from": "2025-01-14"}) 형태로 호출!
→ get_complaints("2025-01-14") 같은 단순 문자열 전달 절대 금지!

#### ✅ 올바른 호출 예시:
- 전체 민원 조회: 빈 객체 사용
- 날짜별 검색: date_from 파라미터에 YYYY-MM-DD 형식 날짜 전달
- 기간 검색: date_from과 date_to 모두 사용
- 카테고리 검색: category 파라미터 사용 (소음/진동, 도로/교통, 환경/위생 등)
- 키워드 검색: keyword 파라미터 사용
- 상태별 검색: status 파라미터 사용 (pending, in_progress, resolved)
- 복합 검색: 여러 파라미터 조합 사용

#### ❌ 잘못된 호출 예시:
- 콜론 구분자 사용 금지 (date_from:2025-01-14 같은 형태)
- 단순 문자열 전달 금지
- 빈 파라미터 사용 금지 (반드시 빈 객체라도 전달)

**절대 규칙**: 
1. get_complaints는 반드시 JSON 객체로만 호출하세요!
2. 날짜 검색 시: date_from 또는 date_to 키를 사용하고 값은 YYYY-MM-DD 형식!
3. 키워드 검색 시: keyword 키 사용!
4. 카테고리 검색 시: category 키 사용!
5. 빈 파라미터일 때도 반드시 빈 객체 전달!
6. 콜론(:) 구분자나 문자열 직접 전달 절대 금지!

예: 2025년 1월 14일 이후 민원 검색 → date_from 키에 "2025-01-14" 값 전달

### 📖 read_file (파일 읽기)
- **목적**: 파일 내용 읽기
- **파라미터**: {"path": "파일경로"}
- **예시**: read_file({"path": "/path/to/file.txt"})

### 📝 write_file (파일 쓰기)
- **목적**: 파일 생성 및 저장
- **파라미터**: {"path": "파일경로", "content": "내용"}
- **예시**: write_file({"path": "/path/to/file.txt", "content": "파일 내용"})

### 💻 execute_command (명령어 실행)
- **목적**: 시스템 명령어 실행
- **파라미터**: {"command": "실행할 명령어"}
- **예시**: execute_command({"command": "ls -la"})

### 🗃️ query (데이터베이스 쿼리)
- **목적**: SQL 쿼리 실행
- **파라미터**: {"sql": "SQL문"}
- **예시**: query({"sql": "SELECT * FROM users LIMIT 10"})

### 🎨 create_mermaid_diagram (다이어그램 생성)
- **목적**: Mermaid 다이어그램 생성
- **필수 파라미터**: {"content": "다이어그램 내용", "output_path": "저장경로"}
- **올바른 예시**: 
  * create_mermaid_diagram({"content": "graph TD; A-->B;", "output_path": "/tmp/diagram.png"})
  * create_mermaid_diagram({"content": "flowchart LR; Start-->End;", "output_path": "./output/flow.png"})
- **잘못된 예시**: create_mermaid_diagram({"content": "graph TD; A-->B;"}) ❌ (output_path 누락)

## 🔧 MCP 도구 호출 형식 (간단하고 자연스럽게!)

**중요**: MCP 도구는 **간단한 위치 기반 파라미터**로 호출하세요!

### ✅ 권장 호출 형식 (자연스러움):
- 민원 조회: get_complaint(2)
- 민원 상태 변경: update_complaint_status(3, "처리중")  
- 민원 답변 추가: reply_to_complaint(1, "assistant", "답변 내용")

### 🔄 대안 호출 형식 (객체):
- 민원 조회: get_complaint({"id": "2"})
- 민원 상태 변경: update_complaint_status({"id": "3", "status": "처리중"})  
- 민원 답변 추가: reply_to_complaint({"id": "1", "officer_id": "assistant", "reply_content": "답변 내용"})

**기억하세요**: **간단한 형태**를 우선 사용하세요! 함수처럼 자연스럽게 호출하면 됩니다.

## 📋 전체 도구 목록 (${mcpTools.length}개):
${mcpTools.map(tool => {
  const serverName = toolToServerMapping.get(tool.name) || 'unknown';
  return `- ${tool.name} (${serverName}서버): ${tool.description || '설명 없음'}`;
}).join('\n')}

## ⚡ 즉시 실행 지침

사용자가 다음을 요청하면:
1. **웹 검색/Web search** → 즉시 web_search 호출
2. **문서 검색/Document search** → 즉시 rag_search 호출  
3. **파일 작업/File operations** → 즉시 read_file/write_file 호출
4. **명령어 실행/Command execution** → 즉시 execute_command 호출
5. **데이터베이스/Database** → 즉시 query 호출
6. **다이어그램 생성/Diagram creation** → 즉시 create_mermaid_diagram 호출 (output_path 필수!)
7. **민원 검색** → 즉시 get_complaints 호출 (JSON 객체 필수!)

🚨 **민원 검색 특별 규칙**:
- "날짜 이후/이전" → date_from/date_to 키 사용
- "카테고리별" → category 키 사용
- "키워드 검색" → keyword 키 사용
- "상태별" → status 키 사용

**예시**: "2025년 1월 14일 이후 민원" → get_complaints({"date_from": "2025-01-14"})

**중요**: 일반적인 응답을 제공하지 마세요. **도구를 즉시 사용하세요**.

현재 사용자 요청에는 도구 사용이 필요합니다 - 적절한 도구 호출을 지금 실행하세요!

🔥 **특별 지시**: 사용자가 "2025년 1월 14일 이후에 접수된 민원을 보여주세요"라고 하면:
→ 반드시 get_complaints({"date_from": "2025-01-14"})로 호출하세요!
→ get_complaints("2025-01-14")는 절대 금지입니다!

## ⚠️ 절대 금지 사항

1. **도구 호출 없이 일반 답변 제공 금지**: 검색 요청 시 반드시 도구 호출
2. **빈 파라미터 사용 금지**: {}, {"query": ""}, {"sql": ""} 등 사용 금지
3. **필수 파라미터 누락 금지**: create_mermaid_diagram의 output_path 등 필수 파라미터 반드시 포함
4. **추측 정보 추가 금지**: 검색 결과에 없는 내용을 추측해서 추가하지 마세요
5. **"검색해드리겠습니다"라고만 말하고 실제 도구 호출하지 않기 금지**

## ✅ 핵심 체크리스트 (반드시 준수!)

1. **도구 호출 최우선** - 검색/작업 요청 시 반드시 먼저 도구 호출!
2. **파라미터 필수 입력** - 빈 객체 {} 또는 빈 문자열 "" 절대 금지!
3. **필수 파라미터 확인** - 각 도구의 필수 파라미터를 반드시 포함!
4. **한국어 응답** 필수
5. **사실 기반** - 검색 결과에 없는 정보는 창작하지 않음

🔥 **즉시 실행 원칙**: 
- "웹에서 검색" → 즉시 web_search({"query": "구체적인검색어"}) 호출
- "문서에서 찾기" → 즉시 rag_search({"query": "구체적인검색어"}) 호출
- "다이어그램 생성" → 즉시 create_mermaid_diagram({"content": "내용", "output_path": "경로"}) 호출
- 절대로 "검색해드리겠습니다"라고만 말하지 마세요!

**기억하세요**: 광범위한 답변보다는 질의에 정확히 맞는 답변이 더 가치 있습니다. 도구를 적극적으로 사용하여 정확한 정보를 제공하세요.`;

  return systemPrompt;
}

// MCP 스키마에 맞게 도구 인자 검증 및 변환 - 개선된 버전
function validateAndTransformArgs(args: any, inputSchema: any, toolName: string): any {
  if (!inputSchema || !inputSchema.properties) {
    console.log(`⚠️ ${toolName}: 스키마가 없거나 properties가 없음, 원본 인자 반환`);
    return args;
  }

  const transformed: any = {};
  const properties = inputSchema.properties;
  const required = inputSchema.required || [];

  console.log(`🔧 ${toolName} 스키마 검증 시작:`, { 
    properties: Object.keys(properties), 
    required,
    inputArgs: args 
  });

  // 필수 필드 확인 및 변환
  for (const [key, schema] of Object.entries(properties)) {
    const propSchema = schema as any;
    
    if (args.hasOwnProperty(key)) {
      // 값이 있는 경우 타입 변환
      let value = args[key];
      
      // 빈 문자열이나 null 체크
      if (value === '' || value === null || value === undefined) {
        if (required.includes(key)) {
          console.error(`❌ ${toolName}.${key}: 필수 파라미터가 비어있음`);
          
          // 기본값 제공 시도
          if (key === 'output_path' && toolName === 'create_mermaid_diagram') {
            value = `/tmp/diagram_${Date.now()}.png`;
            console.log(`🔧 ${toolName}.${key}: 기본 경로 설정 -> ${value}`);
          } else if (key === 'query' && (toolName === 'web_search' || toolName === 'rag_search')) {
            throw new Error(`${toolName}의 query 파라미터는 필수이며 빈 문자열일 수 없습니다.`);
          } else {
            throw new Error(`${toolName}의 ${key} 파라미터는 필수입니다.`);
          }
        } else {
          continue; // 선택적 파라미터는 건너뛰기
        }
      }
      
      // 타입 변환
      if (propSchema.type === 'string' && typeof value !== 'string') {
        value = String(value);
      } else if (propSchema.type === 'number' && typeof value !== 'number') {
        const num = Number(value);
        if (!isNaN(num)) {
          value = num;
        }
      } else if (propSchema.type === 'boolean' && typeof value !== 'boolean') {
        value = Boolean(value);
      } else if (propSchema.type === 'array' && !Array.isArray(value)) {
        if (typeof value === 'string') {
          try {
            value = JSON.parse(value);
          } catch {
            value = [value];
          }
        } else {
          value = [value];
        }
      } else if (propSchema.type === 'object' && typeof value !== 'object') {
        if (typeof value === 'string') {
          try {
            value = JSON.parse(value);
          } catch {
            value = { value };
          }
        }
      }
      
      transformed[key] = value;
      console.log(`✅ ${toolName}.${key}: ${typeof args[key]} -> ${typeof value}`, value);
    } else if (required.includes(key)) {
      // 필수 파라미터가 누락된 경우
      console.error(`❌ ${toolName}: 필수 파라미터 누락 - ${key}`);
      
      // 기본값 제공 시도
      if (key === 'output_path' && toolName === 'create_mermaid_diagram') {
        transformed[key] = `/tmp/diagram_${Date.now()}.png`;
        console.log(`🔧 ${toolName}.${key}: 기본 경로 설정 -> ${transformed[key]}`);
      } else {
        throw new Error(`${toolName}의 필수 파라미터 ${key}가 누락되었습니다.`);
      }
    }
  }

  // 추가 파라미터 처리 (스키마에 없는 파라미터)
  for (const [key, value] of Object.entries(args)) {
    if (!properties.hasOwnProperty(key)) {
      console.log(`⚠️ ${toolName}: 스키마에 없는 파라미터 ${key} 무시`);
    }
  }

  console.log(`🎯 ${toolName} 최종 변환된 인자:`, transformed);
  return transformed;
}

// MCP 도구를 Ollama 형식으로 변환 (dolphin-mcp 스타일)

function convertMCPToolsToOllamaFormat(mcpTools: any[]): OllamaTool[] {
  console.log(`🔄 MCP 도구 ${mcpTools.length}개를 Ollama 형식으로 변환 중...`);
  
  const ollamaTools: OllamaTool[] = [];
  
  for (const mcpTool of mcpTools) {
    try {
      // DynamicStructuredTool에서 정보 추출
      const toolName = mcpTool.name;
      const toolDescription = mcpTool.description || `${toolName} 도구`;
      
      // 스키마에서 파라미터 정보 추출
      let parameters: {
        type: string;
        properties: any;
        required: string[];
      } = {
        type: 'object',
        properties: {},
        required: []
      };
      
      if (mcpTool.schema && mcpTool.schema.shape) {
        // Zod 스키마에서 파라미터 추출
        const shape = mcpTool.schema.shape;
        const properties: any = {};
        const required: string[] = [];
        
        for (const [key, value] of Object.entries(shape)) {
          if (value && typeof value === 'object') {
            const zodType = value as any;
            
            // Zod 타입에 따른 JSON Schema 생성
            if (zodType._def) {
              const def = zodType._def;
              
              if (def.typeName === 'ZodString') {
                properties[key] = {
                  type: 'string',
                  description: def.description || `${key} 파라미터`
                };
              } else if (def.typeName === 'ZodNumber') {
                properties[key] = {
                  type: 'number',
                  description: def.description || `${key} 파라미터`
                };
              } else if (def.typeName === 'ZodBoolean') {
                properties[key] = {
                  type: 'boolean',
                  description: def.description || `${key} 파라미터`
                };
              } else if (def.typeName === 'ZodArray') {
                properties[key] = {
                  type: 'array',
                  description: def.description || `${key} 파라미터`
                };
              } else {
                properties[key] = {
                  type: 'string',
                  description: def.description || `${key} 파라미터`
                };
              }
              
              // 필수 필드 체크
              if (!def.isOptional) {
                required.push(key);
              }
            }
          }
        }
        
        parameters = {
          type: 'object',
          properties,
          required
        };
      }
      
      // 도구별 명확한 설명 추가
      let enhancedDescription = toolDescription;
      if (toolName === 'web_search') {
        enhancedDescription = `실시간 웹 검색 도구. 반드시 {"query": "구체적인검색어"} 형식으로 호출하세요. 빈 객체나 빈 문자열 금지!`;
      } else if (toolName === 'rag_search') {
        enhancedDescription = `문서 검색 도구. 반드시 {"query": "구체적인검색어"} 형식으로 호출하세요. 빈 객체나 빈 문자열 금지!`;
      }

      const ollamaTool: OllamaTool = {
        type: 'function',
        function: {
          name: toolName,
          description: enhancedDescription,
          parameters: parameters
        }
      };
      
      ollamaTools.push(ollamaTool);
      console.log(`✅ 도구 변환 완료: ${toolName}`);
      
    } catch (error) {
      console.error(`❌ 도구 변환 실패: ${mcpTool.name}`, error);
    }
  }
  
  console.log(`🔧 총 ${ollamaTools.length}개 도구를 Ollama 형식으로 변환 완료`);
  return ollamaTools;
}

// Ollama API 직접 호출 (dolphin-mcp 스타일)
async function callOllamaDirectly(
  messages: OllamaMessage[],
  tools: OllamaTool[],
  model: string,
  baseURL: string,
  temperature: number = 0.7
): Promise<AsyncGenerator<string, void, unknown>> {
  console.log(`🦙 Ollama 직접 API 호출: ${baseURL}/api/chat`);
  console.log(`📝 메시지 수: ${messages.length}, 도구 수: ${tools.length}`);
  
  // Ollama Structured Outputs를 위한 JSON 스키마 정의
  const responseSchema = {
    type: 'object',
    properties: {
      content: {
        type: 'string',
        description: '사용자에게 전달할 텍스트 응답'
      },
      tool_calls: {
        type: 'array',
        description: '호출할 도구들의 목록',
        items: {
          type: 'object',
          properties: {
            id: {
              type: 'string',
              description: '도구 호출 고유 ID'
            },
            type: {
              type: 'string',
              enum: ['function'],
              description: '도구 호출 타입'
            },
            function: {
              type: 'object',
              properties: {
                name: {
                  type: 'string',
                  description: '호출할 함수 이름'
                },
                arguments: {
                  type: 'object',
                  description: '함수에 전달할 인수들'
                }
              },
              required: ['name', 'arguments']
            }
          },
          required: ['type', 'function']
        }
      }
    },
    required: ['content']
  };

  const requestBody = {
    model: model,
    messages: messages,
    stream: false, // 완성형 응답으로 변경
    // Ollama Structured Outputs를 위한 format 파라미터 추가
    format: "json",
    // mainLogic.js와 동일하게 tools와 tool_choice를 최상위 레벨에 배치
    ...(tools && tools.length > 0 ? { tools: tools } : {}),
    ...(tools && tools.length > 0 ? { tool_choice: "auto" } : {}),
    options: {
      temperature: temperature,
      num_predict: 4096,
      top_k: 40,
      top_p: 0.9,
      repeat_penalty: 1.1
    }
  };
  
  console.log(`📤 Ollama 요청 데이터:`, {
    model: requestBody.model,
    messageCount: requestBody.messages.length,
    toolCount: requestBody.tools ? requestBody.tools.length : 0,
    options: requestBody.options
  });
  
  return (async function* () {
    try {
      const response = await fetch(`${baseURL}/api/chat`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(requestBody),
      });
      
      if (!response.ok) {
        let errorMessage = `Ollama API 호출 실패: ${response.status} ${response.statusText}`;
        try {
          const errorText = await response.text();
          console.error('🔍 Ollama 오류 응답:', errorText);
          errorMessage += ` - ${errorText}`;
        } catch (e) {
          console.error('🔍 Ollama 오류 응답 읽기 실패:', e);
        }
        throw new Error(errorMessage);
      }
      
      // 완성형 응답 처리
      const responseData = await response.json();
      console.log(`📤 Ollama 완성형 응답:`, responseData);
      
      if (responseData.message && responseData.message.content) {
        let messageContent = responseData.message.content;
        
        // Structured Outputs JSON 파싱 시도
        try {
          const structuredResponse = JSON.parse(messageContent);
          console.log(`✅ 구조화된 JSON 응답 파싱 성공:`, structuredResponse);
          
          // 직접 도구 호출 형식 감지 (Ollama가 올바른 형식을 따르지 않는 경우)
          if (structuredResponse.name && structuredResponse.arguments) {
            console.log(`🔧 직접 도구 호출 형식 감지: ${structuredResponse.name}`);
            
            // 사용자에게 도구 호출 시작을 알림
            yield `data: ${JSON.stringify({ content: `${structuredResponse.name} 도구를 실행합니다...` })}\n\n`;
            
            // 도구 호출 시작 이벤트
            yield `${JSON.stringify({ 
              type: 'tool_call_start', 
              toolName: structuredResponse.name,
              args: structuredResponse.arguments,
              server: toolToServerMapping.get(structuredResponse.name) || 'unknown'
            })}\n`;
            
            try {
              // MCP 도구 실행
              const mcpTools = await loadMCPTools();
              const mcpTool = mcpTools.find(tool => tool.name === structuredResponse.name);
              
              if (mcpTool) {
                let args = structuredResponse.arguments;
                console.log(`🔧 직접 호출 원본 도구 인자:`, args);
                console.log(`🔧 직접 호출 MCP 도구 스키마:`, mcpTool.inputSchema);
                
                // MCP 스키마에 맞게 인자 변환
                args = validateAndTransformArgs(args, mcpTool.inputSchema, structuredResponse.name);
                console.log(`🔧 직접 호출 변환된 도구 인자:`, args);
                
                const result = await mcpTool.invoke(args);
                
                // 도구 호출 완료 이벤트 (서버 정보 포함)
                yield `${JSON.stringify({ 
                  type: 'tool_call_end', 
                  toolName: structuredResponse.name,
                  result: result,
                  server: toolToServerMapping.get(structuredResponse.name) || 'unknown'
                })}\n`;
                
                // 도구 결과를 텍스트로도 표시
                yield `data: ${JSON.stringify({ content: `\n✅ ${structuredResponse.name} 완료\n` })}\n\n`;
                
              } else {
                throw new Error(`도구를 찾을 수 없습니다: ${structuredResponse.name}`);
              }
              
            } catch (toolError) {
              console.error(`❌ 직접 호출 도구 실행 실패: ${structuredResponse.name}`, toolError);
              console.error(`❌ 직접 호출 실패한 도구 인자:`, structuredResponse.arguments);
              if (toolError instanceof Error) {
                console.error(`❌ 직접 호출 오류 스택:`, toolError.stack);
              }
              
              // 도구 호출 오류 이벤트 (서버 정보 포함)
              yield `${JSON.stringify({ 
                type: 'tool_call_error', 
                toolName: structuredResponse.name,
                error: toolError instanceof Error ? toolError.message : String(toolError),
                args: structuredResponse.arguments,
                server: toolToServerMapping.get(structuredResponse.name) || 'unknown'
              })}\n`;
              
              // 오류를 텍스트로도 표시
              yield `data: ${JSON.stringify({ content: `\n❌ ${structuredResponse.name} 실패: ${toolError instanceof Error ? toolError.message : String(toolError)}\n` })}\n\n`;
            }
            
          }
          
          // 구조화된 응답에서 tool_calls 처리 (우선순위)
          if (structuredResponse.tool_calls && Array.isArray(structuredResponse.tool_calls) && structuredResponse.tool_calls.length > 0) {
            console.log(`🔧 구조화된 도구 호출 ${structuredResponse.tool_calls.length}개 감지`);
            
            const toolResults: any[] = [];
            
            for (const toolCall of structuredResponse.tool_calls) {
              console.log(`🔧 Ollama 도구 호출: ${toolCall.function.name}`);
              
              // 도구 호출 시작 이벤트
              yield `${JSON.stringify({ 
                type: 'tool_call_start', 
                toolName: toolCall.function.name,
                args: toolCall.function.arguments,
                server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
              })}\n`;
              
              try {
                // MCP 도구 실행
                const mcpTools = await loadMCPTools();
                const mcpTool = mcpTools.find(tool => tool.name === toolCall.function.name);
                
                if (mcpTool) {
                  let args = toolCall.function.arguments;
                  console.log(`🔧 구조화된 원본 도구 인자:`, args);
                  console.log(`🔧 구조화된 MCP 도구 스키마:`, mcpTool.inputSchema);
                  
                  // MCP 스키마에 맞게 인자 변환
                  args = validateAndTransformArgs(args, mcpTool.inputSchema, toolCall.function.name);
                  console.log(`🔧 구조화된 변환된 도구 인자:`, args);
                  
                  const result = await mcpTool.invoke(args);
                  
                  // 도구 결과 저장 (최종 응답 생성용)
                  toolResults.push({
                    name: toolCall.function.name,
                    args: args,
                    result: result
                  });
                  
                  // 도구 호출 완료 이벤트 (서버 정보 포함)
                  yield `${JSON.stringify({ 
                    type: 'tool_call_end', 
                    toolName: toolCall.function.name,
                    result: result,
                    server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
                  })}\n`;
                  
                  // 도구 결과를 텍스트로도 표시
                  yield `data: ${JSON.stringify({ content: `\n✅ ${toolCall.function.name} 완료\n` })}\n\n`;
                  
                } else {
                  throw new Error(`도구를 찾을 수 없습니다: ${toolCall.function.name}`);
                }
                
              } catch (toolError) {
                console.error(`❌ 구조화된 도구 실행 실패: ${toolCall.function.name}`, toolError);
                console.error(`❌ 구조화된 실패한 도구 인자:`, toolCall.function.arguments);
                if (toolError instanceof Error) {
                  console.error(`❌ 구조화된 오류 스택:`, toolError.stack);
                }
                
                // 오류 결과도 저장 (최종 응답 생성용)
                toolResults.push({
                  name: toolCall.function.name,
                  args: toolCall.function.arguments,
                  error: toolError instanceof Error ? toolError.message : String(toolError)
                });
                
                // 도구 호출 오류 이벤트 (서버 정보 포함)
                yield `${JSON.stringify({ 
                  type: 'tool_call_error', 
                  toolName: toolCall.function.name,
                  error: toolError instanceof Error ? toolError.message : String(toolError),
                  args: toolCall.function.arguments,
                  server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
                })}\n`;
                
                // 오류를 텍스트로도 표시
                yield `data: ${JSON.stringify({ content: `\n❌ ${toolCall.function.name} 실패: ${toolError instanceof Error ? toolError.message : String(toolError)}\n` })}\n\n`;
              }
            }
            
            // 도구 실행 완료 후 최종 응답 생성
            if (toolResults.length > 0) {
              console.log(`🔄 도구 실행 완료, 최종 응답 생성 중...`);
              
              // 도구 결과를 포함한 메시지 생성
              const toolResultsText = toolResults.map(result => {
                if (result.error) {
                  return `도구 "${result.name}" 실행 오류: ${result.error}`;
                } else {
                  return `도구 "${result.name}" 실행 결과:\n${JSON.stringify(result.result, null, 2)}`;
                }
              }).join('\n\n');
              
              // 원래 사용자 메시지 찾기 (시스템 프롬프트 제외)
              const userMessages = messages.filter(msg => msg.role === 'user');
              const originalUserMessage = userMessages[userMessages.length - 1]?.content || '요청';
              
              // 최종 응답 생성을 위한 새로운 메시지 배열 (시스템 프롬프트 없이)
              const followUpMessages = [
                {
                  role: 'user' as const,
                  content: `사용자 질문: "${originalUserMessage}"\n\n도구 실행 결과:\n${toolResultsText}\n\n위 도구 실행 결과를 바탕으로 사용자 질문에 대해 도움이 되는 자연스러운 응답을 작성해주세요.`
                }
              ];
              
              // 최종 응답 생성 요청 (JSON 형식 제거, 도구 없이)
              const followUpRequest = {
                model: model,
                messages: followUpMessages,
                stream: false,
                options: {
                  temperature: temperature,
                  num_predict: 4096,
                  top_k: 40,
                  top_p: 0.9,
                  repeat_penalty: 1.1
                }
              };
              
              console.log(`📤 최종 응답 생성 요청...`);
              
              try {
                const followUpResponse = await fetch(`${baseURL}/api/chat`, {
                  method: 'POST',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                  body: JSON.stringify(followUpRequest),
                });
                
                if (followUpResponse.ok) {
                  const followUpData = await followUpResponse.json();
                  console.log(`📤 최종 응답 수신:`, followUpData);
                  
                  if (followUpData.message && followUpData.message.content) {
                    // JSON 형식이 아니므로 바로 텍스트로 처리
                    yield `data: ${JSON.stringify({ content: `\n${followUpData.message.content}` })}\n\n`;
                  }
                } else {
                  console.error('❌ 최종 응답 생성 실패:', followUpResponse.status);
                  yield `data: ${JSON.stringify({ content: '\n도구 실행이 완료되었습니다.' })}\n\n`;
                }
              } catch (followUpError) {
                console.error('❌ 최종 응답 생성 오류:', followUpError);
                yield `data: ${JSON.stringify({ content: '\n도구 실행이 완료되었습니다.' })}\n\n`;
              }
            }
          } else if (structuredResponse.content) {
            // tool_calls가 없고 정상적인 content 필드가 있는 경우만 출력
            console.log(`📝 구조화된 텍스트 응답 (도구 호출 없음): ${structuredResponse.content.substring(0, 100)}...`);
            yield `data: ${JSON.stringify({ content: structuredResponse.content })}\n\n`;
          } else {
            // tool_calls도 없고 content도 없는 경우 기본 메시지 제공
            console.log(`⚠️ tool_calls와 content 필드가 모두 없습니다. 기본 응답을 제공합니다.`);
            yield `data: ${JSON.stringify({ content: "요청을 처리했습니다." })}\n\n`;
          }
          
        } catch (structuredParseError) {
          // 구조화된 JSON이 아닌 경우, 기존 방식의 tool_calls가 없을 때만 일반 텍스트로 처리
          if (!responseData.message.tool_calls || responseData.message.tool_calls.length === 0) {
            console.log(`📝 일반 텍스트 응답 (도구 호출 없음): ${messageContent.substring(0, 100)}...`);
            yield `data: ${JSON.stringify({ content: messageContent })}\n\n`;
          } else {
            console.log(`🔧 구조화된 JSON 파싱 실패, 기존 방식 tool_calls로 처리 예정`);
          }
        }
      }
      
      // 기존 방식의 tool_calls 처리 (fallback)
      if (responseData.message && responseData.message.tool_calls) {
        console.log(`🔧 기존 방식 도구 호출 ${responseData.message.tool_calls.length}개 감지`);
        
        const toolResults: any[] = [];
        
        for (const toolCall of responseData.message.tool_calls) {
          console.log(`🔧 Ollama 도구 호출: ${toolCall.function.name}`);
          
          // 도구 호출 시작 이벤트
          yield `${JSON.stringify({ 
            type: 'tool_call_start', 
            toolName: toolCall.function.name,
            args: typeof toolCall.function.arguments === 'string' 
              ? JSON.parse(toolCall.function.arguments) 
              : toolCall.function.arguments,
            server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
          })}\n`;
          
          try {
            // MCP 도구 실행
            console.log(`🔧 MCP 도구 로딩 시작...`);
            const mcpTools = await loadMCPTools();
            const mcpTool = mcpTools.find(tool => tool.name === toolCall.function.name);
            
            if (mcpTool) {
              let args = typeof toolCall.function.arguments === 'string' 
                ? JSON.parse(toolCall.function.arguments) 
                : toolCall.function.arguments;
              
              console.log(`🔧 원본 도구 인자:`, args);
              console.log(`🔧 MCP 도구 스키마:`, mcpTool.inputSchema);
              
              // MCP 스키마에 맞게 인자 변환
              args = validateAndTransformArgs(args, mcpTool.inputSchema, toolCall.function.name);
              console.log(`🔧 변환된 도구 인자:`, args);
              
              const result = await mcpTool.invoke(args);
              
              // 도구 결과 저장 (최종 응답 생성용)
              toolResults.push({
                name: toolCall.function.name,
                args: args,
                result: result
              });
              
              // 도구 호출 완료 이벤트 (서버 정보 포함)
              yield `${JSON.stringify({ 
                type: 'tool_call_end', 
                toolName: toolCall.function.name,
                result: result,
                server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
              })}\n`;
              
              // 도구 결과를 텍스트로도 표시
              yield `data: ${JSON.stringify({ content: `\n✅ ${toolCall.function.name} 완료\n` })}\n\n`;
              
            } else {
              throw new Error(`도구를 찾을 수 없습니다: ${toolCall.function.name}`);
            }
            
          } catch (toolError) {
            console.error(`❌ 도구 실행 실패: ${toolCall.function.name}`, toolError);
            console.error(`❌ 실패한 도구 인자:`, toolCall.function.arguments);
            if (toolError instanceof Error) {
              console.error(`❌ 오류 스택:`, toolError.stack);
            }
            
            // 오류 결과도 저장 (최종 응답 생성용)
            toolResults.push({
              name: toolCall.function.name,
              args: toolCall.function.arguments,
              error: toolError instanceof Error ? toolError.message : String(toolError)
            });
            
            // 도구 호출 오류 이벤트 (서버 정보 포함)
            yield `${JSON.stringify({ 
              type: 'tool_call_error', 
              toolName: toolCall.function.name,
              error: toolError instanceof Error ? toolError.message : String(toolError),
              args: toolCall.function.arguments,
              server: toolToServerMapping.get(toolCall.function.name) || 'unknown'
            })}\n`;
            
            // 오류를 텍스트로도 표시
            yield `data: ${JSON.stringify({ content: `\n❌ ${toolCall.function.name} 실패: ${toolError instanceof Error ? toolError.message : String(toolError)}\n` })}\n\n`;
          }
        }
        
        // 도구 실행 완료 후 최종 응답 생성
        if (toolResults.length > 0) {
          console.log(`🔄 기존 방식 도구 실행 완료, 최종 응답 생성 중...`);
          
          // 도구 결과를 포함한 메시지 생성
          const toolResultsText = toolResults.map(result => {
            if (result.error) {
              return `도구 "${result.name}" 실행 오류: ${result.error}`;
            } else {
              return `도구 "${result.name}" 실행 결과:\n${JSON.stringify(result.result, null, 2)}`;
            }
          }).join('\n\n');
          
          // 원래 사용자 메시지 찾기 (시스템 프롬프트 제외)
          const userMessages = messages.filter(msg => msg.role === 'user');
          const originalUserMessage = userMessages[userMessages.length - 1]?.content || '요청';
          
          // 최종 응답 생성을 위한 새로운 메시지 배열 (시스템 프롬프트 없이)
          const followUpMessages = [
            {
              role: 'user' as const,
              content: `사용자 질문: "${originalUserMessage}"\n\n도구 실행 결과:\n${toolResultsText}\n\n위 도구 실행 결과를 바탕으로 사용자 질문에 대해 도움이 되는 자연스러운 응답을 작성해주세요.`
            }
          ];
          
          // 최종 응답 생성 요청 (JSON 형식 제거, 도구 없이)
          const followUpRequest = {
            model: model,
            messages: followUpMessages,
            stream: false,
            options: {
              temperature: temperature,
              num_predict: 4096,
              top_k: 40,
              top_p: 0.9,
              repeat_penalty: 1.1
            }
          };
          
          console.log(`📤 기존 방식 최종 응답 생성 요청...`);
          
          try {
            const followUpResponse = await fetch(`${baseURL}/api/chat`, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify(followUpRequest),
            });
            
            if (followUpResponse.ok) {
              const followUpData = await followUpResponse.json();
              console.log(`📤 기존 방식 최종 응답 수신:`, followUpData);
              
              if (followUpData.message && followUpData.message.content) {
                // JSON 형식이 아니므로 바로 텍스트로 처리
                yield `data: ${JSON.stringify({ content: `\n${followUpData.message.content}` })}\n\n`;
              }
            } else {
              console.error('❌ 기존 방식 최종 응답 생성 실패:', followUpResponse.status);
              yield `data: ${JSON.stringify({ content: '\n도구 실행이 완료되었습니다.' })}\n\n`;
            }
          } catch (followUpError) {
            console.error('❌ 기존 방식 최종 응답 생성 오류:', followUpError);
            yield `data: ${JSON.stringify({ content: '\n도구 실행이 완료되었습니다.' })}\n\n`;
          }
        }
      }
      
    } catch (error) {
      console.error('❌ Ollama API 호출 오류:', error);
      yield `data: ${JSON.stringify({ content: `❌ 오류가 발생했습니다: ${error instanceof Error ? error.message : String(error)}` })}\n\n`;
    }
  })();
}

// 설정 파일 경로
const getAirunConfigPath = () => {
  const homeDir = os.homedir();
  return path.join(homeDir, '.airun', 'airun.conf');
};

const getMCPConfigPath = () => {
  return path.join(process.cwd(), 'mcp_config.json');
};

// airun.conf에서 설정 읽기
const getAirunConfig = () => {
  const configPath = getAirunConfigPath();
  const config: any = {};
  
  // 1. process.env에서 환경변수 우선 로드 (Next.js 환경)
  for (const [key, value] of Object.entries(process.env)) {
    if (value) {
      config[key] = value;
    }
  }
  
  // 2. airun.conf 파일에서 추가 설정 로드 (있는 경우)
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            // process.env에 있는 값을 덮어쓰지 않음 (process.env 우선)
            if (!config[match[1]]) {
              config[match[1]] = match[2];
            }
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  // 환경변수 로드 상태 로그
  console.log('🔧 환경변수 로드 상태:', {
    NEXT_PUBLIC_OLLAMA_PROXY_SERVER: config.NEXT_PUBLIC_OLLAMA_PROXY_SERVER ? '✅ 설정됨' : '❌ 없음',
    OLLAMA_BASE_URL: config.OLLAMA_BASE_URL ? '✅ 설정됨' : '❌ 없음',
    OPENAI_API_KEY: config.OPENAI_API_KEY ? '✅ 설정됨' : '❌ 없음',
    ANTHROPIC_API_KEY: config.ANTHROPIC_API_KEY ? '✅ 설정됨' : '❌ 없음'
  });
  
  return config;
};

// MCP 설정 로드
const loadMCPConfig = () => {
  const configPath = getMCPConfigPath();
  const defaultConfig = {
    "hamonize": {
      "command": "node",
      "args": ["./mcp_tools/hamonize/dist/index.js"],
      "transport": "stdio"
    },
    "postgres": {
      "transport": "stdio",
      "command": "node",
      "args": ["./mcp_tools/mcp-postgres-server/build/index.js"]
    }
  };

  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      return JSON.parse(content);
    } catch (error) {
      console.error('MCP 설정 로드 실패:', error);
      return defaultConfig;
    }
  }
  
  return defaultConfig;
};

// 개선된 MCP 클라이언트 클래스 (tools/route.ts에서 가져옴)
class ImprovedMCPClient {
  private serverName: string;
  private config: any;
  private childProcess?: ChildProcess;
  private isInitialized = false;
  private messageId = 1;

  constructor(serverName: string, config: any) {
    this.serverName = serverName;
    this.config = config;
  }

  async initialize(): Promise<boolean> {
    const { command, args, transport, env } = this.config;
    
    try {
      console.log(`🔄 ${this.serverName} MCP 서버 초기화 시작...`);
      
      const spawnEnv = { ...process.env };
      if (env && typeof env === 'object') {
        Object.assign(spawnEnv, env);
      }

      this.childProcess = spawn(command, args, {
        stdio: ['pipe', 'pipe', 'pipe'],
        cwd: process.cwd(),
        env: spawnEnv
      });

      this.childProcess.stderr?.on('data', (data: Buffer) => {
        console.error(`${this.serverName} stderr:`, data.toString());
      });

      // 60초 타임아웃으로 초기화
      const initResult = await Promise.race([
        this.sendRequest('initialize', {
          protocolVersion: '2024-11-05',
          capabilities: {
            roots: { listChanged: true },
            sampling: {}
          },
          clientInfo: {
            name: 'airun-mcp-chat-client',
            version: '1.0.0'
          }
        }),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('초기화 시간 초과')), 60000)
        )
      ]);

      this.isInitialized = true;
      console.log(`✅ ${this.serverName} MCP 서버 초기화 완료`);
      return true;
    } catch (error) {
      console.error(`❌ ${this.serverName} MCP 서버 초기화 실패:`, error);
      this.cleanup();
      return false;
    }
  }

  async listTools(): Promise<any[]> {
    if (!this.isInitialized) {
      console.warn(`${this.serverName}: 서버가 초기화되지 않음`);
      return [];
    }

    try {
      // 30초 타임아웃으로 도구 목록 가져오기
      const response = await Promise.race([
        this.sendRequest('tools/list', {}),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('도구 목록 요청 시간 초과')), 30000)
        )
      ]) as any;

      return Array.isArray(response?.tools) ? response.tools : [];
    } catch (error) {
      console.error(`${this.serverName} 도구 목록 가져오기 실패:`, error);
      return [];
    }
  }

  async callTool(toolName: string, args: any): Promise<any> {
    if (!this.isInitialized) {
      throw new Error(`${this.serverName}: 서버가 초기화되지 않음`);
    }

    try {
      const response = await Promise.race([
        this.sendRequest('tools/call', {
          name: toolName,
          arguments: args
        }),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('도구 호출 시간 초과')), 30000)
        )
      ]) as any;

      return response;
    } catch (error) {
      console.error(`${this.serverName} 도구 호출 실패 (${toolName}):`, error);
      throw error;
    }
  }

  private async sendRequest(method: string, params: any = {}): Promise<any> {
    if (!this.childProcess || !this.childProcess.stdin) {
      throw new Error(`${this.serverName} 서버가 실행 중이지 않습니다.`);
    }

    const request: any = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method
    };

    if (params !== null && params !== undefined) {
      request.params = params;
    }

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error(`${this.serverName} 요청 시간 초과`));
      }, 30000);

      let buffer = '';
      const onData = (data: Buffer) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                this.childProcess?.stdout?.off('data', onData);
                
                if (response.error) {
                  reject(new Error(`${this.serverName}: ${response.error.message}`));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류는 무시하고 계속 진행
            }
          }
        }
      };

      this.childProcess?.stdout?.on('data', onData);
      this.childProcess?.stdin?.write(JSON.stringify(request) + '\n');
    });
  }

  cleanup() {
    if (this.childProcess) {
      try {
        this.childProcess.kill();
        console.log(`${this.serverName} MCP 서버 종료됨`);
      } catch (error) {
        console.error(`${this.serverName} 서버 종료 실패:`, error);
      }
      this.childProcess = undefined;
    }
    this.isInitialized = false;
  }
}

// 개선된 MCP 도구 클라이언트 관리자
class ImprovedMCPToolsClient {
  private clients: Map<string, ImprovedMCPClient> = new Map();

  async initialize(mcpConfig: any): Promise<boolean> {
    console.log('🔄 MCP 서버들 초기화 시작...');
    
    const initPromises = [];
    
    for (const [serverName, config] of Object.entries(mcpConfig)) {
      const initPromise = this.initializeServer(serverName, config as any);
      initPromises.push(initPromise);
    }
    
    try {
      await Promise.race([
        Promise.allSettled(initPromises),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('전체 초기화 시간 초과')), 120000) // 2분 타임아웃
        )
      ]);
      
      // 초기화 완료를 위한 대기 시간
      await new Promise(resolve => setTimeout(resolve, 3000));
      
      console.log(`✅ MCP 초기화 완료. 활성 서버 수: ${this.clients.size}`);
      return this.clients.size > 0;
    } catch (error) {
      console.error('❌ MCP 서버 초기화 중 전체 오류:', error);
      return this.clients.size > 0;
    }
  }

  private async initializeServer(serverName: string, config: any): Promise<void> {
    try {
      const client = new ImprovedMCPClient(serverName, config);
      const success = await client.initialize();
      
      if (success) {
        this.clients.set(serverName, client);
        console.log(`✅ ${serverName} 서버 초기화 성공`);
      } else {
        console.log(`❌ ${serverName} 서버 초기화 실패`);
      }
    } catch (error) {
      console.error(`❌ ${serverName} 서버 초기화 중 오류:`, error);
    }
  }

  async getAllTools(): Promise<any[]> {
    const allTools: any[] = [];
    const usedToolNames = new Set<string>();
    
    for (const [serverName, client] of this.clients) {
      try {
        const tools = await client.listTools();
        
        for (const mcpTool of tools) {
          // 도구 이름 중복 체크
          if (usedToolNames.has(mcpTool.name)) {
            console.warn(`⚠️ 도구 이름 중복: ${mcpTool.name} (${serverName}에서 발견, 첫 번째 도구만 사용)`);
            continue;
          }
          
          usedToolNames.add(mcpTool.name);
          
          // 도구 이름과 서버 이름의 매핑 저장
          toolToServerMapping.set(mcpTool.name, serverName);
          
          const toolFunction = tool(
            async (input: any) => {
              try {
                const result = await client.callTool(mcpTool.name, input);
                
                // MCP 응답에서 텍스트 내용 추출
                if (result && result.content && Array.isArray(result.content)) {
                  const textContent = result.content
                    .filter((item: any) => item && item.type === 'text' && item.text)
                    .map((item: any) => item.text)
                    .join('\n');
                  
                  return textContent || `${mcpTool.name} 도구가 실행되었습니다.`;
                } else if (typeof result === 'string') {
                  return result;
                } else {
                  // 다른 형태의 응답은 JSON 문자열로 반환
                  return JSON.stringify(result);
                }
              } catch (error) {
                console.error(`도구 실행 실패 (${mcpTool.name}):`, error);
                throw error;
              }
            },
            {
              name: mcpTool.name,
              description: mcpTool.description || '설명 없음',
              schema: this.convertInputSchemaToOpenAI(mcpTool.inputSchema)
            }
          );
          
          allTools.push(toolFunction);
        }
      } catch (error) {
        console.error(`${serverName} 도구 목록 가져오기 실패:`, error);
      }
    }
    
    console.log(`✅ 중복 제거 완료: ${allTools.length}개의 고유 도구`);
    return allTools;
  }

  private convertInputSchemaToZod(inputSchema: any): any {
    if (!inputSchema || !inputSchema.properties) {
      return z.object({});
    }

    const zodSchema: any = {};
    
    for (const [key, value] of Object.entries(inputSchema.properties)) {
      const prop = value as any;
      
      if (prop.type === 'string') {
        zodSchema[key] = z.string().describe(prop.description || '');
      } else if (prop.type === 'number') {
        zodSchema[key] = z.number().describe(prop.description || '');
      } else if (prop.type === 'boolean') {
        zodSchema[key] = z.boolean().describe(prop.description || '');
      } else if (prop.type === 'array') {
        // 배열의 경우 items 정보가 있으면 사용, 없으면 any로 설정
        if (prop.items) {
          zodSchema[key] = z.array(this.convertSinglePropertyToZod(prop.items)).describe(prop.description || '');
        } else {
          zodSchema[key] = z.array(z.any()).describe(prop.description || '');
        }
      } else if (prop.type === 'object') {
        if (prop.properties) {
          zodSchema[key] = this.convertInputSchemaToZod(prop).describe(prop.description || '');
        } else {
          zodSchema[key] = z.object({}).describe(prop.description || '');
        }
      } else {
        zodSchema[key] = z.any().describe(prop.description || '');
      }
      
      if (!inputSchema.required || !inputSchema.required.includes(key)) {
        zodSchema[key] = zodSchema[key].optional();
      }
    }
    
    return z.object(zodSchema);
  }

  private convertSinglePropertyToZod(prop: any): z.ZodType<any> {
    if (prop.type === 'string') {
      return z.string();
    } else if (prop.type === 'number') {
      return z.number();
    } else if (prop.type === 'boolean') {
      return z.boolean();
    } else if (prop.type === 'array') {
      if (prop.items) {
        return z.array(this.convertSinglePropertyToZod(prop.items));
      } else {
        return z.array(z.any());
      }
    } else if (prop.type === 'object') {
      if (prop.properties) {
        return this.convertInputSchemaToZod(prop);
      } else {
        return z.object({});
      }
    } else {
      return z.any();
    }
  }

  private convertInputSchemaToOpenAI(inputSchema: any): any {
    if (!inputSchema) {
      return { type: 'object', properties: {} };
    }

    const convertProperty = (prop: any): any => {
      if (!prop || typeof prop !== 'object') {
        return { type: 'string' };
      }

      const result: any = {
        type: prop.type || 'string'
      };

      if (prop.description) {
        result.description = prop.description;
      }

      if (prop.type === 'array') {
        if (prop.items) {
          result.items = convertProperty(prop.items);
        } else {
          // OpenAI API는 배열에 items가 필수이므로 기본값 제공
          result.items = { type: 'string' };
        }
      } else if (prop.type === 'object' && prop.properties) {
        result.properties = {};
        for (const [key, value] of Object.entries(prop.properties)) {
          result.properties[key] = convertProperty(value);
        }
        if (prop.required && Array.isArray(prop.required)) {
          result.required = prop.required;
        }
      }

      return result;
    };

    const schema = {
      type: 'object',
      properties: {}
    };

    if (inputSchema.properties) {
      schema.properties = {};
      for (const [key, value] of Object.entries(inputSchema.properties)) {
        (schema.properties as any)[key] = convertProperty(value);
      }
    }

    if (inputSchema.required && Array.isArray(inputSchema.required)) {
      (schema as any).required = inputSchema.required;
    }

    return schema;
  }

  cleanup() {
    for (const [serverName, client] of this.clients) {
      try {
        client.cleanup();
      } catch (error) {
        console.error(`${serverName} 정리 중 오류:`, error);
      }
    }
    this.clients.clear();
  }
}

// 전역 MCP 클라이언트 임포트
import { loadMCPTools, cleanupGlobalMCPClient } from '@/lib/mcpClient';

// 실제 MCP 도구들을 동적으로 로드하여 사용
// (하드코딩된 도구들을 제거하고 실제 MCP 서버에서 로드된 도구들을 사용)

// LLM 모델 생성 함수
function createLLMModel(provider: string, model: string, config: any, temperature?: number) {
  switch (provider) {
    case 'openai':
      return new ChatOpenAI({
        apiKey: config.OPENAI_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'anthropic':
      return new ChatAnthropic({
        apiKey: config.ANTHROPIC_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'ollama':
      // Ollama - OpenAI 호환 API 사용 (환경변수 서버 주소 사용)
      console.log(`🦙 Ollama 프로바이더를 사용합니다. 모델: ${model}`);
      
      // 환경변수에서 Ollama 서버 주소 가져오기 - /v1 경로 자동 추가
      let ollamaBaseURL = config.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || config.OLLAMA_BASE_URL || 'http://localhost:11434';
      
      // /v1 경로가 없으면 추가
      if (!ollamaBaseURL.endsWith('/v1')) {
        ollamaBaseURL = ollamaBaseURL.replace(/\/$/, '') + '/v1';
      }
      
      console.log(`🔗 Ollama 서버: ${ollamaBaseURL}`);
      
      if (!config.NEXT_PUBLIC_OLLAMA_PROXY_SERVER && !config.OLLAMA_BASE_URL) {
        console.warn('⚠️ NEXT_PUBLIC_OLLAMA_PROXY_SERVER 또는 OLLAMA_BASE_URL 환경변수가 설정되지 않았습니다. 로컬 서버를 사용합니다.');
      }
      
      // Ollama 도구 호출 권장 모델 (airun-chat 커스텀 모델 포함)
      const recommendedToolModels = ['airun-chat', 'devstral', 'gemma3', 'gemma3:27b', 'llama3.2', 'llama3.1', 'llama3:latest', 'mistral', 'qwen2.5', 'codegemma', 'gemma2'];
      
      // 도구 호출 최적화 확인
      const isRecommendedModel = recommendedToolModels.some(recommended => 
        model.toLowerCase().includes(recommended.toLowerCase())
      );
      
      if (!isRecommendedModel) {
        console.log(`⚠️ ${model}은 도구 호출에 최적화되지 않을 수 있습니다. 권장 모델: ${recommendedToolModels.join(', ')}`);
      } else {
        console.log(`✅ ${model}은 도구 호출을 잘 지원하는 모델입니다.`);
      }
      
      console.log(`🔧 Ollama ChatOpenAI 생성:`, {
        modelName: model,
        baseURL: ollamaBaseURL,
        temperature: temperature || 0.7
      });
      
      return new ChatOpenAI({
        openAIApiKey: 'sk-dummy', // Ollama는 API 키가 필요하지 않지만 ChatOpenAI에서는 형식이 필요
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: ollamaBaseURL,
          apiKey: 'sk-dummy', // configuration 내부에서도 API 키 설정
        },
      });
    
    case 'gemini':
      // Gemini - Google AI Studio API 사용
      console.log(`🔮 Gemini 프로바이더를 사용합니다. 모델: ${model}`);
      if (!config.GEMINI_API_KEY) {
        throw new Error('Gemini API 키가 설정되지 않았습니다. GEMINI_API_KEY 환경변수를 설정해주세요.');
      }
      return new ChatOpenAI({
        apiKey: config.GEMINI_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: config.GEMINI_BASE_URL || 'https://generativelanguage.googleapis.com/v1beta/openai/',
        }
      });
    
    case 'vllm':
      // VLLM - OpenAI 호환 API 사용 (환경변수 서버 주소 사용)
      console.log(`🚀 VLLM 프로바이더를 사용합니다. 모델: ${model}`);
      
      // 환경변수에서 VLLM 서버 주소 가져오기
      const vllmBaseURL = config.NEXT_PUBLIC_VLLM_SERVER || config.VLLM_BASE_URL || 'http://localhost:8000/v1';
      console.log(`🔗 VLLM 서버: ${vllmBaseURL}`);
      
      if (!config.NEXT_PUBLIC_VLLM_SERVER && !config.VLLM_BASE_URL) {
        console.warn('⚠️ NEXT_PUBLIC_VLLM_SERVER 또는 VLLM_BASE_URL 환경변수가 설정되지 않았습니다. 로컬 서버를 사용합니다.');
      }
      
      // VLLM 도구 호출 지원 확인
      console.log(`✅ ${model}은 VLLM 서버에서 도구 호출을 지원합니다.`);
      
      return new ChatOpenAI({
        apiKey: 'vllm', // VLLM은 API 키가 필요하지 않음
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: vllmBaseURL,
        }
      });
    
    case 'claude':
      // Claude - Anthropic과 동일하게 처리
      console.log(`🤖 Claude 프로바이더를 사용합니다 (Anthropic API). 모델: ${model}`);
      return new ChatAnthropic({
        apiKey: config.CLAUDE_API_KEY || config.ANTHROPIC_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'groq':
      // Groq - OpenAI 호환 API (도구 호출 지원 모델 자동 선택)
      console.log(`⚡ Groq 프로바이더를 사용합니다. 모델: ${model}`);
      if (!config.GROQ_API_KEY) {
        throw new Error('Groq API 키가 설정되지 않았습니다. GROQ_API_KEY 환경변수를 설정해주세요.');
      }
      
      // Groq 도구 호출 지원 모델 확인 및 교체
      let groqModel = model;
      const nonToolModels = ['mixtral-8x7b-32768', 'llama2-70b-4096', 'gemma-7b-it'];
      const toolSupportedModels = ['llama3-groq-70b-8192-tool-use-preview', 'llama-3.1-70b-versatile', 'llama3-70b-8192'];
      
      if (nonToolModels.includes(model)) {
        groqModel = 'llama3-groq-70b-8192-tool-use-preview';
        console.log(`⚠️ ${model}은 도구 호출을 지원하지 않습니다. ${groqModel}로 변경합니다.`);
      } else if (!toolSupportedModels.includes(model)) {
        // 알려지지 않은 모델인 경우 안전하게 도구 호출 지원 모델로 변경
        groqModel = 'llama3-groq-70b-8192-tool-use-preview';
        console.log(`⚠️ ${model}의 도구 호출 지원 여부를 확인할 수 없습니다. ${groqModel}로 변경합니다.`);
      }
      
      return new ChatOpenAI({
        apiKey: config.GROQ_API_KEY,
        modelName: groqModel,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: 'https://api.groq.com/openai/v1',
        }
      });
    
    case 'deepseek':
      // DeepSeek - OpenAI 호환 API
      console.log(`🧠 DeepSeek 프로바이더를 사용합니다. 모델: ${model}`);
      if (!config.DEEPSEEK_API_KEY) {
        throw new Error('DeepSeek API 키가 설정되지 않았습니다. DEEPSEEK_API_KEY 환경변수를 설정해주세요.');
      }
      return new ChatOpenAI({
        apiKey: config.DEEPSEEK_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: 'https://api.deepseek.com',
        }
      });
    
    default:
      // 기본적으로 OpenAI 호환 API로 처리
      console.warn(`❓ 알 수 없는 프로바이더 ${provider}, OpenAI 호환 API로 처리합니다.`);
      
      // 환경변수 이름 우선순위 확인
      const apiKey = config[`${provider.toUpperCase()}_API_KEY`];
      let baseURL = config[`${provider.toUpperCase()}_BASE_URL`];
      
      // 특별한 환경변수 이름 확인
      if (provider === 'ollama') {
        baseURL = config.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || config.OLLAMA_BASE_URL || baseURL;
        // Ollama의 경우 /v1 경로 자동 추가
        if (baseURL && !baseURL.endsWith('/v1')) {
          baseURL = baseURL.replace(/\/$/, '') + '/v1';
        }
      } else if (provider === 'vllm') {
        baseURL = config.NEXT_PUBLIC_VLLM_SERVER || config.VLLM_BASE_URL || baseURL;
      }
      
      // 기본 포트 설정
      if (!baseURL) {
        if (provider === 'ollama') {
          baseURL = 'http://localhost:11434/v1';
        } else if (provider === 'vllm') {
          baseURL = 'http://localhost:8000/v1';
        } else {
          baseURL = `http://localhost:${config.API_PORT || 5500}/v1`;
        }
      }
      
      if (!apiKey && !['ollama', 'vllm'].includes(provider)) {
        console.warn(`⚠️ ${provider.toUpperCase()}_API_KEY가 설정되지 않았습니다.`);
      }
      
      console.log(`🔗 ${provider} 서버: ${baseURL}`);
      
      return new ChatOpenAI({
        apiKey: apiKey || 'default',
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
        configuration: {
          baseURL: baseURL,
        }
      });
  }
}

// 스트리밍 응답 생성 함수
const createStreamingResponse = (generator: () => AsyncGenerator<string>) => {
  console.log('🌊 createStreamingResponse 함수 시작');
  return new Response(
    new ReadableStream({
      async start(controller) {
        console.log('📺 ReadableStream start 호출됨');
        try {
          for await (const chunk of generator()) {
            controller.enqueue(new TextEncoder().encode(chunk));
          }
        } catch (error) {
          console.error('스트리밍 오류:', error);
          controller.enqueue(new TextEncoder().encode(`\n\n오류: ${(error as Error).message}`));
        } finally {
          controller.close();
        }
      },
    }),
    {
      headers: {
        'Content-Type': 'text/event-stream',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
        'Access-Control-Allow-Origin': '*',
        'Access-Control-Allow-Headers': 'Cache-Control',
      },
    }
  );
};

// 위치 기반 파라미터 파싱 함수
function parsePositionalArgs(input: string, toolName: string, inputSchema: any): any {
  console.log(`🔧 위치 기반 파라미터 파싱: ${toolName} <- "${input}"`);
  
  // 쉼표로 분리된 파라미터 처리
  const parts = input.split(',').map(part => part.trim());
  
  if (toolName === 'get_complaints') {
    // get_complaints("date_from:2025-01-14") -> {date_from: "2025-01-14"}
    // get_complaints("keyword:소음") -> {keyword: "소음"}
    // get_complaints("category:소음/진동,status:pending") -> {category: "소음/진동", status: "pending"}
    const result: any = {};
    
    for (const part of parts) {
      if (part.includes(':')) {
        const [key, value] = part.split(':', 2);
        result[key.trim()] = value.trim();
      } else if (part.trim()) {
        // 키:값 형태가 아닌 경우 키워드로 처리
        result.keyword = part.trim();
      }
    }
    
    console.log(`🔄 get_complaints 파싱 결과:`, result);
    return result;
  }
  
  else if (toolName === 'get_complaint') {
    // get_complaint(2) -> {id: "2"}
    return { id: parts[0] };
  } 
  
  else if (toolName === 'update_complaint_status') {
    // update_complaint_status(2, "처리중") -> {id: "2", status: "처리중"}
    if (parts.length >= 2) {
      return { id: parts[0], status: parts[1] };
    }
    return { id: parts[0] };
  } 
  
  else if (toolName === 'reply_to_complaint') {
    // reply_to_complaint(1, "assistant", "답변") -> {id: "1", officer_id: "assistant", reply_content: "답변"}
    if (parts.length >= 3) {
      return { 
        id: parts[0], 
        officer_id: parts[1], 
        reply_content: parts.slice(2).join(',').trim() // 나머지는 답변 내용으로
      };
    } else if (parts.length === 2) {
      return { id: parts[0], officer_id: parts[1], reply_content: "자동 생성된 답변" };
    }
    return { id: parts[0] };
  }
  
  else if (toolName === 'process_complaint_workflow') {
    // process_complaint_workflow(1) -> {complaint_id: "1"}
    return { complaint_id: parts[0] };
  }
  
  // 기타 도구는 첫 번째 속성에 할당
  if (inputSchema && inputSchema.properties) {
    const firstProp = Object.keys(inputSchema.properties)[0];
    if (firstProp) {
      return { [firstProp]: input };
    }
  }
  
  return { value: input };
}

// MCP 도구를 LangChain DynamicTool 형식으로 변환
function convertMCPToolsToLangChainFormat(mcpTools: any[]): any[] {
  console.log(`🔄 MCP 도구 ${mcpTools.length}개를 LangChain DynamicTool 형식으로 변환 중...`);
  
  return mcpTools.map(tool => {
    try {
      // LangChain DynamicTool 형식으로 변환
      const dynamicTool = new DynamicTool({
        name: tool.name,
        description: tool.description || `MCP 도구: ${tool.name}`,
        func: async (input: string) => {
          try {
            console.log(`🔧 DynamicTool 도구 호출: ${tool.name}`, input);
            
            // 입력 파라미터 파싱 - 위치 기반 파라미터 지원
            let args = {};
            if (typeof input === 'string') {
              try {
                // JSON 파싱 시도
                args = JSON.parse(input);
              } catch {
                // JSON이 아닌 경우 - 위치 기반 파라미터 처리
                args = parsePositionalArgs(input, tool.name, tool.inputSchema);
              }
            } else if (typeof input === 'object') {
              args = input;
            }
            
            console.log(`🔄 파싱된 파라미터:`, args);
            
            // MCP 도구 실행
            const result = await tool.invoke(args);
            console.log(`✅ DynamicTool 도구 완료: ${tool.name}`);
            
            // 결과를 문자열로 변환
            if (typeof result === 'string') {
              return result;
            } else if (typeof result === 'object') {
              return JSON.stringify(result, null, 2);
            } else {
              return String(result);
            }
          } catch (error) {
            console.error(`❌ DynamicTool 도구 오류: ${tool.name}`, error);
            throw error;
          }
        }
      });
      
      console.log(`✅ DynamicTool 변환 완료: ${tool.name}`);
      return dynamicTool;
    } catch (error) {
      console.error(`❌ 도구 변환 실패: ${tool.name}`, error);
      // 기본 형태로 fallback
      const fallbackTool = new DynamicTool({
        name: tool.name,
        description: tool.description || `MCP 도구: ${tool.name}`,
        func: async (input: string) => {
          return `도구 ${tool.name} 실행 실패: 변환 오류`;
        }
      });
      return fallbackTool;
    }
  }).filter(tool => tool !== null);
}

// MCP 스키마를 OpenAI Function Calling 형식으로 변환
function convertMCPSchemaToOpenAIFormat(inputSchema: any): any {
  if (!inputSchema || !inputSchema.properties) {
    return {
      type: 'object',
      properties: {},
      required: []
    };
  }
  
  const convertProperty = (prop: any): any => {
    if (!prop || typeof prop !== 'object') {
      return { type: 'string' };
    }

    const converted: any = {
      type: prop.type || 'string',
      description: prop.description || ''
    };

    // 배열 타입 처리
    if (prop.type === 'array') {
      converted.items = prop.items ? convertProperty(prop.items) : { type: 'string' };
    }
    
    // 객체 타입 처리
    if (prop.type === 'object' && prop.properties) {
      converted.properties = {};
      for (const [key, value] of Object.entries(prop.properties)) {
        converted.properties[key] = convertProperty(value);
      }
      if (prop.required) {
        converted.required = prop.required;
      }
    }
    
    // enum 처리
    if (prop.enum) {
      converted.enum = prop.enum;
    }
    
    return converted;
  };

  const result = {
    type: 'object',
    properties: {} as any,
    required: inputSchema.required || []
  };

  for (const [propName, propSchema] of Object.entries(inputSchema.properties)) {
    result.properties[propName] = convertProperty(propSchema);
  }

  return result;
}



// LangGraph 에이전트로 처리
async function* processWithLangGraph(
  message: string,
  files: any[],
  provider: string,
  model: string,
  temperature?: number,
  history?: Array<{role: 'user' | 'assistant', content: string}>,
  sessionId?: string
) {
  console.log(`🚀 LangGraph 에이전트 처리 시작: ${provider}/${model}`);
  
  // 도구-서버 매핑 초기화
  await initializeToolServerMapping();
  
  // Ollama인 경우 직접 API 호출 사용 (dolphin-mcp 스타일)
  if (provider === 'ollama') {
    console.log(`🦙 Ollama 프로바이더 감지 - 직접 API 호출 모드 사용`);
    
    try {
      // 설정 로드
      const config = getAirunConfig();
      
      // Ollama 서버 URL 설정
      let ollamaBaseURL = config.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || config.OLLAMA_BASE_URL || 'http://localhost:11434';
      
      // /v1 경로 제거 (직접 API 호출이므로)
      if (ollamaBaseURL.endsWith('/v1')) {
        ollamaBaseURL = ollamaBaseURL.slice(0, -3);
      }
      
      console.log(`🔗 Ollama 서버: ${ollamaBaseURL}`);
      
      // MCP 도구 로드 및 변환
      const mcpTools = await loadMCPTools();
      
      // 모든 MCP 도구를 Ollama에서 사용 (제한 없음)
      console.log(`🎯 모든 MCP 도구 사용: ${mcpTools.length}개`);
      console.log(`📋 도구 목록: ${mcpTools.map(t => t.name).join(', ')}`);
      
      const ollamaTools = convertMCPToolsToOllamaFormat(mcpTools);
      
      // 메시지 형식 변환
      const ollamaMessages: OllamaMessage[] = [];
      
      // 모든 MCP 도구들로부터 동적 시스템 프롬프트 생성
      const systemPrompt = generateDynamicSystemPrompt(mcpTools);

      ollamaMessages.push({
        role: 'system',
        content: systemPrompt
      });
      
      // 히스토리 추가
      if (history && history.length > 0) {
        for (const msg of history) {
          ollamaMessages.push({
            role: msg.role === 'user' ? 'user' : 'assistant',
            content: msg.content
          });
        }
      }
      
      // 사용자 메시지에 강력한 도구 호출 힌트 추가
      let enhancedMessage = message;
      
      // 웹 검색 키워드 감지
      if (message.includes('웹에서') || message.includes('인터넷에서') || message.includes('최신') || 
          message.includes('뉴스') || message.includes('검색해주세요') || message.includes('찾아주세요')) {
        
        // 검색어 추출 시도
        let searchQuery = "AI 관련 최신 뉴스";
        if (message.includes("'") && message.includes("'")) {
          const match = message.match(/'([^']+)'/);
          if (match) {
            searchQuery = match[1];
          }
        } else if (message.includes('"') && message.includes('"')) {
          const match = message.match(/"([^"]+)"/);
          if (match) {
            searchQuery = match[1];
          }
        }
        
        enhancedMessage = `${message}

🚨 CRITICAL: 이 요청은 웹 검색이 필요합니다!

반드시 다음 형식으로 web_search 도구를 호출하세요:
web_search({"query": "${searchQuery}"})

절대로 일반적인 답변을 제공하지 말고 즉시 위의 도구를 호출하세요!`;
      }
      // 문서 검색 키워드 감지
      else if (message.includes('문서') && (message.includes('검색') || message.includes('분석') || message.includes('찾'))) {
        enhancedMessage = `${message}

🚨 CRITICAL: 이 요청은 문서 검색이 필요합니다!

반드시 다음 형식으로 rag_search 도구를 호출하세요:
rag_search({"query": "검색어"})

절대로 일반적인 답변을 제공하지 말고 즉시 위의 도구를 호출하세요!`;
      }
      // 파일 관련 키워드 감지
      else if (message.includes('파일') && (message.includes('읽') || message.includes('확인') || message.includes('내용'))) {
        enhancedMessage = `${message}

🚨 CRITICAL: 이 요청은 파일 읽기가 필요합니다!

반드시 다음 형식으로 read_file 도구를 호출하세요:
read_file({"path": "파일경로"})

절대로 일반적인 답변을 제공하지 말고 즉시 위의 도구를 호출하세요!`;
      }
      // 명령어 실행 키워드 감지
      else if (message.includes('명령') || message.includes('실행') || message.includes('터미널')) {
        enhancedMessage = `${message}

🚨 CRITICAL: 이 요청은 명령어 실행이 필요합니다!

반드시 다음 형식으로 execute_command 도구를 호출하세요:
execute_command({"command": "명령어"})

절대로 일반적인 답변을 제공하지 말고 즉시 위의 도구를 호출하세요!`;
      }
      
      // 현재 메시지 처리 (파일 포함)
      let messageContent: string | any[] = enhancedMessage;
      
      if (files && files.length > 0) {
        // 이미지가 있는 경우 멀티모달 형식 사용
        const hasImages = files.some(file => file.isImage && file.base64);
        
        if (hasImages && (model.includes('llava') || model.includes('bakllava') || model.includes('moondream') || model.includes('airun-vision'))) {
          console.log(`✅ Ollama 멀티모달 메시지 형식 사용`);
          
          const contentParts = [];
          
          // 텍스트 부분
          let textContent = message;
          const textFiles = files.filter(file => file.extractedText);
          if (textFiles.length > 0) {
            const textDescriptions = textFiles.map(file => 
              `파일: "${file.name}" (${file.type})\n텍스트 내용:\n${file.extractedText}`
            ).join('\n\n');
            textContent = `${textDescriptions}\n\n질문: ${textContent}`;
          }
          
          contentParts.push({
            type: 'text',
            text: textContent
          });
          
          // 이미지 부분들
          const imageFiles = files.filter(file => file.isImage && file.base64);
          for (const imageFile of imageFiles) {
            contentParts.push({
              type: 'image_url',
              image_url: {
                url: `data:${imageFile.type};base64,${imageFile.base64}`
              }
            });
          }
          
          messageContent = contentParts;
        } else {
          // 텍스트 설명으로 대체
          const fileDescriptions = files.map(file => {
            let desc = `파일: "${file.name}" (${file.type})`;
            if (file.extractedText) {
              desc += `\n텍스트 내용:\n${file.extractedText}`;
            } else if (file.isImage) {
              desc += `\n이미지 파일이 첨부되었습니다. (크기: ${(file.size / 1024).toFixed(1)}KB)`;
            }
            return desc;
          }).join('\n\n');
          
          messageContent = `${fileDescriptions}\n\n질문: ${message}`;
        }
      }
      
      // 현재 메시지 추가
      ollamaMessages.push({
        role: 'user',
        content: messageContent
      });
      
      console.log(`📤 Ollama 직접 호출 시작: ${ollamaMessages.length}개 메시지, ${ollamaTools.length}개 도구`);
      
      // Ollama API 직접 호출
      const responseGenerator = await callOllamaDirectly(
        ollamaMessages,
        ollamaTools,
        model,
        ollamaBaseURL,
        temperature || 0.7
      );
      
      // 응답 스트리밍
      for await (const chunk of responseGenerator) {
        yield chunk;
      }
      
      // 완료 신호
      yield `data: [DONE]\n\n`;
      return;
      
    } catch (error) {
      console.error('❌ Ollama 직접 API 호출 실패:', error);
      yield `data: ${JSON.stringify({ content: `❌ Ollama 직접 API 호출 실패: ${error instanceof Error ? error.message : String(error)}` })}\n\n`;
      yield `data: [DONE]\n\n`;
      return;
    }
  }
  
  try {
    // 설정 로드
    const config = getAirunConfig();
    console.log(`🔧 설정 로드 완료. API 키 상태:`, {
      OPENAI_API_KEY: config.OPENAI_API_KEY ? `✅ ${config.OPENAI_API_KEY.substring(0, 10)}...` : '❌ 없음',
      ANTHROPIC_API_KEY: config.ANTHROPIC_API_KEY ? `✅ ${config.ANTHROPIC_API_KEY.substring(0, 10)}...` : '❌ 없음'
    });
    
    // LLM 모델 생성
    console.log(`🤖 LLM 모델 생성 중: ${provider}/${model}`);
    const llm = createLLMModel(provider, model, config, temperature);
    console.log(`✅ LLM 모델 생성 완료: ${provider}/${model}`);
    
    // MCP 도구 로드
    const mcpTools = await loadMCPTools();
    console.log(`🔧 MCP 도구 ${mcpTools.length}개 로드 완료`);
    
    // 프로바이더별 도구 형식 변환
    const langchainTools = convertMCPToolsToLangChainFormat(mcpTools);
    console.log(`🔧 LangChain 형식 도구 ${langchainTools.length}개 변환 완료`);
    
    // LangGraph 에이전트 생성
    console.log('🔧 LangGraph 에이전트 생성 중...');
    const agent = createReactAgent({
      llm,
      tools: langchainTools,
    });
    
    // console.log('✅ LangGraph 에이전트 생성 완료');
    
    // 히스토리를 메시지 형식으로 변환
    const messages = [];
    if (history && history.length > 0) {
      for (const msg of history) {
        messages.push({
          role: msg.role === 'user' ? 'human' : 'assistant',
          content: msg.content,
        });
      }
    }
    
    // 프로바이더별 이미지 지원 확인
    const getProviderImageSupport = (provider: string, model: string) => {
      switch (provider) {
        case 'openai':
          // GPT-4 Vision 모델들만 이미지 지원
          return model.includes('gpt-4') && (model.includes('vision') || model.includes('gpt-4o') || model.includes('gpt-4-turbo'));
        case 'anthropic':
        case 'claude':
          // Claude 3 모델들은 이미지 지원
          return model.includes('claude-3') || model.includes('sonnet') || model.includes('haiku') || model.includes('opus');
        case 'gemini':
          // Gemini Pro Vision 모델들은 이미지 지원
          return model.includes('vision') || model.includes('pro') || model.includes('gemini-1.5');
        case 'ollama':
          // Ollama의 멀티모달 모델들 (llava, bakllava 등)
          return model.includes('llava') || model.includes('bakllava') || model.includes('moondream') || model.includes('airun-vision');
        case 'groq':
          // Groq는 현재 이미지 지원이 제한적
          return model.includes('llava') || model.includes('vision');
        case 'vllm':
          // VLLM은 멀티모달 모델 지원 가능
          return model.includes('llava') || model.includes('vision') || model.includes('instructblip');
        default:
          return false;
      }
    };
    
    // 현재 메시지 준비 (프로바이더별 이미지 지원)
    let messageContent: any = message;
    
    if (files && files.length > 0) {
      // 이미지가 있는 경우 프로바이더별 처리
      const hasImages = files.some(file => file.isImage && file.base64);
      const supportsImages = getProviderImageSupport(provider, model);
      
      console.log(`🖼️ 이미지 파일 감지: ${hasImages}`);
      console.log(`📊 사용 중인 프로바이더: ${provider}, 모델: ${model}`);
      console.log(`🎨 이미지 지원 여부: ${supportsImages}`);
      
      if (hasImages && supportsImages) {
        console.log(`✅ 멀티모달 메시지 형식 사용 (이미지 + 텍스트)`);
        
        // 모든 프로바이더에 대해 OpenAI 호환 형식 사용 (LangChain이 자동 변환)
        console.log(`🔄 OpenAI 호환 멀티모달 형식 사용`);
        const contentParts = [];
        
        // 텍스트 부분
        let textContent = message;
        const textFiles = files.filter(file => file.extractedText);
        if (textFiles.length > 0) {
          const textDescriptions = textFiles.map(file => 
            `파일: "${file.name}" (${file.type})\n텍스트 내용:\n${file.extractedText}`
          ).join('\n\n');
          textContent = `${textDescriptions}\n\n질문: ${textContent}`;
        }
        
        contentParts.push({
          type: 'text',
          text: textContent
        });
        
        // 이미지 부분들 (OpenAI 호환 형식 - LangChain이 각 프로바이더에 맞게 변환)
        const imageFiles = files.filter(file => file.isImage && file.base64);
        console.log(`🖼️ 처리할 이미지 파일 수: ${imageFiles.length}`);
        
        for (const imageFile of imageFiles) {
          console.log(`📷 이미지 추가: ${imageFile.name}, 타입: ${imageFile.type}`);
          contentParts.push({
            type: 'image_url',
            image_url: {
              url: `data:${imageFile.type};base64,${imageFile.base64}`
            }
          });
        }
        
        messageContent = contentParts;
        
      } else if (hasImages && !supportsImages) {
        // 이미지가 있지만 지원하지 않는 경우 - 텍스트 설명으로 대체
        console.log(`⚠️ ${provider}/${model}은 이미지를 지원하지 않습니다. 텍스트 설명으로 대체합니다.`);
        
        const fileDescriptions = files.map(file => {
          let desc = `파일: "${file.name}" (${file.type})`;
          if (file.extractedText) {
            desc += `\n텍스트 내용:\n${file.extractedText}`;
          } else if (file.isImage) {
            desc += `\n이미지 파일이 첨부되었습니다. (크기: ${(file.size / 1024).toFixed(1)}KB)`;
          }
          return desc;
        }).join('\n\n');
        
        messageContent = `${fileDescriptions}\n\n질문: ${message}`;
        
      } else {
        // 텍스트 파일만 있는 경우
        const fileDescriptions = files.map(file => {
          let desc = `파일: "${file.name}" (${file.type})`;
          if (file.extractedText) {
            desc += `\n텍스트 내용:\n${file.extractedText}`;
          }
          return desc;
        }).join('\n\n');
        
        messageContent = `${fileDescriptions}\n\n질문: ${message}`;
      }
    }
    
    // 현재 메시지 추가
    messages.push({
      role: 'human' as const,
      content: messageContent,
    });
    
    // console.log('📤 LangGraph 에이전트 스트리밍 실행 중...');
    // console.log('📝 전송할 메시지 수:', messages.length);
    // console.log('📝 마지막 메시지 내용:', messages[messages.length - 1]);
    
    // 에이전트 스트리밍 실행
    console.log('🔍 streamEvents 시작...');
    const stream = agent.streamEvents({
      messages: messages,
    }, { version: "v2" });
    
    let hasContent = false;
    let finalResponse = '';
    let sentContent = ''; // 중복 방지를 위한 전송 내용 추적
    let assistantResponse = ''; // 세션 저장을 위한 assistant 응답 수집
    let eventCount = 0;
    
    for await (const event of stream) {
      eventCount++;
      if (eventCount <= 5 || eventCount % 10 === 0) {
        // console.log(`🎯 이벤트 #${eventCount}:`, event.event, event.name || 'no-name');
      }
      
      // 오직 chat_model_stream 이벤트만 처리하여 중복 방지
      if (event.event === 'on_chat_model_stream') {
        // console.log('💬 Chat model stream:', event.data);
        const chunk = event.data?.chunk;
        // console.log('🔍 청크 상세:', {
        //   hasChunk: !!chunk,
        //   chunkType: typeof chunk,
        //   content: chunk?.content,
        //   contentType: typeof chunk?.content,
        //   chunkKeys: chunk ? Object.keys(chunk) : 'no chunk'
        // });
        
        if (chunk && chunk.content) {
          let contentToSend = '';
          
          // 프로바이더별 특화 처리를 위한 현재 프로바이더 정보 로깅
          // console.log(`🎯 ${provider} 프로바이더의 청크 처리 중...`);
          
          // 1. Anthropic의 배열 형태 응답 처리 [{"type":"text","text":"..."}]
          if (Array.isArray(chunk.content)) {
            // console.log('📝 Anthropic 배열 형태 content:', chunk.content);
            contentToSend = chunk.content
              .filter((item: any) => item && item.type === 'text' && item.text)
              .map((item: any) => item.text)
              .join('');
          } 
          // 2. 문자열인 경우 (대부분의 프로바이더들 - OpenAI, Groq, DeepSeek 등)
          else if (typeof chunk.content === 'string') {
            // console.log(`📝 ${provider} 문자열 형태 content:`, chunk.content.substring(0, 50) + '...');
            contentToSend = chunk.content;
          }
          // 3. Anthropic 단일 객체 형태 {"type":"text","text":"..."}
          else if (typeof chunk.content === 'object' && chunk.content.type === 'text' && chunk.content.text) {
            // console.log('📝 Anthropic 단일 객체 형태 content:', chunk.content);
            contentToSend = chunk.content.text;
          }
          // 4. Gemini 형태 응답 처리 (candidates[0].content.parts[0].text)
          else if (typeof chunk.content === 'object' && chunk.content.parts && Array.isArray(chunk.content.parts)) {
            // console.log('📝 Gemini 형태 content:', chunk.content);
            contentToSend = chunk.content.parts
              .filter((part: any) => part && part.text)
              .map((part: any) => part.text)
              .join('');
          }
          // 5. Ollama 메시지 형태 처리 (message.content)
          else if (typeof chunk.content === 'object' && chunk.content.message && chunk.content.message.content) {
            // console.log('📝 Ollama 메시지 형태 content:', chunk.content);
            contentToSend = chunk.content.message.content;
          }
          // 6. Groq 특화 처리 (choices[0].delta.content 등)
          else if (provider === 'groq' && typeof chunk.content === 'object') {
            // console.log('📝 Groq 특화 처리:', chunk.content);
            
            if (chunk.content.choices && Array.isArray(chunk.content.choices) && chunk.content.choices[0]) {
              const choice = chunk.content.choices[0];
              if (choice.delta && choice.delta.content) {
                contentToSend = choice.delta.content;
              } else if (choice.message && choice.message.content) {
                contentToSend = choice.message.content;
              }
            } else if (chunk.content.delta && chunk.content.delta.content) {
              contentToSend = chunk.content.delta.content;
            }
          }
          // 7. Ollama 특화 처리 (done, response 필드 등)
          else if (provider === 'ollama' && typeof chunk.content === 'object') {
            // console.log('📝 Ollama 특화 처리:', chunk.content);
            
            if (chunk.content.response) {
              contentToSend = chunk.content.response;
            } else if (chunk.content.message && typeof chunk.content.message === 'string') {
              contentToSend = chunk.content.message;
            } else if (chunk.content.done && chunk.content.total_duration) {
              // Ollama 완료 메시지는 무시
              contentToSend = '';
            }
          }
          // 8. VLLM 특화 처리 (OpenAI 호환 형태)
          else if (provider === 'vllm' && typeof chunk.content === 'object') {
            // console.log('📝 VLLM 특화 처리:', chunk.content);
            
            if (chunk.content.choices && Array.isArray(chunk.content.choices) && chunk.content.choices[0]) {
              const choice = chunk.content.choices[0];
              if (choice.delta && choice.delta.content) {
                contentToSend = choice.delta.content;
              } else if (choice.message && choice.message.content) {
                contentToSend = choice.message.content;
              }
            } else if (chunk.content.delta && chunk.content.delta.content) {
              contentToSend = chunk.content.delta.content;
            } else if (chunk.content.text) {
              contentToSend = chunk.content.text;
            }
          }
          // 9. 기타 객체 형태 - 텍스트 관련 필드 찾기
          else if (typeof chunk.content === 'object') {
            // console.log('📝 기타 객체 형태 content:', chunk.content);
            
            // 일반적인 텍스트 필드들 시도
            if (chunk.content.text) {
              contentToSend = chunk.content.text;
            } else if (chunk.content.content) {
              contentToSend = chunk.content.content;
            } else if (chunk.content.message) {
              contentToSend = typeof chunk.content.message === 'string' ? chunk.content.message : chunk.content.message.content || '';
            } else if (chunk.content.response) {
              contentToSend = chunk.content.response;
            } else if (chunk.content.output) {
              contentToSend = chunk.content.output;
            } else {
              // console.log('⚠️ 처리할 수 없는 content 형태:', chunk.content);
              contentToSend = '';
            }
          }
          // 10. 기타 타입은 빈 문자열로 처리
          else {
            // console.log('⚠️ 지원하지 않는 content 타입:', typeof chunk.content, chunk.content);
            contentToSend = '';
          }
          
          // 빈 문자열이 아닌 경우에만 전송
          if (contentToSend && contentToSend.trim() !== '') {
            hasContent = true;
            sentContent += contentToSend;
            assistantResponse += contentToSend; // 세션 저장용 응답 수집
            
            const dataToSend = `data: ${JSON.stringify({ content: contentToSend })}\n\n`;
            // console.log('📤 SSE 데이터 전송:', dataToSend.trim());
            yield dataToSend;
          }
        }
      } else if (event.event === 'on_chain_end' && event.name === 'LangGraph') {
        // console.log('🏁 Agent 완료:', event.data);
        // 완성된 응답 추출 (스트리밍이 없었을 경우를 위해)
        const output = event.data?.output;
        if (output && output.messages && output.messages.length > 0) {
          const lastMessage = output.messages[output.messages.length - 1];
          if (lastMessage && lastMessage.content && !hasContent) {
            finalResponse = lastMessage.content;
            assistantResponse = finalResponse; // 세션 저장용 응답 설정
            console.log('📝 완성된 응답 저장:', finalResponse.substring(0, 100) + '...');
          }
        }
      } else if (event.event === 'on_tool_start') {
        console.log('🔧 도구 시작:', event.name);
        const toolName = event.name;
        
        // 도구 이름으로 실제 서버 이름 찾기
        const serverName = toolToServerMapping.get(toolName) || 'unknown';
        
        // 도구 호출 시작 이벤트를 별도 JSON으로 전송
        yield `${JSON.stringify({ 
          type: 'tool_call_start', 
          toolName: toolName,
          args: event.data?.input || {},
          server: serverName
        })}\n`;
        
        // 사용자에게 보여줄 텍스트도 전송
        yield `data: ${JSON.stringify({ content: `\n🔧 호출 : ${toolName}...\n` })}\n\n`;
      } else if (event.event === 'on_tool_end') {
        console.log('✅도구 완료:', event.name);
        const toolName = event.name;
        
        // 도구 호출 완료 이벤트를 별도 JSON으로 전송
        yield `${JSON.stringify({ 
          type: 'tool_call_end', 
          toolName: toolName,
          result: event.data?.output || '완료'
        })}\n`;
        
        // 사용자에게 보여줄 텍스트도 전송
        // yield `data: ${JSON.stringify({ content: `✅ 도구 완료: ${toolName}\n` })}\n\n`;
      }
      
      // 다른 모든 이벤트들은 로깅만 수행 (중복 방지)
      // if (!['on_chat_model_stream', 'on_chain_end', 'on_tool_start', 'on_tool_end'].includes(event.event)) {
      //   console.log(`📝 기타 이벤트: ${event.event} ${event.name || 'no-name'}`);
      // }
    }
    
    // 완성된 응답이 있고 스트리밍이 없었다면 스트리밍 형태로 전송
    if (finalResponse && !hasContent) {
      console.log('📤 완성된 응답을 스트리밍으로 전송');
      const words = finalResponse.split(' ');
      for (let i = 0; i < words.length; i++) {
        const word = words[i] + (i < words.length - 1 ? ' ' : '');
        yield `data: ${JSON.stringify({ content: word })}\n\n`;
        await new Promise(resolve => setTimeout(resolve, 30)); // 30ms 지연
      }
      hasContent = true;
      assistantResponse = finalResponse; // 세션 저장용 응답 설정
    }
    
    // 이미지 처리 중 응답이 없는 경우 디버깅 정보
    if (!hasContent && files && files.some(f => f.isImage)) {
      console.log('⚠️ 이미지가 포함된 요청에서 응답이 없습니다.');
      console.log(`📊 이미지 파일 수: ${files.filter(f => f.isImage).length}`);
      console.log(`🔧 프로바이더: ${provider}, 모델: ${model}`);
      console.log(`🎨 이미지 지원: ${getProviderImageSupport(provider, model)}`);
    }
    
    // 세션 저장 (로컬 처리 시에도 세션 저장)
    if (sessionId && assistantResponse.trim()) {
      try {
        console.log(`💾 로컬 처리 세션 저장 시도: ${sessionId}`);
        
        // 현재 히스토리에 새 메시지들 추가
        const updatedHistory = [...(history || [])];
        updatedHistory.push({ role: 'user', content: message });
        updatedHistory.push({ role: 'assistant', content: assistantResponse });
        
        // 외부 API 서버의 세션 저장 엔드포인트 호출
        const apiServerUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || getNextjsServerUrl();
        const saveSessionUrl = `${apiServerUrl}/api/v1/sessions/${sessionId}`;
        
        // 세션 데이터 구성
        const sessionData = {
          id: sessionId,
          title: generateSessionTitle(updatedHistory),
          lastMessage: getLastMessage(updatedHistory),
          messageCount: updatedHistory.length,
          provider: provider,
          model: model,
          type: 'assistant',
          history: updatedHistory,
          updatedAt: new Date().toISOString()
        };
        
        // 로컬 파일 시스템에 세션 저장 (외부 API 서버 없이도 동작하도록)
        await saveSessionToLocal(sessionData);
        
        console.log(`✅ 로컬 세션 저장 완료: ${sessionId}`);
      } catch (sessionError) {
        console.error('❌ 세션 저장 실패:', sessionError);
        // 세션 저장 실패해도 응답은 계속 진행
      }
    }
    
    console.log('✅ LangGraph 에이전트 스트리밍 처리 완료');
    console.log(`📊 전송된 총 내용 길이: ${sentContent.length} 문자`);
    
    if (!hasContent) {
      yield `data: ${JSON.stringify({ content: '응답을 생성할 수 없습니다.' })}\n\n`;
    }
    
    // 스트리밍 종료 신호
    yield `data: [DONE]\n\n`;
    
  } catch (error) {
    console.error('❌ LangGraph 에이전트 처리 실패:', error);
    console.error('❌ 에러 타입:', typeof error);
    console.error('❌ 에러 메시지:', error instanceof Error ? error.message : error);
    console.error('❌ 에러 스택:', error instanceof Error ? error.stack : 'N/A');
    console.error('❌ 프로바이더/모델:', `${provider}/${model}`);
    console.error('❌ 메시지 내용:', message.substring(0, 100) + '...');
    
    // 추가 오류 정보
    if (error && typeof error === 'object') {
      console.error('❌ 오류 객체 상세:', {
        name: (error as any).name,
        code: (error as any).code,
        status: (error as any).status,
        statusText: (error as any).statusText,
        url: (error as any).url,
        config: (error as any).config ? {
          baseURL: (error as any).config.baseURL,
          url: (error as any).config.url,
          method: (error as any).config.method
        } : 'No config'
      });
    }
    
    // 프로바이더별 특화 오류 메시지
    let errorMessage = '';
    if (error instanceof Error) {
      const errorStr = error.message.toLowerCase();
      console.error('❌ 원본 에러 메시지 (소문자):', errorStr);
      
      if (provider === 'groq') {
        if (errorStr.includes('tool_use_failed') || errorStr.includes('function')) {
          errorMessage = `🔧 Groq 도구 호출 실패: ${model} 모델이 도구 호출을 지원하지 않습니다. 도구 호출 지원 모델(llama3-groq-70b-8192-tool-use-preview)을 사용하세요.`;
        } else if (errorStr.includes('api key') || errorStr.includes('401')) {
          errorMessage = `🔑 Groq API 키 오류: GROQ_API_KEY 환경변수를 확인하세요.`;
        } else if (errorStr.includes('rate limit')) {
          errorMessage = `⏱️ Groq API 요청 제한: 잠시 후 다시 시도하세요.`;
        } else {
          errorMessage = `❌ Groq 오류: ${error.message}`;
        }
      } else if (provider === 'ollama') {
        if (errorStr.includes('connect') || errorStr.includes('econnrefused')) {
          errorMessage = `🔌 Ollama 연결 실패: Ollama 서버가 실행 중인지 확인하세요. NEXT_PUBLIC_OLLAMA_PROXY_SERVER 환경변수를 확인하세요.`;
        } else if (errorStr.includes('model')) {
          errorMessage = `🤖 Ollama 모델 오류: ${model} 모델이 설치되지 않았거나 로드되지 않았습니다. 'ollama pull ${model}' 명령으로 모델을 다운로드하세요.`;
        } else {
          errorMessage = `❌ Ollama 오류: ${error.message}`;
        }
      } else if (provider === 'vllm') {
        if (errorStr.includes('connect') || errorStr.includes('econnrefused')) {
          errorMessage = `🔌 VLLM 연결 실패: VLLM 서버가 실행 중인지 확인하세요. NEXT_PUBLIC_VLLM_SERVER 환경변수를 확인하세요.`;
        } else if (errorStr.includes('model')) {
          errorMessage = `🤖 VLLM 모델 오류: ${model} 모델이 로드되지 않았거나 사용할 수 없습니다. VLLM 서버에서 모델이 올바르게 설정되었는지 확인하세요.`;
        } else if (errorStr.includes('timeout')) {
          errorMessage = `⏱️ VLLM 시간 초과: 요청 처리 시간이 초과되었습니다. 모델이 크거나 서버가 바쁜 상태일 수 있습니다.`;
        } else {
          errorMessage = `❌ VLLM 오류: ${error.message}`;
        }
      } else if (provider === 'anthropic') {
        if (errorStr.includes('api key') || errorStr.includes('401')) {
          errorMessage = `🔑 Anthropic API 키 오류: ANTHROPIC_API_KEY 환경변수를 확인하세요.`;
        } else if (errorStr.includes('rate limit')) {
          errorMessage = `⏱️ Anthropic API 요청 제한: 잠시 후 다시 시도하세요.`;
        } else {
          errorMessage = `❌ Anthropic 오류: ${error.message}`;
        }
      } else if (provider === 'openai') {
        if (errorStr.includes('api key') || errorStr.includes('401')) {
          errorMessage = `🔑 OpenAI API 키 오류: OPENAI_API_KEY 환경변수를 확인하세요.`;
        } else if (errorStr.includes('rate limit')) {
          errorMessage = `⏱️ OpenAI API 요청 제한: 잠시 후 다시 시도하세요.`;
        } else {
          errorMessage = `❌ OpenAI 오류: ${error.message}`;
        }
      } else if (provider === 'gemini') {
        if (errorStr.includes('api key') || errorStr.includes('401')) {
          errorMessage = `🔑 Gemini API 키 오류: GEMINI_API_KEY 환경변수를 확인하세요.`;
        } else if (errorStr.includes('rate limit')) {
          errorMessage = `⏱️ Gemini API 요청 제한: 잠시 후 다시 시도하세요.`;
        } else {
          errorMessage = `❌ Gemini 오류: ${error.message}`;
        }
      } else {
        // 일반적인 오류 처리
        if (errorStr.includes('tool_use_failed') || errorStr.includes('function')) {
          errorMessage = `🔧 도구 호출 실패: ${provider} 프로바이더 또는 ${model} 모델이 도구 호출을 지원하지 않습니다.`;
        } else if (errorStr.includes('api key') || errorStr.includes('401')) {
          errorMessage = `🔑 API 키 오류: ${provider.toUpperCase()}_API_KEY 환경변수를 확인하세요.`;
        } else if (errorStr.includes('connect') || errorStr.includes('network')) {
          errorMessage = `🔌 연결 오류: ${provider} 서버에 연결할 수 없습니다.`;
        } else {
          errorMessage = `❌ ${provider} 오류: ${error.message}`;
        }
      }
    } else {
      errorMessage = `❌ 알 수 없는 오류가 발생했습니다.`;
    }
    
    yield `data: ${JSON.stringify({ content: errorMessage })}\n\n`;
    yield `data: [DONE]\n\n`;
  }
}

// 로컬 파일 시스템에 세션 저장 (외부 API 서버 없이도 동작)
async function saveSessionToLocal(sessionData: any) {
  console.log(`💾 세션 저장 시작: ${sessionData.id}`);
  
  const sessionsDir = path.join(os.homedir(), '.airun', 'sessions');
  
  // 디렉토리 생성 (존재하지 않으면)
  if (!fs.existsSync(sessionsDir)) {
    fs.mkdirSync(sessionsDir, { recursive: true });
    console.log(`📁 세션 디렉토리 생성: ${sessionsDir}`);
  }
  
  const sessionFile = path.join(sessionsDir, `${sessionData.id}.json`);
  
  try {
    // 세션 파일에 저장
    fs.writeFileSync(sessionFile, JSON.stringify(sessionData, null, 2), 'utf8');
    console.log(`✅ 세션 파일 저장 완료: ${sessionFile}`);
    console.log(`📊 세션 데이터:`, {
      id: sessionData.id,
      historyLength: sessionData.history?.length || 0,
      title: sessionData.title
    });
  } catch (error) {
    console.error(`❌ 세션 파일 저장 실패: ${sessionFile}`, error);
  }
}

// 세션 제목 생성 함수 (로컬용)
function generateSessionTitle(history: any[]): string {
  if (!history || history.length === 0) return '새 대화';
  
  const firstUserMessage = history.find(msg => msg.role === 'user' && msg.content);
  if (firstUserMessage && firstUserMessage.content) {
    let content = firstUserMessage.content.trim();
    
    // 특수 문자 제거 및 정리
    content = content.replace(/[^\w\s가-힣]/g, ' ').replace(/\s+/g, ' ').trim();
    
    // 질문 형태로 변환
    if (content.length > 30) {
      content = content.substring(0, 30) + '...';
    }
    
    // 빈 문자열이면 기본값 반환
    if (!content) {
      return '새 대화';
    }
    
    return content;
  }
  
  return '새 대화';
}

// 마지막 메시지 가져오기 함수 (로컬용)
function getLastMessage(history: any[]): string {
  if (!history || history.length === 0) return '새 대화';
  
  // 마지막 assistant 메시지 우선 찾기
  const lastAssistantMessage = [...history].reverse().find(msg => msg.role === 'assistant' && msg.content);
  if (lastAssistantMessage && lastAssistantMessage.content) {
    let content = lastAssistantMessage.content.trim();
    
    // 마크다운 문법 제거
    content = content.replace(/[#*`_~\[\]()]/g, '').replace(/\s+/g, ' ').trim();
    
    if (content.length > 50) {
      content = content.substring(0, 50) + '...';
    }
    
    return content || '새 대화';
  }
  
  // assistant 메시지가 없으면 마지막 user 메시지
  const lastUserMessage = [...history].reverse().find(msg => msg.role === 'user' && msg.content);
  if (lastUserMessage && lastUserMessage.content) {
    let content = lastUserMessage.content.trim();
    content = content.replace(/[^\w\s가-힣]/g, ' ').replace(/\s+/g, ' ').trim();
    
    if (content.length > 50) {
      content = content.substring(0, 50) + '...';
    }
    
    return content || '새 대화';
  }
  
  return '새 대화';
}

// POST: 채팅 메시지 처리 (LangGraph + MCP 도구 사용)
export async function POST(request: NextRequest) {
  try {
    console.log('📥 MCP 채팅 요청 수신');
    const contentType = request.headers.get('content-type') || '';
    console.log('📋 Content-Type:', contentType);
    
    let message: string;
    let provider: string;
    let model: string;
    let temperature: number | undefined;
    let sessionId: string | undefined;
    let history: Array<{role: 'user' | 'assistant', content: string}> = [];
    let files: any[] = [];
    
    // Content-Type에 따라 다른 방식으로 파싱
    if (contentType.includes('multipart/form-data')) {
      console.log('📦 FormData 요청으로 파싱');
      console.time('FormData 파싱 시간');
      
      // 메모리 사용량 체크
      const beforeMem = process.memoryUsage();
      console.log('📊 파싱 전 메모리:', `${Math.round(beforeMem.heapUsed / 1024 / 1024)}MB`);
      
      let formData: FormData;
      try {
        // FormData 파싱을 더 효율적으로 처리
        console.log('🔄 FormData 파싱 시작...');
        
        // 요청 크기 확인
        const contentLength = request.headers.get('content-length');
        if (contentLength) {
          const sizeMB = parseInt(contentLength) / 1024 / 1024;
          console.log(`📏 요청 크기: ${sizeMB.toFixed(2)}MB`);
          
          // 50MB 초과 시 거부
          if (sizeMB > 50) {
            throw new Error(`요청 크기가 너무 큽니다: ${sizeMB.toFixed(2)}MB. 최대 50MB까지 지원됩니다.`);
          }
        }
        
        // 타임아웃을 30초로 늘리고 진행 상황 모니터링
        const parsePromise = request.formData();
        const timeoutPromise = new Promise((_, reject) => 
          setTimeout(() => reject(new Error('FormData 파싱 타임아웃 (30초)')), 30000)
        );
        
        formData = await Promise.race([parsePromise, timeoutPromise]) as FormData;
        
        console.timeEnd('FormData 파싱 시간');
        
        // 파싱 후 메모리 사용량 체크
        const afterMem = process.memoryUsage();
        console.log('📊 파싱 후 메모리:', `${Math.round(afterMem.heapUsed / 1024 / 1024)}MB`);
        console.log('📈 메모리 증가량:', `${Math.round((afterMem.heapUsed - beforeMem.heapUsed) / 1024 / 1024)}MB`);
        console.log('✅ FormData 파싱 완료');
        
      } catch (error) {
        console.error('❌ FormData 파싱 실패:', error);
        
        // 구체적인 오류 메시지 제공
        if (error instanceof Error) {
          if (error.message.includes('타임아웃')) {
            return NextResponse.json(
              { error: '파일 처리 시간이 초과되었습니다. 이미지 크기를 줄이거나 파일 개수를 줄여주세요.' },
              { status: 408 }
            );
          } else if (error.message.includes('크기가 너무')) {
            return NextResponse.json(
              { error: error.message },
              { status: 413 }
            );
          }
        }
        
        return NextResponse.json(
          { error: `파일 처리 실패: ${error instanceof Error ? error.message : error}` },
          { status: 400 }
        );
      }
      
      message = formData.get('message') as string;
      provider = formData.get('provider') as string;
      model = formData.get('model') as string;
      console.log(`📝 기본 파라미터: provider=${provider}, model=${model}, message=${message?.substring(0, 50)}...`);
      temperature = parseFloat(formData.get('temperature') as string) || undefined;
      sessionId = formData.get('sessionId') as string || undefined;
      const historyString = formData.get('history') as string;
      
      if (historyString) {
        try {
          history = JSON.parse(historyString);
        } catch (error) {
          console.error('히스토리 파싱 실패:', error);
        }
      }
      
      // 다중 파일 처리 - 더 효율적으로 개선
      let fileIndex = 0;
      console.log('📁 파일 처리 시작...');
      const maxFiles = 10; // 최대 파일 수 제한
      
      while (fileIndex < maxFiles) {
        const file = formData.get(`file_${fileIndex}`) as File | null;
        if (!file) break;
        
        console.log(`📎 파일 ${fileIndex} 처리: ${file.name} (${file.type}, ${(file.size / 1024 / 1024).toFixed(2)}MB)`);
        
        // 개별 파일 크기 제한 (20MB)
        if (file.size > 20 * 1024 * 1024) {
          console.warn(`⚠️ 파일 크기 초과, 건너뜀: ${file.name} (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
          fileIndex++;
          continue;
        }
        
        const fileInfo: any = {
          name: file.name,
          type: file.type,
          size: file.size,
          file: file
        };
        
        // 텍스트 추출 데이터
        const extractedText = formData.get(`file_${fileIndex}_text`) as string | null;
        if (extractedText) {
          fileInfo.extractedText = extractedText;
        }
        
        // Base64 데이터 (이미지) - 메모리 효율적으로 처리
        const base64Data = formData.get(`file_${fileIndex}_base64`) as string | null;
        if (base64Data) {
          console.log(`📷 Base64 데이터 수신: ${file.name}, 원본 길이: ${(base64Data.length / 1024 / 1024).toFixed(2)}MB`);
          
          // Base64 데이터 크기 제한 (15MB)
          if (base64Data.length > 15 * 1024 * 1024) {
            console.warn(`⚠️ Base64 데이터 크기 초과, 건너뜀: ${file.name}`);
            fileIndex++;
            continue;
          }
          
          // Data URL에서 base64 부분만 추출 (data:image/jpeg;base64, 제거)
          const base64Only = base64Data.includes(',') ? base64Data.split(',')[1] : base64Data;
          fileInfo.base64 = base64Only;
          fileInfo.isImage = true;
          console.log(`📷 이미지 Base64 데이터 처리 완료: ${file.name}, 처리된 길이: ${(base64Only.length / 1024 / 1024).toFixed(2)}MB`);
        }
        
        files.push(fileInfo);
        console.log(`✅ 파일 ${fileIndex} 처리 완료: ${file.name}`);
        fileIndex++;
      }
      
      if (fileIndex >= maxFiles) {
        console.warn(`⚠️ 최대 파일 수(${maxFiles}개) 제한으로 일부 파일이 무시되었습니다.`);
      }
      
      console.log(`📁 전체 파일 처리 완료: ${files.length}개 파일`);
      
    } else {
      console.log('📋 JSON 요청으로 파싱');
      const body = await request.json();
      message = body.message;
      provider = body.provider;
      model = body.model;
      temperature = body.temperature;
      sessionId = body.sessionId;
      history = body.history || [];
      
      // JSON 요청에서 파일 처리
      if (body.files && Array.isArray(body.files)) {
        console.log(`📁 JSON 파일 처리 시작: ${body.files.length}개 파일`);
        
        body.files.forEach((fileData: any, index: number) => {
          console.log(`📎 파일 ${index} 처리: ${fileData.name} (${fileData.type}, ${(fileData.size / 1024 / 1024).toFixed(2)}MB)`);
          
          const fileInfo: any = {
            name: fileData.name,
            type: fileData.type,
            size: fileData.size
          };
          
          // 텍스트 추출 데이터
          if (fileData.extractedText) {
            fileInfo.extractedText = fileData.extractedText;
          }
          
          // Base64 데이터 (이미지)
          if (fileData.base64) {
            console.log(`📷 Base64 데이터 수신: ${fileData.name}, 길이: ${(fileData.base64.length / 1024 / 1024).toFixed(2)}MB`);
            
            // Base64 데이터 크기 제한 (15MB)
            if (fileData.base64.length > 15 * 1024 * 1024) {
              console.warn(`⚠️ Base64 데이터 크기 초과, 건너뜀: ${fileData.name}`);
              return;
            }
            
            // Data URL에서 base64 부분만 추출 (data:image/jpeg;base64, 제거)
            const base64Only = fileData.base64.includes(',') ? fileData.base64.split(',')[1] : fileData.base64;
            fileInfo.base64 = base64Only;
            fileInfo.isImage = true;
            console.log(`📷 이미지 Base64 데이터 처리 완료: ${fileData.name}, 처리된 길이: ${(base64Only.length / 1024 / 1024).toFixed(2)}MB`);
          }
          
          files.push(fileInfo);
          console.log(`✅ 파일 ${index} 처리 완료: ${fileData.name}`);
        });
        
        console.log(`📁 JSON 파일 처리 완료: ${files.length}개 파일`);
      } else {
        console.log('📄 JSON 요청에 파일이 없습니다.');
      }
    }
    
    console.log(`📍 파싱된 요청 데이터:`, {
      message: message?.substring(0, 100) + (message?.length > 100 ? '...' : ''),
      provider,
      model,
      temperature,
      sessionId,
      historyLength: history.length,
      fileCount: files.length
    });
    
    if (!message || message.trim() === '') {
      return NextResponse.json(
        { error: '메시지가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // 세션 ID가 없으면 새로 생성 (assistant- 접두어 사용)
    if (!sessionId) {
      sessionId = `assistant-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      console.log(`🆕 새 세션 ID 생성: ${sessionId}`);
    }
    
    // 모든 프로바이더 지원 (제한 제거)
    console.log(`📍 요청된 프로바이더: ${provider}, 모델: ${model}, 세션: ${sessionId}`);
    
    console.log(`🦜 LangGraph 채팅 요청: ${message}`);
    if (files.length > 0) {
      console.log(`📎 첨부 파일 ${files.length}개:`, files.map(f => {
        const info = `${f.name} (${f.type})`;
        if (f.isImage && f.base64) {
          return `${info} [이미지: ${(f.base64.length / 1024 / 1024).toFixed(2)}MB]`;
        } else if (f.extractedText) {
          return `${info} [텍스트: ${f.extractedText.length}자]`;
        }
        return info;
      }).join(', '));
    }

    // 스트리밍 응답 생성
    return createStreamingResponse(async function* () {
      yield* processWithLangGraph(message, files, provider, model, temperature, history, sessionId);
    });
    
  } catch (error) {
    console.error('MCP 채팅 처리 실패:', error);
    return NextResponse.json(
      { error: '채팅 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

// GET: 채팅 상태 확인
export async function GET() {
  try {
    const config = getAirunConfig();
    
    const mcpTools = await loadMCPTools();
    
    return NextResponse.json({
      available: true,
      mode: 'mcp-dynamic',
      provider: config.USE_LLM || 'anthropic',
      model: config.ANTHROPIC_MODEL || config.OPENAI_MODEL || 'default',
      supportedProviders: ['openai', 'anthropic', 'ollama', 'gemini', 'vllm', 'claude', 'groq'],
      toolsCount: mcpTools.length,
      tools: mcpTools.map(tool => tool.name),
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('LangGraph 채팅 상태 확인 실패:', error);
    return NextResponse.json(
      { error: '채팅 상태 확인에 실패했습니다.' },
      { status: 500 }
    );
  }
}

// 서버 종료 시 정리
process.on('SIGINT', async () => {
  cleanupGlobalMCPClient();
});

process.on('SIGTERM', async () => {
  cleanupGlobalMCPClient();
}); 