import { NextRequest, NextResponse } from 'next/server';
import { withTokenAuth } from '../../middlewares';

const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

// API 키 목록 조회
export async function GET(req: NextRequest) {
  return withTokenAuth(req, async (req, user) => {
    try {
      console.log('API 서버에 API 키 목록 요청:', `${API_SERVER}/api-keys`);
      
      // API 서버에 API 키 목록 요청
      const apiResponse = await fetch(`${API_SERVER}/api-keys`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${req.cookies.get('auth_token')?.value}`
        }
      });
      
      if (!apiResponse.ok) {
        console.error('API 서버 응답 오류:', {
          status: apiResponse.status,
          statusText: apiResponse.statusText
        });
        
        let errorMessage = 'API 키 목록을 불러오는데 실패했습니다.';
        try {
          const errorData = await apiResponse.json();
          if (errorData.error?.message) {
            errorMessage = errorData.error.message;
          }
        } catch (parseError) {
          console.error('응답 파싱 오류:', parseError);
        }
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: apiResponse.status }
        );
      }
      
      const apiData = await apiResponse.json();
      console.log('API 키 목록 조회 성공');
      
      return NextResponse.json({
        success: true,
        data: {
          keys: apiData.data
        }
      });
    } catch (error) {
      console.error('API 키 목록 조회 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: 'API 키 목록을 불러오는 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
}

// 새 API 키 생성
export async function POST(req: NextRequest) {
  return withTokenAuth(req, async (req, user) => {
    try {
      const body = await req.json();
      const { name } = body;
      
      if (!name) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: 'API 키 이름이 필요합니다.'
            }
          },
          { status: 400 }
        );
      }
      
      console.log('API 서버에 API 키 생성 요청:', {
        url: `${API_SERVER}/api-keys`,
        name
      });
      
      // API 서버에 API 키 생성 요청
      const apiResponse = await fetch(`${API_SERVER}/api-keys`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${req.cookies.get('auth_token')?.value}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          name,
        })
      });
      
      if (!apiResponse.ok) {
        console.error('API 서버 응답 오류:', {
          status: apiResponse.status,
          statusText: apiResponse.statusText
        });
        
        let errorMessage = 'API 키 생성에 실패했습니다.';
        try {
          const errorData = await apiResponse.json();
          if (errorData.error?.message) {
            errorMessage = errorData.error.message;
          }
        } catch (parseError) {
          console.error('응답 파싱 오류:', parseError);
        }
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: apiResponse.status }
        );
      }
      
      const apiData = await apiResponse.json();
      console.log('API 키 생성 성공');
      
      return NextResponse.json({
        success: true,
        data: apiData.data
      });
    } catch (error) {
      console.error('API 키 생성 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: 'API 키 생성 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 