import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

const getAirunConfigPath = () => {
  const homeDir = os.homedir();
  return path.join(homeDir, '.airun', 'airun.conf');
};

// airun.conf에서 설정 읽기
const getAirunConfig = () => {
  const configPath = getAirunConfigPath();
  const config: any = {};
  
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            config[match[1]] = match[2];
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  return config;
};

// app.py 패턴을 적용한 개선된 서버 연결 체크
class ImprovedServerClient {
  private static modelCache = new Map<string, { data: any[], timestamp: number }>();
  private static CACHE_DURATION = 60000; // 1분 캐싱

  // 서버 연결 상태를 미리 체크하는 함수 (app.py 패턴)
  static async checkServerConnection(url: string, timeout: number = 5000): Promise<boolean> {
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), timeout);
      
      const response = await fetch(url, {
        method: 'GET',
        signal: controller.signal,
        headers: { 'Content-Type': 'application/json' }
      });
      
      clearTimeout(timeoutId);
      return response.ok;
    } catch (error) {
      // 연결 실패는 debug 레벨로 로깅 (app.py 패턴)
      console.debug(`서버 연결 체크 실패: ${url} - ${(error as Error).message}`);
      return false;
    }
  }

  // 캐시에서 모델 데이터 가져오기
  static getCachedModels(cacheKey: string): any[] | null {
    const cached = this.modelCache.get(cacheKey);
    if (cached && (Date.now() - cached.timestamp) < this.CACHE_DURATION) {
      console.debug(`캐시에서 ${cacheKey} 모델 반환`);
      return cached.data;
    }
    return null;
  }

  // 캐시에 모델 데이터 저장
  static setCachedModels(cacheKey: string, models: any[]): void {
    this.modelCache.set(cacheKey, {
      data: models,
      timestamp: Date.now()
    });
  }
}

// app.py 패턴을 적용한 개선된 Ollama 모델 목록 가져오기
const getOllamaModels = async (): Promise<any[]> => {
  const cacheKey = 'ollama-models';
  
  // 캐시 확인
  const cachedModels = ImprovedServerClient.getCachedModels(cacheKey);
  if (cachedModels) {
    return cachedModels;
  }

  const defaultModels = [
    { id: 'hamonize:latest', name: 'AI.RUN Chat' },
    { id: 'airun-think:latest', name: 'AI.RUN Think' },
    { id: 'airun-vision:latest', name: 'AI.RUN Vision' }
  ];

  try {
    const ollamaUrl = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || "https://api.hamonize.com/ollama";
    
    // app.py 패턴: 먼저 서버 연결 상태 체크
    const isConnected = await ImprovedServerClient.checkServerConnection(`${ollamaUrl}/api/tags`, 3000);
    
    if (!isConnected) {
      console.debug('Ollama 서버에 연결할 수 없음, 기본 모델 사용');
      ImprovedServerClient.setCachedModels(cacheKey, defaultModels);
      return defaultModels;
    }

    // app.py 패턴: 타임아웃 10초로 제한
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 10000);

    const response = await fetch(`${ollamaUrl}/api/tags`, {
      method: 'GET',
      headers: { 'Content-Type': 'application/json' },
      signal: controller.signal
    });

    clearTimeout(timeoutId);

    if (response.ok) {
      const data = await response.json();
      const models = (data.models || []).map((model: any) => ({
        id: model.name,
        name: model.name,
        size: model.size,
        modified_at: model.modified_at
      }));
      
      if (models.length > 0) {
        console.debug(`Ollama에서 ${models.length}개 모델 로드 성공`);
        ImprovedServerClient.setCachedModels(cacheKey, models);
        return models;
      }
    }
  } catch (error) {
    // app.py 패턴: 상세한 에러는 debug 레벨로만 로깅
    console.debug('Ollama 모델 조회 중 오류:', (error as Error).message);
  }
  
  // 기본 모델 반환 및 캐싱
  console.debug('Ollama 기본 모델 반환');
  ImprovedServerClient.setCachedModels(cacheKey, defaultModels);
  return defaultModels;
};

// app.py 패턴을 적용한 개선된 VLLM 모델 목록 가져오기
const getVllmModels = async (): Promise<any[]> => {
  const cacheKey = 'vllm-models';
  
  // 캐시 확인
  const cachedModels = ImprovedServerClient.getCachedModels(cacheKey);
  if (cachedModels) {
    return cachedModels;
  }

  const defaultModels = [
    { id: 'meta-llama/Llama-3.1-8B-Instruct', name: 'Llama 3.1 8B Instruct' }
  ];

  try {
    const vllmServer = process.env.NEXT_PUBLIC_VLLM_SERVER || "https://api.hamonize.com/vllm";
    
    // app.py 패턴: 먼저 서버 연결 상태 체크
    const isConnected = await ImprovedServerClient.checkServerConnection(`${vllmServer}/v1/models`, 3000);
    
    if (!isConnected) {
      console.debug('VLLM 서버에 연결할 수 없음, 기본 모델 사용');
      ImprovedServerClient.setCachedModels(cacheKey, defaultModels);
      return defaultModels;
    }

    // app.py 패턴: 타임아웃 10초로 제한
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 10000);

    const response = await fetch(`${vllmServer}/v1/models`, {
      method: 'GET',
      headers: { 'Content-Type': 'application/json' },
      signal: controller.signal
    });

    clearTimeout(timeoutId);

    if (response.ok) {
      const data = await response.json();
      const models = (data.data || []).map((model: any) => ({
        id: model.id,
        name: model.id,
        owned_by: model.owned_by
      }));
      
      if (models.length > 0) {
        console.debug(`VLLM에서 ${models.length}개 모델 로드 성공`);
        ImprovedServerClient.setCachedModels(cacheKey, models);
        return models;
      }
    }
  } catch (error) {
    // app.py 패턴: 상세한 에러는 debug 레벨로만 로깅
    console.debug('VLLM 모델 조회 중 오류:', (error as Error).message);
  }
  
  // 기본 모델 반환 및 캐싱
  console.debug('VLLM 기본 모델 반환');
  ImprovedServerClient.setCachedModels(cacheKey, defaultModels);
  return defaultModels;
};

// 프로바이더별 기본 모델 정의 (실제 AI 시스템과 일치)
const getStaticModels = (provider: string) => {
  const staticModels: any = {
    openai: [
      { id: 'gpt-5', name: 'GPT-5' },
      { id: 'gpt-5-mini', name: 'GPT-5 Mini' },
      { id: 'gpt-5-nano', name: 'GPT-5 Nano' },
      { id: 'gpt-4.1', name: 'GPT-4.1' },
      { id: 'gpt-4.1-mini', name: 'GPT-4.1 Mini' },
      { id: 'gpt-4.1-nano', name: 'GPT-4.1 Nano' },
      { id: 'gpt-4o-mini', name: 'GPT-4 Omni Mini' },
      { id: 'gpt-4o', name: 'GPT-4 Omni' },
      { id: 'gpt-3.5-turbo', name: 'GPT-3.5 Turbo' }
    ],
    anthropic: [
      { id: 'claude-sonnet-4-20250514', name: 'Claude Sonnet 4' },
      { id: 'claude-opus-4-20250514', name: 'Claude Opus 4' },
      { id: 'claude-3-7-sonnet-latest', name: 'Claude 3.7 Sonnet Latest' },
      { id: 'claude-3-sonnet-20240229', name: 'Claude 3 Sonnet' },
      { id: 'claude-3-opus-20240229', name: 'Claude 3 Opus' },
      { id: 'claude-3-haiku-20240307', name: 'Claude 3 Haiku' }
    ],
    gemini: [
      { id: 'gemini-2.5-pro-preview-05-06', name: 'Gemini 2.5 Pro Preview' },
      { id: 'gemini-1.5-pro', name: 'Gemini 1.5 Pro' },
      { id: 'gemma-3-27b-it', name: 'Gemma 3 27B IT' }
    ],
    groq: [
      { id: 'llama-3.3-70b-versatile', name: 'Llama 3.3 70B Versatile' },
      { id: 'llama3-70b-8192', name: 'Llama 3 70B 8192' },
      { id: 'gemma2-9b-it', name: 'Gemma 2 9B IT' },
      { id: 'gemma-7b-it', name: 'Gemma 7B IT' }
    ]
  };

  return staticModels[provider] || [];
};

// app.py 패턴을 적용한 개선된 프로바이더 목록 및 실제 모델 정의
const getProviders = async () => {
  const config = getAirunConfig();
  
  // app.py 패턴: 동적 모델 목록을 병렬로 가져오기 (타임아웃 적용)
  const [ollamaModels, vllmModels] = await Promise.allSettled([
    Promise.race([
      getOllamaModels(),
      new Promise<any[]>((_, reject) => 
        setTimeout(() => reject(new Error('Ollama 모델 로드 시간 초과')), 15000)
      )
    ]),
    Promise.race([
      getVllmModels(),
      new Promise<any[]>((_, reject) => 
        setTimeout(() => reject(new Error('VLLM 모델 로드 시간 초과')), 15000)
      )
    ])
  ]);

  // app.py 패턴: 실패한 요청에 대해서는 기본값 사용
  const finalOllamaModels = ollamaModels.status === 'fulfilled' ? ollamaModels.value : [
    { id: 'hamonize:latest', name: 'AI.RUN Chat' },
    { id: 'airun-think:latest', name: 'AI.RUN Think' }, 
    { id: 'airun-vision:latest', name: 'AI.RUN Vision' }
  ];

  const finalVllmModels = vllmModels.status === 'fulfilled' ? vllmModels.value : [
    { id: 'meta-llama/Llama-3.1-8B-Instruct', name: 'Llama 3.1 8B Instruct' }
  ];

  const providers: any = {};

  // 기본 모델 설정 (실제 AI 시스템과 일치)
  const defaultModels = {
    openai: 'gpt-4.1-mini',
    anthropic: 'claude-3-7-sonnet-latest',
    gemini: 'gemini-2.5-pro-preview-05-06',
    groq: 'llama-3.3-70b-versatile',
    ollama: 'hamonize:latest',
    vllm: 'unsloth/gemma-3-12b-it-unsloth-bnb-4bit'
  };

  // API 키가 설정된 프로바이더만 포함
  const providerConfigs = [
    {
      key: 'ollama',
      name: 'Ollama',
      displayName: 'Ollama (로컬)',
      models: finalOllamaModels,
      requiresApiKey: false,
      apiKeyConfigured: true,
      isDynamic: true
    },
    {
      key: 'openai',
      name: 'OpenAI',
      displayName: 'OpenAI',
      models: getStaticModels('openai'),
      requiresApiKey: true,
      apiKeyConfigured: !!config.OPENAI_API_KEY,
      isDynamic: false
    },
    {
      key: 'anthropic',
      name: 'Anthropic',
      displayName: 'Claude',
      models: getStaticModels('anthropic'),
      requiresApiKey: true,
      apiKeyConfigured: !!config.ANTHROPIC_API_KEY,
      isDynamic: false
    },
    {
      key: 'gemini',
      name: 'Google',
      displayName: 'Gemini',
      models: getStaticModels('gemini'),
      requiresApiKey: true,
      apiKeyConfigured: !!config.GEMINI_API_KEY,
      isDynamic: false
    },
    {
      key: 'groq',
      name: 'Groq',
      displayName: 'Groq',
      models: getStaticModels('groq'),
      requiresApiKey: true,
      apiKeyConfigured: !!config.GROQ_API_KEY,
      isDynamic: false
    },
    {
      key: 'vllm',
      name: 'VLLM',
      displayName: 'VLLM (분산처리)',
      models: finalVllmModels,
      requiresApiKey: false,
      apiKeyConfigured: !!config.VLLM_SERVER,
      isDynamic: true
    }
  ];

  // API 키가 있거나 키가 필요없는 프로바이더만 포함
  for (const providerConfig of providerConfigs) {
    if (!providerConfig.requiresApiKey || providerConfig.apiKeyConfigured) {
      providers[providerConfig.key] = {
        name: providerConfig.name,
        displayName: providerConfig.displayName,
        models: providerConfig.models,
        requiresApiKey: providerConfig.requiresApiKey,
        apiKeyConfigured: providerConfig.apiKeyConfigured,
        isDynamic: providerConfig.isDynamic,
        isAvailable: true
      };
    }
  }

  // 현재 선택된 프로바이더 표시
  const currentProvider = config.USE_LLM || 'ollama';
  if (providers[currentProvider]) {
    providers[currentProvider].isCurrent = true;
  }

  return providers;
};

// GET: app.py 패턴을 적용한 개선된 프로바이더 목록 조회
export async function GET() {
  try {
    console.debug('프로바이더 목록 조회 시작 (Improved Pattern)');
    
    // app.py 패턴: 전체 작업에 대한 타임아웃 적용
    const providers = await Promise.race([
      getProviders(),
      new Promise((_, reject) => 
        setTimeout(() => reject(new Error('프로바이더 목록 조회 시간 초과')), 30000)
      )
    ]) as any;
    
    const responseData = {
      success: true,
      data: providers,
      timestamp: new Date().toISOString(),
      implementation: 'Improved Providers API (app.py pattern)',
      features: [
        '3초 연결 체크',
        '10초 요청 타임아웃',
        '1분 응답 캐싱',
        'Graceful fallback',
        '병렬 모델 로딩'
      ]
    };

    console.debug('프로바이더 목록 조회 완료');
    return NextResponse.json(responseData);
    
  } catch (error) {
    console.error('프로바이더 목록 조회 실패:', error);
    
    // app.py 패턴: 실패 시 최소한의 기본 프로바이더 반환
    const fallbackProviders = {
      ollama: {
        name: 'Ollama',
        displayName: 'Ollama (로컬)', 
        models: [
          { id: 'hamonize:latest', name: 'AI.RUN Chat' },
          { id: 'airun-think:latest', name: 'AI.RUN Think' },
          { id: 'airun-vision:latest', name: 'AI.RUN Vision' }
        ],
        requiresApiKey: false,
        apiKeyConfigured: true,
        isDynamic: true,
        isAvailable: true,
        isCurrent: true
      }
    };

    return NextResponse.json({
      success: true,
      data: fallbackProviders,
      timestamp: new Date().toISOString(),
      implementation: 'Improved Providers API (app.py pattern - Fallback)',
      note: '일부 서버 연결 실패로 기본 설정 반환'
    });
  }
}

// POST: 프로바이더 설정 업데이트
export async function POST(request: NextRequest) {
  try {
    const { provider, model } = await request.json();
    
    if (!provider) {
      return NextResponse.json(
        { 
          success: false, 
          error: { message: '프로바이더가 필요합니다.' } 
        },
        { status: 400 }
      );
    }

    const providers = await getProviders();
    
    if (!providers[provider]) {
      return NextResponse.json(
        { 
          success: false, 
          error: { message: '유효하지 않은 프로바이더입니다.' } 
        },
        { status: 400 }
      );
    }

    // 여기에 실제 설정 업데이트 로직을 구현할 수 있습니다.
    // 현재는 성공 응답만 반환합니다.
    
    return NextResponse.json({
      success: true,
      data: {
        provider,
        model,
        updated: true
      },
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('프로바이더 설정 업데이트 실패:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: { 
          message: '프로바이더 설정 업데이트 중 오류가 발생했습니다.' 
        } 
      },
      { status: 500 }
    );
  }
} 