import { NextResponse } from 'next/server';

const RAG_API_BASE_URL = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';

export async function GET() {
  try {
    const response = await fetch(`${RAG_API_BASE_URL}/models/finetune/status`);
    
    if (!response.ok) {
      // RAG 서버 연결 실패 또는 HTTP 오류 처리
      const errorMessage = response.status === 0 || !response.status
        ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
        : `RAG API 호출 실패: ${response.status} ${response.statusText}`;
      
      return NextResponse.json(
        { success: false, error: errorMessage },
        { status: response.status || 503 }
      );
    }
    
    const data = await response.json();
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('RAG API 파인튜닝 상태 확인 실패:', error);
    
    // 연결 오류인지 확인
    const isConnectionError = error instanceof Error && 
      (error.message.includes('ECONNREFUSED') || 
       error.message.includes('fetch failed') ||
       (error.cause && typeof error.cause === 'object' && 'code' in error.cause && error.cause.code === 'ECONNREFUSED'));
    
    const errorMessage = isConnectionError
      ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
      : '파인튜닝 상태를 확인할 수 없습니다.';
    
    return NextResponse.json(
      { success: false, error: errorMessage },
      { status: isConnectionError ? 503 : 500 }
    );
  }
} 