import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();

// 파일 업로드 및 RAG 문서 추가 API
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      // formData 형식으로 전송된 데이터 파싱
      const formData = await req.formData();
      const file = formData.get('file') as File;
      
      if (!file) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '파일이 제공되지 않았습니다.'
            }
          }, 
          { status: 400 }
        );
      }
      
      // 토큰 또는 API 키 설정
      const headers: Record<string, string> = {};

      if (authType === 'token') {
        const token = req.cookies.get('auth_token')?.value;
        if (token) {
          headers['Authorization'] = `Bearer ${token}`;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: '인증 토큰이 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else if (authType === 'apikey') {
        const apiKey = req.headers.get('x-api-key');
        if (apiKey) {
          headers['X-API-Key'] = apiKey;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: 'API 키가 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '인증이 필요합니다.'
            }
          }, 
          { status: 401 }
        );
      }
      
      // 파일 데이터를 바이너리 형태로 준비
      const bytes = await file.arrayBuffer();
      const buffer = Buffer.from(bytes);
      
      // API 서버로 파일 업로드 요청을 위한 FormData 생성
      const apiFormData = new FormData();
      const blob = new Blob([buffer], { type: file.type });
      apiFormData.append('file', blob, file.name);
      
      // 사용자 정보 디버깅
      console.log('사용자 정보 확인:', {
        user: user,
        userId: user?.id,
        authType: authType
      });
      
      if (user?.id) {
        apiFormData.append('user_id', user.id);
        console.log('사용자 ID를 FormData에 추가:', user.id);
      } else {
        console.log('사용자 ID가 없습니다. user 객체:', user);
      }
      
      // API 서버에 파일 업로드 요청
      console.log('API 서버에 파일 업로드 요청 전송:', {
        url: `${API_SERVER}/rag/upload`,
        method: 'POST',
        file: {
          name: file.name,
          type: file.type,
          size: file.size
        },
        userId: user?.id
      });
      
      try {
        const apiResponse = await fetch(`${API_SERVER}/rag/upload`, {
          method: 'POST',
          headers: headers,
          body: apiFormData
        });
        
        if (!apiResponse.ok) {
          const errorText = await apiResponse.text();
          console.error('API 서버 파일 업로드 응답 오류:', {
            status: apiResponse.status,
            statusText: apiResponse.statusText,
            error: errorText
          });
          
          return NextResponse.json(
            {
              success: false,
              error: {
                message: `파일 업로드 실패: ${apiResponse.statusText}`,
                details: errorText
              }
            }, 
            { status: apiResponse.status }
          );
        }
        
        const apiData = await apiResponse.json();
        console.log('API 서버 파일 업로드 응답 성공:', apiData);
        
        return NextResponse.json({
          success: true,
          data: {
            id: apiData.id || `file-${Date.now()}`,
            name: file.name,
            type: file.type,
            size: file.size,
            message: '파일이 성공적으로 업로드되었습니다.'
          }
        });
      } catch (error) {
        console.error('API 서버 파일 업로드 요청 실패:', error);
        
        // 모의 응답 생성 (개발 목적)
        return NextResponse.json({
          success: true,
          data: {
            id: `file-${Date.now()}`,
            name: file.name,
            type: file.type,
            size: file.size,
            message: '파일이 성공적으로 업로드되었습니다. (모의 응답)'
          }
        });
      }
    } catch (error) {
      console.error('파일 업로드 요청 처리 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '파일 업로드 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 