'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { getBusinessServerUrl } from '@/config/serverConfig';
import { 
  FileText, 
  Download, 
  Edit3, 
  Eye,
  RefreshCw,
  Plus,
  Search,
  Filter,
  Calendar,
  Clock,
  Star,
  Award,
  CheckCircle,
  AlertCircle,
  XCircle,
  ArrowRight,
  Trash2,
  Copy,
  Share,
  Settings,
  Zap,
  Target,
  Users,
  X
} from 'lucide-react';

interface Proposal {
  id: string;
  title: string;
  announcement_id: string;
  announcement_title: string;
  company_id: string;
  company_name: string;
  status: 'draft' | 'generating' | 'completed' | 'failed';
  template_type: string;
  content_sections: string[];
  generation_progress: number;
  created_at: string;
  updated_at: string;
  file_url?: string;
  success_probability: number;
  estimated_budget: number;
  key_features: string[];
}

interface ProposalTemplate {
  id: string;
  name: string;
  description: string;
  sections: string[];
  category: string;
  estimated_time: number;
}

function BusinessProposalsContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [proposals, setProposals] = useState<Proposal[]>([]);
  const [templates, setTemplates] = useState<ProposalTemplate[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [selectedTemplate, setSelectedTemplate] = useState<string>('');
  const [generationInProgress, setGenerationInProgress] = useState<string | null>(null);

  useEffect(() => {
    fetchProposals();
    fetchTemplates();

    // Check if we came from matching page
    const announcementId = searchParams.get('announcement');
    const companyId = searchParams.get('company');
    if (announcementId && companyId) {
      setShowCreateModal(true);
    }
  }, [searchParams]);

  const fetchProposals = async () => {
    try {
      setLoading(true);
      const baseUrl = getBusinessServerUrl();
      const response = await fetch(`${baseUrl}/proposals`);
      
      if (response.ok) {
        const data = await response.json();
        setProposals(data);
      } else {
        setMockProposals();
      }
    } catch (error) {
      console.error('Error fetching proposals:', error);
      setMockProposals();
    } finally {
      setLoading(false);
    }
  };

  const fetchTemplates = async () => {
    try {
      const baseUrl = getBusinessServerUrl();
      const response = await fetch(`${baseUrl}/proposal-templates`);
      
      if (response.ok) {
        const data = await response.json();
        setTemplates(data);
      } else {
        setMockTemplates();
      }
    } catch (error) {
      console.error('Error fetching templates:', error);
      setMockTemplates();
    }
  };

  const setMockProposals = () => {
    setProposals([
      {
        id: 'prop1',
        title: 'AI 기반 스마트팩토리 구축 제안서',
        announcement_id: 'bizinfo_real_20250911154638_2074',
        announcement_title: 'AI 기반 스마트팩토리 구축 지원사업',
        company_id: 'comp1',
        company_name: '(주)테크이노베이션',
        status: 'completed',
        template_type: 'technical_rd',
        content_sections: ['개요', '기술현황', '추진계획', '예산계획', '기대효과'],
        generation_progress: 100,
        created_at: '2025-09-11T09:30:00Z',
        updated_at: '2025-09-11T10:45:00Z',
        file_url: '/proposals/prop1.pdf',
        success_probability: 87,
        estimated_budget: 500000000,
        key_features: ['AI 통합', '실시간 모니터링', 'IoT 센서 네트워크']
      },
      {
        id: 'prop2',
        title: '인도네시아 전시회 참가 제안서',
        announcement_id: 'bizinfo_real_20250911154638_2073',
        announcement_title: '[부산] 2025년 인도네시아 산업기계 전시회 부산관 참가 지원',
        company_id: 'comp1',
        company_name: '(주)테크이노베이션',
        status: 'generating',
        template_type: 'export_support',
        content_sections: ['사업개요', '해외진출계획', '전시참가계획', '마케팅전략'],
        generation_progress: 65,
        created_at: '2025-09-11T11:15:00Z',
        updated_at: '2025-09-11T11:30:00Z',
        success_probability: 94,
        estimated_budget: 200000000,
        key_features: ['해외진출', '전시회 참가', 'B2B 네트워킹']
      },
      {
        id: 'prop3',
        title: '친환경 에너지 솔루션 R&D 제안서',
        announcement_id: 'bizinfo_real_20250911154638_2075',
        announcement_title: '중소기업 기술혁신 R&D 지원',
        company_id: 'comp2',
        company_name: '그린에너지솔루션',
        status: 'draft',
        template_type: 'rd_innovation',
        content_sections: ['연구배경', '연구목표', '연구내용', '추진체계', '성과목표'],
        generation_progress: 25,
        created_at: '2025-09-11T08:45:00Z',
        updated_at: '2025-09-11T09:00:00Z',
        success_probability: 78,
        estimated_budget: 300000000,
        key_features: ['친환경기술', '에너지효율', '탄소중립']
      }
    ]);
  };

  const setMockTemplates = () => {
    setTemplates([
      {
        id: 'technical_rd',
        name: '기술개발 R&D 제안서',
        description: '기술개발 및 연구개발 과제를 위한 표준 템플릿',
        sections: ['사업개요', '기술현황분석', '연구개발계획', '추진체계', '예산계획', '기대효과'],
        category: 'R&D',
        estimated_time: 45
      },
      {
        id: 'export_support',
        name: '수출지원 사업 제안서',
        description: '해외진출 및 수출지원 사업을 위한 템플릿',
        sections: ['사업개요', '해외시장분석', '진출전략', '마케팅계획', '위험관리', '성과지표'],
        category: '수출지원',
        estimated_time: 30
      },
      {
        id: 'rd_innovation',
        name: '혁신기술 개발 제안서',
        description: '혁신기술 개발 및 사업화를 위한 종합 템플릿',
        sections: ['혁신배경', '기술개발목표', '연구개발내용', '사업화계획', '파급효과'],
        category: '혁신',
        estimated_time: 60
      },
      {
        id: 'startup_support',
        name: '창업지원 사업 제안서',
        description: '창업기업 및 스타트업 지원사업용 템플릿',
        sections: ['창업아이템', '시장분석', '사업모델', '팀구성', '재무계획', '성장전략'],
        category: '창업',
        estimated_time: 35
      }
    ]);
  };

  const generateProposal = async (templateId: string, announcementId?: string, companyId?: string) => {
    try {
      const proposalId = Date.now().toString();
      setGenerationInProgress(proposalId);
      
      const baseUrl = getBusinessServerUrl();
      const response = await fetch(`${baseUrl}/proposals/generate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          template_id: templateId,
          announcement_id: announcementId,
          company_id: companyId
        })
      });
      
      if (response.ok) {
        const data = await response.json();
        // Refresh proposals list
        fetchProposals();
      } else {
        // Mock successful generation
        setTimeout(() => {
          fetchProposals();
          setGenerationInProgress(null);
        }, 3000);
      }
    } catch (error) {
      console.error('Error generating proposal:', error);
    } finally {
      setShowCreateModal(false);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed': return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'generating': return <RefreshCw className="w-5 h-5 text-blue-600 animate-spin" />;
      case 'draft': return <Edit3 className="w-5 h-5 text-yellow-600" />;
      case 'failed': return <XCircle className="w-5 h-5 text-red-600" />;
      default: return <AlertCircle className="w-5 h-5 text-gray-600" />;
    }
  };

  const getStatusText = (status: string) => {
    switch (status) {
      case 'completed': return '완료';
      case 'generating': return '생성중';
      case 'draft': return '초안';
      case 'failed': return '실패';
      default: return '알 수 없음';
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200';
      case 'generating': return 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200';
      case 'draft': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200';
      case 'failed': return 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200';
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 100000000) {
      return `${(amount / 100000000).toFixed(1)}억원`;
    } else if (amount >= 10000) {
      return `${(amount / 10000).toFixed(1)}만원`;
    }
    return `${amount.toLocaleString()}원`;
  };

  const filteredProposals = proposals.filter(proposal => {
    const matchesSearch = proposal.title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         proposal.company_name?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = !statusFilter || proposal.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  if (loading) {
    return (
      <div className="container mx-auto px-6 py-8">
        <div className="flex items-center justify-center min-h-64">
          <RefreshCw className="w-8 h-8 animate-spin text-gray-400" />
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header Actions */}
      <div className="flex justify-end mb-6">
        <button 
          onClick={() => setShowCreateModal(true)}
          className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
        >
          <Plus className="w-4 h-4 mr-2" />
          새 제안서 생성
        </button>
      </div>

      {/* Filters */}
      <div className="flex flex-col sm:flex-row gap-4 mb-6">
        <div className="flex-1 relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
          <input
            type="text"
            placeholder="제안서 제목이나 기업명으로 검색..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-200 rounded-lg dark:border-gray-700 dark:bg-gray-800"
          />
        </div>
        
        <select
          value={statusFilter}
          onChange={(e) => setStatusFilter(e.target.value)}
          className="px-4 py-2 border border-gray-200 rounded-lg dark:border-gray-700 dark:bg-gray-800"
        >
          <option value="">전체 상태</option>
          <option value="completed">완료</option>
          <option value="generating">생성중</option>
          <option value="draft">초안</option>
          <option value="failed">실패</option>
        </select>
      </div>

      {/* Proposals Grid */}
      {filteredProposals.length === 0 ? (
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-8">
          <div className="text-center">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
              {searchTerm || statusFilter ? '검색 결과가 없습니다' : '제안서가 없습니다'}
            </h3>
            <p className="text-gray-600 dark:text-gray-400 mb-4">
              {searchTerm || statusFilter ? 
                '다른 검색어나 필터를 시도해보세요.' : 
                'AI를 활용하여 첫 번째 사업제안서를 생성해보세요.'
              }
            </p>
            {!searchTerm && !statusFilter && (
              <button 
                onClick={() => setShowCreateModal(true)}
                className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors mx-auto"
              >
                <Plus className="w-4 h-4 mr-2" />
                새 제안서 생성
              </button>
            )}
          </div>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredProposals.map((proposal) => (
            <div key={proposal.id} className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="p-6">
                {/* Header */}
                <div className="flex items-start justify-between mb-4">
                  <div className="flex items-center space-x-2">
                    {getStatusIcon(proposal.status)}
                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(proposal.status)}`}>
                      {getStatusText(proposal.status)}
                    </span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <button className="p-1 text-gray-400 hover:text-gray-600 rounded">
                      <Share className="w-4 h-4" />
                    </button>
                    <button className="p-1 text-gray-400 hover:text-gray-600 rounded">
                      <Copy className="w-4 h-4" />
                    </button>
                    <button className="p-1 text-gray-400 hover:text-red-600 rounded">
                      <Trash2 className="w-4 h-4" />
                    </button>
                  </div>
                </div>

                {/* Title and Company */}
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2 line-clamp-2">
                  {proposal.title}
                </h3>
                <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">
                  {proposal.company_name}
                </p>
                <p className="text-xs text-gray-500 mb-4 line-clamp-1">
                  {proposal.announcement_title}
                </p>

                {/* Progress Bar (for generating status) */}
                {proposal.status === 'generating' && (
                  <div className="mb-4">
                    <div className="flex justify-between text-sm text-gray-600 dark:text-gray-400 mb-1">
                      <span>생성 진행률</span>
                      <span>{proposal.generation_progress}%</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-2 dark:bg-gray-700">
                      <div 
                        className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                        style={{ width: `${proposal.generation_progress}%` }}
                      ></div>
                    </div>
                  </div>
                )}

                {/* Stats */}
                <div className="grid grid-cols-2 gap-4 mb-4">
                  <div className="text-center">
                    <div className="text-lg font-bold text-green-600">
                      {proposal.success_probability}%
                    </div>
                    <div className="text-xs text-gray-600 dark:text-gray-400">성공 확률</div>
                  </div>
                  <div className="text-center">
                    <div className="text-lg font-bold text-blue-600">
                      {formatCurrency(proposal.estimated_budget)}
                    </div>
                    <div className="text-xs text-gray-600 dark:text-gray-400">예상 예산</div>
                  </div>
                </div>

                {/* Key Features */}
                <div className="mb-4">
                  <div className="flex flex-wrap gap-1">
                    {proposal.key_features.slice(0, 3).map((feature, index) => (
                      <span key={index} className="inline-block bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300 text-xs px-2 py-1 rounded">
                        {feature}
                      </span>
                    ))}
                    {proposal.key_features.length > 3 && (
                      <span className="inline-block bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300 text-xs px-2 py-1 rounded">
                        +{proposal.key_features.length - 3}
                      </span>
                    )}
                  </div>
                </div>

                {/* Actions */}
                <div className="flex space-x-2">
                  {proposal.status === 'completed' && (
                    <>
                      <button className="flex-1 flex items-center justify-center px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm">
                        <Eye className="w-4 h-4 mr-1" />
                        미리보기
                      </button>
                      <button className="flex items-center justify-center px-3 py-2 border border-gray-200 rounded-lg hover:bg-gray-50 dark:border-gray-700 dark:hover:bg-gray-700 transition-colors">
                        <Download className="w-4 h-4" />
                      </button>
                    </>
                  )}
                  {proposal.status === 'draft' && (
                    <button className="flex-1 flex items-center justify-center px-3 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm">
                      <Edit3 className="w-4 h-4 mr-1" />
                      편집 계속
                    </button>
                  )}
                  {proposal.status === 'generating' && (
                    <button disabled className="flex-1 flex items-center justify-center px-3 py-2 bg-gray-400 text-white rounded-lg cursor-not-allowed text-sm">
                      <RefreshCw className="w-4 h-4 mr-1 animate-spin" />
                      생성중...
                    </button>
                  )}
                  {proposal.status === 'failed' && (
                    <button className="flex-1 flex items-center justify-center px-3 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors text-sm">
                      <RefreshCw className="w-4 h-4 mr-1" />
                      다시 생성
                    </button>
                  )}
                </div>

                {/* Timestamp */}
                <div className="mt-4 pt-4 border-t border-gray-200 dark:border-gray-700">
                  <div className="flex items-center justify-between text-xs text-gray-500">
                    <span>생성일: {new Date(proposal.created_at).toLocaleDateString('ko-KR')}</span>
                    <span>수정일: {new Date(proposal.updated_at).toLocaleDateString('ko-KR')}</span>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}

      {/* Create Proposal Modal */}
      {showCreateModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6 border-b border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <h2 className="text-xl font-semibold text-gray-900 dark:text-white">새 제안서 생성</h2>
                <button 
                  onClick={() => setShowCreateModal(false)}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                >
                  <X className="w-6 h-6" />
                </button>
              </div>
            </div>

            <div className="p-6">
              <div className="mb-6">
                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">템플릿 선택</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  {templates.map((template) => (
                    <div 
                      key={template.id}
                      className={`border rounded-lg p-4 cursor-pointer transition-colors ${
                        selectedTemplate === template.id 
                          ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' 
                          : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600'
                      }`}
                      onClick={() => setSelectedTemplate(template.id)}
                    >
                      <div className="flex items-start justify-between mb-2">
                        <h4 className="font-medium text-gray-900 dark:text-white">{template.name}</h4>
                        <div className="flex items-center text-sm text-gray-500">
                          <Clock className="w-4 h-4 mr-1" />
                          {template.estimated_time}분
                        </div>
                      </div>
                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-3">{template.description}</p>
                      <div className="flex flex-wrap gap-1">
                        {template.sections.slice(0, 3).map((section, index) => (
                          <span key={index} className="inline-block bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300 text-xs px-2 py-1 rounded">
                            {section}
                          </span>
                        ))}
                        {template.sections.length > 3 && (
                          <span className="inline-block bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300 text-xs px-2 py-1 rounded">
                            +{template.sections.length - 3}
                          </span>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              <div className="flex justify-end space-x-3">
                <button 
                  onClick={() => setShowCreateModal(false)}
                  className="px-4 py-2 text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-white transition-colors"
                >
                  취소
                </button>
                <button 
                  onClick={() => selectedTemplate && generateProposal(
                    selectedTemplate, 
                    searchParams.get('announcement') || undefined,
                    searchParams.get('company') || undefined
                  )}
                  disabled={!selectedTemplate || generationInProgress !== null}
                  className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
                >
                  {generationInProgress ? (
                    <>
                      <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                      생성중...
                    </>
                  ) : (
                    <>
                      <Zap className="w-4 h-4 mr-2" />
                      AI 제안서 생성
                    </>
                  )}
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default function BusinessProposals() {
  return (
    <Suspense fallback={
      <div className="container mx-auto px-6 py-8">
        <div className="flex items-center justify-center min-h-64">
          <RefreshCw className="w-8 h-8 animate-spin text-gray-400" />
        </div>
      </div>
    }>
      <BusinessProposalsContent />
    </Suspense>
  );
}