'use client';

import { useState, useRef, useEffect, type FormEvent, type ChangeEvent, useCallback } from 'react';
import type { Message, SupportIssue } from '@/types/support';
import { externalApiClient } from '@/lib/apiClient';
// import { useAuth } from '@/hooks/useAuth'; // 로그인 없이도 접근 가능하도록 변경
import { supportApi } from '@/lib/apiClient';
import { SupportRenderer } from '@/components/SupportRenderer';
import { getRagServerUrl } from '@/config/serverConfig';
import { BackgroundJobStatus } from '@/components/BackgroundJobStatus';
import type { BackgroundJob } from '@/types/support';
import { getAuthHeaders } from '@/utils/api';
// --- START: Icon Components ---
const SendIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="m22 2-7 20-4-9-9-4Z" />
    <path d="M22 2 11 13" />
  </svg>
);
const UploadIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4" />
    <polyline points="17 8 12 3 7 8" />
    <line x1="12" y1="3" x2="12" y2="15" />
  </svg>
);
const UserIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2" />
    <circle cx="12" cy="7" r="4" />
  </svg>
);
const BotIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M12 8V4H8" />
    <rect width="16" height="12" x="4" y="8" rx="2" />
    <path d="M2 14h2" />
    <path d="M20 14h2" />
    <path d="M15 13v2" />
    <path d="M9 13v2" />
  </svg>
);
const PlusCircleIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <circle cx="12" cy="12" r="10" />
    <path d="M8 12h8" />
    <path d="M12 8v8" />
  </svg>
);
const XIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M18 6 6 18" />
    <path d="m6 6 12 12" />
  </svg>
);
const TrashIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M3 6h18" />
    <path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6" />
    <path d="M8 6V4c0-1 1-2 2-2h4c0 1 1 2 2 2v2" />
  </svg>
);
const SpinnerIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M21 12a9 9 0 1 1-6.219-8.56" />
  </svg>
);
const ClockIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <circle cx="12" cy="12" r="10" />
    <polyline points="12 6 12 12 16 14" />
  </svg>
);
const CheckCircleIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14" />
    <polyline points="22 4 12 14.01 9 11.01" />
  </svg>
);
const XCircleIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <circle cx="12" cy="12" r="10" />
    <path d="m15 9-6 6" />
    <path d="m9 9 6 6" />
  </svg>
);
// --- END: Icon Components ---

// --- START: UI Components (Consistent Styling) ---
const Input = (props: React.InputHTMLAttributes<HTMLInputElement>) => <input {...props} className={`flex h-10 w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 px-3 py-2 text-sm text-gray-900 dark:text-gray-100 placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-blue-500 ${props.className}`} />;

const Textarea = (props: React.TextareaHTMLAttributes<HTMLTextAreaElement>) => <textarea {...props} className={`flex min-h-[80px] w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 px-3 py-2 text-sm text-gray-900 dark:text-gray-100 placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-blue-500 ${props.className}`} />;

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'outline' | 'default';
}
const Button = ({ className, variant, ...props }: ButtonProps) => {
  const baseClasses = 'inline-flex items-center justify-center rounded-md text-sm font-medium transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:pointer-events-none disabled:opacity-50 h-9 px-4';
  const variantClasses = variant === 'outline' ? 'border border-gray-300 dark:border-gray-600 hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-700 dark:text-gray-200' : 'bg-blue-600 text-white hover:bg-blue-700';
  return <button className={`${baseClasses} ${variantClasses} ${className}`} {...props} />;
};

const Badge = ({ className, ...props }: React.HTMLAttributes<HTMLSpanElement>) => <span {...props} className={`inline-flex items-center rounded-full border px-2.5 py-0.5 text-xs font-semibold focus:outline-none ${className}`} />;

const Modal = ({ children, onClose }: { children: React.ReactNode; onClose: () => void }) => (
  <div className="fixed inset-0 bg-black bg-opacity-60 z-50 flex justify-center items-start pt-32 p-4 transition-opacity duration-300 ease-in-out overflow-y-auto" onClick={onClose}>
    <div className="relative bg-white dark:bg-gray-800 rounded-lg shadow-xl w-full max-w-5xl max-h-[85vh] flex flex-col border border-gray-200 dark:border-gray-700 my-8 shadow-2xl" onClick={(e) => e.stopPropagation()}>
      <div className="flex-grow overflow-y-auto">{children}</div>
      <button onClick={onClose} className="absolute top-2 right-2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 bg-transparent rounded-full p-1">
        <XIcon className="h-6 w-6" />
      </button>
    </div>
  </div>
);

// 확인/경고 모달 컴포넌트
const ConfirmModal = ({
  isOpen,
  title,
  message,
  onConfirm,
  onCancel,
  confirmText = '확인',
  cancelText = '취소',
  type = 'confirm' // "confirm" | "alert"
}: {
  isOpen: boolean;
  title: string;
  message: string;
  onConfirm: () => void;
  onCancel: () => void;
  confirmText?: string;
  cancelText?: string;
  type?: 'confirm' | 'alert';
}) => {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg max-w-md w-full mx-4">
        <div className="p-4">
          <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100 mb-3">{title}</h3>
          <p className="text-sm text-gray-600 dark:text-gray-400 mb-4 whitespace-pre-line">{message}</p>
          <div className="flex gap-2 justify-end">
            {type === 'confirm' && (
              <Button variant="outline" onClick={onCancel} className="px-3 py-1.5 text-sm">
                {cancelText}
              </Button>
            )}
            <Button onClick={onConfirm} className={`px-3 py-1.5 text-sm ${type === 'alert' ? 'bg-blue-500 hover:bg-blue-600' : 'bg-red-500 hover:bg-red-600'} text-white`}>
              {confirmText}
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

// --- Type definitions for component props ---
interface Reply {
  id: string;
  content: string;
  authorName: string;
  authorRole: string;
  createdAt: string;
}

interface TicketFormData {
  title: string;
  content: string;
  images: File[];
}

interface TicketsViewProps {
  user: any;
  onSelectIssue: (issue: SupportIssue) => void;
}

interface ChatViewProps {
  messages: Message[];
  input: string;
  isLoading: boolean;
  user: any;
  selectedIssue: SupportIssue | null;
  isUploading: boolean;
  files: File[];
  uploadProgress: Record<string, { status: string; message?: string }>;
  embeddingStatuses: {[filename: string]: {step: string; step_label: string; isProcessing: boolean}};
  isDragging: boolean;
  setInput: (value: string) => void;
  handleSubmit: (e: FormEvent) => void;
  handleFileUpload: (filesToUpload: File[]) => void;
  setFiles: React.Dispatch<React.SetStateAction<File[]>>;
  removeFile: (index: number) => void;
  setIsDragging: React.Dispatch<React.SetStateAction<boolean>>;
  handleNewChat: () => void;
  handleFileChange: (e: ChangeEvent<HTMLInputElement>) => void;
  uploadedDocuments: UploadedDocument[];
  onRemoveDocument: (document: UploadedDocument) => Promise<void>;
  similarTickets: any[];
  showAlert: (title: string, message: string) => void;
  isSyncing: boolean;
  onSyncTickets: () => void;
  handleJobComplete: (job: BackgroundJob) => void;
  refreshTrigger: number;
}

// --- Main Support Page Component ---
// 연락처 정보 상수 (UI와 AI 프롬프트에서 공통 사용)
const CONTACT_INFO = {
  representative: '김형채',
  company: '(주)인베슘',
  phone: '02-2039-3977',
  email: 'sales@invesume.com'
};

// 업로드된 문서 정보 타입
interface UploadedDocument {
  id: number;
  filename: string;
  filepath: string;
  filesize?: number;
  mimetype?: string;
  user_id?: string;
  created_at: string;
  file_exists?: boolean; // 파일 존재 여부
}

// 파일 검증 유틸리티 함수
const validateFiles = (files: File[]) => {
  const supportedTypes = [
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'text/plain',
    'text/markdown',
    'application/vnd.hancom.hwp',
    'application/vnd.hancom.hwpx',
    'application/vnd.ms-powerpoint',
    'application/vnd.openxmlformats-officedocument.presentationml.presentation',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'text/csv',
    'application/json'
  ];

  const supportedExtensions = ['pdf', 'doc', 'docx', 'txt', 'md', 'hwp', 'hwpx', 'ppt', 'pptx', 'xls', 'xlsx', 'csv', 'json'];

  const validFiles = files.filter((file) => {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    return supportedTypes.includes(file.type) || (fileExtension && supportedExtensions.includes(fileExtension));
  });

  const invalidFiles = files.filter((file) => {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    return !supportedTypes.includes(file.type) && (!fileExtension || !supportedExtensions.includes(fileExtension));
  });

  return { validFiles, invalidFiles };
};

export default function SupportPage() {
  // 로컬 스토리지 기반 사용자 정보 (로그인 없이도 접근 가능)
  const [user, setUser] = useState<any>(null);
  const [selectedIssue, setSelectedIssue] = useState<SupportIssue | null>(null);
  const [messages, setMessages] = useState<Message[]>([]);
  const [input, setInput] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [files, setFiles] = useState<File[]>([]);
  const [uploadProgress, setUploadProgress] = useState<Record<string, { status: string; message?: string }>>({});
  const [isUploading, setIsUploading] = useState(false);
  const [isDragging, setIsDragging] = useState(false);
  const [currentSessionId, setCurrentSessionId] = useState<string | null>(null);
  const [uploadedDocuments, setUploadedDocuments] = useState<UploadedDocument[]>([]);
  const [similarTickets, setSimilarTickets] = useState<any[]>([]);
  const [isSyncing, setIsSyncing] = useState(false);
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [embeddingStatuses, setEmbeddingStatuses] = useState<{[filename: string]: {step: string; step_label: string; isProcessing: boolean}}>({});

  // 모바일 탭 상태 관리 (tickets: 티켓 목록, chat: 채팅)
  const [activeTab, setActiveTab] = useState<'tickets' | 'chat'>('tickets');

  // RAG 서버에서 임베딩 상태 가져오기 (support 사용자 전용)
  const fetchEmbeddingStatuses = async () => {
    try {
      const response = await fetch(`${getRagServerUrl()}/rag/admin/all-statuses`);
      if (response.ok) {
        const data = await response.json();
        const statusMap: {[filename: string]: {step: string; step_label: string; isProcessing: boolean}} = {};

        if (data.documents && typeof data.documents === 'object') {
          // 새로운 API 응답 구조에 맞게 처리
          Object.entries(data.documents).forEach(([filename, status]: [string, any]) => {
            statusMap[filename] = {
              step: status.step,
              step_label: status.step_label,
              isProcessing: status.isProcessing
            };
          });
        }

        setEmbeddingStatuses(statusMap);
      }
    } catch (error) {
      console.error('임베딩 상태 조회 실패:', error);
    }
  };

  // 백그라운드 작업 완료 시 콜백
  const handleJobComplete = async (job: BackgroundJob) => {
    console.log('백그라운드 작업 완료:', job);

    if (job.type === 'file_upload') {
      // 파일 업로드 작업 완료 시 상태 업데이트
      const fileName = job.filename;

      // 현재 파일이 files 목록에 있는지 확인하고 상태 업데이트
      setFiles((prevFiles) => {
        const fileExists = prevFiles.some(f => f.name === fileName);
        if (fileExists) {
          // 파일이 아직 목록에 있다면 성공 상태로 표시 후 제거
          const fileToUpdate = prevFiles.find(f => f.name === fileName);
          if (fileToUpdate) {
            const fileId = `${fileToUpdate.name}-${fileToUpdate.lastModified}`;

            if (job.status === 'completed') {
              // 성공 상태로 업데이트
              setUploadProgress((prev) => ({
                ...prev,
                [fileId]: { status: 'success', message: '임베딩 완료!' }
              }));

              // 3초 후에 파일 목록에서 제거
              setTimeout(() => {
                setFiles((prev) => prev.filter(f => f.name !== fileName));
                setUploadProgress((prev) => {
                  const newProgress = { ...prev };
                  delete newProgress[fileId];
                  return newProgress;
                });
              }, 3000);
            } else if (job.status === 'failed') {
              // 실패 상태로 업데이트
              setUploadProgress((prev) => ({
                ...prev,
                [fileId]: { status: 'error', message: job.error || '임베딩 실패' }
              }));
            }
          }
        }
        return prevFiles;
      });

      // 문서 목록 새로고침
      await loadUploadedDocuments();
    } else if (job.type === 'ticket_sync') {
      // 티켓 동기화 완료 시
      setIsSyncing(false);
      // 문서 목록 새로고침
      await loadUploadedDocuments();
    }
  };

  // 모달 상태 관리
  const [modalState, setModalState] = useState({
    isOpen: false,
    type: 'alert' as 'alert' | 'confirm',
    title: '',
    message: '',
    onConfirm: () => {},
    onCancel: () => {}
  });

  // 모달 헬퍼 함수들
  const showAlert = (title: string, message: string) => {
    setModalState({
      isOpen: true,
      type: 'alert',
      title,
      message,
      onConfirm: () => setModalState((prev) => ({ ...prev, isOpen: false })),
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  const showConfirm = (title: string, message: string, onConfirm: () => void) => {
    setModalState({
      isOpen: true,
      type: 'confirm',
      title,
      message,
      onConfirm: () => {
        setModalState((prev) => ({ ...prev, isOpen: false }));
        onConfirm();
      },
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  // 업로드된 문서 목록 로드 및 자동 정리
  const loadUploadedDocuments = async () => {
    try {
      // 고객지원 페이지에서는 shared 폴더의 문서만 조회
      let url = '/api/support/documents?checkFiles=true';

      // 로그인하지 않은 사용자도 shared 문서는 볼 수 있음
      if (!user || !user.username) {
        // 로그인하지 않은 사용자도 shared 문서 조회 가능
        // console.log('비로그인 사용자 - shared 문서 조회');
      } else if (user.role === 'admin' || user.role === 'manager') {
        // 관리자/매니저도 shared 문서 조회
        // console.log('관리자/매니저 - shared 문서 조회');
      } else {
        // 일반 사용자도 shared 문서 조회
        // console.log('일반 사용자 - shared 문서 조회');
      }

      // 1. 파일 존재 여부 확인과 함께 문서 목록 조회
      const headers = getAuthHeaders();
      const response = await fetch(url, {
        credentials: 'include',
        headers
      });

      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          const documents = result.data;

          // 2. 누락된 파일이 있는지 확인
          const missingFiles = documents.filter((doc: UploadedDocument) => doc.file_exists === false);

          // 관리자만 자동 정리 수행
          if (missingFiles.length > 0 && user && (user.role === 'admin' || user.role === 'manager')) {
            // console.log(`[AUTO_CLEANUP] Found ${missingFiles.length} missing files, cleaning up...`);

            // 3. 자동으로 누락된 파일들 정리
            try {
              const cleanupResponse = await fetch('/api/support/documents', {
                method: 'PATCH',
                headers: {
                  'Content-Type': 'application/json'
                },
                credentials: 'include',
                body: JSON.stringify({ action: 'cleanup-missing-files' })
              });

              if (cleanupResponse.ok) {
                const cleanupResult = await cleanupResponse.json();
                if (cleanupResult.success && cleanupResult.removedCount > 0) {
                  // console.log(`[AUTO_CLEANUP] Removed ${cleanupResult.removedCount} missing file records`);

                  // 4. 정리 후 다시 문서 목록 로드 (정리된 목록)
                  const finalResponse = await fetch(url, {
                    credentials: 'include'
                  });

                  if (finalResponse.ok) {
                    const finalResult = await finalResponse.json();
                    if (finalResult.success) {
                      setUploadedDocuments(finalResult.data);
                    }
                  }
                  return;
                }
              }
            } catch (cleanupError) {
              console.error('자동 정리 실패:', cleanupError);
              // 정리 실패해도 원본 목록은 표시
              setUploadedDocuments(documents);
              return;
            }
          } else if (missingFiles.length > 0) {
            // 일반 사용자나 로그인하지 않은 사용자의 경우 누락 파일을 제외하고 표시
            const validDocuments = documents.filter((doc: UploadedDocument) => doc.file_exists !== false);
            setUploadedDocuments(validDocuments);
            return;
          }

          // 누락된 파일이 없거나 정리가 필요없는 경우
          setUploadedDocuments(documents);
        }
      }
    } catch (error) {
      console.error('문서 목록 로드 실패:', error);
    }
  };

  // 사용자 정보 로드 (로컬 스토리지 기반, 로그인하지 않은 사용자도 접근 가능)
  useEffect(() => {
    const loadUserInfo = () => {
      const token = localStorage.getItem('userToken');
      const username = localStorage.getItem('username');
      const userId = localStorage.getItem('userId');
      const apiKey = localStorage.getItem('apiKey');

      // anonymous 사용자 정보가 있으면 제거
      if (username === 'anonymous' || userId === 'anonymous') {
        localStorage.removeItem('userToken');
        localStorage.removeItem('username');
        localStorage.removeItem('userId');
        localStorage.removeItem('userEmail');
        localStorage.removeItem('userRole');
        localStorage.removeItem('apiKey');
        console.log('Anonymous 사용자 정보 제거됨');
        setUser(null);
        return;
      }

      if (token && username && userId && apiKey && username !== 'anonymous') {
        setUser({
          id: userId,
          username,
          email: localStorage.getItem('userEmail') || '',
          role: localStorage.getItem('userRole') || 'user'
        });
        // console.log('사용자 인증 완료:', username);
      } else {
        // 로그인하지 않은 사용자도 고객센터 접근 가능 (제한된 기능)
        console.log('비로그인 사용자로 고객센터 접근');
        setUser(null);
      }
    };

    loadUserInfo();
  }, []);

  // 사용자 정보가 로드된 후 문서 목록 로드 (자동 정리 포함)
  useEffect(() => {
    // 로그인 여부와 관계없이 문서 목록 로드 (역할에 따라 다른 문서 조회)
    loadUploadedDocuments(); // 자동으로 파일 존재 여부 확인 및 정리
  }, [user]);

  // 임베딩 상태를 주기적으로 업데이트
  useEffect(() => {
    fetchEmbeddingStatuses(); // 초기 로드
    const interval = setInterval(fetchEmbeddingStatuses, 3000); // 3초마다 업데이트

    return () => clearInterval(interval);
  }, []);

  // 임베딩 완료된 파일 자동 제거
  useEffect(() => {
    const timers: NodeJS.Timeout[] = [];

    files.forEach((file) => {
      const embeddingStatus = embeddingStatuses[file.name];
      if (embeddingStatus && embeddingStatus.step === 'done') {
        const fileId = `${file.name}-${file.lastModified}`;
        const currentProgress = uploadProgress[fileId];

        // 이미 제거 타이머가 설정된 파일은 스킵
        if (!currentProgress || currentProgress.status !== 'removing') {
          // 제거 예정 상태로 마킹
          setUploadProgress((prev) => ({
            ...prev,
            [fileId]: { status: 'removing', message: '임베딩 완료!' }
          }));

          // 3초 후 자동 제거
          const timer = setTimeout(() => {
            setFiles((prev) => prev.filter(f => f.name !== file.name));
            setUploadProgress((prev) => {
              const newProgress = { ...prev };
              delete newProgress[fileId];
              return newProgress;
            });
          }, 3000);

          timers.push(timer);
        }
      }
    });

    return () => {
      timers.forEach(timer => clearTimeout(timer));
    };
  }, [embeddingStatuses, files, uploadProgress]);

  const handleSubmit = async (e: FormEvent) => {
    e.preventDefault();
    if (!input.trim() || isLoading) return;

    const userMessage = input.trim();
    setInput('');
    setMessages((prev) => [...prev, { role: 'user', content: userMessage }]);
    setIsLoading(true);

    try {
      // 세션 ID가 없으면 support- 접두사로 새로 생성
      let sessionId = currentSessionId;
      if (!sessionId) {
        sessionId = `support-${Date.now()}-${Math.random().toString(36).substring(2, 9)}`;
        setCurrentSessionId(sessionId);
      }

      // 유사 티켓 검색 기능 복원 (UI 표시용)
      try {
        const similarResponse = await fetch('/api/support/similar-tickets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'include',
          body: JSON.stringify({ query: userMessage, limit: 3 })
        });

        if (similarResponse.ok) {
          const similarData = await similarResponse.json();
          if (similarData.success && similarData.data?.similarTickets?.length > 0) {
            setSimilarTickets(similarData.data.similarTickets);
          } else {
            setSimilarTickets([]);
          }
        } else {
          setSimilarTickets([]);
        }
      } catch (error) {
        console.error('유사 티켓 검색(UI) 실패:', error);
        setSimilarTickets([]);
      }

      // AI 답변 생성 (모든 사용자 RAG 사용 가능)
      const response = await externalApiClient.post('/chat/sync', {
        prompt: userMessage, // 프롬프트 간소화
        options: {
          sessionId: sessionId, // support- 접두사 세션 ID 전달
          rag: true, // 모든 사용자 RAG 사용 가능
          ragSearchScope: 'all', // 서포트에서는 개인 + 공용 문서 모두 검색
          temperature: 0.7,
          max_tokens: 1000,
          type: 'support', // 세션 타입 명시
          userId: user?.id ? parseInt(user.id) : null, // 사용자 ID를 숫자로 변환
          username: user?.username, // 사용자명은 별도로 전달
          model: 'airun-chat:latest' // JSON 형식 응답을 위해 airun-chat 모델 사용
        }
      });

      // AI 응답 처리 - externalApiClient는 ApiResponse<T> 형태로 래핑됨
      let aiResponse = '';

      console.log('[채팅창] AI 응답 전체:', response); // 디버깅용

      // ApiResponse 구조 확인 - 중첩된 data 구조 처리
      if (response?.success && response?.data) {
        console.log('[채팅창] response.data 상세 분석:', {
          hasDirectResponse: !!response.data.response,
          hasNestedResponse: !!response.data.data?.response,
          directResponseType: typeof response.data.response,
          nestedResponseType: typeof response.data.data?.response,
          nestedResponseValue: response.data.data?.response,
          hasDirectHistory: !!response.data.history,
          hasNestedHistory: !!response.data.data?.history,
          nestedHistoryLength: Array.isArray(response.data.data?.history) ? response.data.data.history.length : 'not array'
        });

        // 1차: 중첩된 data.data.response에서 추출
        if (response.data.data?.response && response.data.data.response.trim()) {
          aiResponse = response.data.data.response;
          console.log('[채팅창] data.data.response에서 추출 성공');
        }
        // 2차: 직접 data.response에서 추출
        else if (response.data.response && response.data.response.trim()) {
          aiResponse = response.data.response;
          console.log('[채팅창] data.response에서 추출 성공');
        }
        // 3차: 중첩된 data.data.history에서 추출
        else if (response.data.data?.history && Array.isArray(response.data.data.history)) {
          const history = response.data.data.history;
          console.log('[채팅창] 중첩된 history 구조:', {
            historyExists: !!history,
            historyType: typeof history,
            isArray: Array.isArray(history),
            historyLength: history.length
          });

          const lastAssistant = [...history].reverse().find((msg) => msg.role === 'assistant' && msg.content);
          if (lastAssistant) {
            aiResponse = lastAssistant.content;
            console.log('[채팅창] 중첩된 history에서 추출 성공');
          } else {
            console.log('[채팅창] 중첩된 history에서 assistant 메시지를 찾을 수 없음');
          }
        }
        // 4차: 직접 data.history에서 추출
        else if (response.data.history && Array.isArray(response.data.history)) {
          const history = response.data.history;
          console.log('[채팅창] 직접 history 구조:', {
            historyExists: !!history,
            historyType: typeof history,
            isArray: Array.isArray(history),
            historyLength: history.length
          });

          const lastAssistant = [...history].reverse().find((msg) => msg.role === 'assistant' && msg.content);
          if (lastAssistant) {
            aiResponse = lastAssistant.content;
            console.log('[채팅창] 직접 history에서 추출 성공');
          } else {
            console.log('[채팅창] 직접 history에서 assistant 메시지를 찾을 수 없음');
          }
        } else {
          console.log('[채팅창] 모든 경로에서 응답을 찾을 수 없음');
        }
      } else {
        console.log('[채팅창] API 응답이 성공하지 않았거나 data가 없음:', {
          success: response?.success,
          hasData: !!response?.data,
          error: response?.error
        });
      }

      if (aiResponse && aiResponse !== '죄송합니다. 응답을 생성하는 중 문제가 발생했습니다.') {
        setMessages((prev) => [...prev, { role: 'assistant', content: aiResponse }]);
      } else {
        throw new Error('AI 응답을 파싱할 수 없습니다.');
      }
    } catch (error) {
      console.error('Chat error:', error);
      setMessages((prev) => [
        ...prev,
        {
          role: 'assistant',
          content: '죄송합니다. 응답 생성 중 오류가 발생했습니다. 잠시 후 다시 시도해주세요.'
        }
      ]);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSelectIssue = (issue: SupportIssue) => {
    setSelectedIssue(issue);
    // ... logic to set messages for ChatView
  };

  const handleFileUpload = async (filesToUpload: File[]) => {
    if (!filesToUpload || filesToUpload.length === 0) {
      return;
    }

    if (!user?.username) {
      showAlert('업로드 실패', '로그인이 필요합니다.');
      return;
    }

    // 백그라운드 작업으로 전환
    for (const file of filesToUpload) {
      try {
        const fileId = `${file.name}-${file.lastModified}`;

        // 업로드 시작 상태로 업데이트
        setUploadProgress((prev) => ({
          ...prev,
          [fileId]: { status: 'uploading', message: '임베딩 준비 중...' }
        }));

        // 파일을 Base64로 변환
        const fileData = await new Promise<{
          name: string;
          size: number;
          type: string;
          content: string;
        }>((resolve, reject) => {
          const reader = new FileReader();
          reader.onload = () => {
            const result = reader.result as string;
            const base64Content = result.split(',')[1]; // data:type;base64, 제거
            resolve({
              name: file.name,
              size: file.size,
              type: file.type,
              content: base64Content
            });
          };
          reader.onerror = reject;
          reader.readAsDataURL(file);
        });

        const response = await fetch('/api/support/background-jobs', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          credentials: 'include',
          body: JSON.stringify({
            type: 'file_upload',
            userId: 'support',
            filename: file.name,
            fileData: fileData,
            metadata: {
              pageType: 'support'
            }
          })
        });

        if (response.ok) {
          const result = await response.json();
          if (result.success) {
            console.log(`파일 ${file.name} 백그라운드 업로드 시작:`, result.jobId);

            // 백그라운드 작업 시작됨 상태로 업데이트
            setUploadProgress((prev) => ({
              ...prev,
              [fileId]: { status: 'processing', message: '임베딩 진행 중...' }
            }));

            // 즉시 백그라운드 작업 상태 새로고침 트리거
            setRefreshTrigger((prev) => prev + 1);
          } else {
            // 백그라운드 작업 시작 실패
            setUploadProgress((prev) => ({
              ...prev,
              [fileId]: { status: 'error', message: '임베딩 시작 실패' }
            }));
          }
        } else {
          // API 요청 실패
          setUploadProgress((prev) => ({
            ...prev,
            [fileId]: { status: 'error', message: 'API 요청 실패' }
          }));
        }
      } catch (error) {
        console.error('백그라운드 작업 시작 실패:', error);
        const fileId = `${file.name}-${file.lastModified}`;
        setUploadProgress((prev) => ({
          ...prev,
          [fileId]: { status: 'error', message: '업로드 실패' }
        }));
        showAlert('업로드 실패', `${file.name} 업로드 작업을 시작할 수 없습니다.`);
      }
    }
  };

  const handleFileChange = (e: ChangeEvent<HTMLInputElement>) => {
    const selectedFiles = Array.from(e.target.files || []);
    const { validFiles, invalidFiles } = validateFiles(selectedFiles);

    if (invalidFiles.length > 0) {
      showAlert('파일 형식 오류', `지원하지 않는 파일 형식입니다: ${invalidFiles.map((f) => f.name).join(', ')}\n\n지원 형식: PDF, DOC, DOCX, TXT, MD, HWP, HWPX, PPT, PPTX, XLS, XLSX, CSV, JSON`);
    }

    if (validFiles.length > 0) {
      setFiles((prev) => [...prev, ...validFiles]);
      const newProgress: Record<string, { status: string }> = {};
      validFiles.forEach((f) => {
        newProgress[`${f.name}-${f.lastModified}`] = { status: 'pending' };
      });
      setUploadProgress((prev) => ({ ...prev, ...newProgress }));
      handleFileUpload(validFiles);
    }
    e.target.value = '';
  };

  const removeFile = (indexToRemove: number) => {
    const fileToRemove = files[indexToRemove];
    if (!fileToRemove) return;
    const fileId = `${fileToRemove.name}-${fileToRemove.lastModified}`;

    setFiles((prev) => prev.filter((_, i) => i !== indexToRemove));
    setUploadProgress((prev) => {
      const newProgress = { ...prev };
      delete newProgress[fileId];
      return newProgress;
    });
  };

  // 업로드된 문서 삭제 함수
  const removeUploadedDocument = async (document: UploadedDocument) => {
    showConfirm('문서 삭제 확인', `"${document.filename}" 문서를 삭제하시겠습니까?`, async () => {
      try {
        // 1. RAG 시스템에서 파일 삭제
        const ragServerUrl = getRagServerUrl();
        const ragResponse = await fetch(`${ragServerUrl}/delete`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            file_paths: [`${document.filepath}/${document.filename}`],
            user_id: 'support' // RAG 시스템에서 user_id로 사용됨
          })
        });

        if (!ragResponse.ok) {
          const errorData = await ragResponse.json().catch(() => ({ message: '알 수 없는 서버 오류' }));
          throw new Error(errorData.message || 'RAG 시스템 삭제 실패');
        }

        // 2. 데이터베이스에서 문서 정보 삭제 (filename과 user_id 사용)
        const dbResponse = await fetch('/api/support/documents', {
          method: 'DELETE',
          headers: {
            'Content-Type': 'application/json'
          },
          credentials: 'include', // 쿠키 포함
          body: JSON.stringify({
            filename: document.filename,
            id: 'support' // API에서 user_id로 사용됨
          })
        });

        if (!dbResponse.ok) {
          const errorData = await dbResponse.json().catch(() => ({ message: '알 수 없는 서버 오류' }));
          throw new Error(errorData.message || 'DB 삭제 실패');
        }

        // 성공 시 문서 목록 새로고침
        await loadUploadedDocuments();
        console.log('문서 삭제 완료:', document.filename);
      } catch (error) {
        console.error('문서 삭제 실패:', error);
        showAlert('삭제 실패', `문서 삭제에 실패했습니다: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
      }
    });
  };

  // 티켓 DB를 RAG와 동기화하는 함수
  const handleSyncTicketsToRag = async () => {
    if (!user?.username) {
      showAlert('동기화 실패', '로그인이 필요합니다.');
      return;
    }

    console.log('[handleSyncTicketsToRag] 티켓 동기화 시작');
    setIsSyncing(true);
    try {
      const response = await fetch('/api/support/background-jobs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify({
          type: 'ticket_sync',
          userId: 'support',
          filename: 'support-tickets-knowledge-base.md',
          metadata: {
            pageType: 'support'
          }
        })
      });

      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          console.log('[handleSyncTicketsToRag] 티켓 동기화 백그라운드 작업 시작:', result.jobId);
          // 알림창 제거 - 백그라운드 작업 상태에서 확인 가능
          // 즉시 백그라운드 작업 상태 새로고침 트리거
          console.log('[handleSyncTicketsToRag] refreshTrigger 업데이트 전:', refreshTrigger);
          setRefreshTrigger((prev) => {
            console.log('[handleSyncTicketsToRag] refreshTrigger 업데이트:', prev, '->', prev + 1);
            return prev + 1;
          });
        } else {
          throw new Error(result.error || '백그라운드 작업 시작 실패');
        }
      } else {
        const errorData = await response.json().catch(() => ({ error: '서버 오류' }));
        throw new Error(errorData.error || '백그라운드 작업 시작 실패');
      }
    } catch (error) {
      console.error('티켓 DB 동기화 실패:', error);
      showAlert('동기화 실패', `티켓 DB 동기화에 실패했습니다: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
    } finally {
      setIsSyncing(false);
    }
  };

  return (
    <div className="h-full flex flex-col bg-[var(--body-bg)]">
      <header className="flex-none p-2 text-center border-b border-[var(--card-border)] shadow-sm bg-[var(--header-bg)]">{/* <h2 className="text-xl font-bold">AI 기술 지원 대시보드</h2> */}</header>

      {/* 모바일 탭 네비게이션 */}
      <div className="md:hidden flex-none bg-[var(--card-bg)] border-b border-[var(--card-border)]" role="tablist" aria-label="페이지 탭 네비게이션">
        <div className="flex">
          <button
            onClick={() => setActiveTab('tickets')}
            role="tab"
            aria-selected={activeTab === 'tickets'}
            aria-controls="tickets-panel"
            className={`flex-1 py-3 px-4 text-sm font-medium text-center transition-colors min-h-[44px] ${activeTab === 'tickets' ? 'bg-blue-500 text-white' : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700'}`}
          >
            고객센터
          </button>
          <button onClick={() => setActiveTab('chat')} role="tab" aria-selected={activeTab === 'chat'} aria-controls="chat-panel" className={`flex-1 py-3 px-4 text-sm font-medium text-center transition-colors min-h-[44px] ${activeTab === 'chat' ? 'bg-blue-500 text-white' : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700'}`}>
            문의하기
          </button>
        </div>
      </div>

      <div className="flex-1 flex min-h-0 p-2 md:p-4 gap-2 md:gap-4 bg-[var(--body-bg)]">
        {/* 데스크톱: 두 열 레이아웃, 모바일: 탭 기반 단일 뷰 */}

        {/* Left Column: Tickets - 데스크톱에서는 항상 표시, 모바일에서는 탭에 따라 표시 */}
        <div className={`${activeTab === 'tickets' ? 'flex' : 'hidden'} md:flex w-full md:w-2/5 flex-col rounded-lg border border-[var(--sidebar-border)] shadow-sm overflow-hidden`} style={{ backgroundColor: 'var(--sidebar-bg)' }} role="tabpanel" id="tickets-panel" aria-labelledby="tickets-tab">
          <TicketsView
            user={user}
            onSelectIssue={(issue) => {
              handleSelectIssue(issue);
              // 모바일에서 티켓 선택 시 자동으로 채팅 탭으로 이동
              if (window.innerWidth < 768) {
                setActiveTab('chat');
              }
            }}
          />
        </div>

        {/* Right Column: Chat - 데스크톱에서는 항상 표시, 모바일에서는 탭에 따라 표시 */}
        <div className={`${activeTab === 'chat' ? 'flex' : 'hidden'} md:flex w-full md:w-3/5 flex-col rounded-lg border border-[var(--sidebar-border)] shadow-sm overflow-hidden`} style={{ backgroundColor: 'var(--sidebar-bg)' }} role="tabpanel" id="chat-panel" aria-labelledby="chat-tab">
          <ChatView
            messages={messages}
            input={input}
            isLoading={isLoading}
            user={user}
            selectedIssue={selectedIssue}
            isUploading={isUploading}
            files={files}
            uploadProgress={uploadProgress}
            embeddingStatuses={embeddingStatuses}
            isDragging={isDragging}
            setInput={setInput}
            handleSubmit={handleSubmit}
            handleFileUpload={handleFileUpload}
            setFiles={setFiles}
            removeFile={removeFile}
            setIsDragging={setIsDragging}
            handleNewChat={() => {
              setMessages([]);
              setSelectedIssue(null);
              setSimilarTickets([]);
              // 업로드 진행 상태도 초기화
              setFiles([]);
              setUploadProgress({});
            }}
            handleFileChange={handleFileChange}
            uploadedDocuments={uploadedDocuments}
            onRemoveDocument={removeUploadedDocument}
            similarTickets={similarTickets}
            showAlert={showAlert}
            isSyncing={isSyncing}
            onSyncTickets={handleSyncTicketsToRag}
            handleJobComplete={handleJobComplete}
            refreshTrigger={refreshTrigger}
          />
        </div>
      </div>

      {/* 확인/경고 모달 */}
      <ConfirmModal isOpen={modalState.isOpen} type={modalState.type} title={modalState.title} message={modalState.message} onConfirm={modalState.onConfirm} onCancel={modalState.onCancel} />
    </div>
  );
}

// --- Child Components ---
function TicketsView({ user, onSelectIssue }: TicketsViewProps) {
  const [issues, setIssues] = useState<SupportIssue[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalIssues, setTotalIssues] = useState(0);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedIssue, setSelectedIssue] = useState<SupportIssue | null>(null);
  const [isCreating, setIsCreating] = useState(false);
  const issuesPerPage = 10;

  // 모달 상태 관리
  const [modalState, setModalState] = useState({
    isOpen: false,
    type: 'alert' as 'alert' | 'confirm',
    title: '',
    message: '',
    onConfirm: () => {},
    onCancel: () => {}
  });

  // 모달 헬퍼 함수
  const showAlert = (title: string, message: string) => {
    setModalState({
      isOpen: true,
      type: 'alert',
      title,
      message,
      onConfirm: () => setModalState((prev) => ({ ...prev, isOpen: false })),
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  const fetchIssues = useCallback(async () => {
    setIsLoading(true);
    try {
      const params = new URLSearchParams({
        userName: user?.username || 'guest',
        userRole: user?.role || 'guest',
        page: String(currentPage),
        limit: String(issuesPerPage),
        searchTerm: searchTerm
      });
      const response = await fetch(`/api/support/issues?${params.toString()}`);
      const data = await response.json();
      setIssues(data.issues || []);
      setTotalIssues(data.totalCount || 0);
    } catch (error) {
      console.error('Failed to fetch issues:', error);
    } finally {
      setIsLoading(false);
    }
  }, [user, currentPage, searchTerm]);

  // showConfirm 함수를 컴포넌트 레벨에서 정의
  const showConfirm = (title: string, message: string, onConfirm: () => void) => {
    setModalState({
      isOpen: true,
      type: 'confirm',
      title,
      message,
      onConfirm: () => {
        setModalState((prev) => ({ ...prev, isOpen: false }));
        onConfirm();
      },
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  // 티켓 삭제 함수
  const handleDeleteIssue = async (issue: SupportIssue) => {
    const isAdmin = user?.role === 'admin' || user?.role === 'manager';
    const isAuthor = issue.author.name?.toLowerCase() === user?.username?.toLowerCase();

    if (!isAdmin && !isAuthor) {
      showAlert('삭제 권한 없음', '이 티켓을 삭제할 권한이 없습니다.');
      return;
    }

    const confirmMessage = isAdmin ? `"${issue.title}" 티켓을 삭제하시겠습니까?\n\n작성자: ${issue.author.name}\n등록일: ${new Date(issue.createdAt).toLocaleDateString()}\n\n※ 관리자 권한으로 삭제합니다.` : `"${issue.title}" 티켓을 삭제하시겠습니까?\n\n답변이 있는 경우에도 모두 삭제됩니다.`;

    showConfirm('티켓 삭제 확인', confirmMessage, async () => {
      try {
        const response = await fetch(`/api/support/issues/${issue.id}`, {
          method: 'DELETE',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            userRole: user?.role || 'user',
            userName: user?.username
          })
        });

        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || '티켓 삭제 실패');
        }

        showAlert('삭제 완료', '티켓이 성공적으로 삭제되었습니다.');
        fetchIssues(); // 목록 새로고침
      } catch (error) {
        console.error('티켓 삭제 실패:', error);
        showAlert('삭제 실패', `티켓 삭제에 실패했습니다: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
      }
    });
  };

  useEffect(() => {
    fetchIssues();
  }, [fetchIssues]);

  const handleIssueClick = (issue: SupportIssue) => {
    setSelectedIssue(issue);
    onSelectIssue(issue);
  };

  const handleCloseDetailModal = () => {
    setSelectedIssue(null);
    fetchIssues();
  };

  const handleCreateTicket = async (data: TicketFormData) => {
    try {
      const formData = new FormData();
      formData.append('title', data.title);
      formData.append('question', data.content);
      formData.append(
        'author',
        JSON.stringify({
          name: user?.username || 'guest',
          role: user?.role || 'user'
        })
      );

      if (data.images && data.images.length > 0) {
        // 현재는 첫 번째 이미지만 사용 (API가 단일 이미지만 지원)
        formData.append('image', data.images[0]);
      }

      const response = await fetch('/api/support/issues', {
        method: 'POST',
        body: formData
      });

      if (!response.ok) {
        throw new Error('티켓 생성 실패');
      }

      const result = await response.json();
      setSelectedIssue(result);
      onSelectIssue(result);
      fetchIssues();
      setIsCreating(false);
    } catch (error) {
      console.error('Failed to create ticket:', error);
      showAlert('티켓 생성 실패', '티켓 생성에 실패했습니다.');
    }
  };

  return (
    <>
      <div className="p-3 md:p-4 border-b border-[var(--sidebar-border)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        <div className="flex items-center justify-between gap-2 md:gap-4 mb-3 md:mb-4">
          <h2 className="text-base md:text-lg font-bold flex-shrink-0">고객센터</h2>
        </div>

        {/* 연락처 정보 섹션 */}
        <div className="mb-3 md:mb-4 p-2 md:p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
          <h3 className="text-xs md:text-sm font-semibold text-blue-900 dark:text-blue-100 mb-2">📞 문의 연락처</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-1 md:gap-2 text-xs text-blue-700 dark:text-blue-200">
            <div>
              <strong>담당자:</strong> {CONTACT_INFO.representative}
            </div>
            <div>
              <strong>회사:</strong> {CONTACT_INFO.company}
            </div>
            <div>
              <strong>전화:</strong> {CONTACT_INFO.phone}
            </div>
            <div>
              <strong>이메일:</strong> {CONTACT_INFO.email}
            </div>
          </div>
        </div>

        <div className="flex flex-col gap-2 md:gap-0 md:flex-row md:items-center md:justify-between pt-3 md:pt-5">
          <h3 className="text-sm md:text-md font-medium">티켓 목록</h3>
          <div className="flex flex-col sm:flex-row gap-2 md:flex-grow md:min-w-0 md:ml-4">
            <Input type="search" placeholder="제목 또는 내용으로 검색..." value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} className="w-full h-8 md:h-9 text-sm" />
            <Button onClick={() => setIsCreating(true)} className="!h-8 md:!h-9 px-2 md:px-3 flex items-center justify-center gap-1 text-sm whitespace-nowrap">
              <PlusCircleIcon className="h-3 w-3 md:h-4 md:w-4" />
              <span className="hidden sm:inline">새 요청</span>
              <span className="sm:hidden">요청</span>
            </Button>
          </div>
        </div>
      </div>

      <div className="flex-1 overflow-y-auto p-1 md:p-2" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        {isLoading ? (
          <div className="text-center p-4 md:p-8 text-sm md:text-base text-gray-500">로딩 중...</div>
        ) : issues.length === 0 ? (
          <div className="text-center p-4 md:p-8 text-sm md:text-base text-gray-500">조건에 맞는 티켓이 없습니다.</div>
        ) : (
          <div className="space-y-1 md:space-y-0.5">
            {issues.map((issue) => {
              const isAdmin = user?.role === 'admin' || user?.role === 'manager';
              const isAuthor = issue.author.name?.toLowerCase() === user?.username?.toLowerCase();
              const canDelete = isAdmin || isAuthor;

              return (
                <div key={issue.id} className="p-2 md:p-2 border-b dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700/50 rounded-md" style={{ backgroundColor: 'var(--body-bg)' }}>
                  <div className="flex flex-col sm:flex-row sm:justify-between sm:items-start gap-1 sm:gap-0 mb-1">
                    <div className="flex-1 cursor-pointer" onClick={() => handleIssueClick(issue)}>
                      <p className="font-semibold text-sm truncate pr-2">{issue.title}</p>
                    </div>
                    <div className="flex items-center gap-1 md:gap-2 self-start sm:self-auto flex-shrink-0">
                      <Badge className={`text-xs px-1.5 py-0.5 ${issue.status.toLowerCase() === 'open' ? 'border-orange-300 bg-orange-100 text-orange-800 dark:border-orange-600 dark:bg-orange-900/40 dark:text-orange-300' : 'border-green-300 bg-green-100 text-green-800 dark:border-green-600 dark:bg-green-900/40 dark:text-green-300'}`}>
                        {issue.status.toLowerCase() === 'open' ? '대기중' : '완료'}
                      </Badge>
                      {canDelete && (
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            handleDeleteIssue(issue);
                          }}
                          className="text-gray-400 hover:text-red-600 dark:hover:text-red-400 p-1 rounded transition-colors"
                          title={isAdmin ? '관리자 권한으로 삭제' : '내 티켓 삭제'}
                        >
                          <TrashIcon className="h-3 w-3 md:h-4 md:w-4" />
                        </button>
                      )}
                    </div>
                  </div>
                  <div className="cursor-pointer" onClick={() => handleIssueClick(issue)}>
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      {user?.role === 'admin' ? `${issue.author.name} • ` : ''}
                      {new Date(issue.createdAt).toLocaleDateString()}
                    </p>
                  </div>
                </div>
              );
            })}
          </div>
        )}
      </div>

      <div className="p-2 md:p-4 border-t border-[var(--sidebar-border)] flex flex-col sm:flex-row justify-between items-center gap-2 sm:gap-0" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        <span className="text-xs md:text-sm text-gray-500 order-2 sm:order-1">총 {totalIssues}개</span>
        <div className="flex gap-1 md:gap-2 order-1 sm:order-2">
          <Button onClick={() => setCurrentPage((p) => Math.max(1, p - 1))} disabled={currentPage === 1 || isLoading} className="px-2 md:px-4 py-1 md:py-2 text-xs md:text-sm h-8 md:h-9">
            이전
          </Button>
          <span className="flex items-center justify-center text-xs md:text-sm font-medium w-8 md:w-10 h-8 md:h-9 bg-gray-100 dark:bg-gray-700 rounded">
            {currentPage}/{Math.ceil(totalIssues / issuesPerPage) || 1}
          </span>
          <Button onClick={() => setCurrentPage((p) => p + 1)} disabled={currentPage >= Math.ceil(totalIssues / issuesPerPage) || isLoading} className="px-2 md:px-4 py-1 md:py-2 text-xs md:text-sm h-8 md:h-9">
            다음
          </Button>
        </div>
      </div>

      {selectedIssue && (
        <Modal onClose={handleCloseDetailModal}>
          <TicketDetail ticket={selectedIssue} onClose={handleCloseDetailModal} />
        </Modal>
      )}

      {isCreating && (
        <Modal onClose={() => setIsCreating(false)}>
          <TicketForm onSubmit={handleCreateTicket} onClose={() => setIsCreating(false)} />
        </Modal>
      )}

      {/* 확인/경고 모달 */}
      <ConfirmModal isOpen={modalState.isOpen} type={modalState.type} title={modalState.title} message={modalState.message} onConfirm={modalState.onConfirm} onCancel={modalState.onCancel} />
    </>
  );
}

function TicketForm({ onSubmit, initialData, onClose }: { onSubmit: (data: TicketFormData) => void; initialData?: Partial<TicketFormData>; onClose: () => void }) {
  const [formData, setFormData] = useState<TicketFormData>({
    title: initialData?.title || '',
    content: initialData?.content || '',
    images: initialData?.images || []
  });
  const imageInputRef = useRef<HTMLInputElement>(null);

  const handleImageUpload = (e: ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    const imageFiles = files.filter((file) => file.type.startsWith('image/'));
    setFormData((prev) => ({
      ...prev,
      images: [...prev.images, ...imageFiles]
    }));
    e.target.value = '';
  };

  const removeImage = (index: number) => {
    setFormData((prev) => ({
      ...prev,
      images: prev.images.filter((_, i) => i !== index)
    }));
  };

  return (
    <div className="p-6 max-h-[80vh] overflow-y-auto">
      <div className="mb-4">
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">제목</label>
        <Input value={formData.title} onChange={(e) => setFormData((prev) => ({ ...prev, title: e.target.value }))} placeholder="제목을 입력하세요" className="w-full" />
      </div>
      <div className="mb-4">
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">내용</label>
        <Textarea value={formData.content} onChange={(e) => setFormData((prev) => ({ ...prev, content: e.target.value }))} placeholder="내용을 입력하세요" className="min-h-[350px] resize-none" />
      </div>
      <div className="mb-4">
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">이미지 첨부</label>
        <div className="flex flex-wrap gap-2 mb-2">
          {formData.images.map((image, index) => (
            <div key={index} className="relative group">
              <img src={URL.createObjectURL(image)} alt={`첨부 ${index + 1}`} className="w-24 h-24 object-cover rounded" />
              <button onClick={() => removeImage(index)} className="absolute top-1 right-1 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity">
                <XIcon className="h-4 w-4" />
              </button>
            </div>
          ))}
        </div>
        <Button type="button" variant="outline" onClick={() => imageInputRef.current?.click()} className="w-full">
          이미지 추가
        </Button>
        <input ref={imageInputRef} type="file" accept="image/*" multiple onChange={handleImageUpload} className="hidden" />
      </div>
      <div className="flex justify-end gap-2">
        <Button variant="outline" onClick={onClose}>
          취소
        </Button>
        <Button onClick={() => onSubmit(formData)} disabled={!formData.title.trim() || !formData.content.trim()}>
          등록
        </Button>
      </div>
    </div>
  );
}

// 이미지 뷰어 모달 컴포넌트
interface ImageViewerProps {
  image: string | null;
  onClose: () => void;
}

const ImageViewer = ({ image, onClose }: ImageViewerProps) => {
  if (!image) return null;

  return (
    <div className="fixed inset-0 bg-black/90 z-50 flex items-center justify-center cursor-zoom-out" onClick={onClose}>
      <div className="relative w-full h-full flex items-center justify-center p-4">
        <img src={image} alt="원본 이미지" className="max-w-full max-h-full object-contain cursor-default" onClick={(e) => e.stopPropagation()} />
        <button onClick={onClose} className="fixed top-6 right-6 z-[60] flex items-center justify-center w-12 h-12 bg-black/50 hover:bg-black/70 text-white hover:text-gray-200 rounded-full transition-all duration-200 shadow-lg border border-white/20" aria-label="이미지 닫기">
          <XIcon className="h-8 w-8" />
        </button>
      </div>
    </div>
  );
};

function TicketDetail({ ticket, onClose }: { ticket: SupportIssue; onClose: () => void }) {
  // 로컬 스토리지 기반 사용자 정보
  const [user, setUser] = useState<any>(null);

  // 사용자 정보 로드
  useEffect(() => {
    const token = localStorage.getItem('userToken');
    const username = localStorage.getItem('username');
    const userId = localStorage.getItem('userId');

    if (token && username && userId) {
      setUser({
        id: userId,
        username,
        email: localStorage.getItem('userEmail') || '',
        role: localStorage.getItem('userRole') || 'user'
      });
    }
  }, []);
  const [replyContent, setReplyContent] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [currentTicket, setCurrentTicket] = useState(ticket);
  const [selectedImage, setSelectedImage] = useState<string | null>(null);
  const [replyImages, setReplyImages] = useState<File[]>([]);
  const [isGeneratingAI, setIsGeneratingAI] = useState(false);
  const imageInputRef = useRef<HTMLInputElement>(null);

  // 티켓 삭제 함수
  const handleDeleteTicket = async () => {
    const isAdmin = user?.role === 'admin' || user?.role === 'manager';
    const isAuthor = currentTicket.author.name?.toLowerCase() === user?.username?.toLowerCase();

    if (!isAdmin && !isAuthor) {
      showAlert('삭제 권한 없음', '이 티켓을 삭제할 권한이 없습니다.');
      return;
    }

    const confirmMessage = isAdmin ? `"${currentTicket.title}" 티켓을 삭제하시겠습니까?\n\n작성자: ${currentTicket.author.name}\n등록일: ${new Date(currentTicket.createdAt).toLocaleDateString()}\n\n※ 관리자 권한으로 삭제합니다.` : `"${currentTicket.title}" 티켓을 삭제하시겠습니까?\n\n답변이 있는 경우에도 모두 삭제됩니다.`;

    showConfirm('티켓 삭제 확인', confirmMessage, async () => {
      try {
        const response = await fetch(`/api/support/issues/${currentTicket.id}`, {
          method: 'DELETE',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            userRole: user?.role || 'user',
            userName: user?.username
          })
        });

        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || '티켓 삭제 실패');
        }

        showAlert('삭제 완료', '티켓이 성공적으로 삭제되었습니다.');
        onClose(); // 상세 화면 닫기
      } catch (error) {
        console.error('티켓 삭제 실패:', error);
        showAlert('삭제 실패', `티켓 삭제에 실패했습니다: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
      }
    });
  };

  // 모달 상태 관리
  const [modalState, setModalState] = useState({
    isOpen: false,
    type: 'alert' as 'alert' | 'confirm',
    title: '',
    message: '',
    onConfirm: () => {},
    onCancel: () => {}
  });

  // 모달 헬퍼 함수들
  const showAlert = (title: string, message: string) => {
    setModalState({
      isOpen: true,
      type: 'alert',
      title,
      message,
      onConfirm: () => setModalState((prev) => ({ ...prev, isOpen: false })),
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  const showConfirm = (title: string, message: string, onConfirm: () => void) => {
    setModalState({
      isOpen: true,
      type: 'confirm',
      title,
      message,
      onConfirm: () => {
        setModalState((prev) => ({ ...prev, isOpen: false }));
        onConfirm();
      },
      onCancel: () => setModalState((prev) => ({ ...prev, isOpen: false }))
    });
  };

  const fetchTicketDetails = async () => {
    try {
      const response = await fetch(`/api/support/issues/${ticket.id}`);
      if (!response.ok) throw new Error('Failed to fetch ticket details');
      const data = await response.json();
      setCurrentTicket(data);
    } catch (error) {
      console.error('Error fetching ticket details:', error);
    }
  };

  useEffect(() => {
    fetchTicketDetails();
  }, [ticket.id]);

  const handleSubmitReply = async (e: FormEvent) => {
    e.preventDefault();
    if (!replyContent.trim() || isSubmitting) return;

    setIsSubmitting(true);
    try {
      const formData = new FormData();
      formData.append('content', replyContent);
      formData.append('authorName', user?.username || 'admin');
      formData.append('authorRole', user?.role || 'admin');

      replyImages.forEach((image) => {
        formData.append('images', image);
      });

      const response = await fetch(`/api/support/issues/${ticket.id}`, {
        method: 'POST',
        body: formData
      });

      if (!response.ok) throw new Error('Failed to submit reply');

      setReplyContent('');
      setReplyImages([]);
      await fetchTicketDetails();
    } catch (error) {
      console.error('Error submitting reply:', error);
      showAlert('답변 등록 실패', '답변 등록에 실패했습니다.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleReplyImageUpload = (e: ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    const imageFiles = files.filter((file) => file.type.startsWith('image/'));
    setReplyImages((prev) => [...prev, ...imageFiles]);
    e.target.value = '';
  };

  const removeReplyImage = (index: number) => {
    setReplyImages((prev) => prev.filter((_, i) => i !== index));
  };

  const generateAIResponse = async () => {
    if (isGeneratingAI) return;

    console.log('[AI답변생성] 함수 시작');
    setIsGeneratingAI(true);
    try {
      console.log('[AI답변생성] 1단계: 유사 티켓 검색 시작');
      // 1. 유사한 티켓 검색
      const similarResponse = await fetch('/api/support/similar-tickets', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include', // 쿠키 기반 인증 사용
        body: JSON.stringify({
          query: `${currentTicket.title} ${currentTicket.content}`,
          limit: 3
        })
      });

      console.log('[AI답변생성] 유사 티켓 검색 API 호출 완료:', similarResponse.ok);
      let similarTicketsContext = '';
      if (similarResponse.ok) {
        const similarData = await similarResponse.json();
        console.log('[AI답변생성] 유사 티켓 데이터:', similarData);
        if (similarData.success && similarData.data.similarTickets.length > 0) {
          const contextParts = similarData.data.similarTickets.map((ticket: any, index: number) => {
            let ticketContext = `[유사 티켓 ${index + 1}]\n제목: ${ticket.title}\n질문: ${ticket.question}`;

            if (ticket.replies && ticket.replies.length > 0) {
              const answers = ticket.replies
                .filter((reply: any) => reply.author_role === 'admin' || reply.author_role === 'support')
                .map((reply: any) => reply.content)
                .join('\n\n');

              if (answers) {
                ticketContext += `\n답변: ${answers}`;
              }
            }

            return ticketContext;
          });

          similarTicketsContext = `=== 유사한 이전 티켓들 ===\n${contextParts.join('\n\n')}\n========================\n\n`;
        }
      }

      console.log('[AI답변생성] 유사 티켓 검색 완료, chat/sync로 통합 처리 시작');

      // chat/sync 엔드포인트에서 RAG 검색과 답변 생성을 한 번에 처리
      // 티켓 내용을 프롬프트에 직접 포함하여 관련 문서 검색 및 답변 생성
      const supportPrompt = `고객 지원 담당자로서 다음 문의에 대한 답변을 작성해주세요.

고객 문의: "${currentTicket.title}"
상세 내용: "${currentTicket.content}"

${similarTicketsContext}

지시사항:
- 문의 내용을 정확히 파악하고 도움이 되는 구체적인 답변을 제공하세요
- 관련 문서나 매뉴얼 내용이 있다면 활용하여 상세한 답변을 작성하세요
- 이전 유사한 문의의 해결 방법이 있다면 참고하세요
- 문의 제목이나 내용을 반복하지 말고, 바로 답변 내용만 작성하세요

연락처 정보 (답변 마지막에 포함):
- 담당자: ${CONTACT_INFO.representative} (${CONTACT_INFO.company})
- 연락처: ${CONTACT_INFO.phone}
- 이메일: ${CONTACT_INFO.email}

답변 형식:
1. 간단한 인사와 문의 확인
2. 구체적인 해결 방법 또는 설명
3. 추가 지원 안내

고객이 이해하기 쉽고 실용적인 답변을 작성해주세요.`;

      console.log('[AI답변생성] chat/sync 통합 처리 중...');
      console.log('[AI답변생성] 프롬프트 길이:', supportPrompt.length);

      const aiResponse = await externalApiClient.post('/chat/sync', {
        prompt: supportPrompt,
        options: {
          sessionId: `support-answer-${Date.now()}-${Math.random().toString(36).substring(2, 9)}`,
          rag: true, // RAG 검색 활성화 - 티켓 내용으로 관련 문서 검색
          ragSearchScope: 'all', // 모든 문서에서 검색 (개인 + 공용)
          temperature: 0.7,
          max_tokens: 2000,
          type: 'support',
          userId: user?.id ? parseInt(user.id) : null,
          username: user?.username || 'demo',
          model: 'airun-chat:latest' // JSON 형식 응답을 위해 airun-chat 모델 사용
        }
      });

      // 상세한 응답 구조 디버깅
      console.log('[AI답변생성] 전체 응답 구조:', {
        success: aiResponse?.success,
        hasData: !!aiResponse?.data,
        dataKeys: aiResponse?.data ? Object.keys(aiResponse.data) : [],
        hasResponse: !!aiResponse?.data?.response,
        responseValue: aiResponse?.data?.response,
        dataStructure: aiResponse?.data,
        fullResponse: aiResponse
      });

      // AI 응답 처리 - externalApiClient는 ApiResponse<T> 형태로 래핑됨
      let generatedResponse = '';

      // ApiResponse 구조 확인 - 중첩된 data 구조 처리
      if (aiResponse?.success && aiResponse?.data) {
        console.log('[AI답변생성] aiResponse.data 상세 분석:', {
          hasDirectResponse: !!aiResponse.data.response,
          hasNestedResponse: !!aiResponse.data.data?.response,
          directResponseType: typeof aiResponse.data.response,
          nestedResponseType: typeof aiResponse.data.data?.response,
          nestedResponseValue: aiResponse.data.data?.response,
          hasDirectHistory: !!aiResponse.data.history,
          hasNestedHistory: !!aiResponse.data.data?.history,
          nestedHistoryLength: Array.isArray(aiResponse.data.data?.history) ? aiResponse.data.data.history.length : 'not array'
        });

        // 1차: 중첩된 data.data.response에서 추출
        if (aiResponse.data.data?.response && aiResponse.data.data.response.trim()) {
          generatedResponse = aiResponse.data.data.response;
          console.log('[AI답변생성] data.data.response에서 추출 성공');
        }
        // 2차: 직접 data.response에서 추출
        else if (aiResponse.data.response && aiResponse.data.response.trim()) {
          generatedResponse = aiResponse.data.response;
          console.log('[AI답변생성] data.response에서 추출 성공');
        }
        // 3차: 중첩된 data.data.history에서 추출
        else if (aiResponse.data.data?.history && Array.isArray(aiResponse.data.data.history)) {
          const history = aiResponse.data.data.history;
          console.log('[AI답변생성] 중첩된 history 구조:', {
            historyExists: !!history,
            historyType: typeof history,
            isArray: Array.isArray(history),
            historyLength: history.length
          });

          const lastAssistant = [...history].reverse().find((msg) => msg.role === 'assistant' && msg.content);
          if (lastAssistant) {
            generatedResponse = lastAssistant.content;
            console.log('[AI답변생성] 중첩된 history에서 추출 성공');
          } else {
            console.log('[AI답변생성] 중첩된 history에서 assistant 메시지를 찾을 수 없음');
          }
        }
        // 4차: 직접 data.history에서 추출
        else if (aiResponse.data.history && Array.isArray(aiResponse.data.history)) {
          const history = aiResponse.data.history;
          console.log('[AI답변생성] 직접 history 구조:', {
            historyExists: !!history,
            historyType: typeof history,
            isArray: Array.isArray(history),
            historyLength: history.length
          });

          const lastAssistant = [...history].reverse().find((msg) => msg.role === 'assistant' && msg.content);
          if (lastAssistant) {
            generatedResponse = lastAssistant.content;
            console.log('[AI답변생성] 직접 history에서 추출 성공');
          } else {
            console.log('[AI답변생성] 직접 history에서 assistant 메시지를 찾을 수 없음');
          }
        } else {
          console.log('[AI답변생성] 모든 경로에서 응답을 찾을 수 없음');
        }
      } else {
        console.log('[AI답변생성] API 응답이 성공하지 않았거나 data가 없음:', {
          success: aiResponse?.success,
          hasData: !!aiResponse?.data,
          error: aiResponse?.error
        });
      }

      // 최종 확인: 응답이 없는 경우 기본 메시지
      if (!generatedResponse || !generatedResponse.trim()) {
        generatedResponse = 'AI 답변을 생성할 수 없습니다. 잠시 후 다시 시도해주세요.';
        console.log('[AI답변생성] 응답이 없어 기본 메시지 사용');
      }

      console.log('[AI답변생성] 최종 답변:', generatedResponse ? `${generatedResponse.substring(0, 100)}...` : 'empty');

      // 생성된 답변을 텍스트 영역에 설정
      if (replyContent.trim()) {
        showConfirm('AI 답변 덮어쓰기', '기존에 작성된 답변이 있습니다. AI가 생성한 답변으로 덮어쓰시겠습니까?', () => setReplyContent(generatedResponse));
      } else {
        setReplyContent(generatedResponse);
      }
    } catch (error) {
      console.error('[AI답변생성] 에러 발생:', error);
      showAlert('AI 답변 생성 실패', 'AI 답변 생성 중 오류가 발생했습니다. 잠시 후 다시 시도해주세요.');
    } finally {
      console.log('[AI답변생성] 함수 종료');
      setIsGeneratingAI(false);
    }
  };

  return (
    <>
      <div className="p-6">
        {/* 사용자 요청 영역 */}
        <div className="mb-6 bg-blue-50 dark:bg-blue-900/10 p-5 border-l-4 border-blue-500">
          <div className="flex items-center gap-2 mb-3">
            <UserIcon className="h-5 w-5 text-blue-600 dark:text-blue-400" />
            <h4 className="text-lg font-semibold text-blue-800 dark:text-blue-300">고객 문의</h4>
          </div>

          <div className="flex justify-between items-start mb-4">
            <h3 className="text-xl font-semibold text-gray-900 dark:text-gray-100">{currentTicket.title}</h3>
            <div className="flex items-center gap-2">
              <Badge className={currentTicket.status.toLowerCase() === 'open' ? 'bg-orange-100 text-orange-800 dark:bg-orange-900/40 dark:text-orange-300' : 'bg-green-100 text-green-800 dark:bg-green-900/40 dark:text-green-300'}>{currentTicket.status.toLowerCase() === 'open' ? '대기중' : '답변완료'}</Badge>
              {/* 삭제 버튼 - 관리자 또는 작성자만 */}
              {user && (user.role === 'admin' || user.role === 'manager' || currentTicket.author.name?.toLowerCase() === user.username?.toLowerCase()) && (
                <button onClick={handleDeleteTicket} className="text-gray-400 hover:text-red-600 dark:hover:text-red-400 p-1 rounded transition-colors" title={user.role === 'admin' || user.role === 'manager' ? '관리자 권한으로 삭제' : '내 티켓 삭제'}>
                  <TrashIcon className="h-5 w-5" />
                </button>
              )}
            </div>
          </div>

          <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
            작성자: {currentTicket.author.name} • {new Date(currentTicket.createdAt).toLocaleDateString()}
          </p>

          <div className="bg-white dark:bg-gray-800 p-4 mb-4">
            <p className="whitespace-pre-wrap text-gray-900 dark:text-gray-100">{currentTicket.content}</p>
          </div>

          {currentTicket.images && currentTicket.images.length > 0 && (
            <div className="mb-4">
              <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">첨부 이미지</h5>
              <div className="flex flex-wrap gap-4 justify-center">
                {currentTicket.images.map((image, index) => (
                  <div key={index} className="relative group">
                    <div className="aspect-video w-full max-w-4xl bg-gray-100 dark:bg-gray-800 overflow-hidden rounded-lg">
                      <img src={image} alt={`첨부 ${index + 1}`} className="w-full h-full object-contain cursor-pointer hover:opacity-90 transition-opacity" onClick={() => setSelectedImage(image)} />
                    </div>
                    <div className="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity bg-black bg-opacity-30 rounded-lg">
                      <button onClick={() => setSelectedImage(image)} className="bg-white dark:bg-gray-800 text-gray-800 dark:text-gray-200 px-4 py-2 rounded-full text-sm font-medium shadow-lg hover:bg-opacity-90 transition-all transform hover:scale-105">
                        원본 크기로 보기
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>

        {/* 답변 영역 */}
        {currentTicket.replies && currentTicket.replies.length > 0 && (
          <div className="mb-6 bg-green-50 dark:bg-green-900/10 p-5 border-l-4 border-green-500">
            <div className="flex items-center gap-2 mb-4">
              <BotIcon className="h-5 w-5 text-green-600 dark:text-green-400" />
              <h4 className="text-lg font-semibold text-green-800 dark:text-green-300">답변 내역</h4>
              <Badge className="bg-green-100 text-green-800 dark:bg-green-900/40 dark:text-green-300">{currentTicket.replies.length}개</Badge>
            </div>
            <div className="space-y-4">
              {currentTicket.replies.map((reply, index) => (
                <div key={index} className="bg-white dark:bg-gray-800 p-4">
                  <div className="flex justify-between items-start mb-3">
                    <div className="flex items-center gap-2">
                      <span className="text-sm font-medium text-gray-900 dark:text-gray-100">{reply.authorName}</span>
                      <Badge className="bg-blue-100 text-blue-800 dark:bg-blue-900/40 dark:text-blue-300">{reply.authorRole}</Badge>
                    </div>
                    <span className="text-xs text-gray-500 dark:text-gray-400">{new Date(reply.createdAt).toLocaleString()}</span>
                  </div>
                  <div className="text-sm text-gray-900 dark:text-gray-100">
                    <SupportRenderer content={reply.content} />
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* 관리자/매니저용 답변 작성 영역 */}
        {(user?.role === 'admin' || user?.role === 'manager') && (
          <div className="mb-6 bg-purple-50 dark:bg-purple-900/10 p-5 border-l-4 border-purple-500">
            <div className="flex items-center gap-2 mb-4">
              <BotIcon className="h-5 w-5 text-purple-600 dark:text-purple-400" />
              <h4 className="text-lg font-semibold text-purple-800 dark:text-purple-300">답변 작성</h4>
            </div>

            <form onSubmit={handleSubmitReply} className="bg-white dark:bg-gray-800 p-4">
              <div className="mb-4">
                <div className="flex items-center justify-between mb-3">
                  <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">답변 내용</label>
                  <Button type="button" variant="outline" onClick={generateAIResponse} disabled={isGeneratingAI || isSubmitting} className="!h-7 px-2 text-sm flex items-center gap-1 bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-800 text-blue-700 dark:text-blue-300 hover:bg-blue-100 dark:hover:bg-blue-900/40">
                    {isGeneratingAI ? (
                      <>
                        <SpinnerIcon className="h-3 w-3 animate-spin" />
                        생성중...
                      </>
                    ) : (
                      <>
                        <BotIcon className="text-base font-bold h-5 w-5" />
                        AI 답변 생성
                      </>
                    )}
                  </Button>
                </div>
                <Textarea value={replyContent} onChange={(e) => setReplyContent(e.target.value)} placeholder={isGeneratingAI ? 'AI가 답변을 생성하고 있습니다. 잠시만 기다려주세요...' : "답변을 입력하세요... (우측 상단의 'AI 답변 생성' 버튼을 클릭하면 AI가 답변을 자동으로 생성해드립니다)"} className="min-h-[300px]" disabled={isSubmitting || isGeneratingAI} />
              </div>

              {/* 이미지 첨부 영역 */}
              <div className="mb-4">
                <div className="flex items-center gap-2 mb-2">
                  <Button type="button" variant="outline" onClick={() => imageInputRef.current?.click()} className="!h-9 px-3 flex items-center gap-1" disabled={isSubmitting}>
                    <UploadIcon className="h-4 w-4" />
                    이미지 첨부
                  </Button>
                  {replyImages.length > 0 && <span className="text-sm text-gray-500">{replyImages.length}개의 이미지 선택됨</span>}
                </div>
                <input ref={imageInputRef} type="file" accept="image/*" multiple onChange={handleReplyImageUpload} className="hidden" />
                {replyImages.length > 0 && (
                  <div className="flex flex-wrap gap-4 mt-2">
                    {replyImages.map((image, index) => (
                      <div key={index} className="relative group">
                        <div className="w-24 h-24 overflow-hidden bg-gray-100 dark:bg-gray-800">
                          <img src={URL.createObjectURL(image)} alt={`첨부 ${index + 1}`} className="w-full h-full object-cover" />
                        </div>
                        <button type="button" onClick={() => removeReplyImage(index)} className="absolute -top-1 -right-1 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity">
                          <XIcon className="h-4 w-4" />
                        </button>
                      </div>
                    ))}
                  </div>
                )}
              </div>

              <div className="flex justify-between">
                <div>
                  {/* 관리자/매니저용 삭제 버튼 */}
                  <Button type="button" variant="outline" onClick={handleDeleteTicket} disabled={isGeneratingAI || isSubmitting} className="border-red-300 text-red-700 hover:bg-red-50 dark:border-red-600 dark:text-red-400 dark:hover:bg-red-900/20">
                    티켓 삭제
                  </Button>
                </div>
                <div className="flex gap-2">
                  <Button variant="outline" onClick={onClose} disabled={isGeneratingAI}>
                    닫기
                  </Button>
                  <Button type="submit" disabled={!replyContent.trim() || isSubmitting || isGeneratingAI}>
                    {isSubmitting ? '등록 중...' : '답변 등록'}
                  </Button>
                </div>
              </div>
            </form>
          </div>
        )}

        {/* 사용자용 하단 버튼 */}
        {user?.role !== 'admin' && user?.role !== 'manager' && (
          <div className="flex justify-between">
            <div>
              {/* 일반 사용자의 티켓 삭제 버튼 (작성자인 경우만) */}
              {user && currentTicket.author.name?.toLowerCase() === user.username?.toLowerCase() && (
                <Button variant="outline" onClick={handleDeleteTicket} className="border-red-300 text-red-700 hover:bg-red-50 dark:border-red-600 dark:text-red-400 dark:hover:bg-red-900/20">
                  티켓 삭제
                </Button>
              )}
            </div>
            <Button variant="outline" onClick={onClose}>
              닫기
            </Button>
          </div>
        )}
      </div>

      <ImageViewer image={selectedImage} onClose={() => setSelectedImage(null)} />

      {/* 확인/경고 모달 */}
      <ConfirmModal isOpen={modalState.isOpen} type={modalState.type} title={modalState.title} message={modalState.message} onConfirm={modalState.onConfirm} onCancel={modalState.onCancel} />
    </>
  );
}

// 백그라운드 작업 상태 섹션 컴포넌트 (항상 표시)
function BackgroundJobStatusSection({ userId, onJobComplete, refreshTrigger }: { userId?: string; onJobComplete?: (job: BackgroundJob) => void; refreshTrigger: number }) {
  return (
    <div className="mb-6">
      <BackgroundJobStatus
        userId={userId}
        onJobComplete={onJobComplete}
        refreshTrigger={refreshTrigger}
        hideFileUpload={true} // 파일 업로드 작업은 별도 UI에서 처리
      />
    </div>
  );
}

function ChatView(props: ChatViewProps) {
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const [isDragging, setIsDragging] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [thinkingDots, setThinkingDots] = useState('');
  const [isKnowledgeBaseExpanded, setIsKnowledgeBaseExpanded] = useState(true);

  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (props.isLoading) {
      let count = 0;
      interval = setInterval(() => {
        count = (count + 1) % 4;
        setThinkingDots('.'.repeat(count));
      }, 300);
    }
    return () => {
      if (interval) clearInterval(interval);
    };
  }, [props.isLoading]);

  useEffect(() => {
    if (props.messages.length > 0) {
      messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    }
  }, [props.messages]);

  const handleDragOver = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(true);
  };

  const handleDragLeave = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
  };

  const handleDrop = async (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);

    console.log('드롭 이벤트 처리 시작...');
    console.log('dataTransfer.files 길이:', e.dataTransfer.files?.length);

    // 직접 dataTransfer.files 사용 (가장 안정적인 방법)
    const allFiles = Array.from(e.dataTransfer.files);
    console.log(
      '드롭된 파일들:',
      allFiles.map((f) => `${f.name} (${f.size} bytes, ${f.type})`)
    );

    if (allFiles.length === 0) {
      console.log('드롭된 파일이 없습니다.');
      return;
    }

    // 지원되는 파일만 필터링
    const supportedExtensions = ['pdf', 'txt', 'md', 'doc', 'docx', 'hwp', 'hwpx', 'ppt', 'pptx', 'xls', 'xlsx', 'csv', 'json'];

    const validFiles = allFiles.filter((file) => {
      const fileExt = file.name.split('.').pop()?.toLowerCase();
      const hasExtension = file.name.includes('.') && !file.name.endsWith('.');
      const isNotHidden = !file.name.startsWith('.');
      const isSupportedExtension = fileExt && supportedExtensions.includes(fileExt);
      const isValidFile = hasExtension && isNotHidden && isSupportedExtension;

      console.log(`파일 ${file.name}: 확장자=${fileExt}, 지원여부=${isSupportedExtension}, 결과=${isValidFile ? 'INCLUDE' : 'EXCLUDE'}`);

      return isValidFile;
    });

    const invalidFiles = allFiles.filter((file) => !validFiles.includes(file));

    console.log(
      '필터링 후 최종 파일 목록:',
      validFiles.map((f) => f.name)
    );

    if (invalidFiles.length > 0) {
      props.showAlert(
        '파일 형식 오류',
        `지원하지 않는 파일 형식입니다: ${invalidFiles.map((f) => f.name).join(', ')}

지원 형식: PDF, DOC, DOCX, TXT, MD, HWP, HWPX, PPT, PPTX, XLS, XLSX, CSV, JSON`
      );
    }

    if (validFiles.length === 0) {
      console.log('업로드할 수 있는 파일이 없습니다.');
      return;
    }

    if (!props.user?.username) {
      props.showAlert('오류', '로그인이 필요합니다.');
      return;
    }

    console.log(`총 ${validFiles.length}개 파일을 업로드합니다.`);

    // 각 파일을 순차적으로 업로드
    for (const file of validFiles) {
      try {
        await externalApiClient.uploadFile(
          file,
          'support',
          'support',
          (progress, status) => {
            // props.setFiles 업데이트 로직 (기존 방식과 호환)
            props.setFiles((prev) => {
              const index = prev.findIndex((f) => f.name === file.name);
              const fileWithStatus = {
                ...file,
                status,
                progress
              };
              if (index !== -1) {
                const updatedFiles = [...prev];
                updatedFiles[index] = fileWithStatus;
                return updatedFiles;
              } else {
                return [...prev, fileWithStatus];
              }
            });
          },
          validFiles.length > 1 // 다중 파일 업로드 여부
        );
      } catch (error) {
        console.error(`파일 업로드 실패: ${file.name}`, error);
        props.showAlert('업로드 오류', `${file.name} 업로드에 실패했습니다.`);
      }
    }
  };

  const handleUpload = (filesToUpload: File[]) => {
    if (filesToUpload.length === 0 || props.isUploading) return;
    props.handleFileUpload(filesToUpload);
  };

  const removeFile = (index: number) => {
    props.removeFile(index);
  };

  return (
    <div className="flex-1 flex flex-col h-full" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
      {/* Knowledge Base - 관리자/매니저만 접근 가능하고 모바일에서는 숨김 */}
      {(props.user?.role === 'admin' || props.user?.role === 'manager') && (
        <div className="hidden md:block border-b border-[var(--sidebar-border)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
          <button onClick={() => setIsKnowledgeBaseExpanded(!isKnowledgeBaseExpanded)} className="w-full p-4 text-left focus:outline-none hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors">
            <div className="flex items-center justify-between">
              <h3 className="flex items-center gap-2 text-md font-semibold text-gray-900 dark:text-gray-100">
                <UploadIcon className="h-5 w-5" />
                문서 관리
              </h3>
              <svg className={`w-5 h-5 transition-transform ${isKnowledgeBaseExpanded ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
              </svg>
            </div>
            <p className="text-sm pl-7 text-gray-500 dark:text-gray-400">AI 답변의 기반이 될 문서 관리</p>
          </button>

          {isKnowledgeBaseExpanded && (
            <div className="px-6 py-4 border-t border-[var(--sidebar-border)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
              {/* 상단 버튼 영역 - 관리자 전용 */}
              <div className="flex justify-end mb-4">
                <Button onClick={props.onSyncTickets} variant="outline" disabled={props.isSyncing} className="!h-9 px-3 flex items-center gap-1 bg-purple-200 dark:bg-purple-600 border-purple-200 dark:border-purple-800 text-purple-700 dark:text-purple-300 hover:bg-purple-100 dark:hover:bg-purple-600/40">
                  {props.isSyncing ? (
                    <>
                      <SpinnerIcon className="h-4 w-4 animate-spin" />
                      동기화 중...
                    </>
                  ) : (
                    '요청 DB 동기화'
                  )}
                </Button>
              </div>

              {/* 모바일: 세로 배치, 데스크톱: 가로 배치 */}
              <div className="flex flex-col md:flex-row gap-4 md:gap-6 px-2 md:px-4">
                {/* 파일 업로드 영역 */}
                <div className="w-full md:w-1/2">
                  <div className={`border-2 border-dashed rounded-lg p-6 mb-3 transition-colors ${isDragging ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 bg-gray-50 dark:bg-gray-800'}`} onDragOver={handleDragOver} onDragLeave={handleDragLeave} onDrop={handleDrop}>
                    <div className="text-center">
                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-3">파일을 가져오거나 아래 버튼을 클릭하세요.</p>
                      <Button type="button" onClick={() => fileInputRef.current?.click()} variant="outline" disabled={props.isUploading} className="bg-gray-200 dark:bg-gray-900">
                        파일 선택
                      </Button>
                    </div>
                  </div>
                  <input ref={fileInputRef} type="file" accept=".pdf,.doc,.docx,.txt,.md,.hwp,.hwpx,.ppt,.pptx,.xls,.xlsx,.csv,.json" multiple onChange={props.handleFileChange} className="hidden" />
                  {props.files.length > 0 && (
                    <div className="space-y-2">
                      {props.files.map((file, index) => {
                        const fileId = `${file.name}-${file.lastModified}`;
                        const progress = props.uploadProgress[fileId];
                        const embeddingStatus = props.embeddingStatuses[file.name];

                        // RAG 서버의 실제 임베딩 상태 우선 적용
                        let displayStatus = progress?.status || 'pending';
                        let displayMessage = progress?.message || '';
                        let displayIcon = null;

                        // 제거 예정 상태가 있으면 우선 적용
                        if (progress?.status === 'removing') {
                          displayStatus = 'success';
                          displayMessage = '임베딩 완료!';
                          displayIcon = <CheckCircleIcon className="h-4 w-4 text-green-500 flex-shrink-0" />;
                        } else if (embeddingStatus) {
                          if (embeddingStatus.isProcessing) {
                            displayStatus = 'processing';
                            displayMessage = embeddingStatus.step_label || '임베딩 진행 중...';
                            displayIcon = <SpinnerIcon className="h-4 w-4 text-orange-500 animate-spin flex-shrink-0" />;
                          } else if (embeddingStatus.step === 'done') {
                            displayStatus = 'success';
                            displayMessage = '임베딩 완료!';
                            displayIcon = <CheckCircleIcon className="h-4 w-4 text-green-500 flex-shrink-0" />;
                          } else if (embeddingStatus.step === 'error' || embeddingStatus.step === 'failed') {
                            displayStatus = 'error';
                            displayMessage = '임베딩 실패';
                            displayIcon = <XCircleIcon className="h-4 w-4 text-red-500 flex-shrink-0" />;
                          }
                        } else {
                          // 기존 업로드 진행 상태 표시
                          if (!progress || progress.status === 'pending') {
                            displayIcon = <ClockIcon className="h-4 w-4 text-gray-500 flex-shrink-0" />;
                          } else if (progress.status === 'uploading') {
                            displayIcon = <SpinnerIcon className="h-4 w-4 text-blue-500 animate-spin flex-shrink-0" />;
                          } else if (progress.status === 'processing') {
                            displayIcon = <SpinnerIcon className="h-4 w-4 text-orange-500 animate-spin flex-shrink-0" />;
                          } else if (progress.status === 'success') {
                            displayIcon = <CheckCircleIcon className="h-4 w-4 text-green-500 flex-shrink-0" />;
                          } else if (progress.status === 'error') {
                            displayIcon = <XCircleIcon className="h-4 w-4 text-red-500 flex-shrink-0" />;
                          }
                        }

                        return (
                          <div key={fileId} className="flex items-center justify-between bg-gray-50 dark:bg-gray-800 p-2 rounded-lg">
                            <div className="flex items-center gap-2 overflow-hidden">
                              {displayIcon}
                              <span className="text-sm truncate" title={file.name}>
                                {file.name}
                              </span>
                            </div>
                            <div className="flex items-center gap-2">
                              {displayMessage && (
                                <span className={`text-xs truncate ${
                                  displayStatus === 'error' ? 'text-red-500' :
                                  displayStatus === 'success' ? 'text-green-500' :
                                  'text-blue-500'
                                }`} title={displayMessage}>
                                  {displayMessage}
                                </span>
                              )}
                              <button
                                type="button"
                                onClick={() => removeFile(index)}
                                className="text-gray-400 hover:text-red-600 dark:hover:text-red-400 ml-2 flex-shrink-0"
                                disabled={displayStatus === 'uploading' || displayStatus === 'processing'}
                              >
                                <TrashIcon className="h-4 w-4" />
                              </button>
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  )}
                </div>

                {/* 업로드된 문서 목록 */}
                <div className="w-full md:w-1/2">
                  <p className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">업로드된 문서 ({props.uploadedDocuments.length}개)</p>
                  {props.uploadedDocuments.length > 0 ? (
                    <div className="space-y-2 max-h-48 overflow-y-auto">
                      {props.uploadedDocuments.map((doc) => (
                        <div key={doc.id} className="flex items-center justify-between bg-green-50 dark:bg-green-900/20 p-1 rounded-lg border border-green-200 dark:border-green-800">
                          <div className="flex items-center gap-2 overflow-hidden">
                            <CheckCircleIcon className="h-4 w-4 text-green-500 flex-shrink-0" />
                            <div className="min-w-0">
                              <span className="text-xs text-green-700 dark:text-green-300 truncate block" title={doc.filename}>
                                {doc.filename}
                              </span>
                              <span className="text-xs text-green-600 dark:text-green-400">
                                {new Date(doc.created_at).toLocaleString()}
                                {doc.filesize && ` • ${(doc.filesize / 1024 / 1024).toFixed(1)}MB`}
                              </span>
                            </div>
                          </div>
                          <button type="button" onClick={() => props.onRemoveDocument(doc)} className="text-green-400 hover:text-red-600 dark:hover:text-red-400 ml-2 flex-shrink-0" title="문서 삭제">
                            <TrashIcon className="h-4 w-4" />
                          </button>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-xs text-center p-4 text-gray-500 dark:text-gray-400 bg-gray-50 dark:bg-gray-800 rounded-lg">업로드된 문서가 없습니다.</div>
                  )}
                </div>
              </div>
              {/* 백그라운드 작업 상태 표시 - 작업이 있을 때만 */}
              <div className="px-4 pt-4">
                <BackgroundJobStatusSection userId="support" onJobComplete={props.handleJobComplete} refreshTrigger={props.refreshTrigger} />
              </div>
            </div>
          )}
        </div>
      )}

      <div className="flex-1 flex flex-col px-2 md:px-8 py-2 md:py-4 overflow-hidden" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        <div className="px-3 md:px-6 py-3 md:py-4 border-b border-[var(--sidebar-border)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
          <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-2 sm:gap-0 mb-3">
            <h2 className="text-lg md:text-xl font-bold">어떤 문제가 있으신가요?</h2>
            <button onClick={props.handleNewChat} className="px-3 py-1.5 bg-blue-500 text-white rounded hover:bg-blue-600 text-sm disabled:opacity-50 self-start sm:self-auto" disabled={props.isLoading}>
              새 대화
            </button>
          </div>
          <div className="bg-blue-50 dark:bg-blue-900/20 p-3 rounded-lg border border-blue-200 dark:border-blue-700">
            <div className="flex items-start gap-2">
              <BotIcon className="h-5 w-5 text-blue-600 mt-0.5 flex-shrink-0" />
              <div className="text-sm text-blue-800 dark:text-blue-200">
                <p className="font-medium mb-1">인공지능 챗봇</p>
                <p className="text-xs text-blue-700 dark:text-blue-300 leading-relaxed">
                  • 궁금한 점을 자유롭게 질문해주세요
                  <br />
                  • 업로드된 매뉴얼과 FAQ를 기반으로 정확한 답변을 제공합니다
                  <br />• 복잡한 문제는 상단의 "새 요청" 버튼으로 티켓을 등록해주세요
                </p>
              </div>
            </div>
          </div>
        </div>
        {/* 유사 티켓 표시 섹션 */}
        {props.similarTickets.length > 0 && (
          <div className="mb-4 mx-3 md:mx-6 p-2 md:p-3 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
            <h4 className="text-sm font-medium text-blue-800 dark:text-blue-300 mb-2">🔍 유사한 이전 티켓 ({props.similarTickets.length}개)</h4>
            <div className="space-y-2 max-h-32 overflow-y-auto">
              {props.similarTickets.map((ticket: any, index: number) => (
                <div key={ticket.id} className="bg-white dark:bg-gray-800 p-2 rounded border border-blue-200 dark:border-blue-700">
                  <div className="text-xs font-medium text-blue-700 dark:text-blue-300 mb-1">{ticket.title}</div>
                  <div
                    className="text-xs text-gray-600 dark:text-gray-400"
                    style={{
                      display: '-webkit-box',
                      WebkitLineClamp: 2,
                      WebkitBoxOrient: 'vertical',
                      overflow: 'hidden'
                    }}
                  >
                    {ticket.question}
                  </div>
                  {ticket.replies && ticket.replies.length > 0 && <div className="text-xs text-green-600 dark:text-green-400 mt-1">✓ 해결된 티켓 (답변 {ticket.replies.length}개)</div>}
                </div>
              ))}
            </div>
            <div className="text-xs text-blue-600 dark:text-blue-400 mt-2">위 티켓들을 참고하여 더 정확한 답변을 제공합니다.</div>
          </div>
        )}

        <div className="flex-1 overflow-y-auto rounded-md border border-[var(--sidebar-border)] px-3 md:px-8 py-2 md:py-4 space-y-3 md:space-y-4 mb-2 md:mb-4 max-h-[50vh] mx-3 md:mx-6" style={{ backgroundColor: 'var(--body-bg)' }}>
          {props.messages.length === 0 ? (
            <div className="h-full flex items-center justify-center">
              <div className="text-center text-gray-500 dark:text-gray-400">
                <BotIcon className="h-12 w-12 mx-auto mb-3 text-gray-400" />
                <p className="text-base font-medium mb-2">무엇이든 물어보세요!</p>
                <p className="text-sm">아래 입력창에 질문을 입력해주세요</p>
              </div>
            </div>
          ) : (
            props.messages.map((msg, i) => (
              <div key={i} className={`flex items-start gap-3 ${msg.role === 'user' ? 'justify-end' : ''}`}>
                {msg.role === 'assistant' && <BotIcon className="h-6 w-6 text-blue-600 flex-shrink-0" />}
                <div className={`max-w-[75%] rounded-lg px-4 py-2 text-sm ${msg.role === 'user' ? 'bg-gray-200 text-gray-900' : 'bg-gray-200 dark:bg-gray-700 text-gray-900 dark:text-gray-100'}`}>{msg.role === 'assistant' ? <SupportRenderer content={msg.content} /> : <p className="whitespace-pre-wrap">{msg.content}</p>}</div>
                {msg.role === 'user' && <UserIcon className="h-6 w-6 text-gray-400 flex-shrink-0" />}
              </div>
            ))
          )}
          {props.isLoading && (
            <div className="flex items-start gap-3">
              <BotIcon className="h-6 w-6 text-blue-600 flex-shrink-0" />
              <div className="max-w-[70%] rounded-lg px-4 py-2 bg-gray-200 dark:bg-gray-700">
                <div className="flex items-center space-x-2">
                  <span className="text-sm text-gray-900 dark:text-gray-100">Think{thinkingDots}</span>
                </div>
              </div>
            </div>
          )}
          <div ref={messagesEndRef} />
        </div>
        <form onSubmit={props.handleSubmit} className="flex gap-2 mx-3 md:mx-6">
          <Input value={props.input} onChange={(e) => props.setInput(e.target.value)} placeholder="질문을 입력하세요..." disabled={props.isLoading} className="text-sm md:text-base" />
          <Button type="submit" disabled={props.isLoading || !props.input.trim()} className="px-2 md:px-4">
            <SendIcon className="h-4 w-4 md:h-5 md:w-5" />
          </Button>
        </form>
      </div>
    </div>
  );
}
