'use client';

import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import GraphRAGVisualization from './GraphRAGVisualization';

interface Chunk {
  id: string;
  content: string;
  metadata?: {
    page?: number;
    chunk_index?: number;
    file_path?: string;
  };
}

interface GraphRAGFilePreviewModalProps {
  isOpen: boolean;
  onClose: () => void;
  selectedDoc: any;
  chunks: Chunk[];
  onProcessGraphRAG?: (documentId: string) => Promise<void>;
}

const GraphRAGFilePreviewModal: React.FC<GraphRAGFilePreviewModalProps> = ({
  isOpen,
  onClose,
  selectedDoc,
  chunks,
  onProcessGraphRAG
}) => {
  const [activeTab, setActiveTab] = useState('content');
  const [isProcessing, setIsProcessing] = useState(false);
  const [hasGraphRAGData, setHasGraphRAGData] = useState(false);

  // GraphRAG 데이터 존재 여부 확인
  const checkGraphRAGData = async (documentId: string) => {
    if (!documentId) return;
    
    try {
      const response = await fetch(`/api/v1/rag/graphrag/${documentId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': localStorage.getItem('apiKey') || ''
        }
      });

      if (response.ok) {
        const result = await response.json();
        setHasGraphRAGData(result.success && result.data && result.data.entities && result.data.entities.length > 0);
      } else if (response.status === 404) {
        // GraphRAG 데이터가 없음 (정상)
        setHasGraphRAGData(false);
      } else {
        console.warn('GraphRAG API 엔드포인트가 아직 활성화되지 않았습니다.');
        setHasGraphRAGData(false);
      }
    } catch (error) {
      console.warn('GraphRAG 기능이 아직 활성화되지 않았습니다:', error);
      setHasGraphRAGData(false);
    }
  };

  // GraphRAG 처리 실행
  const handleProcessGraphRAG = async () => {
    if (!selectedDoc?.id) return;

    setIsProcessing(true);
    try {
      if (onProcessGraphRAG) {
        await onProcessGraphRAG(selectedDoc.id);
      } else {
        // 기본 GraphRAG 처리 요청
        const response = await fetch('/api/v1/rag/graphrag/process', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-API-Key': localStorage.getItem('apiKey') || ''
          },
          body: JSON.stringify({
            document_id: selectedDoc.id,
            user_id: 'admin' // 또는 현재 사용자 ID
          })
        });

        if (!response.ok) {
          if (response.status === 404) {
            throw new Error('GraphRAG API가 아직 활성화되지 않았습니다. API 서버를 확인해주세요.');
          }
          throw new Error(`GraphRAG 처리 실패: ${response.status}`);
        }

        const result = await response.json();
        if (!result.success) {
          throw new Error(result.error?.message || 'GraphRAG 처리 실패');
        }
      }
      
      // 처리 완료 후 데이터 다시 확인
      await checkGraphRAGData(selectedDoc.id);
      setActiveTab('graphrag'); // 자동으로 GraphRAG 탭으로 전환
    } catch (error) {
      console.error('GraphRAG 처리 오류:', error);
      alert(error instanceof Error ? error.message : 'GraphRAG 처리 중 오류가 발생했습니다.');
    } finally {
      setIsProcessing(false);
    }
  };

  // 선택된 문서 변경 시 GraphRAG 데이터 확인
  useEffect(() => {
    if (selectedDoc?.id) {
      checkGraphRAGData(selectedDoc.id);
    }
  }, [selectedDoc?.id]);

  if (!isOpen || !selectedDoc) return null;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-7xl w-full h-[90vh] overflow-hidden">
        <DialogHeader>
          <DialogTitle className="flex items-center justify-between">
            <span>{selectedDoc.file_name || selectedDoc.name}</span>
            <div className="flex items-center space-x-2">
              {selectedDoc.file_size && (
                <Badge variant="secondary">
                  {(selectedDoc.file_size / 1024).toFixed(1)} KB
                </Badge>
              )}
              {selectedDoc.upload_date && (
                <Badge variant="outline">
                  {new Date(selectedDoc.upload_date).toLocaleDateString('ko-KR')}
                </Badge>
              )}
            </div>
          </DialogTitle>
        </DialogHeader>

        <div className="flex-1 overflow-hidden">
          <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full h-full flex flex-col">
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="content">문서 내용</TabsTrigger>
              <TabsTrigger value="graphrag" className="relative">
                GraphRAG 관계도
                {hasGraphRAGData && (
                  <div className="absolute -top-1 -right-1 w-2 h-2 bg-green-500 rounded-full"></div>
                )}
              </TabsTrigger>
            </TabsList>

            <TabsContent value="content" className="flex-1 overflow-hidden mt-4">
              <div className="grid grid-cols-1 gap-4 h-full overflow-auto">
                {chunks.length > 0 ? (
                  chunks.map((chunk, index) => (
                    <div 
                      key={chunk.id || index} 
                      className="rounded-lg p-4"
                      style={{
                        border: '1px solid var(--card-border)',
                        backgroundColor: 'var(--header-bg)'
                      }}
                    >
                      <div className="flex justify-between items-center mb-2">
                        <span 
                          className="text-sm font-medium"
                          style={{ color: 'var(--text-secondary)' }}
                        >
                          청크 {index + 1}
                          {chunk.metadata?.page && ` (페이지 ${chunk.metadata.page})`}
                        </span>
                        <span 
                          className="text-xs"
                          style={{ color: 'var(--text-muted)' }}
                        >
                          {chunk.content.length} 글자
                        </span>
                      </div>
                      <div 
                        className="text-sm whitespace-pre-wrap max-h-32 overflow-y-auto"
                        style={{ color: 'var(--text-primary)' }}
                      >
                        {chunk.content}
                      </div>
                    </div>
                  ))
                ) : (
                  <div 
                    className="flex items-center justify-center h-full"
                    style={{ color: 'var(--text-muted)' }}
                  >
                    <div className="text-center">
                      <svg 
                        className="w-16 h-16 mx-auto mb-4" 
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                        style={{ color: 'var(--text-muted)' }}
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                      </svg>
                      <p>문서 내용을 불러올 수 없습니다.</p>
                    </div>
                  </div>
                )}
              </div>
            </TabsContent>

            <TabsContent value="graphrag" className="flex-1 overflow-hidden mt-4">
              <div className="h-full">
                {hasGraphRAGData ? (
                  <GraphRAGVisualization 
                    documentId={selectedDoc.id}
                    width={900}
                    height={500}
                  />
                ) : (
                  <div className="flex items-center justify-center h-full">
                    <div className="text-center">
                      <svg 
                        className="w-16 h-16 mx-auto mb-4" 
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                        style={{ color: 'var(--text-muted)' }}
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                      </svg>
                      <h3 
                        className="text-lg font-medium mb-2"
                        style={{ color: 'var(--text-primary)' }}
                      >
                        GraphRAG 관계 분석
                      </h3>
                      <p 
                        className="mb-4"
                        style={{ color: 'var(--text-secondary)' }}
                      >
                        이 문서의 엔티티와 관계를 분석하여 시각적으로 표시합니다.
                      </p>
                      <Button 
                        onClick={handleProcessGraphRAG}
                        disabled={isProcessing}
                        className="transition-colors"
                        style={{
                          backgroundColor: 'var(--btn-primary-bg)',
                          color: 'var(--btn-primary-fg)'
                        }}
                        onMouseEnter={(e) => {
                          if (!isProcessing) {
                            e.currentTarget.style.backgroundColor = 'var(--btn-primary-hover)';
                          }
                        }}
                        onMouseLeave={(e) => {
                          if (!isProcessing) {
                            e.currentTarget.style.backgroundColor = 'var(--btn-primary-bg)';
                          }
                        }}
                      >
                        {isProcessing ? (
                          <>
                            <div 
                              className="animate-spin rounded-full h-4 w-4 border-b-2 mr-2"
                              style={{ borderBottomColor: 'white' }}
                            ></div>
                            분석 중...
                          </>
                        ) : (
                          'GraphRAG 분석 시작'
                        )}
                      </Button>
                      <p 
                        className="text-xs mt-2"
                        style={{ color: 'var(--text-muted)' }}
                      >
                        처리 시간은 문서 크기에 따라 1-5분 소요될 수 있습니다.
                      </p>
                    </div>
                  </div>
                )}
              </div>
            </TabsContent>
          </Tabs>
        </div>

        <div className="flex justify-end space-x-2 pt-4 border-t">
          <Button variant="outline" onClick={onClose}>
            닫기
          </Button>
          {hasGraphRAGData && (
            <Button 
              onClick={handleProcessGraphRAG}
              disabled={isProcessing}
              variant="outline"
            >
              {isProcessing ? '재분석 중...' : 'GraphRAG 재분석'}
            </Button>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
};

export default GraphRAGFilePreviewModal;