// AI 모델 메타데이터 정의
export interface ModelMetadata {
  parameters: string;           // 파라미터 수 (예: "7B", "13B")
  contextLength: number;        // 컨텍스트 길이
  supportsTools: boolean;       // 도구 호출 지원
  supportsVision: boolean;      // 이미지 인식 지원
  family: string;              // 모델 패밀리 (예: "Llama", "Gemma")
  provider: string;            // 제공자 (예: "Meta", "Google")
  description?: string;        // 모델 설명
  source: string;             // 정보 출처
}

// 모델 메타데이터 매핑
export const MODEL_METADATA: Record<string, ModelMetadata> = {
  // Gemma 3 시리즈 - https://ai.google.dev/gemma/docs/core/model_card_3
  'gemma3': {
    parameters: '4B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: true,
    family: 'Gemma 3',
    provider: 'Google',
    description: 'Multimodal model with function calling and 128K context window',
    source: 'ai.google.dev/gemma/docs/core/model_card_3'
  },
  'gemma3:1b': {
    parameters: '1B',
    contextLength: 32000,
    supportsTools: true,
    supportsVision: true,
    family: 'Gemma 3',
    provider: 'Google',
    description: 'Lightweight multimodal model with function calling',
    source: 'ai.google.dev/gemma/docs/core/model_card_3'
  },
  'gemma3:4b': {
    parameters: '4B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: true,
    family: 'Gemma 3',
    provider: 'Google',
    description: 'Multimodal model with function calling and 128K context window',
    source: 'ai.google.dev/gemma/docs/core/model_card_3'
  },
  'gemma3:12b': {
    parameters: '12B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: true,
    family: 'Gemma 3',
    provider: 'Google',
    description: 'Large multimodal model with function calling and 128K context window',
    source: 'ai.google.dev/gemma/docs/core/model_card_3'
  },
  'gemma3:27b': {
    parameters: '27B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: true,
    family: 'Gemma 3',
    provider: 'Google',
    description: 'Largest multimodal model with function calling and 128K context window',
    source: 'ai.google.dev/gemma/docs/core/model_card_3'
  },

  // Llama 3.2 시리즈
  'llama3.2': {
    parameters: '3B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Instruction-tuned model with tool calling',
    source: 'ollama.com/library/llama3.2'
  },
  'llama3.2:1b': {
    parameters: '1B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Lightweight instruction-tuned model',
    source: 'ollama.com/library/llama3.2'
  },
  'llama3.2:3b': {
    parameters: '3B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Instruction-tuned model with tool calling',
    source: 'ollama.com/library/llama3.2'
  },
  'llama3.2-vision': {
    parameters: '11B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: true,
    family: 'Llama',
    provider: 'Meta',
    description: 'Multimodal model with vision and tool calling',
    source: 'ollama.com/library/llama3.2-vision'
  },

  // Llama 3.1 시리즈
  'llama3.1': {
    parameters: '8B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Instruction-tuned model with extended context',
    source: 'ollama.com/library/llama3.1'
  },
  'llama3.1:8b': {
    parameters: '8B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Instruction-tuned model with extended context',
    source: 'ollama.com/library/llama3.1'
  },
  'llama3.1:70b': {
    parameters: '70B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Large instruction-tuned model',
    source: 'ollama.com/library/llama3.1'
  },
  'llama3.1:405b': {
    parameters: '405B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Llama',
    provider: 'Meta',
    description: 'Largest instruction-tuned model',
    source: 'ollama.com/library/llama3.1'
  },

  // Qwen 2.5 시리즈
  'qwen2.5': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Multilingual model with strong reasoning',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5vl': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: true,
    family: 'Qwen VL',
    provider: 'Alibaba',
    description: 'Multimodal vision-language model',
    source: 'ollama.com/library/qwen2.5-vl'
  },
  'qwen2.5vl:latest': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: true,
    family: 'Qwen VL',
    provider: 'Alibaba',
    description: 'Multimodal vision-language model',
    source: 'ollama.com/library/qwen2.5-vl'
  },
  'qwen2.5:0.5b': {
    parameters: '0.5B',
    contextLength: 32768,
    supportsTools: false,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Ultra-lightweight multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:1.5b': {
    parameters: '1.5B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Lightweight multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:3b': {
    parameters: '3B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Balanced multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:7b': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Multilingual model with strong reasoning',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:14b': {
    parameters: '14B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Large multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:32b': {
    parameters: '32B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Very large multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },
  'qwen2.5:72b': {
    parameters: '72B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Qwen',
    provider: 'Alibaba',
    description: 'Largest multilingual model',
    source: 'ollama.com/library/qwen2.5'
  },

  // Mistral 시리즈
  'mistral': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Mistral',
    provider: 'Mistral AI',
    description: 'Efficient instruction-following model',
    source: 'ollama.com/library/mistral'
  },
  'mistral-nemo': {
    parameters: '12B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Mistral',
    provider: 'Mistral AI',
    description: 'Advanced reasoning model with extended context',
    source: 'ollama.com/library/mistral-nemo'
  },
  'mistral-large': {
    parameters: '123B',
    contextLength: 128000,
    supportsTools: true,
    supportsVision: false,
    family: 'Mistral',
    provider: 'Mistral AI',
    description: 'Flagship model with advanced capabilities',
    source: 'ollama.com/library/mistral-large'
  },

  // CodeStral/Devstral 시리즈
  'codestral': {
    parameters: '22B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Codestral',
    provider: 'Mistral AI',
    description: 'Code generation and completion model',
    source: 'ollama.com/library/codestral'
  },
  'devstral': {
    parameters: '22B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'Codestral',
    provider: 'Mistral AI',
    description: 'Development-focused code model',
    source: 'ollama.com/library/codestral'
  },

  // Phi 시리즈
  'phi3': {
    parameters: '3.8B',
    contextLength: 128000,
    supportsTools: false,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'Small language model optimized for efficiency',
    source: 'ollama.com/library/phi3'
  },
  'phi3:mini': {
    parameters: '3.8B',
    contextLength: 128000,
    supportsTools: false,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'Compact model for resource-constrained environments',
    source: 'ollama.com/library/phi3'
  },
  'phi3:medium': {
    parameters: '14B',
    contextLength: 128000,
    supportsTools: false,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'Balanced model with good performance',
    source: 'ollama.com/library/phi3'
  },
  'phi4': {
    parameters: '14B',
    contextLength: 16384,
    supportsTools: true,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'State-of-the-art open model built upon synthetic datasets and academic resources',
    source: 'ollama.com/library/phi4'
  },
  'phi4:14b': {
    parameters: '14B',
    contextLength: 16384,
    supportsTools: true,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'State-of-the-art open model built upon synthetic datasets and academic resources',
    source: 'ollama.com/library/phi4'
  },
  'phi4:latest': {
    parameters: '14B',
    contextLength: 16384,
    supportsTools: true,
    supportsVision: false,
    family: 'Phi',
    provider: 'Microsoft',
    description: 'State-of-the-art open model built upon synthetic datasets and academic resources',
    source: 'ollama.com/library/phi4'
  },

  // Deepseek 시리즈
  'deepseek-coder': {
    parameters: '6.7B',
    contextLength: 16384,
    supportsTools: false,
    supportsVision: false,
    family: 'DeepSeek',
    provider: 'DeepSeek',
    description: 'Specialized code generation model',
    source: 'ollama.com/library/deepseek-coder'
  },
  'deepseek-r1': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'DeepSeek R1',
    provider: 'DeepSeek',
    description: 'Reasoning model with chain-of-thought capabilities',
    source: 'ollama.com/library/deepseek-r1'
  },
  'deepseek-r1:latest': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: true,
    supportsVision: false,
    family: 'DeepSeek R1',
    provider: 'DeepSeek',
    description: 'Reasoning model with chain-of-thought capabilities',
    source: 'ollama.com/library/deepseek-r1'
  },

  // Orca 시리즈
  'orca-mini': {
    parameters: '3B',
    contextLength: 2048,
    supportsTools: false,
    supportsVision: false,
    family: 'Orca',
    provider: 'Microsoft',
    description: 'Lightweight reasoning model',
    source: 'ollama.com/library/orca-mini'
  },

  // Neural Chat
  'neural-chat': {
    parameters: '7B',
    contextLength: 32768,
    supportsTools: false,
    supportsVision: false,
    family: 'Neural Chat',
    provider: 'Intel',
    description: 'Conversational AI model optimized for chat',
    source: 'ollama.com/library/neural-chat'
  },

  // Vicuna
  'vicuna': {
    parameters: '7B',
    contextLength: 2048,
    supportsTools: false,
    supportsVision: false,
    family: 'Vicuna',
    provider: 'LMSYS',
    description: 'Instruction-following model based on LLaMA',
    source: 'ollama.com/library/vicuna'
  },

  // Alpaca
  'alpaca': {
    parameters: '7B',
    contextLength: 2048,
    supportsTools: false,
    supportsVision: false,
    family: 'Alpaca',
    provider: 'Stanford',
    description: 'Instruction-following model for research',
    source: 'ollama.com/library/alpaca'
  }
};

// 모델명에서 메타데이터를 찾는 함수
export function getModelMetadata(modelName: string): ModelMetadata | null {
  // 사용자 정의 모델의 베이스 모델 매핑
  const customModelMapping: Record<string, string> = {
    'airun-chat:latest': 'devstral',
    'airun-think:latest': 'deepseek-r1:latest', 
    'airun-vision:latest': 'qwen2.5vl:latest'
  };

  // 사용자 정의 모델인 경우 베이스 모델의 메타데이터 사용
  if (customModelMapping[modelName]) {
    const baseModel = customModelMapping[modelName];
    const baseMetadata = getModelMetadata(baseModel);
    if (baseMetadata) {
      return {
        ...baseMetadata,
        family: `${baseMetadata.family} (Custom)`,
        description: `Custom fine-tuned model based on ${baseModel}`,
        source: 'custom-model'
      };
    }
  }

  // 정확한 매치 시도
  if (MODEL_METADATA[modelName]) {
    return MODEL_METADATA[modelName];
  }

  // 베이스 모델명으로 매치 시도 (태그 제거)
  const baseModelName = modelName.split(':')[0];
  if (MODEL_METADATA[baseModelName]) {
    return MODEL_METADATA[baseModelName];
  }

  // 패밀리별 패턴 매칭
  const lowerModelName = modelName.toLowerCase();
  
  // Gemma 패밀리
  if (lowerModelName.includes('gemma3')) {
    if (lowerModelName.includes('1b')) {
      return MODEL_METADATA['gemma3:1b'];
    } else if (lowerModelName.includes('4b')) {
      return MODEL_METADATA['gemma3:4b'];
    } else if (lowerModelName.includes('12b')) {
      return MODEL_METADATA['gemma3:12b'];
    } else if (lowerModelName.includes('27b')) {
      return MODEL_METADATA['gemma3:27b'];
    }
    return MODEL_METADATA['gemma3'];
  }

  // Llama 패밀리
  if (lowerModelName.includes('llama3.2')) {
    if (lowerModelName.includes('vision')) {
      return MODEL_METADATA['llama3.2-vision'];
    } else if (lowerModelName.includes('1b')) {
      return MODEL_METADATA['llama3.2:1b'];
    } else if (lowerModelName.includes('3b')) {
      return MODEL_METADATA['llama3.2:3b'];
    }
    return MODEL_METADATA['llama3.2'];
  }

  if (lowerModelName.includes('llama3.1')) {
    if (lowerModelName.includes('405b')) {
      return MODEL_METADATA['llama3.1:405b'];
    } else if (lowerModelName.includes('70b')) {
      return MODEL_METADATA['llama3.1:70b'];
    } else if (lowerModelName.includes('8b')) {
      return MODEL_METADATA['llama3.1:8b'];
    }
    return MODEL_METADATA['llama3.1'];
  }

  // Qwen 패밀리
  if (lowerModelName.includes('qwen2.5vl') || lowerModelName.includes('qwen-vl')) {
    return MODEL_METADATA['qwen2.5vl'];
  }
  
  if (lowerModelName.includes('qwen2.5') || lowerModelName.includes('qwen3')) {
    if (lowerModelName.includes('0.5b')) {
      return MODEL_METADATA['qwen2.5:0.5b'];
    } else if (lowerModelName.includes('1.5b')) {
      return MODEL_METADATA['qwen2.5:1.5b'];
    } else if (lowerModelName.includes('3b')) {
      return MODEL_METADATA['qwen2.5:3b'];
    } else if (lowerModelName.includes('7b')) {
      return MODEL_METADATA['qwen2.5:7b'];
    } else if (lowerModelName.includes('14b')) {
      return MODEL_METADATA['qwen2.5:14b'];
    } else if (lowerModelName.includes('32b')) {
      return MODEL_METADATA['qwen2.5:32b'];
    } else if (lowerModelName.includes('72b')) {
      return MODEL_METADATA['qwen2.5:72b'];
    }
    return MODEL_METADATA['qwen2.5'];
  }

  // DeepSeek 패밀리
  if (lowerModelName.includes('deepseek')) {
    if (lowerModelName.includes('r1')) {
      return MODEL_METADATA['deepseek-r1'];
    } else if (lowerModelName.includes('coder')) {
      return MODEL_METADATA['deepseek-coder'];
    }
    return MODEL_METADATA['deepseek-coder'];
  }

  // Mistral 패밀리
  if (lowerModelName.includes('mistral') || lowerModelName.includes('codestral') || lowerModelName.includes('devstral')) {
    if (lowerModelName.includes('nemo')) {
      return MODEL_METADATA['mistral-nemo'];
    } else if (lowerModelName.includes('large')) {
      return MODEL_METADATA['mistral-large'];
    } else if (lowerModelName.includes('codestral') || lowerModelName.includes('devstral')) {
      return MODEL_METADATA['devstral'];
    }
    return MODEL_METADATA['mistral'];
  }

  // 다른 패밀리들도 유사하게 처리...
  
  return null;
}

// 평균 응답 시간 계산 함수
export function calculateAverageResponseTime(results: any): number {
  const categories = ['math', 'writing', 'coding', 'reasoning', 'grammar'];
  let totalTime = 0;
  let count = 0;

  categories.forEach(category => {
    if (results[category]?.response_time_ms) {
      totalTime += results[category].response_time_ms;
      count++;
    }
  });

  return count > 0 ? Math.round(totalTime / count) : 0;
}

// Ollama API를 통해 동적으로 메타데이터 가져오기
export async function getModelMetadataFromOllama(modelName: string): Promise<ModelMetadata | null> {
  try {
    const response = await fetch(`/api/ollama-metadata?model=${encodeURIComponent(modelName)}`);
    if (!response.ok) {
      return null;
    }
    
    const data = await response.json();
    
    // 베이스 모델이 있으면 해당 메타데이터 사용
    if (data.baseModel) {
      const baseMetadata = getModelMetadata(data.baseModel);
      if (baseMetadata) {
        return {
          ...baseMetadata,
          description: `Custom model based on ${data.baseModel}`,
          source: 'ollama-api'
        };
      }
    }
    
    // 직접 추출한 정보로 메타데이터 생성
    return {
      parameters: data.parameters || 'Unknown',
      contextLength: data.contextLength || 2048,
      supportsTools: false, // 기본값
      supportsVision: false, // 기본값
      family: data.family || 'Unknown',
      provider: 'Custom',
      description: 'Custom model from Ollama',
      source: 'ollama-api'
    };
  } catch (error) {
    console.error('Error fetching Ollama metadata:', error);
    return null;
  }
}

// 확장된 메타데이터 조회 함수 (Ollama API 포함)
export async function getModelMetadataExtended(modelName: string): Promise<ModelMetadata | null> {
  // 먼저 정적 메타데이터에서 찾기
  const staticMetadata = getModelMetadata(modelName);
  if (staticMetadata) {
    return staticMetadata;
  }
  
  // 정적 메타데이터에 없으면 Ollama API에서 동적으로 가져오기
  return await getModelMetadataFromOllama(modelName);
} 