/**
 * A2A (Agent-to-Agent) Protocol Adapter
 * Bridges A2A protocol with existing MCP infrastructure
 */

interface A2AAgent {
  id: string;
  name: string;
  capabilities: string[];
  endpoint: string;
  protocol: 'websocket' | 'http' | 'stdio';
  trustLevel: 'public' | 'trusted' | 'private';
}

interface A2AMessage {
  id: string;
  from: string;
  to: string;
  type: 'request' | 'response' | 'notification';
  method?: string;
  params?: any;
  result?: any;
  error?: any;
  timestamp: Date;
}

interface A2ACapability {
  name: string;
  description: string;
  inputSchema: any;
  outputSchema: any;
  authRequired: boolean;
}

class A2AAdapter {
  private registeredAgents: Map<string, A2AAgent> = new Map();
  private capabilities: Map<string, A2ACapability> = new Map();
  private mcpClient: any; // MCP 클라이언트 참조
  private messageHandlers: Map<string, Function> = new Map();

  constructor(mcpClient: any) {
    this.mcpClient = mcpClient;
    this.setupDefaultHandlers();
  }

  /**
   * MCP 도구들을 A2A 능력으로 노출
   */
  async exposeToolsToA2A(): Promise<A2ACapability[]> {
    const mcpTools = await this.mcpClient.getTools();
    const a2aCapabilities: A2ACapability[] = [];

    for (const tool of mcpTools) {
      const capability: A2ACapability = {
        name: `mcp.${tool.name}`,
        description: tool.description,
        inputSchema: tool.inputSchema,
        outputSchema: { type: 'object', properties: { result: { type: 'string' } } },
        authRequired: tool.authRequired || false
      };

      this.capabilities.set(capability.name, capability);
      a2aCapabilities.push(capability);
    }

    return a2aCapabilities;
  }

  /**
   * A2A 에이전트를 네트워크에 등록
   */
  async registerAgent(agent: A2AAgent): Promise<boolean> {
    try {
      // 에이전트 검증
      const isValid = await this.validateAgent(agent);
      if (!isValid) {
        throw new Error(`Agent validation failed: ${agent.id}`);
      }

      // 에이전트 등록
      this.registeredAgents.set(agent.id, agent);
      
      console.log(`A2A Agent registered: ${agent.name} (${agent.id})`);
      return true;
    } catch (error) {
      console.error(`Failed to register A2A agent:`, error);
      return false;
    }
  }

  /**
   * Ollama 모델들을 A2A 에이전트로 등록
   */
  async registerOllamaAsA2AAgent(ollamaConfig: any): Promise<boolean> {
    try {
      const agent: A2AAgent = {
        id: `ollama-${ollamaConfig.model}`,
        name: `Ollama ${ollamaConfig.model}`,
        capabilities: [
          'text-generation',
          'conversation',
          'tool-calling',
          ...(ollamaConfig.model.includes('vision') ? ['image-analysis'] : [])
        ],
        endpoint: ollamaConfig.endpoint || 'http://localhost:11434',
        protocol: 'http',
        trustLevel: 'private' // 로컬 모델이므로 높은 신뢰도
      };

      return await this.registerAgent(agent);
    } catch (error) {
      console.error('Failed to register Ollama as A2A agent:', error);
      return false;
    }
  }

  /**
   * A2A 메시지 처리
   */
  async handleA2AMessage(message: A2AMessage): Promise<A2AMessage | null> {
    try {
      switch (message.type) {
        case 'request':
          return await this.handleRequest(message);
        case 'notification':
          await this.handleNotification(message);
          return null;
        default:
          throw new Error(`Unsupported message type: ${message.type}`);
      }
    } catch (error) {
      return this.createErrorResponse(message, error);
    }
  }

  /**
   * A2A 요청을 MCP 도구 호출로 변환
   */
  private async handleRequest(message: A2AMessage): Promise<A2AMessage> {
    const { method, params } = message;
    
    if (!method) {
      throw new Error('Method is required for A2A requests');
    }

    // MCP 도구 호출인지 확인
    if (method.startsWith('mcp.')) {
      const toolName = method.substring(4); // 'mcp.' 제거
      const result = await this.mcpClient.callTool(toolName, params);
      
      return {
        id: message.id,
        from: 'airun-mcp-agent',
        to: message.from,
        type: 'response',
        result,
        timestamp: new Date()
      };
    }

    // 다른 A2A 에이전트로 라우팅
    const targetAgent = this.registeredAgents.get(message.to);
    if (targetAgent) {
      return await this.forwardToAgent(targetAgent, message);
    }

    throw new Error(`Unknown method or target: ${method}`);
  }

  /**
   * 외부 A2A 에이전트에게 메시지 전달
   */
  private async forwardToAgent(agent: A2AAgent, message: A2AMessage): Promise<A2AMessage> {
    switch (agent.protocol) {
      case 'http':
        return await this.forwardViaHTTP(agent, message);
      case 'websocket':
        return await this.forwardViaWebSocket(agent, message);
      default:
        throw new Error(`Unsupported protocol: ${agent.protocol}`);
    }
  }

  /**
   * HTTP를 통한 A2A 메시지 전달
   */
  private async forwardViaHTTP(agent: A2AAgent, message: A2AMessage): Promise<A2AMessage> {
    const response = await fetch(`${agent.endpoint}/a2a/message`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'A2A-Protocol': '1.0',
        'A2A-Agent-Id': 'airun-mcp-agent'
      },
      body: JSON.stringify(message)
    });

    if (!response.ok) {
      throw new Error(`HTTP request failed: ${response.status}`);
    }

    return await response.json();
  }

  /**
   * WebSocket을 통한 A2A 메시지 전달
   */
  private async forwardViaWebSocket(agent: A2AAgent, message: A2AMessage): Promise<A2AMessage> {
    // WebSocket 연결 관리 로직
    return new Promise((resolve, reject) => {
      const ws = new WebSocket(agent.endpoint);
      
      ws.onopen = () => {
        ws.send(JSON.stringify(message));
      };
      
      ws.onmessage = (event) => {
        const response = JSON.parse(event.data);
        ws.close();
        resolve(response);
      };
      
      ws.onerror = (error) => {
        reject(error);
      };
    });
  }

  /**
   * A2A 에이전트 검증
   */
  private async validateAgent(agent: A2AAgent): Promise<boolean> {
    try {
      // 기본 필드 검증
      if (!agent.id || !agent.name || !agent.endpoint) {
        return false;
      }

      // 엔드포인트 연결 테스트
      const pingResponse = await fetch(`${agent.endpoint}/a2a/ping`, {
        method: 'GET',
        headers: { 'A2A-Protocol': '1.0' }
      });

      return pingResponse.ok;
    } catch (error) {
      console.warn(`Agent validation failed for ${agent.id}:`, error);
      return false;
    }
  }

  /**
   * 오류 응답 생성
   */
  private createErrorResponse(originalMessage: A2AMessage, error: any): A2AMessage {
    return {
      id: originalMessage.id,
      from: 'airun-mcp-agent',
      to: originalMessage.from,
      type: 'response',
      error: {
        code: -32603,
        message: error.message || 'Internal error'
      },
      timestamp: new Date()
    };
  }

  /**
   * 기본 메시지 핸들러 설정
   */
  private setupDefaultHandlers(): void {
    this.messageHandlers.set('capabilities.list', async () => {
      return Array.from(this.capabilities.values());
    });

    this.messageHandlers.set('agents.list', async () => {
      return Array.from(this.registeredAgents.values());
    });

    this.messageHandlers.set('ping', async () => {
      return { status: 'ok', timestamp: new Date() };
    });
  }

  /**
   * 알림 메시지 처리
   */
  private async handleNotification(message: A2AMessage): Promise<void> {
    console.log('A2A Notification received:', message);
    // 알림 처리 로직 구현
  }

  /**
   * A2A 네트워크 상태 조회
   */
  async getNetworkStatus(): Promise<any> {
    return {
      registeredAgents: this.registeredAgents.size,
      capabilities: this.capabilities.size,
      networkHealth: 'healthy', // 실제 상태 체크 로직 필요
      lastUpdate: new Date()
    };
  }
}

export { A2AAdapter };
export type { A2AAgent, A2AMessage, A2ACapability }; 