import { spawn, ChildProcess } from 'child_process';
import * as fs from 'fs';
import * as path from 'path';

// MCP 설정 로드 함수
const loadMCPConfig = () => {
  try {
    const configPath = path.join(process.cwd(), 'mcp_config.json');
    
    if (fs.existsSync(configPath)) {
      const configData = fs.readFileSync(configPath, 'utf8');
      return JSON.parse(configData);
    } else {
      // 설정 파일이 없으면 빈 설정 반환
      return {};
    }
  } catch (error) {
    console.error('MCP 설정 로드 실패:', error);
    return {};
  }
};

class ImprovedMCPClient {
  private serverName: string;
  private config: any;
  private childProcess?: ChildProcess;
  private isInitialized = false;
  private messageId = 1;

  constructor(serverName: string, config: any) {
    this.serverName = serverName;
    this.config = config;
  }

  async initialize(): Promise<boolean> {
    const { command, args, transport, env } = this.config;
    
    try {
      console.log(`🔄 ${this.serverName} MCP 서버 초기화 시작...`);
      
      const spawnEnv = { ...process.env };
      if (env && typeof env === 'object') {
        Object.assign(spawnEnv, env);
      }

      this.childProcess = spawn(command, args, {
        stdio: ['pipe', 'pipe', 'pipe'],
        cwd: process.cwd(),
        env: spawnEnv
      });

      this.childProcess.stderr?.on('data', (data: Buffer) => {
        console.error(`${this.serverName} stderr:`, data.toString());
      });

      // 60초 타임아웃으로 초기화
      const initResult = await Promise.race([
        this.sendRequest('initialize', {
          protocolVersion: '2024-11-05',
          capabilities: {
            roots: { listChanged: true },
            sampling: {}
          },
          clientInfo: {
            name: 'airun-global-mcp-client',
            version: '1.0.0'
          }
        }),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('초기화 시간 초과')), 60000)
        )
      ]);

      this.isInitialized = true;
      console.log(`✅ ${this.serverName} MCP 서버 초기화 완료`);
      return true;
    } catch (error) {
      console.error(`❌ ${this.serverName} MCP 서버 초기화 실패:`, error);
      this.cleanup();
      return false;
    }
  }

  async listTools(): Promise<any[]> {
    if (!this.isInitialized) {
      throw new Error(`${this.serverName} 서버가 초기화되지 않았습니다`);
    }

    try {
      const result = await this.sendRequest('tools/list');
      return result.tools || [];
    } catch (error) {
      console.error(`${this.serverName} 도구 목록 가져오기 실패:`, error);
      return [];
    }
  }

  async callTool(toolName: string, args: any): Promise<any> {
    if (!this.isInitialized) {
      throw new Error(`${this.serverName} 서버가 초기화되지 않았습니다`);
    }

    try {
      return await this.sendRequest('tools/call', { name: toolName, arguments: args });
    } catch (error) {
      console.error(`${this.serverName} 도구 호출 실패:`, error);
      throw error;
    }
  }

  private async sendRequest(method: string, params: any = {}): Promise<any> {
    if (!this.childProcess || !this.childProcess.stdin) {
      throw new Error('MCP 프로세스가 시작되지 않았습니다');
    }

    const request = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method,
      params
    };

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error(`${method} 요청 시간 초과`));
      }, 30000);

      const onData = (data: Buffer) => {
        try {
          const lines = data.toString().split('\n').filter(line => line.trim());
          for (const line of lines) {
            const response = JSON.parse(line);
            if (response.id === request.id) {
              clearTimeout(timeout);
              this.childProcess?.stdout?.off('data', onData);
              
              if (response.error) {
                reject(new Error(response.error.message || 'MCP 서버 오류'));
              } else {
                resolve(response.result);
              }
              return;
            }
          }
        } catch (parseError) {
          // JSON 파싱 실패는 무시 (부분 데이터일 수 있음)
        }
      };

      this.childProcess!.stdout?.on('data', onData);
      this.childProcess!.stdin?.write(JSON.stringify(request) + '\n');
    });
  }

  cleanup() {
    if (this.childProcess) {
      this.childProcess.kill();
      this.childProcess = undefined;
    }
    this.isInitialized = false;
  }
}

class GlobalMCPToolsClient {
  private clients: Map<string, ImprovedMCPClient> = new Map();
  private isInitialized = false;

  async initialize(mcpConfig?: any): Promise<boolean> {
    if (this.isInitialized) {
      console.log('✅ MCP 클라이언트 이미 초기화됨');
      return true;
    }

    const config = mcpConfig || loadMCPConfig();
    console.log('🔄 전역 MCP 클라이언트 초기화 시작...');
    
    const initPromises = [];
    
    for (const [serverName, serverConfig] of Object.entries(config)) {
      const initPromise = this.initializeServer(serverName, serverConfig as any);
      initPromises.push(initPromise);
    }
    
    try {
      await Promise.race([
        Promise.allSettled(initPromises),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('전체 초기화 시간 초과')), 120000) // 2분 타임아웃
        )
      ]);
      
      // 초기화 완료를 위한 대기 시간
      await new Promise(resolve => setTimeout(resolve, 3000));
      
      console.log(`✅ 전역 MCP 초기화 완료. 활성 서버 수: ${this.clients.size}`);
    } catch (error) {
      console.error('❌ 전역 MCP 서버 초기화 중 전체 오류:', error);
    }
    
    // 일부 서버라도 성공했으면 초기화 완료로 간주
    const hasActiveServers = this.clients.size > 0;
    if (hasActiveServers) {
      this.isInitialized = true;
      console.log(`✅ 전역 MCP 클라이언트 초기화 완료: ${this.clients.size}개 서버 활성`);
    } else {
      console.warn('⚠️ 전역 MCP 클라이언트 초기화: 활성 서버 없음');
    }
    
    return hasActiveServers;
  }

  private async initializeServer(serverName: string, config: any): Promise<void> {
    try {
      const client = new ImprovedMCPClient(serverName, config);
      const success = await client.initialize();
      
      if (success) {
        this.clients.set(serverName, client);
        console.log(`✅ ${serverName} 서버 초기화 성공`);
      } else {
        console.log(`❌ ${serverName} 서버 초기화 실패`);
      }
    } catch (error) {
      console.error(`❌ ${serverName} 서버 초기화 중 오류:`, error);
    }
  }

  async getAllTools(): Promise<any[]> {
    if (!this.isInitialized) {
      throw new Error('MCP 클라이언트가 초기화되지 않았습니다');
    }

    const allTools: any[] = [];
    const duplicateCheck = new Set<string>();

    for (const [serverName, client] of this.clients.entries()) {
      try {
        const tools = await client.listTools();
        
        for (const tool of tools) {
          const toolName = tool.name;
          
          if (duplicateCheck.has(toolName)) {
            console.warn(`⚠️ 도구 이름 중복: ${toolName} (${serverName}에서 발견, 첫 번째 도구만 사용)`);
            continue;
          }
          
          duplicateCheck.add(toolName);
          
          // 도구에 서버 정보 추가
          const enhancedTool = {
            ...tool,
            server: serverName,
            invoke: async (args: any) => {
              return await client.callTool(toolName, args);
            }
          };
          
          allTools.push(enhancedTool);
        }
      } catch (error) {
        console.error(`${serverName} 도구 목록 가져오기 실패:`, error);
      }
    }

    console.log(`✅ 중복 제거 완료: ${allTools.length}개의 고유 도구`);
    return allTools;
  }

  cleanup() {
    for (const client of this.clients.values()) {
      client.cleanup();
    }
    this.clients.clear();
    this.isInitialized = false;
  }

  getInitializationStatus(): boolean {
    return this.isInitialized;
  }

  getServerCount(): number {
    return this.clients.size;
  }
}

// 전역 싱글톤 인스턴스
let globalMCPClient: GlobalMCPToolsClient | null = null;
let isInitializing = false;

// 전역 MCP 클라이언트 가져오기 (싱글톤)
export const getGlobalMCPClient = async (): Promise<GlobalMCPToolsClient | null> => {
  // 이미 초기화된 경우 반환
  if (globalMCPClient && globalMCPClient.getInitializationStatus()) {
    console.log('🔄 기존 전역 MCP 클라이언트 재사용');
    return globalMCPClient;
  }

  // 초기화 중인 경우 대기
  if (isInitializing) {
    console.log('⏳ 전역 MCP 클라이언트 초기화 대기 중...');
    while (isInitializing) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
    console.log('✅ 전역 MCP 클라이언트 초기화 대기 완료');
    return globalMCPClient;
  }

  // 새로 초기화
  console.log('🆕 전역 MCP 클라이언트 생성 중...');
  isInitializing = true;
  try {
    globalMCPClient = new GlobalMCPToolsClient();
    const success = await globalMCPClient.initialize();
    
    if (!success) {
      console.error('❌ 전역 MCP 클라이언트 초기화 실패');
      globalMCPClient = null;
    } else {
      console.log('✅ 새 전역 MCP 클라이언트 생성 완료');
    }
    
    return globalMCPClient;
  } finally {
    isInitializing = false;
  }
};

// MCP 도구 로드 (모든 API에서 공통 사용)
export const loadMCPTools = async (): Promise<any[]> => {
  console.log('🔧 전역 MCP 도구 로딩 시작...');
  console.log(`🔍 현재 전역 클라이언트 상태: ${globalMCPClient ? '존재' : '없음'}`);
  console.log(`🔍 초기화 진행 중: ${isInitializing ? '예' : '아니오'}`);
  
  if (globalMCPClient) {
    console.log(`🔍 기존 클라이언트 초기화 상태: ${globalMCPClient.getInitializationStatus() ? '완료' : '미완료'}`);
    console.log(`🔍 기존 클라이언트 서버 수: ${globalMCPClient.getServerCount()}개`);
  }
  
  try {
    const client = await getGlobalMCPClient();
    
    if (!client) {
      console.error('❌ 전역 MCP 클라이언트를 가져올 수 없습니다');
      return [];
    }
    
    console.log(`✅ 전역 MCP 클라이언트 확보 완료 (서버 수: ${client.getServerCount()}개)`);
    
    const tools = await client.getAllTools();
    console.log(`✅ 전역 MCP 도구 로드 완료: ${tools.length}개`);
    return tools;
  } catch (error) {
    console.error('❌ 전역 MCP 도구 로드 실패:', error);
    return [];
  }
};

// 정리 함수
export const cleanupGlobalMCPClient = () => {
  if (globalMCPClient) {
    globalMCPClient.cleanup();
    globalMCPClient = null;
  }
}; 