import { NextRequest, NextResponse } from 'next/server';

// 인증이 필요하지 않은 경로
const publicPaths = [
  '/',
  '/login',
  '/support',
  '/api/auth/login',
  '/api/auth/logout',
  '/favicon.ico',
  '/_next',
  '/images',
  '/lang',
];

export function middleware(request: NextRequest) {
  // 현재 경로
  const { pathname } = request.nextUrl;
  
  // 공개 경로는 통과
  if (publicPaths.some(path => pathname.startsWith(path))) {
    return NextResponse.next();
  }
  
  // 인증 상태 확인
  const authCookie = request.cookies.get('auth_token');
  const isAuthenticated = !!authCookie?.value;
  
  // 로그인 상태에서 로그인 페이지에 접근하면 홈으로 리디렉션
  if (isAuthenticated && pathname === '/login') {
    return NextResponse.redirect(new URL('/', request.url));
  }
  
  // 인증되지 않은 상태에서 보호된 경로에 접근하면 로그인 페이지로 리디렉션
  if (!isAuthenticated && pathname !== '/login') {
    const loginUrl = new URL('/login', request.url);
    // 로그인 후 원래 페이지로 돌아갈 수 있도록 redirect_url 추가
    loginUrl.searchParams.set('redirect', pathname);
    return NextResponse.redirect(loginUrl);
  }
  
  return NextResponse.next();
}

export const config = {
  matcher: [
    // 모든 경로에 적용 (API 경로, 정적 파일, 이미지는 제외)
    '/((?!api|_next/static|_next/image|favicon.ico|images).*)',
  ],
}; 