import React from 'react';

// 통일된 환경변수 사용
export const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || '/api/v1';

// WebSocket 서버 URL 설정
export const getWebSocketUrl = () => {
  // 환경변수에서 WebSocket URL이 직접 설정된 경우 사용
  if (process.env.NEXT_PUBLIC_WS_SERVER_URL) {
    console.log('[WebSocket] 환경변수에서 직접 설정된 URL 사용:', process.env.NEXT_PUBLIC_WS_SERVER_URL);
    return process.env.NEXT_PUBLIC_WS_SERVER_URL;
  }
  
  // 현재 페이지의 프로토콜 확인
  const isPageSecure = typeof window !== 'undefined' && window.location.protocol === 'https:';
  const currentHost = typeof window !== 'undefined' ? window.location.host : 'localhost:3000';
  
  // API 서버 URL을 기반으로 WebSocket URL 생성
  const apiUrl = process.env.NEXT_PUBLIC_API_SERVER_URL;
  if (apiUrl) {
    // URL 파싱하여 호스트와 포트 추출
    let wsHost;
    let wsProtocol;
    
    try {
      const url = new URL(apiUrl);
      wsHost = url.host;
      
      // 🚀 개선된 프로토콜 판단 로직
      if (isPageSecure) {
        // HTTPS 페이지에서는 Mixed Content 정책에 따라 반드시 WSS 사용
        wsProtocol = 'wss:';
        console.log('[WebSocket] HTTPS 페이지 감지 - WSS 프로토콜 강제 사용');
      } else if (wsHost.includes('localhost') || wsHost.startsWith('127.0.0.1') || wsHost.startsWith('192.168.') || wsHost.startsWith('10.') || wsHost.startsWith('172.')) {
        // localhost 또는 내부 IP인 경우 WS 사용
        wsProtocol = 'ws:';
        console.log('[WebSocket] 로컬/내부 네트워크 감지 - WS 프로토콜 사용');
      } else {
        // 외부 도메인인 경우 원본 API URL의 프로토콜에 따라 결정
        wsProtocol = url.protocol === 'https:' ? 'wss:' : 'ws:';
        console.log('[WebSocket] 외부 도메인 감지 - API URL 프로토콜 기반:', wsProtocol);
      }
      
      const finalUrl = `${wsProtocol}//${wsHost}/wss`;
      console.log('[WebSocket] 최종 WebSocket URL 생성:', finalUrl);
      return finalUrl;
      
    } catch (error) {
      console.warn('[WebSocket] API URL 파싱 실패, fallback 로직 사용:', error);
    }
  }
  
  // fallback: 현재 호스트 사용
  const wsProtocol = isPageSecure ? 'wss:' : 'ws:';
  const fallbackUrl = `${wsProtocol}//${currentHost}/wss`;
  
  console.log('[WebSocket] Fallback URL 사용:', {
    isPageSecure,
    currentHost,
    wsProtocol,
    finalUrl: fallbackUrl
  });
  
  return fallbackUrl;
};

// API 응답 타입 정의
export interface ApiResponse<T = any> {
  success: boolean;
  data?: T;
  error?: {
    code: string;
    message: string;
  };
  timestamp: number;
}

// 인증 정보 캐싱 - 반복적인 호출 방지
let cachedHeaders: Record<string, string> | null = null;
let lastCacheTime = 0;
const CACHE_DURATION = 60000; // 60초 캐시 유효 시간

// 인증 헤더 가져오기
export const getAuthHeaders = (): Record<string, string> => {
  const now = Date.now();
  
  // 캐시된 헤더가 있고 유효 시간 내라면 캐시된 값 반환
  if (cachedHeaders && (now - lastCacheTime < CACHE_DURATION)) {
    return cachedHeaders;
  }
  
  const headers: Record<string, string> = {
    'Content-Type': 'application/json',
  };
  
  if (typeof window !== 'undefined') {
    try {
      const token = localStorage.getItem('userToken');
      const apiKey = localStorage.getItem('apiKey');
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      if (apiKey) {
        headers['X-API-Key'] = apiKey;
      }
      
      // 결과 캐싱
      cachedHeaders = headers;
      lastCacheTime = now;
    } catch (error) {
      console.error('인증 정보 가져오기 오류:', error);
    }
  }
  
  return headers;
};

// 로컬 스토리지 변경 감지 및 캐시 무효화
if (typeof window !== 'undefined') {
  window.addEventListener('storage', (event) => {
    if (event.key === 'userToken' || event.key === 'apiKey') {
      cachedHeaders = null; // 캐시 무효화
    }
  });
}

// API 요청 함수
export const apiRequest = async <T>(
  endpoint: string,
  options: RequestInit = {}
): Promise<ApiResponse<T>> => {
  const url = `${API_SERVER}${endpoint}`;
  const headers = {
    ...await getAuthHeaders(),
    ...options.headers,
  };
  
  const response = await fetch(url, {
    ...options,
    headers,
  });
  
  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`API 요청 실패 (${response.status}): ${errorText}`);
  }
  
  return response.json();
};

// 파일 업로드 전용 함수
export const uploadFile = async (
  endpoint: string,
  file: File,
  userId?: string
): Promise<ApiResponse> => {
  const url = `${API_SERVER}${endpoint}`;
  const formData = new FormData();
  formData.append('file', file);
  
  // 사용자 ID는 API 라우트에서 자동으로 추가되므로 여기서는 추가하지 않음
  // API 라우트의 withAnyAuth 미들웨어에서 사용자 정보를 가져와서 처리함
  
  // FormData를 사용할 때는 Content-Type 헤더를 설정하지 않아야 함
  // 브라우저가 자동으로 multipart/form-data와 boundary를 설정함
  const authHeaders = await getAuthHeaders();
  const headers = {
    'Authorization': authHeaders['Authorization'] || '',
    'X-API-Key': authHeaders['X-API-Key'] || '',
  };
  
  const response = await fetch(url, {
    method: 'POST',
    headers,
    body: formData,
  });
  
  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`파일 업로드 실패 (${response.status}): ${errorText}`);
  }
  
  return response.json();
};

// 스트리밍 API 요청 함수
export const streamRequest = async <T>(
  endpoint: string,
  options: RequestInit = {},
  onChunk: (chunk: string) => void,
  onDone: (completeResponse: string) => void,
  onError: (error: Error) => void
): Promise<void> => {
  const url = `${API_SERVER}${endpoint}`;
  const headers = {
    ...await getAuthHeaders(),
    ...options.headers,
  };
  
  try {
    const response = await fetch(url, {
      ...options,
      headers,
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`API 요청 실패 (${response.status}): ${errorText}`);
    }
    
    if (!response.body) {
      throw new Error('스트리밍 응답을 받을 수 없습니다');
    }
    
    // ReadableStream 처리
    const reader = response.body.getReader();
    const decoder = new TextDecoder('utf-8');
    let completeResponse = '';
    
    while (true) {
      const { value, done } = await reader.read();
      
      if (done) {
        onDone(completeResponse);
        break;
      }
      
      // 청크 디코딩 및 콜백 호출
      const chunk = decoder.decode(value, { stream: true });
      completeResponse += chunk;
      onChunk(chunk);
    }
  } catch (error) {
    if (error instanceof Error) {
      onError(error);
    } else {
      onError(new Error('알 수 없는 오류가 발생했습니다'));
    }
  }
};

// 웹소켓을 통한 스트리밍 관련 타입
export interface StreamCallbacks {
  onConnect: () => void;
  onContent: (content: string) => void;
  onThinking?: (thinking: string) => void;  // thinking 내용 콜백 추가
  onProgress?: (message: string, percent?: number, stage?: string) => void;
  onComplete: (response: any) => void;
  onError: (error: Error) => void;
  onDisconnect: () => void;
}

// 웹소켓 인스턴스 관리
let websocket: WebSocket | null = null;
let wsConnected = false;

interface StreamRequestOptions {
  onConnect?: () => void;
  onContent?: (content: string) => void;
  onComplete?: (data: any) => void;
  onError?: (error: Error) => void;
  onDisconnect?: () => void;
}

// 웹소켓 스트리밍 요청 생성 함수
export const createStreamingRequest = async (
  endpoint: string,
  callbacks: StreamCallbacks
): Promise<WebSocket> => {
  try {
    const headers = await getAuthHeaders();
    const apiKey = headers['X-API-Key'];
    
    // console.log('API 키 확인:', apiKey ? '설정됨' : '설정되지 않음');
    
    if (!apiKey) {
      throw new Error('인증 정보가 없습니다. API 키가 필요합니다.');
    }

    // WebSocket URL 구성 (API 키를 URL 파라미터로 전달)
    const wsBaseUrl = getWebSocketUrl();
    const wsUrl = new URL(`${wsBaseUrl}${endpoint}`);
    wsUrl.searchParams.append('apiKey', apiKey);
    
    // console.log('WebSocket 연결 시도:', wsUrl.toString());
    
    // WebSocket 프로토콜 설정 제거 (선택적으로 사용)
    const ws = new WebSocket(wsUrl.toString());
    
    // 연결 타임아웃 설정
    const connectionTimeout = setTimeout(() => {
      if (ws.readyState !== WebSocket.OPEN) {
        ws.close();
        callbacks.onError(new Error('서버 연결 시간이 초과되었습니다.'));
      }
    }, 5000);

    // 연결 준비 완료 대기
    await new Promise<void>((resolve, reject) => {
      let isConnected = false;

      ws.onopen = () => {
        clearTimeout(connectionTimeout);
        // console.log('WebSocket 연결 성공');
        isConnected = true;
        
        if (callbacks.onConnect) {
          callbacks.onConnect();
        }
        
        resolve();
      };
      
      ws.onerror = (error) => {
        clearTimeout(connectionTimeout);
        const errorMessage = isConnected ? 
          '서버와의 통신 중 오류가 발생했습니다.' : 
          '서버에 연결할 수 없습니다. 서버가 실행 중인지 확인해주세요.';
        
        console.error('WebSocket 오류:', error);
        reject(new Error(errorMessage));
      };
    });
    
    ws.onmessage = async (event) => {
      try {
        const response = JSON.parse(event.data);
        // console.log('WebSocket 메시지 수신:', response);
        
        switch (response.type) {
          case 'connection_established':
            // console.log('서버 연결 확인:', response.message);
            break;
          case 'stream_start':
            // console.log('스트리밍 시작:', response.streamId);
            break;
          case 'stream_content':
            // console.log('스트리밍 컨텐츠 수신:', response.content);
            if (callbacks.onContent) {
              callbacks.onContent(response.content);
            }
            break;
          case 'api_response':
            // console.log('API 응답 수신:', response);
            if (callbacks.onContent) {
              // 전체 응답을 문자열로 전달
              callbacks.onContent(JSON.stringify(response));
            }
            if (callbacks.onComplete) {
              callbacks.onComplete(response);
            }
            break;
          case 'stream_end':
            // console.log('스트리밍 완료:', response.streamId);
            if (callbacks.onComplete) {
              // response.response가 객체일 수 있으므로 안전하게 처리
              const responseData = response.response;
              if (responseData && typeof responseData === 'object') {
                // responseData가 객체인 경우 response 속성을 우선적으로 사용
                const finalResponse = responseData.response || responseData.content || JSON.stringify(responseData);
                callbacks.onComplete(finalResponse);
              } else {
                // responseData가 문자열이면 그대로 사용
                callbacks.onComplete(responseData || '');
              }
            }
            break;
          case 'stream_error':
            console.error('스트리밍 오류:', response.error);
            if (callbacks.onError) {
              callbacks.onError(new Error(response.error));
            }
            break;
          case 'error':
            console.error('WebSocket 에러 응답:', response.error);
            if (callbacks.onError) {
              callbacks.onError(new Error(response.error));
            }
            break;
          case 'pong':
            // console.log('서버로부터 pong 수신');
            break;
        }
      } catch (error) {
        console.error('WebSocket 메시지 처리 중 오류:', error);
        if (callbacks.onError) {
          callbacks.onError(error instanceof Error ? error : new Error('메시지 처리 중 오류가 발생했습니다'));
        }
      }
    };
    
    ws.onerror = (error) => {
      clearTimeout(connectionTimeout);
      console.error('WebSocket 연결 오류:', error);
      if (callbacks.onError) {
        callbacks.onError(new Error('서버와의 연결이 실패했습니다. 잠시 후 다시 시도해주세요.'));
      }
    };
    
    ws.onclose = () => {
      clearTimeout(connectionTimeout);
      console.log('WebSocket 연결 종료');
      if (callbacks.onDisconnect) {
        callbacks.onDisconnect();
      }
    };
    
    return ws;
  } catch (error) {
    console.error('WebSocket 연결 생성 중 오류:', error);
    throw error;
  }
};

// 웹소켓을 통한 스트리밍 채팅 요청 함수 (기존 방식)
export const streamChatRequest = async (
  prompt: string,
  options: any = {},
  callbacks: StreamCallbacks,
  userId?: string
): Promise<void> => {
  try {
    const ws = await createStreamingRequest('/api/v1/chat/stream', callbacks);
    
    // 채팅 요청 전송
    const streamRequest = {
      type: 'api_request',
      endpoint: '/api/v1/chat',
      data: {
        prompt,
        options: {
          ...options,
          sessionId: options.sessionId || 'chat-' + Date.now(),
          temperature: 0.7,
          userId: userId // 사용자 ID 추가
        }
      }
    };

    // console.log('스트리밍 요청 전송:', streamRequest);
    ws.send(JSON.stringify(streamRequest));
  } catch (error) {
    console.error('채팅 요청 중 오류:', error);
    if (callbacks.onError) {
      callbacks.onError(error instanceof Error ? error : new Error('채팅 요청 중 오류가 발생했습니다'));
    }
  }
};

// 웹소켓을 통한 스트리밍 채팅 요청 함수 (api_server.js의 streamChat 사용)
export const streamChatWithWebSocket = async (
  prompt: string,
  options: any = {},
  callbacks: StreamCallbacks,
  userId?: string,
  wsRef?: React.MutableRefObject<WebSocket | null>
): Promise<void> => {
  try {
    const headers = await getAuthHeaders();
    const apiKey = headers['X-API-Key'];
    
    if (!apiKey) {
      throw new Error('인증 정보가 없습니다. API 키가 필요합니다.');
    }

    // WebSocket URL 구성
    const wsBaseUrl = getWebSocketUrl();
    const wsUrl = new URL(wsBaseUrl);
    wsUrl.searchParams.append('apiKey', apiKey);
    
    const ws = new WebSocket(wsUrl.toString());
    
    // 연결 타임아웃 설정
    const connectionTimeout = setTimeout(() => {
      if (ws.readyState !== WebSocket.OPEN) {
        // Clear WebSocket ref on timeout
        if (wsRef) {
          wsRef.current = null;
        }
        ws.close();
        callbacks.onError(new Error('서버 연결 시간이 초과되었습니다.'));
      }
    }, 10000);

    // 연결 준비 완료 대기
    await new Promise<void>((resolve, reject) => {
      let isConnected = false;

      ws.onopen = () => {
        clearTimeout(connectionTimeout);
        // console.log('WebSocket 연결 성공');
        isConnected = true;
        
        // Store WebSocket instance in provided ref for cancellation support
        if (wsRef) {
          wsRef.current = ws;
        }
        
        if (callbacks.onConnect) {
          callbacks.onConnect();
        }
        
        resolve();
      };
      
      ws.onerror = (error) => {
        clearTimeout(connectionTimeout);
        // Clear WebSocket ref on error
        if (wsRef) {
          wsRef.current = null;
        }
        const errorMessage = isConnected ? 
          '서버와의 통신 중 오류가 발생했습니다.' : 
          '서버에 연결할 수 없습니다. 서버가 실행 중인지 확인해주세요.';
        
        console.error('WebSocket 오류:', error);
        reject(new Error(errorMessage));
      };
    });
    
    ws.onmessage = async (event) => {
      try {
        const response = JSON.parse(event.data);
        // console.log('WebSocket 메시지 수신:', response);
        
        switch (response.type) {
          case 'stream_start':
            // console.log('스트리밍 시작:', response.streamId);
            break;
          case 'stream_content':
            // console.log('스트리밍 컨텐츠 수신:', response.content);
            if (callbacks.onContent) {
              callbacks.onContent(response.content);
            }
            // thinking 내용이 있으면 별도 콜백 호출
            if (response.thinking && callbacks.onThinking) {
              console.log('Stream content의 thinking 수신:', response.thinking);
              callbacks.onThinking(response.thinking);
            }
            break;
          case 'thinking':
            // console.log('Thinking 과정 수신:', response.content);
            if (callbacks.onThinking) {
              callbacks.onThinking(response.content);
            }
            break;
          case 'progress':
            // console.log('진행 상황 수신:', response.message);
            // 진행 상황 콜백이 있으면 호출 (onProgress 콜백 추가 필요)
            if ((callbacks as any).onProgress) {
              // percent 값이 객체인 경우 0으로 처리
              const percent = typeof response.percent === 'number' ? response.percent : 0;
              (callbacks as any).onProgress(response.message, percent, response.stage);
            }
            break;
          case 'stream_end':
            // console.log('스트리밍 완료:', response.streamId);
            if (callbacks.onComplete) {
              // response.response가 객체일 수 있으므로 안전하게 처리
              const responseData = response.response;
              if (responseData && typeof responseData === 'object') {
                // responseData가 객체인 경우 전체 객체를 전달하여 downloadInfo와 webReferences 포함
                callbacks.onComplete(responseData);
              } else {
                // responseData가 문자열이면 그대로 사용
                callbacks.onComplete(responseData || '');
              }
            }
            // Clear WebSocket ref before closing
            if (wsRef) {
              wsRef.current = null;
            }
            ws.close();
            break;
          case 'complete':
            // console.log('완료 응답 수신:', response);
            if (callbacks.onComplete) {
              // response.data 전체 객체를 전달하여 downloadInfo에 접근 가능하도록 함
              callbacks.onComplete(response.data);
            }
            // Clear WebSocket ref before closing
            if (wsRef) {
              wsRef.current = null;
            }
            ws.close();
            break;
          case 'error':
            console.error('WebSocket 에러 응답:', response.data);
            if (callbacks.onError) {
              // response.data가 객체일 수 있으므로 문자열로 변환
              const errorMessage = typeof response.data === 'string' 
                ? response.data 
                : JSON.stringify(response.data);
              callbacks.onError(new Error(errorMessage));
            }
            // Clear WebSocket ref before closing
            if (wsRef) {
              wsRef.current = null;
            }
            ws.close();
            break;
        }
      } catch (error) {
        console.error('메시지 파싱 오류:', error);
        if (callbacks.onError) {
          callbacks.onError(error instanceof Error ? error : new Error('메시지 파싱 오류'));
        }
      }
    };

    ws.onclose = () => {
      console.log('WebSocket 연결 종료');
      // Clear WebSocket ref on close
      if (wsRef) {
        wsRef.current = null;
      }
      if (callbacks.onDisconnect) {
        callbacks.onDisconnect();
      }
    };

    // 스트리밍 요청 전송 (streamChat 방식 사용)
    const streamRequest = {
      type: 'streamChat',
      prompt: prompt,
      options: {
        // 기본 옵션
        sessionId: options.sessionId || 'chat-' + Date.now(),
        userId: userId || options.userId,
        username: options.username,
        
        // 모델 및 프로바이더 설정
        provider: options.provider,
        model: options.model,
        settings: {
          ...options.settings,
          USE_LLM: options.provider || options.settings?.USE_LLM,
          model: options.model || options.settings?.model,
          enableSmartResearch: options.enableSmartResearch
        },
        
        // 채팅 파라미터
        temperature: options.temperature || 0.7,
        max_tokens: options.max_tokens || 8000,
        parameters: {
          temperature: options.temperature || 0.7,
          max_tokens: options.max_tokens || 8000,
          ragSearchScope: options.ragSearchScope || 'personal'
        },
        
        // 기능 옵션
        rag: options.rag === true,
        web: options.web === true,
        enableSmartResearch: options.enableSmartResearch === true,
        ragSearchScope: options.ragSearchScope || 'personal',
        
        // Reasoning 옵션
        reasoningLevel: options.reasoningLevel || 'none',
        
        // 멘션 기능 (백엔드에서 문서 내용 처리)
        mentionedDocuments: options.mentionedDocuments || [],
        
        // 이미지 처리
        image: options.image || null,
        
        // 기타 옵션
        history: options.history || [],
        summary: options.summary || '',
        first: options.first !== false,
        type: 'chat'
      }
    };

    // console.log('스트리밍 요청 전송 (streamChat 방식):', {
    //   type: streamRequest.type,
    //   prompt: prompt.substring(0, 100) + '...',
    //   sessionId: streamRequest.options.sessionId,
    //   hasImage: !!streamRequest.options.image,
    //   ragEnabled: streamRequest.options.rag,
    //   webEnabled: streamRequest.options.web,
    //   smartResearchEnabled: streamRequest.options.enableSmartResearch,
    //   provider: streamRequest.options.provider,
    //   model: streamRequest.options.model
    // });
    
    ws.send(JSON.stringify(streamRequest));
  } catch (error) {
    console.error('채팅 요청 중 오류:', error);
    if (callbacks.onError) {
      callbacks.onError(error instanceof Error ? error : new Error('채팅 요청 중 오류가 발생했습니다'));
    }
  }
};

// 웹소켓 연결 닫기
export const closeWebSocketConnection = (): void => {
  if (websocket) {
    websocket.close();
    websocket = null;
    wsConnected = false;
    // console.log('웹소켓 연결이 닫혔습니다');
  }
};

// 동기식 채팅 요청 함수
export const syncChatRequest = async (
  prompt: string,
  options: any = {}
): Promise<ApiResponse> => {
  try {
    // console.log('채팅 요청 전송:', { prompt, options }); // 디버깅용 로그

    const response = await apiRequest('/chat/sync', {
      method: 'POST',
      body: JSON.stringify({
        prompt,
        options: {
          ...options,
          sessionId: options.sessionId || 'chat-' + Date.now(),
          temperature: 0.7
        }
      })
    });

    // console.log('API 응답 상세:', {
    //   success: response.success,
    //   data: response.data,
    //   error: response.error,
    //   timestamp: response.timestamp
    // }); // 디버깅용 로그
    
    return response;
  } catch (error) {
    console.error('동기식 채팅 요청 중 오류:', error);
    throw error;
  }
}; 