import { getVarVal } from '../../configuration.js';

export class Replicate {
    constructor() {
        this.baseUrl = 'https://api.replicate.com/v1';
    }

    async getApiKey() {
        const apiKey = await getVarVal('REPLICATE_API_KEY');
        if (!apiKey) {
            throw new Error('Replicate API 키가 설정되지 않았습니다.');
        }
        return apiKey;
    }

    async describeImage(base64Image) {
        try {
            const apiKey = await this.getApiKey();
            
            // Replicate API 호출 (LLAVA 모델 사용)
            const response = await fetch(`${this.baseUrl}/predictions`, {
                method: 'POST',
                headers: {
                    'Authorization': `Token ${apiKey}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    version: "2facb4a474a0462c15041b78b1ad70952ea46b5ec6ad29583c0b29dbd4249591",
                    input: {
                        image: `data:image/jpeg;base64,${base64Image}`,
                        prompt: "Describe this image in detail, focusing on both visual elements and potential context or meaning."
                    }
                })
            });

            if (!response.ok) {
                throw new Error(`Replicate API 오류: ${response.statusText}`);
            }

            const prediction = await response.json();
            
            // 예측 결과 대기
            const result = await this.waitForPrediction(prediction.id);
            
            return result.output || '이미지 설명을 생성할 수 없습니다.';
        } catch (error) {
            console.error('이미지 설명 생성 중 오류:', error);
            throw error;
        }
    }

    async waitForPrediction(predictionId) {
        const apiKey = await this.getApiKey();
        const maxAttempts = 30;
        const delayMs = 1000;

        for (let attempt = 0; attempt < maxAttempts; attempt++) {
            const response = await fetch(`${this.baseUrl}/predictions/${predictionId}`, {
                headers: {
                    'Authorization': `Token ${apiKey}`
                }
            });

            if (!response.ok) {
                throw new Error(`예측 상태 확인 실패: ${response.statusText}`);
            }

            const prediction = await response.json();

            if (prediction.status === 'succeeded') {
                return prediction;
            } else if (prediction.status === 'failed') {
                throw new Error('이미지 설명 생성 실패');
            }

            // 다음 시도 전 대기
            await new Promise(resolve => setTimeout(resolve, delayMs));
        }

        throw new Error('이미지 설명 생성 시간 초과');
    }
} 