#!/bin/bash

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# ASCII 아트
echo -e "${GREEN}"
cat << "EOF"
__        _____  ____  _  ______  ____   _    ____ _____ ____  
\ \      / / _ \|  _ \| |/ / ___||  _ \ / \  / ___| ____/ ___| 
 \ \ /\ / / | | | |_) | ' /\___ \| |_) / _ \| |   |  _| \___ \ 
  \ V  V /| |_| |  _ <| . \ ___) |  __/ ___ \ |___| |___ ___) |
   \_/\_/  \___/|_| \_\_|\_\____/|_| /_/   \_\____|_____|____/
EOF
echo -e "${NC}"

# 로그 파일 관리
LOG_FILE="web.log"
MAX_LOG_SIZE=10485760  # 10MB
BACKUP_LOGS=3

# 로그 파일 로테이션
rotate_logs() {
    if [ -f "$LOG_FILE" ] && [ $(stat -c%s "$LOG_FILE") -gt $MAX_LOG_SIZE ]; then
        echo -e "${YELLOW}로그 파일이 너무 큽니다. 로테이션을 진행합니다...${NC}"
        
        # 기존 백업 로그들을 순환
        for i in $(seq $((BACKUP_LOGS-1)) -1 1); do
            if [ -f "${LOG_FILE}.${i}" ]; then
                mv "${LOG_FILE}.${i}" "${LOG_FILE}.$((i+1))"
            fi
        done
        
        # 현재 로그를 백업으로 이동
        mv "$LOG_FILE" "${LOG_FILE}.1"
        echo -e "${GREEN}로그 로테이션 완료${NC}"
    fi
}

# Node.js와 npm 설치 확인
if ! command -v node &> /dev/null; then
    echo -e "${RED}Node.js is not installed. Please install Node.js first.${NC}"
    exit 1
fi

if ! command -v npm &> /dev/null; then
    echo -e "${RED}npm is not installed. Please install npm first.${NC}"
    exit 1
fi

echo -e "${GREEN}Node.js version: $(node --version)${NC}"
echo -e "${GREEN}npm version: $(npm --version)${NC}"

# 가상환경 활성화
if [ ! -d "$HOME/.airun_venv" ]; then
    echo -e "${RED}Virtual environment is not installed. Please run install.sh first.${NC}"
    exit 1
fi

source "$HOME/.airun_venv/bin/activate"

# 프로젝트 루트 디렉토리로 이동
cd "$(dirname "$0")/workspaces" || exit 1

# 웹 클라이언트 PID 파일
WEB_PID_FILE="web.pid"

# MCP 서버 빌드 상태 확인
check_mcp_servers() {
    echo -e "${YELLOW}MCP 서버 빌드 상태를 확인합니다...${NC}"
    
    local mcp_servers=(
        "web/mcp_tools/hamonize/dist/index.js"
        "web/mcp_tools/github/dist/index.js"
        "web/mcp_tools/mcp-postgres-server/build/index.js"
        "web/mcp_tools/telegram/build/index.js"
        "web/mcp_tools/filesystem/dist/index.js"
    )
    
    local missing_servers=()
    
    for server in "${mcp_servers[@]}"; do
        if [ ! -f "$server" ]; then
            missing_servers+=("$server")
        fi
    done
    
    if [ ${#missing_servers[@]} -gt 0 ]; then
        echo -e "${RED}❌ 다음 MCP 서버들이 빌드되지 않았습니다:${NC}"
        for server in "${missing_servers[@]}"; do
            echo -e "${RED}  - $server${NC}"
        done
        echo -e "${YELLOW}⚠️  MCP 도구 기능이 제한될 수 있습니다.${NC}"
        echo -e "${YELLOW}    빌드를 위해 'node build.js'를 실행하세요.${NC}"
        return 1
    else
        echo -e "${GREEN}✅ 모든 MCP 서버가 빌드되어 있습니다.${NC}"
        return 0
    fi
}

# 포트를 사용하는 프로세스 종료 함수 (lsof + ss 모두 시도)
kill_process_on_port() {
    local port=$1

    # 1. lsof로 시도
    local pids=$(lsof -ti:"$port" 2>/dev/null)
    if [ -n "$pids" ]; then
        echo -e "${YELLOW}Port $port is in use (lsof). Attempting to terminate...${NC}"
        for pid in $pids; do
            echo -e "${YELLOW}→ Sending SIGTERM to PID $pid${NC}"
            kill -15 "$pid" 2>/dev/null
        done
        sleep 3
        for pid in $pids; do
            if kill -0 "$pid" 2>/dev/null; then
                echo -e "${YELLOW}→ PID $pid still running. Force killing...${NC}"
                kill -9 "$pid" 2>/dev/null
            fi
        done
        if [ -n "$(lsof -ti:$port 2>/dev/null)" ]; then
            echo -e "${RED}❌ Failed to terminate all processes on port $port (lsof). Trying ss...${NC}"
        else
            echo -e "${GREEN}✅ Successfully terminated all processes on port $port.${NC}"
            return 0
        fi
    fi

    # 2. lsof 실패시 ss로 시도
    # ss -tnlp는 "pid=NNN" 패턴을 포함
    local ss_pid=$(ss -tnlp 2>/dev/null | awk -v p=":$port" '$4 ~ p {match($0, /pid=[0-9]+/); if (RSTART > 0) { print substr($0, RSTART+4, RLENGTH-4) }}' | head -n1)
    if [ -n "$ss_pid" ]; then
        echo -e "${YELLOW}Port $port is in use (ss). Attempting to terminate PID $ss_pid...${NC}"
        kill -15 "$ss_pid" 2>/dev/null
        sleep 3
        if kill -0 "$ss_pid" 2>/dev/null; then
            echo -e "${YELLOW}→ PID $ss_pid still running. Force killing...${NC}"
            kill -9 "$ss_pid" 2>/dev/null
        fi
        # 다시 확인
        local still_running=$(ss -tnlp 2>/dev/null | awk -v p=":$port" '$4 ~ p {print 1}')
        if [ -n "$still_running" ]; then
            echo -e "${RED}❌ Failed to terminate process on port $port (ss). Please check manually.${NC}"
            return 1
        else
            echo -e "${GREEN}✅ Successfully terminated process on port $port (ss).${NC}"
            return 0
        fi
    fi

    # 3. 완전히 비었으면
    echo -e "${GREEN}✅ Port $port is free.${NC}"
    return 0
}

# 서버 종료 함수
stop_server() {
    # 웹 서버 종료
    if [ -f "$WEB_PID_FILE" ]; then
        PID=$(cat "$WEB_PID_FILE")
        if kill -0 "$PID" 2>/dev/null; then
            echo -e "${YELLOW}AI.RUN Web Application is shutting down... (PID: $PID)${NC}"
            kill -15 "$PID" 2>/dev/null
            
            # 3초 대기 후 여전히 실행 중이면 강제 종료
            sleep 3
            if kill -0 "$PID" 2>/dev/null; then
                echo -e "${YELLOW}Process $PID is still running. Force killing...${NC}"
                kill -9 "$PID" 2>/dev/null
            fi
            
            rm "$WEB_PID_FILE"
            echo -e "${GREEN}AI.RUN Web Application has been terminated.${NC}"
        else
            echo -e "${RED}AI.RUN Web Application is already terminated.${NC}"
            rm "$WEB_PID_FILE"
        fi
    else
        echo -e "${RED}AI.RUN Web Application is not running.${NC}"
    fi
    
    # 포트 5700 사용 중인 프로세스 종료
    kill_process_on_port 5700
}

# 의존성 설치 및 업데이트 함수
install_dependencies() {
    local dir=$1
    local name=$2
    
    echo -e "${YELLOW}Installing $name dependencies...${NC}"
    
    # package-lock.json이 있으면 npm ci 사용 (더 빠르고 안정적)
    if [ -f "$dir/package-lock.json" ]; then
        npm ci --silent
    else
        npm install --silent
    fi
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to install $name dependencies.${NC}"
        return 1
    fi
    
    echo -e "${GREEN}$name dependencies installed successfully.${NC}"
    return 0
}

# 서버 시작 함수
start_server() {
    # 포트 5700이 사용 중인지 확인하고 종료
    if ! kill_process_on_port 5700; then
        echo -e "${RED}Cannot start server: Port 5700 is still in use.${NC}"
        exit 1
    fi
    
    # NODE_ENV가 설정되지 않았으면 기본값을 production으로 설정
    if [ -z "$NODE_ENV" ]; then
        export NODE_ENV=production
        echo -e "${GREEN}NODE_ENV not set. Defaulting to production mode.${NC}"
    fi
    
    echo -e "${YELLOW}AI.RUN Web Application is starting in $NODE_ENV mode...${NC}"
    
    # 로그 파일 로테이션
    rotate_logs
    
    # MCP 서버 빌드 상태 확인
    check_mcp_servers
    
    # shared 폴더 의존성 확인 및 설치
    if [ -d "shared" ]; then
        cd shared
        if [ ! -d "node_modules" ] || [ "package.json" -nt "node_modules" ]; then
            if ! install_dependencies "$(pwd)" "shared"; then
                exit 1
            fi
        fi
        cd ..
    fi
    
    # 웹 클라이언트 디렉토리로 이동
    cd web || exit 1
    
    # Node.js 의존성 확인 및 설치
    if [ ! -d "node_modules" ] || [ "package.json" -nt "node_modules" ]; then
        if ! install_dependencies "$(pwd)" "web"; then
            exit 1
        fi
    fi
    
    # 환경별 실행 로직
    if [ "$NODE_ENV" = "development" ]; then
        # 개발 환경: 빌드 시도 후 실패하면 개발 모드로 실행
        echo -e "${YELLOW}Development mode: Attempting to build web application...${NC}"
        npm run build
        if [ $? -ne 0 ]; then
            echo -e "${YELLOW}⚠️  Build failed. Starting in development mode...${NC}"
            echo -e "${YELLOW}    Note: This will enable HMR and hot reload features.${NC}"
            nohup npm run dev > "../$LOG_FILE" 2>&1 &
        else
            echo -e "${GREEN}✅ Build successful. Starting in development mode...${NC}"
            nohup npm run dev > "../$LOG_FILE" 2>&1 &
        fi
    else
        # 운영 환경: 빌드 필수
        echo -e "${YELLOW}Production mode: Building web application...${NC}"
        npm run build
        if [ $? -ne 0 ]; then
            echo -e "${RED}❌ Build failed in production mode. Cannot start server.${NC}"
            exit 1
        fi
        echo -e "${GREEN}✅ Build successful. Starting in production mode...${NC}"
        nohup npm run start > "../$LOG_FILE" 2>&1 &
    fi
    
    echo $! > "../$WEB_PID_FILE"
    cd ..
    
    # 서버 시작 확인
    sleep 5
    if [ -f "$WEB_PID_FILE" ]; then
        PID=$(cat "$WEB_PID_FILE")
        if kill -0 "$PID" 2>/dev/null; then
            echo -e "${GREEN}✅ AI.RUN Web Application started successfully. (PID: $PID)${NC}"
            echo -e "${YELLOW}🌐 Web UI: http://localhost:5700${NC}"
            echo -e "${YELLOW}📋 Web log: tail -f workspaces/$LOG_FILE${NC}"
        else
            echo -e "${RED}❌ Failed to start AI.RUN Web Application. Check logs for details.${NC}"
            echo -e "${YELLOW}📋 Log file: workspaces/$LOG_FILE${NC}"
            exit 1
        fi
    fi
}

# 서버 상태 확인 함수
check_status() {
    # 웹 서버 상태
    if [ -f "$WEB_PID_FILE" ]; then
        PID=$(cat "$WEB_PID_FILE")
        if kill -0 "$PID" 2>/dev/null; then
            echo -e "${GREEN}✅ Web Application is running. (PID: $PID)${NC}"
            echo -e "${YELLOW}🌐 Web UI: http://localhost:5700${NC}"
            
            # 메모리 사용량 표시
            local mem_usage=$(ps -o pid,rss,vsz,comm --no-headers -p "$PID" 2>/dev/null)
            if [ -n "$mem_usage" ]; then
                echo -e "${YELLOW}📊 Memory usage: ${mem_usage}${NC}"
            fi
            
            # 로그 파일 크기 표시
            if [ -f "$LOG_FILE" ]; then
                local log_size=$(du -h "$LOG_FILE" | cut -f1)
                echo -e "${YELLOW}📋 Log file size: ${log_size}${NC}"
            fi
        else
            echo -e "${RED}❌ Web Application is terminated.${NC}"
            rm "$WEB_PID_FILE"
        fi
    else
        echo -e "${RED}❌ Web Application is not running.${NC}"
    fi
    
    # MCP 서버 상태 확인
    check_mcp_servers
}

# 명령어 처리
case "$1" in
    start)
        start_server
        ;;
    stop)
        stop_server
        ;;
    status)
        check_status
        ;;
    restart)
        stop_server
        sleep 2
        start_server
        ;;
    logs)
        if [ -f "$LOG_FILE" ]; then
            tail -f "$LOG_FILE"
        else
            echo -e "${RED}Log file not found: $LOG_FILE${NC}"
        fi
        ;;
    *)
        echo "Usage: $0 {start|stop|status|restart|logs}"
        echo ""
        echo "Commands:"
        echo "  start   - Start the web application (default: production mode)"
        echo "  stop    - Stop the web application"
        echo "  status  - Check application status"
        echo "  restart - Restart the web application"
        echo "  logs    - Show real-time logs"
        echo ""
        echo "Environment Variables:"
        echo "  NODE_ENV=production  - Run in production mode (default)"
        echo "  NODE_ENV=development - Run in development mode (enables HMR)"
        echo ""
        echo "Examples:"
        echo "  $0 start                    # Start in production mode"
        echo "  NODE_ENV=development $0 start # Start in development mode"
        exit 1
        ;;
esac

# 가상환경 비활성화
deactivate

exit 0 