#! /usr/bin/env python

from PIL import Image
import os
import sys

def crop_image(input_image_path, output_folder, left, top, right, bottom):
    """
    이미지의 특정 영역을 잘라서 다른 폴더에 저장합니다.
    
    Args:
        input_image_path (str): 입력 이미지 경로
        output_folder (str): 출력 폴더 경로
        left (int): 자를 영역의 왼쪽 x 좌표
        top (int): 자를 영역의 위쪽 y 좌표
        right (int): 자를 영역의 오른쪽 x 좌표
        bottom (int): 자를 영역의 아래쪽 y 좌표
    """
    try:
        # 출력 폴더가 없으면 생성
        if not os.path.exists(output_folder):
            os.makedirs(output_folder)
            print(f"폴더 생성됨: {output_folder}")
        
        # 이미지 열기
        with Image.open(input_image_path) as img:
            # 이미지 자르기
            cropped_img = img.crop((left, top, right, bottom))
            
            # 출력 파일 경로 생성
            filename = os.path.basename(input_image_path)
            name, ext = os.path.splitext(filename)
            output_path = os.path.join(output_folder, f"{name}_cropped{ext}")
            
            # 잘린 이미지 저장
            cropped_img.save(output_path)
            print(f"이미지가 성공적으로 잘려서 저장되었습니다: {output_path}")
            return True
    
    except Exception as e:
        print(f"오류 발생 ({os.path.basename(input_image_path)}): {e}")
        return False

def process_folder(input_folder, left, top, right, bottom):
    """
    폴더 내의 모든 이미지를 처리합니다.
    
    Args:
        input_folder (str): 입력 폴더 경로
        left (int): 자를 영역의 왼쪽 x 좌표
        top (int): 자를 영역의 위쪽 y 좌표
        right (int): 자를 영역의 오른쪽 x 좌표
        bottom (int): 자를 영역의 아래쪽 y 좌표
    """
    # 결과 폴더 경로 생성 (입력 폴더 내의 'result' 하위 폴더)
    output_folder = os.path.join(input_folder, "result")
    
    # 이미지 확장자 목록
    image_extensions = ['.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp']
    
    # 처리된 이미지 수와 성공 수 카운트
    total_images = 0
    successful_images = 0
    
    print(f"폴더 처리 중: {input_folder}")
    
    # 폴더 내의 모든 파일 처리
    for filename in os.listdir(input_folder):
        file_path = os.path.join(input_folder, filename)
        
        # 파일이고 이미지 확장자를 가진 경우에만 처리
        if os.path.isfile(file_path) and any(filename.lower().endswith(ext) for ext in image_extensions):
            total_images += 1
            if crop_image(file_path, output_folder, left, top, right, bottom):
                successful_images += 1
    
    # 결과 출력
    print(f"\n처리 완료: 총 {total_images}개 이미지 중 {successful_images}개 성공")
    if total_images == 0:
        print(f"경고: '{input_folder}' 폴더에 이미지 파일이 없습니다.")

def print_usage():
    """사용법을 출력합니다."""
    print("사용법:")
    print(f"  {sys.argv[0]} [폴더 경로]")
    print("  - 폴더 경로를 입력하면 해당 폴더의 모든 이미지를 처리합니다.")
    print("  - 인수 없이 실행하면 대화형 모드로 실행됩니다.")

if __name__ == "__main__":
    # 자르고 싶은 영역 좌표 (left, top, right, bottom)
    left, top, right, bottom = 97, 405, 1697, 1294
    
    # 명령줄 인수 확인
    if len(sys.argv) > 1:
        # 첫 번째 인수를 폴더 경로로 사용
        input_folder = sys.argv[1]
        
        if os.path.isdir(input_folder):
            # 폴더가 존재하면 바로 폴더 처리 모드로 실행
            process_folder(input_folder, left, top, right, bottom)
        else:
            print(f"오류: '{input_folder}'는 유효한 폴더가 아닙니다.")
            print_usage()
    else:
        # 인수가 없으면 대화형 모드로 실행
        print("이미지 자르기 모드를 선택하세요:")
        print("1. 단일 이미지 처리")
        print("2. 폴더 내 모든 이미지 처리")
        
        mode = input("모드 선택 (1 또는 2): ")
        
        if mode == "1":
            # 단일 이미지 처리
            input_image_path = input("입력 이미지 경로를 입력하세요: ")
            output_folder = input("출력 폴더 경로를 입력하세요: ")
            crop_image(input_image_path, output_folder, left, top, right, bottom)
        
        elif mode == "2":
            # 폴더 내 모든 이미지 처리
            input_folder = input("처리할 이미지가 있는 폴더 경로를 입력하세요: ")
            if os.path.isdir(input_folder):
                process_folder(input_folder, left, top, right, bottom)
            else:
                print(f"오류: '{input_folder}'는 유효한 폴더가 아닙니다.") 