import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs/promises';
import { existsSync } from 'fs';
import chalk from 'chalk';
import os from 'os';
import readline from 'readline';
import { logger } from '../../utils/logger.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 프로젝트 루트 디렉토리 찾기 (services/report 상위 디렉토리)
const projectRoot = path.resolve(__dirname, '../..');

export class PythonBridge {
    constructor(options = {}) {
        this.venvPath = path.join(os.homedir(), '.airun_venv');
        this.pythonPath = process.platform === 'win32' 
            ? path.join(this.venvPath, 'Scripts', 'python.exe')
            : path.join(this.venvPath, 'bin', 'python');
        
        // 현재 실행 중인 디렉토리 대신 스크립트 파일의 실제 위치를 기준으로 경로 설정
        // this.scriptPath = path.join(currentDir, 'services', 'report');
        this.scriptPath = __dirname;
        
        // 프로젝트 루트 디렉토리 설정 (services/report의 상위 디렉토리)
        this.projectRoot = projectRoot;
        
        this.debug = false;  // 디버그 모드 비활성화 (로그 최소화)
        this.initialized = false;
    }

    log(message) {
        if (this.debug) {
            console.log(chalk.cyan(`[PythonBridge] ${message}`));
        }
    }

    async initialize() {
        try {
            // this.log('초기화 시작...');
            
            // 프로젝트 루트 디렉토리 확인
            // this.log(`프로젝트 루트 디렉토리: ${this.projectRoot}`);
            
            // venv 확인
            // this.log(`가상환경 확인: ${this.venvPath}`);
            if (!await fs.access(this.venvPath).then(() => true).catch(() => false)) {
                throw new Error('가상환경(.airun_venv)을 찾을 수 없습니다. install.sh 스크립트를 실행하여 가상환경을 생성해 주세요.');
            }
            
            // Python 실행 파일 확인
            // this.log(`Python 실행 파일 확인: ${this.pythonPath}`);
            if (!await fs.access(this.pythonPath).then(() => true).catch(() => false)) {
                throw new Error('가상환경의 Python을 찾을 수 없습니다. install.sh 스크립트를 실행하여 가상환경을 다시 생성해 주세요.');
            }
            
            // scripts 디렉토리 존재 확인
            // this.log(`스크립트 디렉토리 확인: ${this.scriptPath}`);
            if (!await fs.access(this.scriptPath).then(() => true).catch(() => false)) {
                throw new Error(`스크립트 디렉토리를 찾을 수 없습니다: ${this.scriptPath}`);
            }
            
            // generate_bizplan.py 파일 존재 확인
            const bizplanScriptPath = path.join(this.scriptPath, 'generate_bizplan.py');
            // this.log(`사업계획서 스크립트 확인: ${bizplanScriptPath}`);
            if (!await fs.access(bizplanScriptPath).then(() => true).catch(() => false)) {
                throw new Error(`사업계획서 스크립트를 찾을 수 없습니다: ${bizplanScriptPath}`);
            }
            
            // Python 환경 확인
            // this.log('Python 환경 확인 중...');
            await this.checkPythonEnvironment();
            
            // this.log('초기화 완료');
            return true;
        } catch (error) {
            // console.error(chalk.red('Python bridge initialization failed:'), error);
            throw error;
        }
    }

    async executePythonScript(scriptName, args = []) {
        return new Promise((resolve, reject) => {
            const scriptPath = path.join(this.scriptPath, scriptName);
            
            // 스크립트 파일 존재 여부 로깅
            // this.log(`스크립트 실행: ${scriptPath}`);
            if (!existsSync(scriptPath)) {
                this.log(`경고: 스크립트 파일이 존재하지 않습니다: ${scriptPath}`);
                reject(new Error(`Script file not found: ${scriptPath}`));
                return;
            }

            // Python 프로세스 환경변수 설정
            const env = {
                ...process.env,
                PYTHONUNBUFFERED: '1',  // Python 출력 버퍼링 비활성화
                PYTHONIOENCODING: 'utf-8',  // 출력 인코딩 설정
                FORCE_COLOR: '1',  // 컬러 출력 강제
                NODE_ENV: process.env.NODE_ENV || 'development'
            };

            // 프로세스 생성
            const pythonProcess = spawn(this.pythonPath, [scriptPath, ...args], { env });

            let output = '';
            let error = '';

            // stdout을 라인 단위로 처리
            const stdout = readline.createInterface({
                input: pythonProcess.stdout,
                terminal: false
            });

            let currentSection = null;  // 현재 처리 중인 섹션
            let buffer = '';  // 여러 줄 데이터를 위한 버퍼

            stdout.on('line', (line) => {
                output += line + '\n';

                // JSON 형식의 라인은 출력하지 않음
                if (line.trim().startsWith('{') && line.trim().endsWith('}')) {
                    try {
                        JSON.parse(line);
                        return; // JSON 라인은 출력하지 않음
                    } catch (e) {
                        // JSON이 아닌 경우 계속 진행
                    }
                }

                // 필터링할 DEBUG 메시지 패턴들
                const debugFilterPatterns = [
                    '현재 디렉토리:',
                    '스크립트 디렉토리:',
                    '프로젝트 루트:',
                    'Python 경로:',
                    '[DEBUG] Python 버전:',
                    'sys.path:',
                    '개발 모드: Python 소스 파일 사용',
                    'Python 소스 파일에서 모듈 로드 성공',
                    '최종 사용 모듈:',
                    '개발 모드: 수정된 Python 소스 파일을 사용합니다',
                    'LangChainDeprecationWarning',
                    'from langchain.embeddings import HuggingFaceEmbeddings',
                    'from langchain_community.embeddings import HuggingFaceEmbeddings',
                    'https://python.langchain.com/docs/versions/v0_2/',
                    'from .rag_process import DocumentProcessor'
                ];

                // 필터링 패턴에 해당하는 라인은 출력하지 않음
                const shouldFilter = debugFilterPatterns.some(pattern => line.includes(pattern));
                if (shouldFilter) {
                    return;
                }

                // 진행 상태 메시지만 출력 (🚀, 📊, ✅ 등의 이모지가 포함된 중요한 메시지)
                if (line.trim() && (
                    line.includes('🚀') || line.includes('📊') || 
                    line.includes('✅') || line.includes('📝') || 
                    line.includes('📑') || line.includes('💾') ||
                    line.includes('작성 중') || line.includes('생성 중') ||
                    line.includes('분석 중') || line.includes('처리 중') ||
                    line.includes('완료') || line.includes('시작합니다')
                )) {
                    if (typeof logger !== 'undefined') {
                        logger.debug(line);
                    } else {
                        console.log(line);
                    }
                }
            });

            // stderr도 라인 단위로 처리
            const stderr = readline.createInterface({
                input: pythonProcess.stderr,
                terminal: false
            });

            stderr.on('line', (line) => {
                error += line + '\n';
                
                // 필터링할 DEBUG 메시지 패턴들 (stdout과 동일)
                const debugFilterPatterns = [
                    '현재 디렉토리:',
                    '스크립트 디렉토리:',
                    '프로젝트 루트:',
                    'Python 경로:',
                    '[DEBUG] Python 버전:',
                    'sys.path:',
                    '개발 모드: Python 소스 파일 사용',
                    'Python 소스 파일에서 모듈 로드 성공',
                    '최종 사용 모듈:',
                    '개발 모드: 수정된 Python 소스 파일을 사용합니다',
                    'LangChainDeprecationWarning',
                    'from langchain.embeddings import HuggingFaceEmbeddings',
                    'from langchain_community.embeddings import HuggingFaceEmbeddings',
                    'https://python.langchain.com/docs/versions/v0_2/',
                    'from .rag_process import DocumentProcessor',
                    '=== OpenAI API 요청 ===',
                    '프롬프트:',
                    '시스템 메시지:',
                    '=== 파싱된 JSON ===',
                    '=== AI 원본 응답 ===',
                    'API 파라미터:',
                    ' {',
                    '다이어그램 데이터:',
                    '캐시 파일 내용: [',
                    '검증된 결과: [',
                    '=== AI 응답 내용 ===',
                    '==================',
                    '기능 목록: [',
                    '=== 검색 결과 프롬프트 ===',
                    '## 관련 검색 결과: [',
                    '## 관련 검색 결과:',
                    '========================='
                ];

                // JSON 형식의 라인은 출력하지 않음
                if (line.trim().startsWith('{') && line.trim().endsWith('}')) {
                    try {
                        JSON.parse(line);
                        return; // JSON 라인은 출력하지 않음
                    } catch (e) {
                        // JSON이 아닌 경우 계속 진행
                    }
                }

                // 필터링 패턴에 해당하는 라인은 출력하지 않음
                const shouldFilter = debugFilterPatterns.some(pattern => line.includes(pattern));
                if (shouldFilter) {
                    return;
                }

                // 중요한 오류나 경고 메시지만 출력
                if (line.trim() && (
                    line.includes('[ERROR]') || line.includes('[WARNING]') ||
                    line.includes('❌') || line.includes('⚠️') ||
                    line.includes('Traceback') || line.includes('Exception') ||
                    line.includes('Error:') || line.includes('Failed')
                )) {
                    logger.debug(line);
                }                    
            });

            pythonProcess.on('close', (code) => {
                stdout.close();
                stderr.close();
                
                if (code !== 0) {
                    // 오류 메시지 개선
                    let errorMsg = '알 수 없는 오류';
                    
                    // 오류 메시지에서 중요한 부분 추출
                    if (error.includes('[ERROR]')) {
                        // 모든 ERROR 로그 라인 추출
                        const errorLines = error.split('\n')
                            .filter(line => line.includes('[ERROR]'))
                            .map(line => {
                                // [ERROR] 접두사와 JobID 제거
                                return line.replace(/\[\w+: [^\]]+\] \[ERROR\] /g, '');
                            });
                        
                        if (errorLines.length > 0) {
                            // 마지막 오류 메시지 사용 (일반적으로 가장 구체적인 오류)
                            errorMsg = errorLines[errorLines.length - 1];
                        }
                    }
                    
                    // 특정 패턴의 오류 메시지 추가 검색
                    if (errorMsg === '알 수 없는 오류') {
                        // 'Traceback' 또는 'Error:' 포함된 라인 찾기
                        const tracebackLines = error.split('\n')
                            .filter(line => line.includes('Traceback') || 
                                            line.includes('Error:') || 
                                            line.includes('Exception:') ||
                                            line.includes('referenced before assignment'));
                        
                        if (tracebackLines.length > 0) {
                            // 가장 구체적인 오류 메시지 선택
                            for (const line of tracebackLines) {
                                if (line.includes('Error:') || 
                                    line.includes('Exception:') || 
                                    line.includes('referenced before assignment')) {
                                    errorMsg = line.trim();
                                    break;
                                }
                            }
                        }
                    }
                    
                    // 여전히 오류 메시지를 찾지 못한 경우 마지막 몇 줄 사용
                    if (errorMsg === '알 수 없는 오류') {
                        const lastLines = error.trim().split('\n').slice(-5);
                        if (lastLines.length > 0) {
                            errorMsg = lastLines.join(' | ');
                        }
                    }
                    
                    const errorObject = {
                        code,
                        error: errorMsg,
                        fullError: error.trim(),
                        output: output.trim()
                    };
                    
                    console.error(chalk.red('Python 오류 상세 정보:'), errorMsg);
                    reject(new Error(errorMsg));
                } else {
                    try {
                        // 마지막 줄에서 JSON 찾기
                        const lines = output.trim().split('\n');
                        const lastLine = lines[lines.length - 1];
                        // JSON 파싱만 하고 출력하지 않음
                        let result;
                        try {
                            result = JSON.parse(lastLine);
                        } catch (e) {
                            // JSON 파싱 실패 시 전체 출력에서 JSON 찾기
                            const jsonMatch = output.match(/\{[\s\S]*\}/);
                            if (jsonMatch) {
                                result = JSON.parse(jsonMatch[0]);
                            } else {
                                throw e;
                            }
                        }
                        
                        if (!result.success) {
                            // 오류 메시지 개선
                            const errorMsg = result.error || '알 수 없는 오류';
                            reject(new Error(errorMsg));
                            return;
                        }
                        
                        resolve(result);
                    } catch (e) {
                        // JSON 파싱 오류 개선
                        reject(new Error(`유효하지 않은 응답: ${e.message}`));
                    }
                }
            });
        });
    }

    async checkPythonEnvironment() {
        try {
            // 이미 초기화된 경우 추가 체크 건너뛰기
            if (this.initialized) {
                return true;
            }

            const result = await this.executePythonScript('generate_bizplan.py', ['--check-env']);
            
            if (!result.success) {
                throw new Error(result.error || 'Python environment check failed');
            }
            
            // pyhwp 패키지가 설치되어 있는지 확인
            const packages = result.packages || {};
            if (packages['pyhwp'] === 'not installed') {
                throw new Error('pyhwp package is not installed. Please run: pip install pyhwp');
            }
            
            // 초기화 시에만 버전 정보를 한 번만 출력
            if (!this.initialized) {
                // console.log(`[DEBUG] Python 버전: ${result.python_version}`);
                this.initialized = true;
            }
            
            return true;
        } catch (error) {
            throw new Error(`Python environment is not properly configured: ${error.message}`);
        }
    }
} 