/**
 * @swagger
 * tags:
 *   - name: Report
 *     description: 보고서 생성 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ReportRequest:
 *       type: object
 *       properties:
 *         prompt:
 *           type: string
 *           description: 보고서 생성을 위한 프롬프트
 *         filepath:
 *           type: string
 *           description: 보고서 내용이 담긴 파일 경로
 *         format:
 *           type: string
 *           description: 보고서 출력 형식
 *           enum: [pdf, docx, html]
 *           default: pdf
 *         template:
 *           type: string
 *           description: 사용할 보고서 템플릿
 *           default: simple
 *     ReportResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             file_path:
 *               type: string
 *               description: 생성된 보고서 파일 경로
 *             format:
 *               type: string
 *               description: 보고서 형식
 *             template:
 *               type: string
 *               description: 사용된 템플릿
 *             message:
 *               type: string
 *             timestamp:
 *               type: string
 *               format: date-time
 *     TemplateInfo:
 *       type: object
 *       properties:
 *         name:
 *           type: string
 *           description: 템플릿 이름
 *         type:
 *           type: string
 *           description: 템플릿 유형 (사업계획서, 제안서, 연구개발계획서 등)
 *         structure:
 *           type: object
 *           properties:
 *             topLevelSections:
 *               type: array
 *               items:
 *                 type: string
 *               description: 최상위 섹션 목록
 *             totalPrompts:
 *               type: integer
 *               description: 전체 프롬프트 수
 *             sectionTypes:
 *               type: array
 *               items:
 *                 type: string
 *               description: 섹션 타입 목록
 *         features:
 *           type: object
 *           properties:
 *             hasCharts:
 *               type: boolean
 *               description: 차트 포함 여부
 *             hasTables:
 *               type: boolean
 *               description: 표 포함 여부
 *             hasFlows:
 *               type: boolean
 *               description: 플로우차트 포함 여부
 *             hasRag:
 *               type: boolean
 *               description: RAG 기능 사용 여부
 *             hasWeb:
 *               type: boolean
 *               description: 웹 검색 기능 사용 여부
 *             hasCompetitorAnalysis:
 *               type: boolean
 *               description: 경쟁사 분석 포함 여부
 *             hasDiagrams:
 *               type: boolean
 *               description: 다이어그램 포함 여부
 *         lastModified:
 *           type: string
 *           format: date-time
 *           description: 마지막 수정 일시
 */

/**
 * @swagger
 * /api/v1/report:
 *   post:
 *     summary: 보고서 생성
 *     description: 프롬프트나 파일을 기반으로 보고서를 생성합니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ReportRequest'
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ReportResponse'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/report/status/{jobId}:
 *   get:
 *     summary: 보고서 생성 작업 상태 확인
 *     description: 보고서 생성 요청의 처리 상태를 확인합니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: jobId
 *         required: true
 *         schema:
 *           type: string
 *         description: 작업 ID
 *     responses:
 *       200:
 *         description: 작업 상태 정보
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     jobId:
 *                       type: string
 *                     state:
 *                       type: string
 *                       enum: [completed, failed, active, waiting, delayed]
 *                     file_path:
 *                       type: string
 *                     message:
 *                       type: string
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 */

/**
 * @swagger
 * /api/v1/report/templates:
 *   get:
 *     summary: 사용 가능한 템플릿 목록 조회
 *     description: 모든 사용 가능한 보고서 템플릿의 목록과 상세 정보를 반환합니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 템플릿 목록 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     templates:
 *                       type: array
 *                       items:
 *                         $ref: '#/components/schemas/TemplateInfo'
 *                     default:
 *                       type: string
 *                       description: 기본 템플릿 이름
 *                     count:
 *                       type: integer
 *                       description: 전체 템플릿 수
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/report/templates/{templateName}:
 *   get:
 *     summary: 특정 템플릿 상세 정보 조회
 *     description: 지정된 템플릿의 상세 정보를 반환합니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: templateName
 *         required: true
 *         schema:
 *           type: string
 *         description: 조회할 템플릿 이름 (bizplan, simple, proposal 등)
 *     responses:
 *       200:
 *         description: 템플릿 상세 정보 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   $ref: '#/components/schemas/TemplateInfo'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 템플릿을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/template/upload:
 *   post:
 *     summary: 보고서 템플릿 업로드
 *     description: JSON 파일을 업로드하여 새로운 보고서 템플릿을 추가합니다. 템플릿 이름은 파일명에서 _templates.json 제거하여 자동으로 추출됩니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             properties:
 *               file:
 *                 type: string
 *                 format: binary
 *                 description: 업로드할 템플릿 JSON 파일
 *     responses:
 *       200:
 *         description: 템플릿 업로드 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     success:
 *                       type: boolean
 *                     template:
 *                       $ref: '#/components/schemas/TemplateInfo'
 *                     message:
 *                       type: string
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: 잘못된 요청 또는 파일 처리 오류
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     code:
 *                       type: string
 *                       example: "NO_FILE_RECEIVED"
 *                     message:
 *                       type: string
 *                       example: "파일이 없습니다. 파일을 제공해주세요."
 */

/**
 * @swagger
 * /api/v1/template/{templateName}:
 *   delete:
 *     summary: 보고서 템플릿 삭제
 *     description: 지정된 이름의 템플릿을 삭제합니다. 기본 제공되는 템플릿은 삭제할 수 없습니다.
 *     tags: [Report]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: templateName
 *         required: true
 *         schema:
 *           type: string
 *         description: 삭제할 템플릿 이름
 *     responses:
 *       200:
 *         description: 템플릿 삭제 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     success:
 *                       type: boolean
 *                       example: true
 *                     deletedTemplate:
 *                       type: object
 *                       properties:
 *                         name:
 *                           type: string
 *                           example: "custom_template"
 *                         message:
 *                           type: string
 *                           example: "템플릿 'custom_template'이 성공적으로 삭제되었습니다."
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: 템플릿 삭제 실패
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     code:
 *                       type: string
 *                       example: "TEMPLATE_DELETE_ERROR"
 *                     message:
 *                       type: string
 *                       example: "기본 템플릿은 삭제할 수 없습니다."
 */

export const reportSchemas = {
    ReportRequest: {
        type: 'object',
        properties: {
            prompt: {
                type: 'string',
                description: '보고서 생성을 위한 프롬프트'
            },
            filepath: {
                type: 'string',
                description: '보고서 내용이 담긴 파일 경로'
            },
            format: {
                type: 'string',
                description: '보고서 출력 형식',
                enum: ['pdf', 'docx', 'html'],
                default: 'pdf'
            },
            template: {
                type: 'string',
                description: '사용할 보고서 템플릿 (bizplan, proposal, rnd, marketing, thesis, simple 등)',
                default: 'simple',
                example: 'bizplan'
            }
        }
    }
};